﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.Generic.Preview.MCS.Ctrl
{
    using System;
    using System.Collections;
    using System.Diagnostics;
    using System.IO;
    using System.Net;
    using System.Net.Sockets;
    using System.Threading;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Ctrl;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Viewer;

    public class MCSCtrlConnection : MCSConnection
    {
        // 読み込み間隔
        private const int Interval = 100;

        // パケットキュー
        private Queue _packetQueue = new Queue();

        // 最後のパケット
        private CtrlPacket _lastPacket = null;

        // キューイベント
        private ManualResetEvent _queuingEvent = new ManualResetEvent(false);

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MCSCtrlConnection() { }

        /// <summary>
        /// チャンネルの取得。
        /// </summary>
        public override uint Channel
        {
            get { return _baseChannel + 0x4; }
        }

#if USE_TIMEOUT
        /// <summary>
        /// 接続
        /// </summary>
        public override bool Connect(IPEndPoint ipEndPoint, string hostName)
        {
            bool result = base.Connect(ipEndPoint, hostName);
            if(result)
            {
                Client.ReceiveTimeout = Interval;
                Client.SendTimeout = Interval;
            }
            return result;
        }
#endif

        /// <summary>
        /// パケット送信
        /// </summary>
        public override void SendPacket( CommPacket commPacket)
        {
            CtrlPacket packet = commPacket as CtrlPacket;

            // キューにパケットを追加する
            lock (_packetQueue)
            {
                if (_packetQueue.Count > 512)
                {
                    return;
                }

                _packetQueue.Enqueue(packet);
                _lastPacket = packet;

                _queuingEvent.Set();

                //Debug.WriteLine( string.Format( "+ Queue : {0:C}", _packetQueue.Count ) );
            }

            //ShowMessage("Tool Packet Queue {0}", _packetQueue.Count);
        }

        /// <summary>
        /// パケット数の取得
        /// </summary>
        public int GetPacketCount()
        {
            lock (_packetQueue) { return _packetQueue.Count; }
        }

        /// <summary>
        /// パケット取得
        /// </summary>
        public CtrlPacket GetPacket()
        {
            // キューからパケットを取り出す
            CtrlPacket packet = null;
            lock (_packetQueue)
            {
                if (_packetQueue.Count > 0)
                {
                    packet = (CtrlPacket)_packetQueue.Dequeue();
                    _queuingEvent.Reset();

                    //Debug.WriteLine( string.Format( "- Dequeue : {0:N}", _packetQueue.Count ) );
                }
            }
            return packet;
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// メインループ。
        /// </summary>
        public override void MainLoop()
        {
            CtrlPacket packet = GetPacket();
            if (packet != null)
            {
                // パケットの書き込み
                CommPacketHeader packetHeader = packet.Write(Writer);
                Writer.Flush();
            }

#if USE_TIMEOUT
            // タイムアウトを利用した読み込み待ち
            MCSPacketHeader header = null;
            try
            {
                header = new MCSPacketHeader(Reader);
                Debug.Assert(!header.Error);
            }
            catch(IOException exception)
            {
                Exception inner = exception.InnerException;
                if((inner != null) && (inner is SocketException)){ return; }
                throw exception;
            }
#else
            // ネットワークストリームのDataAvailableを利用した読み込み待ち
            if ( !( (NetworkStream)Reader.BaseStream ).DataAvailable ) {
                Thread.Sleep( Interval );
                return;
            }
            CommPacketHeader header = new CtrlPacketHeader( Reader );
            Debug.Assert( !header.Error, "Header is null" );
#endif

            CtrlPacket replyPacket = CreatePacket(header);
            if (replyPacket != null)
            {
                replyPacket.Read(Reader, header);
                //ShowMessage("  Ctrl : " + packet);
            }
            else
            {
                // スキップ
                Reader.ReadBytes((int)header.Size);
                ShowMessage("  Ctrl : skip unknown message.");
            }
        }

        private CtrlPacket CreatePacket(CommPacketHeader header)
        {
            switch (header.PacketType & Constants.SRE_MESSAGE_MASK)
            {
                case Constants.SNDCTRL_UPDATE_SOUND_HANDLE_INFO:
                    return new UpdateSoundHandleInfoPacket();
            }

            return null;
        }
    }
}
