﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Collections;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.WaveFileFormat
{

    public class WaveFileWav : WaveFile
    {
        WavFormatChunk formatChunk = new WavFormatChunk();
        WavSamplerChunk samplerChunk = new WavSamplerChunk();
        long numSampleFrames;

        public WavFormatChunk FormatChunk
        {
            set { formatChunk = value; }
            get { return formatChunk; }
        }

        public WavSamplerChunk SamplerChunk
        {
            set { samplerChunk = value; }
            get { return samplerChunk; }
        }

        enum FormatTag : short
        {
            Unknown = 0,
            Pcm = 1,
            Adpcm = 2,
            IeeeFloat = 3,
        }

        public class WavFormatChunk
        {
            FormatTag wFormatTag = FormatTag.Pcm;
            ushort wChannels = 1;
            ulong dwSamplesPerSec = 32000;
            ulong dwAvgBytesPerSec = 64000;
            ushort wBlockAlign = 2;
            ushort wBitsPerSample = 16;

            void UpdateField()
            {
                wBlockAlign = (ushort)(wChannels * ((wBitsPerSample + 7) % 8));
                dwAvgBytesPerSec = wBlockAlign * dwSamplesPerSec;
            }
            public ushort ChannelCount
            {
                set
                {
                    wChannels = value;
                    UpdateField();
                }
                get { return wChannels; }
            }
            public ushort SampleBit
            {
                set
                {
                    wBitsPerSample = value;
                    UpdateField();
                }
                get { return wBitsPerSample; }
            }
            public ulong SampleRate
            {
                set
                {
                    dwSamplesPerSec = value;
                    UpdateField();
                }
                get { return dwSamplesPerSec; }
            }
            public void Read(BinaryReader reader)
            {
                wFormatTag = (FormatTag)reader.ReadInt16();
                wChannels = reader.ReadUInt16();
                dwSamplesPerSec = reader.ReadUInt32();
                dwAvgBytesPerSec = reader.ReadUInt32();
                wBlockAlign = reader.ReadUInt16();
                wBitsPerSample = reader.ReadUInt16();
            }
        }

        public class SampleLoop
        {
            Int32 dwIdentifier;
            Int32 dwType;
            Int32 dwStart;
            Int32 dwEnd;
            Int32 dwFraction;
            Int32 dwPlayCount;

            public Int32 StartFrame
            {
                set { dwStart = value; }
                get { return dwStart; }
            }
            public Int32 EndFrame
            {
                set { dwEnd = value; }
                get { return dwEnd; }
            }

            public void Read(BinaryReader reader)
            {
                dwIdentifier = reader.ReadInt32();
                dwType = reader.ReadInt32();
                dwStart = reader.ReadInt32();
                dwEnd = reader.ReadInt32();
                dwFraction = reader.ReadInt32();
                dwPlayCount = reader.ReadInt32();
            }
        }

        public class WavSamplerChunk
        {
            Int32 dwManufacturer = 0;
            Int32 dwProduct = 0;
            Int32 dwSamplePeriod = 0;
            Int32 dwMIDIUnityNote = 60;
            Int32 dwMIDIPitchFraction = 0;
            Int32 dwSMPTEFormat = 0;
            Int32 dwSMPTEOffset = 0;
            //		Int32 cSampleLoops = 0;
            Int32 cbSamplerData = 0;

            SampleLoop[] loops = new SampleLoop[0];

            public bool IsLoop
            {
                get { return loops.Length > 0; }
            }
            public long LoopStartFrame
            {
                set
                {
                    if (loops.Length == 0) loops = new SampleLoop[1];
                    loops[0].StartFrame = (int)value;
                }
                get
                {
                    if (loops.Length == 0) return -1;
                    return loops[0].StartFrame;
                }
            }
            public long LoopEndFrame
            {
                set
                {
                    if (loops.Length == 0) loops = new SampleLoop[1];
                    loops[0].EndFrame = (int)value;
                }
                get
                {
                    if (loops.Length == 0) return -1;
                    return loops[0].EndFrame + 1;
                }
            }
            public int OriginalKey
            {
                set
                {
                    dwMIDIUnityNote = (Int32)value;
                }
                get
                {
                    return (int)dwMIDIUnityNote;
                }
            }

            public void Read(BinaryReader reader)
            {
                dwManufacturer = reader.ReadInt32();
                dwProduct = reader.ReadInt32();
                dwSamplePeriod = reader.ReadInt32();
                dwMIDIUnityNote = reader.ReadInt32();
                dwMIDIPitchFraction = reader.ReadInt32();
                dwSMPTEFormat = reader.ReadInt32();
                dwSMPTEOffset = reader.ReadInt32();
                Int32 cSampleLoops = reader.ReadInt32();
                cbSamplerData = reader.ReadInt32();

                loops = new SampleLoop[cSampleLoops];
                for (int i = 0; i < cSampleLoops; i++)
                {
                    SampleLoop loop = new SampleLoop();
                    loop.Read(reader);
                    loops[i] = loop;
                }
            }
        }

        public override bool IsLoop
        {
            get { return samplerChunk.IsLoop; }
        }
        public override long LoopStartFrame
        {
            set { samplerChunk.LoopStartFrame = value; }
            get { return samplerChunk.LoopStartFrame; }
        }
        public override long LoopEndFrame
        {
            set { samplerChunk.LoopEndFrame = value; }
            get { return samplerChunk.LoopEndFrame; }
        }

        public override int ChannelCount
        {
            set { formatChunk.ChannelCount = (ushort)value; }
            get { return formatChunk.ChannelCount; }
        }
        public override long FrameCount
        {
            set { numSampleFrames = value; }
            get { return numSampleFrames; }
        }
        public override int SampleBit
        {
            set
            {
                formatChunk.SampleBit = (ushort)value;
            }
            get
            {
                return formatChunk.SampleBit;
            }
        }
        public override int SampleRate
        {
            set
            {
                formatChunk.SampleRate = (ushort)value;
            }
            get
            {
                return (int)formatChunk.SampleRate;
            }
        }
        public override int OriginalKey
        {
            set
            {
                samplerChunk.OriginalKey = value;
            }
            get
            {
                return samplerChunk.OriginalKey;
            }
        }
    }

}
