﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Xml;
using System.IO;
using System.Collections;
using System.Collections.Generic;
using NintendoWare.SoundFoundation.Legacies.FileFormat;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{

    public class Nw4rXmlBank : Nw4rXmlFile
    {
        Nw4rXmlBankInstList instList = new Nw4rXmlBankInstList();

        public Nw4rXmlBankInstList InstList
        {
            get { return instList; }
        }

        protected override string FileTitle { get { return "NintendoWare Bank"; } }

        protected override void LoadBody(XmlDocument doc, XmlElement bodyElem)
        {
            instList.Clear();

            XmlElement bankElem = (XmlElement)bodyElem.SelectSingleNode("bank");
            if (bankElem == null)
            {
                throw new Nw4rFileFormatException("Invalid bank file format.");
            }

            XmlElement instArrayElem = (XmlElement)bankElem.SelectSingleNode("inst_array");
            if (instArrayElem != null)
            {
                foreach (XmlElement instElem in instArrayElem.SelectNodes("inst"))
                {
                    Nw4rXmlBankInst inst = Nw4rXmlBankInst.Parse(doc, instElem);
                    if (inst != null)
                    {
                        instList.Add(inst);
                    }
                }
            }
        }

        protected override void SaveBody(XmlDocument doc, XmlElement bodyElem)
        {
            XmlElement bankElem = doc.CreateElement("bank");

            if (instList.Count > 0)
            {
                XmlElement instArrayElem = doc.CreateElement("inst_array");
                instArrayElem.SetAttribute("size", instList.Count.ToString());
                foreach (Nw4rXmlBankInst inst in instList)
                {
                    XmlElement instElem = inst.ToXmlElement(doc);
                    instArrayElem.AppendChild(instElem);
                }

                bankElem.AppendChild(instArrayElem);
            }
            bodyElem.AppendChild(bankElem);
        }
    }

    public class Nw4rXmlBankInstList : IEnumerable
    {
        ArrayList itemList = new ArrayList();

        /******************************************************************************
          interface
         ******************************************************************************/
        IEnumerator IEnumerable.GetEnumerator()
        {
            return itemList.GetEnumerator();
        }

        /******************************************************************************
           public
         ******************************************************************************/
        public int Count
        {
            get { return itemList.Count; }
        }
        public void Add(Nw4rXmlBankInst inst)
        {
            itemList.Add(inst);
        }
        public void Remove(Nw4rXmlBankInst inst)
        {
            itemList.Remove(inst);
        }
        public void Clear()
        {
            itemList.Clear();
        }
    }

    public struct Nw4rXmlBankPcmFileFormat
    {
        public const int Invalid = 0;
        public const int Adpcm = 1;
        public const int Pcm16 = 2;
        public const int Pcm8 = 3;

        readonly static KeyValuePair<int, string>[] strTable =
        {
            new KeyValuePair<int, string>( Invalid, "Invalid" ),
            new KeyValuePair<int, string>( Adpcm, "Adpcm" ),
            new KeyValuePair<int, string>( Pcm16, "Pcm16" ),
            new KeyValuePair<int, string>( Pcm8, "Pcm8" ),
        };

        public static implicit operator int(Nw4rXmlBankPcmFileFormat s) { return s.val_; }
        public static implicit operator Nw4rXmlBankPcmFileFormat(int s) { return new Nw4rXmlBankPcmFileFormat(s); }
        public static Nw4rXmlBankPcmFileFormat Parse(string fileFormat) { return IntConverter.Parse(fileFormat, strTable); }
        public override string ToString() { return IntConverter.ToString(val_, strTable); }

        private Nw4rXmlBankPcmFileFormat(int e) { val_ = e; }
        private int val_;
    }

    public struct Nw4rXmlBankNoteOffType
    {
        public const int Release = 0;
        public const int Ignore = 1;

        readonly static KeyValuePair<int, string>[] strTable =
        {
            new KeyValuePair<int, string>( Release, "release" ),
            new KeyValuePair<int, string>( Ignore, "ignore" ),
        };

        public static implicit operator int(Nw4rXmlBankNoteOffType s) { return s.val_; }
        public static implicit operator Nw4rXmlBankNoteOffType(int s) { return new Nw4rXmlBankNoteOffType(s); }
        public static Nw4rXmlBankNoteOffType Parse(string type) { return IntConverter.Parse(type, strTable); }
        public override string ToString() { return IntConverter.ToString(val_, strTable); }

        private Nw4rXmlBankNoteOffType(int e) { val_ = e; }
        private int val_;
    }

    public struct Nw4rXmlBankAdsrEnvelopeSelect
    {
        public const int Inst = 0;
        public const int KeyRegion = 1;
        public const int VelRegion = 2;
        public const int LowerPriority = 3;

        readonly static KeyValuePair<int, string>[] strTable =
        {
            new KeyValuePair<int, string>( Inst, "Inst" ),
            new KeyValuePair<int, string>( KeyRegion, "KeyRegion" ),
            new KeyValuePair<int, string>( VelRegion, "VelRegion" ),
            new KeyValuePair<int, string>( LowerPriority, "LowerPriority" ),
        };

        public static implicit operator int(Nw4rXmlBankAdsrEnvelopeSelect s) { return s.val_; }
        public static implicit operator Nw4rXmlBankAdsrEnvelopeSelect(int s) { return new Nw4rXmlBankAdsrEnvelopeSelect(s); }
        public static Nw4rXmlBankAdsrEnvelopeSelect Parse(string fileFormat) { return IntConverter.Parse(fileFormat, strTable); }
        public override string ToString() { return IntConverter.ToString(val_, strTable); }

        private Nw4rXmlBankAdsrEnvelopeSelect(int e) { val_ = e; }
        private int val_;
    }

    public class Nw4rXmlBankInst : IEnumerable
    {
        ArrayList keyRegionArray = new ArrayList();
        int index;
        string label = String.Empty;
        bool enabled = true;
        int colorIndex = 0;
        double volume = 1.0;
        int fineTune = 0;
        int coarseTune = 0;
        string comment = String.Empty;
        Nw4rAdsrEnvelope envelope = null;
        Nw4rXmlBankAdsrEnvelopeSelect adsrSelect = Nw4rXmlBankAdsrEnvelopeSelect.LowerPriority;

        /******************************************************************************
          interface
         ******************************************************************************/
        IEnumerator IEnumerable.GetEnumerator()
        {
            return keyRegionArray.GetEnumerator();
        }

        /******************************************************************************
           public
         ******************************************************************************/
        public int PrgNo
        {
            set { index = value; }
            get { return index; }
        }
        public string Label
        {
            set { label = value; }
            get { return label; }
        }
        public bool Enabled
        {
            set { enabled = value; }
            get { return enabled; }
        }
        public int ColorIndex
        {
            set { colorIndex = value; }
            get { return colorIndex; }
        }
        public string Comment
        {
            set { comment = value; }
            get { return comment; }
        }
        public int Volume
        {
            get { return (int)(volume * 127.0); }
            set { volume = (double)value / 127.0; }
        }
        public int FineTune
        {
            get { return fineTune; }
            set { fineTune = value; }
        }
        public int CoarseTune
        {
            get { return coarseTune; }
            set { coarseTune = value; }
        }
        public Nw4rAdsrEnvelope AdsrEnvelope
        {
            get { return envelope; }
            set { envelope = value; }
        }
        public Nw4rXmlBankAdsrEnvelopeSelect AdsrEnvelopeSelect
        {
            get { return adsrSelect; }
            set { adsrSelect = value; }
        }
        public void AddKeyRegion(Nw4rXmlBankInstKeyRegion keyRegion)
        {
            keyRegionArray.Add(keyRegion);
        }
        public void RemoveKeyReigon(Nw4rXmlBankInstKeyRegion keyRegion)
        {
            keyRegionArray.Remove(keyRegion);
        }
        public int KeyRegionCount
        {
            get { return keyRegionArray.Count; }
        }
        public void ClearKeyRegion()
        {
            keyRegionArray.Clear();
        }

        internal static Nw4rXmlBankInst Parse(XmlDocument doc, XmlElement instElem)
        {
            Nw4rXmlBankInst inst = new Nw4rXmlBankInst();
            if (instElem.HasAttribute("name"))
            {
                inst.label = instElem.GetAttribute("name");
            }
            if (instElem.HasAttribute("enabled"))
            {
                inst.enabled = bool.Parse(instElem.GetAttribute("enabled"));
            }
            if (instElem.HasAttribute("colorIndex"))
            {
                inst.colorIndex = int.Parse(instElem.GetAttribute("colorIndex"));
            }
            if (instElem.HasAttribute("prg_no"))
            {
                inst.PrgNo = Int32.Parse(instElem.GetAttribute("prg_no"));
            }
            XmlElement commentElem = (XmlElement)instElem.SelectSingleNode("comment");
            if (commentElem != null)
            {
                inst.Comment = commentElem.InnerText;
            }
            XmlElement volumeElem = (XmlElement)instElem.SelectSingleNode("volume");
            if (volumeElem != null)
            {
                inst.volume = Double.Parse(volumeElem.InnerText);
            }

            XmlElement fineTuneElem = (XmlElement)instElem.SelectSingleNode("fine_tune");
            if (fineTuneElem != null)
            {
                inst.FineTune = Int32.Parse(fineTuneElem.InnerText);
            }

            XmlElement coarseTuneElem = (XmlElement)instElem.SelectSingleNode("coarse_tune");
            if (coarseTuneElem != null)
            {
                inst.CoarseTune = Int32.Parse(coarseTuneElem.InnerText);
            }

            if (instElem.HasAttribute("adsr_envelope_select"))
            {
                inst.adsrSelect = Nw4rXmlBankAdsrEnvelopeSelect.Parse(instElem.GetAttribute("adsr_envelope_select"));
            }
            XmlElement adsrEnvElem = (XmlElement)instElem.SelectSingleNode("adsr_envelope");
            if (adsrEnvElem != null)
            {
                inst.AdsrEnvelope = Nw4rAdsrEnvelope.Parse(doc, adsrEnvElem);
            }

            XmlElement keyRegionArrayElem = (XmlElement)instElem.SelectSingleNode("key_region_array");
            if (keyRegionArrayElem != null)
            {
                foreach (XmlElement keyRegionElem in keyRegionArrayElem.SelectNodes("key_region"))
                {
                    Nw4rXmlBankInstKeyRegion keyRegion = Nw4rXmlBankInstKeyRegion.Parse(doc, keyRegionElem);
                    if (keyRegion != null)
                    {
                        inst.keyRegionArray.Add(keyRegion);
                    }
                }
            }
            return inst;
        }
        internal XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement instElem = doc.CreateElement("inst");
            instElem.SetAttribute("name", Label);

            if (!Enabled)
            {
                instElem.SetAttribute("enabled", Enabled.ToString());
            }

            if (0 != ColorIndex)
            {
                instElem.SetAttribute("colorIndex", ColorIndex.ToString());
            }

            instElem.SetAttribute("prg_no", PrgNo.ToString());
            instElem.SetAttribute("adsr_envelope_select", adsrSelect.ToString());

            XmlElement volumeElem = doc.CreateElement("volume");
            volumeElem.InnerText = volume.ToString();
            instElem.AppendChild(volumeElem);

            XmlElement fineTuneElem = doc.CreateElement("fine_tune");
            fineTuneElem.InnerText = fineTune.ToString();
            instElem.AppendChild(fineTuneElem);

            XmlElement coarseTuneElem = doc.CreateElement("coarse_tune");
            coarseTuneElem.InnerText = coarseTune.ToString();
            instElem.AppendChild(coarseTuneElem);

            if (!String.IsNullOrEmpty(comment))
            {
                XmlElement commentElem = doc.CreateElement("comment");
                commentElem.InnerText = comment;
                instElem.AppendChild(commentElem);
            }

            if (envelope != null)
            {
                XmlElement adsrEnvElem = envelope.ToXmlElement(doc);
                instElem.AppendChild(adsrEnvElem);
            }

            XmlElement keyRegionArrayElem = doc.CreateElement("key_region_array");
            keyRegionArrayElem.SetAttribute("size", keyRegionArray.Count.ToString());
            foreach (Nw4rXmlBankInstKeyRegion keyRegion in keyRegionArray)
            {
                XmlElement keyRegionElem = keyRegion.ToXmlElement(doc);
                keyRegionArrayElem.AppendChild(keyRegionElem);
            }
            instElem.AppendChild(keyRegionArrayElem);

            return instElem;
        }
    }

    public class Nw4rXmlBankInstVelRegion
    {
        int velMin = 0;
        int velMax = 127;
        string srcFilePath;
        Nw4rXmlBankPcmFileFormat fileType = Nw4rXmlBankPcmFileFormat.Invalid;
        Nw4rAdsrEnvelope envelope = null;
        int originalKey = 60;
        int pan = 64;
        double volume = 1.0;
        int fineTune = 0;
        int coarseTune = 0;
        Nw4rXmlBankNoteOffType noteOffType = Nw4rXmlBankNoteOffType.Release;
        int alternateAssign = 0;

        public string FilePath
        {
            get { return srcFilePath; }
            set { srcFilePath = value; }
        }

        public Nw4rAdsrEnvelope AdsrEnvelope
        {
            get { return envelope; }
            set { envelope = value; }
        }
        public int Pan
        {
            get { return pan; }
            set { pan = value; }
        }
        public int Volume
        {
            get { return (int)(volume * 127.0); }
            set { volume = (double)value / 127.0; }
        }
        public int FineTune
        {
            get { return fineTune; }
            set { fineTune = value; }
        }
        public int CoarseTune
        {
            get { return coarseTune; }
            set { coarseTune = value; }
        }

        public Nw4rXmlBankPcmFileFormat FileFormat
        {
            get { return fileType; }
            set { fileType = value; }
        }

        public int OriginalKey
        {
            get { return originalKey; }
            set { originalKey = value; }
        }

        public Nw4rXmlBankNoteOffType NoteOffType
        {
            get { return noteOffType; }
            set { noteOffType = value; }
        }
        public int AlternateAssign
        {
            get { return alternateAssign; }
            set { alternateAssign = value; }
        }

        public int RangeMin
        {
            get { return velMin; }
            set { velMin = value; }
        }
        public int RangeMax
        {
            get { return velMax; }
            set { velMax = value; }
        }
        internal static Nw4rXmlBankInstVelRegion Parse(XmlDocument doc, XmlElement velRegionElem)
        {
            Nw4rXmlBankInstVelRegion velRegion = new Nw4rXmlBankInstVelRegion();

            if (velRegionElem.HasAttribute("range_min"))
            {
                velRegion.RangeMin = Int32.Parse(velRegionElem.GetAttribute("range_min"));
            }
            if (velRegionElem.HasAttribute("range_max"))
            {
                velRegion.RangeMax = Int32.Parse(velRegionElem.GetAttribute("range_max"));
            }

            XmlElement fileElem = (XmlElement)velRegionElem.SelectSingleNode("file");
            if (fileElem != null)
            {
                if (fileElem.HasAttribute("path"))
                {
                    velRegion.FilePath = fileElem.GetAttribute("path");
                }
                if (fileElem.HasAttribute("encode"))
                {
                    velRegion.FileFormat = Nw4rXmlBankPcmFileFormat.Parse(fileElem.GetAttribute("encode"));
                }
            }

            XmlElement adsrEnvElem = (XmlElement)velRegionElem.SelectSingleNode("adsr_envelope");
            if (adsrEnvElem != null)
            {
                velRegion.AdsrEnvelope = Nw4rAdsrEnvelope.Parse(doc, adsrEnvElem);
            }

            XmlElement panElem = (XmlElement)velRegionElem.SelectSingleNode("pan");
            if (panElem != null)
            {
                velRegion.Pan = Int32.Parse(panElem.InnerText);
            }

            XmlElement volumeElem = (XmlElement)velRegionElem.SelectSingleNode("volume");
            if (volumeElem != null)
            {
                velRegion.volume = Double.Parse(volumeElem.InnerText);
            }

            XmlElement fineTuneElem = (XmlElement)velRegionElem.SelectSingleNode("fine_tune");
            if (fineTuneElem != null)
            {
                velRegion.FineTune = Int32.Parse(fineTuneElem.InnerText);
            }

            XmlElement coarseTuneElem = (XmlElement)velRegionElem.SelectSingleNode("coarse_tune");
            if (coarseTuneElem != null)
            {
                velRegion.CoarseTune = Int32.Parse(coarseTuneElem.InnerText);
            }

            XmlElement originalKeyElem = (XmlElement)velRegionElem.SelectSingleNode("original_key");
            if (originalKeyElem != null)
            {
                velRegion.OriginalKey = Int32.Parse(originalKeyElem.InnerText);
            }

            XmlElement noteOffElem = (XmlElement)velRegionElem.SelectSingleNode("note_off");
            if (noteOffElem != null)
            {
                if (noteOffElem.HasAttribute("type"))
                {
                    velRegion.NoteOffType = Nw4rXmlBankNoteOffType.Parse(noteOffElem.GetAttribute("type"));
                }
            }

            XmlElement alternateAssignElem = (XmlElement)velRegionElem.SelectSingleNode("alternate_assign");
            if (alternateAssignElem != null)
            {
                velRegion.AlternateAssign = Int32.Parse(alternateAssignElem.InnerText);
            }

            return velRegion;
        }
        internal XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement velRegionElem = doc.CreateElement("vel_region");
            if (velMin != Int32.MinValue)
            {
                velRegionElem.SetAttribute("range_min", velMin.ToString());
            }
            if (velMax != Int32.MaxValue)
            {
                velRegionElem.SetAttribute("range_max", velMax.ToString());
            }

            XmlElement fileElem = doc.CreateElement("file");
            fileElem.SetAttribute("path", FilePath);
            fileElem.SetAttribute("encode", FileFormat.ToString());
            velRegionElem.AppendChild(fileElem);

            if (envelope != null)
            {
                XmlElement adsrEnvElem = envelope.ToXmlElement(doc);
                velRegionElem.AppendChild(adsrEnvElem);
            }

            XmlElement panElem = doc.CreateElement("pan");
            panElem.InnerText = pan.ToString();
            velRegionElem.AppendChild(panElem);

            XmlElement volumeElem = doc.CreateElement("volume");
            volumeElem.InnerText = volume.ToString();
            velRegionElem.AppendChild(volumeElem);

            XmlElement fineTuneElem = doc.CreateElement("fine_tune");
            fineTuneElem.InnerText = fineTune.ToString();
            velRegionElem.AppendChild(fineTuneElem);

            XmlElement coarseTuneElem = doc.CreateElement("coarse_tune");
            coarseTuneElem.InnerText = coarseTune.ToString();
            velRegionElem.AppendChild(coarseTuneElem);

            XmlElement originalKeyElem = doc.CreateElement("original_key");
            originalKeyElem.InnerText = originalKey.ToString();
            velRegionElem.AppendChild(originalKeyElem);

            XmlElement noteOffElem = doc.CreateElement("note_off");
            noteOffElem.SetAttribute("type", NoteOffType.ToString());
            velRegionElem.AppendChild(noteOffElem);

            XmlElement alternateAssignElem = doc.CreateElement("alternate_assign");
            alternateAssignElem.InnerText = alternateAssign.ToString();
            velRegionElem.AppendChild(alternateAssignElem);

            return velRegionElem;
        }
    }

    public class Nw4rXmlBankInstKeyRegion : IEnumerable
    {
        ArrayList velRegionArray = new ArrayList();
        int keyMin = Int32.MinValue;
        int keyMax = Int32.MaxValue;
        Nw4rAdsrEnvelope envelope = null;
        int pan = 64;
        string comment = String.Empty;

        /******************************************************************************
          interface
         ******************************************************************************/
        IEnumerator IEnumerable.GetEnumerator()
        {
            return velRegionArray.GetEnumerator();
        }

        /******************************************************************************
           public
         ******************************************************************************/
        public int RangeMin
        {
            get { return keyMin; }
            set { keyMin = value; }
        }
        public int RangeMax
        {
            get { return keyMax; }
            set { keyMax = value; }
        }
        public Nw4rAdsrEnvelope AdsrEnvelope
        {
            get { return envelope; }
            set { envelope = value; }
        }
        public int Pan
        {
            get { return pan; }
            set { pan = value; }
        }
        public string Comment
        {
            set { comment = value; }
            get { return comment; }
        }

        public void AddVelRegion(Nw4rXmlBankInstVelRegion velRegion)
        {
            velRegionArray.Add(velRegion);
        }
        public void RemoveVelRegion(Nw4rXmlBankInstVelRegion velRegion)
        {
            velRegionArray.Remove(velRegion);
        }
        public int VelRegionCount
        {
            get { return velRegionArray.Count; }
        }
        public void ClearVelRegion()
        {
            velRegionArray.Clear();
        }
        internal static Nw4rXmlBankInstKeyRegion Parse(XmlDocument doc, XmlElement keyRegionElem)
        {
            Nw4rXmlBankInstKeyRegion keyRegion = new Nw4rXmlBankInstKeyRegion();

            if (keyRegionElem.HasAttribute("range_min"))
            {
                keyRegion.RangeMin = Int32.Parse(keyRegionElem.GetAttribute("range_min"));
            }
            if (keyRegionElem.HasAttribute("range_max"))
            {
                keyRegion.RangeMax = Int32.Parse(keyRegionElem.GetAttribute("range_max"));
            }

            XmlElement commentElem = (XmlElement)keyRegionElem.SelectSingleNode("comment");
            if (commentElem != null)
            {
                keyRegion.Comment = commentElem.InnerText;
            }
            XmlElement adsrEnvElem = (XmlElement)keyRegionElem.SelectSingleNode("adsr_envelope");
            if (adsrEnvElem != null)
            {
                keyRegion.AdsrEnvelope = Nw4rAdsrEnvelope.Parse(doc, adsrEnvElem);
            }

            XmlElement panElem = (XmlElement)keyRegionElem.SelectSingleNode("pan");
            if (panElem != null)
            {
                keyRegion.Pan = Int32.Parse(panElem.InnerText);
            }

            XmlElement velRegionArrayElem = (XmlElement)keyRegionElem.SelectSingleNode("vel_region_array");
            if (velRegionArrayElem != null)
            {
                foreach (XmlElement velRegionElem in velRegionArrayElem.SelectNodes("vel_region"))
                {
                    Nw4rXmlBankInstVelRegion velRegion = Nw4rXmlBankInstVelRegion.Parse(doc, velRegionElem);
                    if (velRegion != null)
                    {
                        keyRegion.velRegionArray.Add(velRegion);
                    }
                }
            }

            return keyRegion;
        }
        internal XmlElement ToXmlElement(XmlDocument doc)
        {
            XmlElement keyRegionElem = doc.CreateElement("key_region");
            if (keyMin != Int32.MinValue)
            {
                keyRegionElem.SetAttribute("range_min", keyMin.ToString());
            }
            if (keyMax != Int32.MaxValue)
            {
                keyRegionElem.SetAttribute("range_max", keyMax.ToString());
            }

            if (envelope != null)
            {
                XmlElement adsrEnvElem = envelope.ToXmlElement(doc);
                keyRegionElem.AppendChild(adsrEnvElem);
            }

            if (!String.IsNullOrEmpty(comment))
            {
                XmlElement commentElem = doc.CreateElement("comment");
                commentElem.InnerText = comment;
                keyRegionElem.AppendChild(commentElem);
            }

            XmlElement panElem = doc.CreateElement("pan");
            panElem.InnerText = pan.ToString();
            keyRegionElem.AppendChild(panElem);

            XmlElement velRegionArrayElem = doc.CreateElement("vel_region_array");
            velRegionArrayElem.SetAttribute("size", velRegionArray.Count.ToString());
            foreach (Nw4rXmlBankInstVelRegion velRegion in velRegionArray)
            {
                XmlElement velRegionElem = velRegion.ToXmlElement(doc);
                velRegionArrayElem.AppendChild(velRegionElem);
            }
            keyRegionElem.AppendChild(velRegionArrayElem);

            return keyRegionElem;
        }
    }

    class Nw4rXmlBankKeySymbolParser
    {
        public static bool ParseSymbol(string symbol, out long x)
        {
            int key;
            if (!Lexer.ParseKeyString(symbol, out key))
            {
                x = 0;
                return false;
            }
            x = key;
            return true;
        }
    }


}

