﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.IO;
using System.Text;

using NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Model;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{
    #region ** WaveSound ファイル

    internal class Nw4rWsdDataFile : Nw4rFileRoot
    {
        private static readonly ushort Version = 0x0103;

        private Nw4rWaveSoundSet _waveSoundSet = null;
        private INw4rComponentCollection _waveFiles = null;

        public Nw4rWsdDataFile() : base("WaveSoundDataFileRoot") { }

        public Nw4rWsdDataFile(Nw4rWaveSoundSet waveSoundSet) : this(waveSoundSet, null) { }
        public Nw4rWsdDataFile(Nw4rWaveSoundSet waveSoundSet, INw4rComponentCollection waveFiles) : base("WaveSoundDataFileRoot")
        {
            if (null == waveSoundSet) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("waveSoundSet")); }

            _waveSoundSet = waveSoundSet;
            _waveFiles = (null == waveFiles) ? waveSoundSet.BinaryFile.Components : waveFiles;


            Nw4rFileHeader header = new Nw4rFileHeader("FileHeader", "RWSD", Version);
            Nw4rWsdDataBlock dataBlock = new Nw4rWsdDataBlock(new Nw4rWsdDataBlockBody(waveSoundSet));

            header.AddBlockEntry(dataBlock);

            AddNode(header);
            AddNode(dataBlock);
        }

        #region ** プロパティ

        public Nw4rWaveSoundSet WaveSoundSet
        {
            get { return _waveSoundSet; }
            set { _waveSoundSet = value; }
        }

        public INw4rComponentCollection WaveFiles
        {
            get { return _waveFiles; }
            set { _waveFiles = value; }
        }

        private Nw4rWsdDataBlock DataBlock
        {
            get { return ChildNodes[Nw4rWsdDataBlock.NodeName] as Nw4rWsdDataBlock; }
        }

        #endregion

        #region ** メソッド

        public static Nw4rWsdDataFile FromFile(string fileName)
        {
            if (null == fileName) { throw new ArgumentNullException("fileName"); }

            Nw4rWsdDataFile root = new Nw4rWsdDataFile();
            root.LoadFile(fileName);

            return root;
        }

        protected void LoadFile(string fileName)
        {
            if (null == fileName) { throw new ArgumentNullException("fileName"); }

            using (FileStream stream = File.OpenRead(fileName))
            {
                Read(BinaryReaderBigEndian.CreateInstance(stream));
            }
        }

        protected override Nw4rFileDataBlock ReadDataBlock(BinaryReader reader)
        {
            Nw4rFileDataBlockHeader header = new Nw4rFileDataBlockHeader();
            header.Read(reader);


            Nw4rFileDataBlock newBlock = null;

            switch (header.Kind)
            {
                case "DATA":
                    newBlock = new Nw4rWsdDataBlock(header);
                    break;

                default:
                    newBlock = new Nw4rFileBinaryDataBlock("UnknownBlock", header);
                    break;
            }

            newBlock.ReadBody(reader);

            return newBlock;
        }

        #endregion
    }

    #region ** WaveSoundFile DataBlock

    internal class Nw4rWsdDataBlock : Nw4rFileDataBlock
    {
        #region ** 固定値

        public const string NodeName = "DataBlock";

        #endregion

        public Nw4rWsdDataBlock(Nw4rFileDataBlockHeader header) : base(NodeName, header) { }

        public Nw4rWsdDataBlock(Nw4rWsdDataBlockBody body) : base(NodeName, new Nw4rFileDataBlockHeader("DATA"), body) { }

        #region ** イベントハンドラ

        protected override void OnReadBody(BinaryReader reader)
        {
            Nw4rWsdDataBlockBody body = new Nw4rWsdDataBlockBody();
            body.Read(reader);

            AddNode(body);
        }

        #endregion
    }

    internal class Nw4rWsdDataBlockBody : Nw4rFileDataBlockBody
    {
        private Nw4rWaveSoundSet _waveSoundSet = null;
        private Nw4rFileInnerOffsetRefTable _table = new Nw4rFileInnerOffsetRefTable();

        public Nw4rWsdDataBlockBody() { }
        public Nw4rWsdDataBlockBody(Nw4rWaveSoundSet waveSoundSet)
        {
            if (null == waveSoundSet) { throw new ArgumentNullException("waveSoundSet"); }
            _waveSoundSet = waveSoundSet;

            BuildNodes(waveSoundSet);
        }

        #region ** イベントハンドラ

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            base.OnWriteBinary(writer);

            // テーブル出力
            _table.Write(writer);

            // データ出力は OnWriteChildNodeBinaries にて実行される
        }

        protected override void OnPostWriteBinary(Nw4rWriteFileEventArgs e)
        {
            // 32byte アライメント調整
            new Nw4rByteAligner(e.Writer.BaseStream).Pad(32);

            base.OnPostWriteBinary(e);
        }

        protected override void OnReadBinary(BinaryReader reader)
        {
            _table.Read(reader);

            foreach (Nw4rFileInnerOffsetRefTableItem item in _table.Items)
            {

                reader.BaseStream.Position = Offset + item.Reference.Size;

                Nw4rWaveSoundData newWaveSoundData = new Nw4rWaveSoundData();
                newWaveSoundData.Read(reader);

                AddNode(newWaveSoundData);

                item.Data = newWaveSoundData;

            }

            new Nw4rByteAligner(reader.BaseStream).SeekForward(32);
        }

        #endregion

        #region ** メソッド

        private void BuildNodes(Nw4rWaveSoundSet waveSoundSet)
        {
            Debug.Assert(null != waveSoundSet);

            foreach (Nw4rWaveSound waveSound in waveSoundSet.Components)
            {

                Nw4rWaveSoundData newNode = new Nw4rWaveSoundData(waveSound);

                _table.AddItem(newNode);
                AddNode(newNode);

            }
        }

        #endregion
    }

    internal class Nw4rWaveSoundData : Nw4rFileNode
    {
        private Nw4rFileDataOffsetRef _referenceWsdInfo = new Nw4rFileDataOffsetRef();
        private Nw4rFileDataOffsetRef _referenceTrackTable = new Nw4rFileDataOffsetRef();
        private Nw4rFileDataOffsetRef _referenceNoteTable = new Nw4rFileDataOffsetRef();

        public Nw4rWaveSoundData() : base("WaveSoundData") { }

        public Nw4rWaveSoundData(Nw4rWaveSound waveSound) : base("WaveSoundData")
        {
            if (null == waveSound) { throw new ArgumentNullException("waveSound"); }

            BuildNodes(waveSound);
        }

        #region ** イベントハンドラ

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            _referenceWsdInfo.ReserveWrite(writer);
            _referenceTrackTable.ReserveWrite(writer);
            _referenceNoteTable.ReserveWrite(writer);
        }

        protected override void OnReadBinary(BinaryReader reader)
        {
            Nw4rWsdWaveSoundInforamtion wsdInfo = new Nw4rWsdWaveSoundInforamtion();
            wsdInfo.Read(reader);
            AddNode(wsdInfo);

            Nw4rWsdTrackTable trackTable = new Nw4rWsdTrackTable();
            trackTable.Read(reader);
            AddNode(trackTable);

            Nw4rWsdNoteTable noteTable = new Nw4rWsdNoteTable();
            noteTable.Read(reader);
            AddNode(noteTable);
        }

        private void OnWsdInfoOffsetFixed(object sender, EventArgs e)
        {
            _referenceWsdInfo.Origin = Parent.Offset;
            _referenceWsdInfo.Size = (sender as Nw4rFileNode).Offset - _referenceWsdInfo.Origin;
            _referenceWsdInfo.Commit();
        }

        private void OnTrackTableOffsetFixed(object sender, EventArgs e)
        {
            _referenceTrackTable.Origin = Parent.Offset;
            _referenceTrackTable.Size = (sender as Nw4rFileNode).Offset - _referenceTrackTable.Origin;
            _referenceTrackTable.Commit();
        }

        private void OnNoteTableOffsetFixed(object sender, EventArgs e)
        {
            _referenceNoteTable.Origin = Parent.Offset;
            _referenceNoteTable.Size = (sender as Nw4rFileNode).Offset - _referenceNoteTable.Origin;
            _referenceNoteTable.Commit();
        }

        #endregion

        #region ** メソッド

        private void BuildNodes(Nw4rWaveSound waveSound)
        {
            Debug.Assert(null != waveSound);

            Nw4rWsdWaveSoundInforamtion wsdInfo = new Nw4rWsdWaveSoundInforamtion(waveSound);
            Nw4rWsdTrackTable trackTable = new Nw4rWsdTrackTable(waveSound);
            Nw4rWsdNoteTable noteTable = new Nw4rWsdNoteTable(waveSound);

            AddNode(wsdInfo);
            AddNode(trackTable);
            AddNode(noteTable);

            wsdInfo.OffsetFixed += OnWsdInfoOffsetFixed;
            trackTable.OffsetFixed += OnTrackTableOffsetFixed;
            noteTable.OffsetFixed += OnNoteTableOffsetFixed;
        }

        #endregion
    }

    #region ** パラメータ

    internal class Nw4rWsdWaveSoundInforamtion : Nw4rFileNode
    {
        private Single _pitch = 0.0f;	// ピッチ
        private Byte _pan = 0;	// パン
        private Byte _surroundPan = 0;	// サラウンドパン
        private Byte _fxSendA = 0;	// エフェクトセンドA
        private Byte _fxSendB = 0;	// エフェクトセンドB
        private Byte _fxSendC = 0;	// エフェクトセンドC
        private Byte _mainSend = 0;	// メインセンド

        public Nw4rWsdWaveSoundInforamtion() : base("WaveSoundInformation") { }

        public Nw4rWsdWaveSoundInforamtion(Nw4rWaveSound waveSound)
            : base("WaveSoundInformation")
        {
            if (null == waveSound) { throw new ArgumentNullException("waveSound"); }

            _pitch = (Single)waveSound.XmlData.Pitch;
            _pan = (Byte)waveSound.XmlData.Pan;
            _surroundPan = (Byte)waveSound.XmlData.SurroundPan;
            _fxSendA = (Byte)waveSound.XmlData.FxSendA;
            _fxSendB = (Byte)waveSound.XmlData.FxSendB;
            _fxSendC = (Byte)waveSound.XmlData.FxSendC;
            _mainSend = (Byte)waveSound.XmlData.MainSend;
        }

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            writer.Write(_pitch);			// ピッチ
            writer.Write(_pan);			// パン
            writer.Write(_surroundPan);	// サラウンドパン
            writer.Write(_fxSendA);		// エフェクトセンドA
            writer.Write(_fxSendB);		// エフェクトセンドB
            writer.Write(_fxSendC);		// エフェクトセンドC
            writer.Write(_mainSend);		// メインセンド

            writer.Write((UInt16)0);								// Padding 2byte

            Nw4rFileDataAddressRef.InvalidValue.Write(writer);	// graphEnvTable
            Nw4rFileDataAddressRef.InvalidValue.Write(writer);	// randomizerTable

            writer.Write((UInt32)0);					// 予約済み
        }

        protected override void OnReadBinary(BinaryReader reader)
        {
            _pitch = reader.ReadSingle();	// ピッチ
            _pan = reader.ReadByte();	// パン
            _surroundPan = reader.ReadByte();	// サラウンドパン
            _fxSendA = reader.ReadByte();	// エフェクトセンドA
            _fxSendB = reader.ReadByte();	// エフェクトセンドB
            _fxSendC = reader.ReadByte();	// エフェクトセンドC
            _mainSend = reader.ReadByte();	// メインセンド

            reader.ReadUInt16();								// Padding 2byte

            Nw4rFileDataAddressRef.InvalidValue.Read(reader);	// graphEnvTable
            Nw4rFileDataAddressRef.InvalidValue.Read(reader);	// randomizerTable

            reader.ReadUInt32();								// 予約済み
        }
    }

    internal class Nw4rWsdTrackTable : Nw4rFileNode
    {
        private Nw4rWaveSound _waveSound = null;

        public Nw4rWsdTrackTable() : base("WaveSoundTrackTable") { }

        public Nw4rWsdTrackTable(Nw4rWaveSound waveSound) : base("WaveSoundTrackTable")
        {
            if (null == waveSound) { throw new ArgumentNullException("waveSound"); }
            _waveSound = waveSound;
        }

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            // トラック情報テーブルの出力
            {
                // トラック数（固定値：１個だけ）
                writer.Write((UInt32)1);

                // トラック情報へのオフセット
                Nw4rFileDataOffsetRef trackInfoReference = new Nw4rFileDataOffsetRef();
                trackInfoReference.ReserveWrite(writer);
                trackInfoReference.Origin = Parent.Parent.Offset;
                trackInfoReference.Commit();
            }

            // トラック情報の出力（１個だけ）
            {
                // ノートイベントテーブルへのオフセットを予約書き込み
                Nw4rFileDataOffsetRef noteEventTableReference = new Nw4rFileDataOffsetRef();
                noteEventTableReference.ReserveWrite(writer);

                // ノートイベントテーブルへのオフセット確定
                noteEventTableReference.Origin = Parent.Parent.Offset;
                noteEventTableReference.Commit();

                // ノートイベントテーブルの出力
                {
                    // ノートイベント数（固定値：１個だけ）
                    writer.Write((UInt32)1);

                    // ノートイベントデータへのオフセット
                    Nw4rFileDataOffsetRef noteEventDataReference = new Nw4rFileDataOffsetRef();
                    noteEventDataReference.ReserveWrite(writer);
                    noteEventDataReference.Origin = Parent.Parent.Offset;
                    noteEventDataReference.Commit();
                }

                // ノートイベントデータの出力
                {
                    writer.Write((Single)0.0f);	// position   (固定値)
                    writer.Write((Single)0.0f);	// length     (固定値)
                    writer.Write((UInt32)0.0f);	// note index (固定値)
                    writer.Write((UInt32)0.0f);	// reserved   (固定値)
                }
            }
        }

        protected override void OnReadBinary(BinaryReader reader)
        {
            // トラック情報テーブルの入力
            {
                // トラック数（固定値：１個だけ）
                uint trackCount = reader.ReadUInt32();
                if (1 != trackCount) { throw new Nw4rFileFormatInternalException("track count is not 1."); }

                // トラック情報へのオフセット
                Nw4rFileDataOffsetRef trackInfoReference = new Nw4rFileDataOffsetRef();
                trackInfoReference.Read(reader);
            }

            // トラック情報の入力（１個だけ）
            {
                // ノートイベントテーブルへのオフセット
                Nw4rFileDataOffsetRef noteEventTableReference = new Nw4rFileDataOffsetRef();
                noteEventTableReference.Read(reader);

                // ノートイベントテーブルの入力
                {
                    // ノートイベント数（固定値：１個だけ）
                    uint noteEventCount = reader.ReadUInt32();
                    if (1 != noteEventCount) { throw new Nw4rFileFormatInternalException("note event count is not 1."); }

                    // ノートイベントデータへのオフセット
                    Nw4rFileDataOffsetRef noteEventDataReference = new Nw4rFileDataOffsetRef();
                    noteEventDataReference.Read(reader);
                }

                // ノートイベントデータの入力
                {
                    reader.ReadSingle();	// position   (固定値)
                    reader.ReadSingle();	// length     (固定値)
                    reader.ReadUInt32();	// note index (固定値)
                    reader.ReadUInt32();	// reserved   (固定値)
                }
            }
        }
    }

    internal class Nw4rWsdNoteTable : Nw4rFileNode
    {
        #region ** 固定値

        public const string NodeName = "WaveSoundNoteTable";

        #endregion

        private Nw4rWaveSound _waveSound = null;

        private Int32 _waveSoundIndex = 0;
        private Byte _envelopeAttack = 0;
        private Byte _envelopeDecay = 0;
        private Byte _envelopeSustain = 0;
        private Byte _envelopeRelease = 0;
        private Byte _envelopeHold = 0;

        public Nw4rWsdNoteTable() : base(NodeName) { }

        public Nw4rWsdNoteTable(Nw4rWaveSound waveSound) : base(NodeName)
        {
            if (null == waveSound) { throw new ArgumentNullException("waveSound"); }
            _waveSound = waveSound;
        }

        #region ** プロパティ

        public new Nw4rWsdDataFile Root
        {
            get { return base.Root as Nw4rWsdDataFile; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnPreWriteBinary(Nw4rWriteFileEventArgs e)
        {
            Update();
        }

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            writer.Write((UInt32)1);		// ノート数（固定値：１個だけ）

            // ノート情報テーブルの出力（１個だけ）
            // オフセット
            Nw4rFileDataOffsetRef noteInfoReference = new Nw4rFileDataOffsetRef();
            noteInfoReference.ReserveWrite(writer);
            noteInfoReference.Origin = Parent.Parent.Offset;
            noteInfoReference.Size = writer.BaseStream.Position - noteInfoReference.Origin;
            noteInfoReference.Commit();

            // ノート情報の出力（１個だけ）
            writer.Write(_waveSoundIndex);						// WaveSound Index
            writer.Write(_envelopeAttack);						// attack
            writer.Write(_envelopeDecay);							// decay
            writer.Write(_envelopeSustain);						// sustain
            writer.Write(_envelopeRelease);						// release
            writer.Write(_envelopeHold);							// hold

            writer.Write(new Byte[3]);							// padding

            writer.Write((Byte)60);								// original key (固定値)
            writer.Write((Byte)127);								// volume       (固定値)
            writer.Write((Byte)64);								// pan          (固定値)
            writer.Write((Byte)0);								// span         (固定値)
            writer.Write((Single)1.0);							// pitch        (固定値)

            Nw4rFileDataAddressRef.InvalidValue.Write(writer);	// lfoTablevRef       (無効値)
            Nw4rFileDataAddressRef.InvalidValue.Write(writer);	// graphEnvTablevRef  (無効値)
            Nw4rFileDataAddressRef.InvalidValue.Write(writer);	// randomizerTableRef (無効値)

            writer.Write((UInt32)0);								// reserved
        }

        protected override void OnReadBinary(BinaryReader reader)
        {
            uint noteCount = reader.ReadUInt32();
            if (1 != noteCount) { throw new Nw4rFileFormatInternalException("note count is not 1."); }

            // ノート情報テーブルの入力（１個だけ）
            // オフセット
            Nw4rFileDataOffsetRef noteInfoReference = new Nw4rFileDataOffsetRef();
            noteInfoReference.Read(reader);

            // ノート情報の入力（１個だけ）
            _waveSoundIndex = reader.ReadInt32();				// WaveSound Index
            _envelopeAttack = reader.ReadByte();				// attack
            _envelopeDecay = reader.ReadByte();				// decay
            _envelopeSustain = reader.ReadByte();				// sustain
            _envelopeRelease = reader.ReadByte();				// release
            _envelopeHold = reader.ReadByte();				// hold

            reader.ReadBytes(3);								// padding

            reader.ReadByte();									// original key (固定値)
            reader.ReadByte();									// volume       (固定値)
            reader.ReadByte();									// pan          (固定値)
            reader.ReadByte();									// span         (固定値)
            reader.ReadSingle();								// pitch        (固定値)

            Nw4rFileDataAddressRef.InvalidValue.Read(reader);	// lfoTablevRef       (無効値)
            Nw4rFileDataAddressRef.InvalidValue.Read(reader);	// graphEnvTablevRef  (無効値)
            Nw4rFileDataAddressRef.InvalidValue.Read(reader);	// randomizerTableRef (無効値)

            reader.ReadUInt32();								// reserved
        }

        #endregion

        #region ** メソッド

        private void Update()
        {
            _waveSoundIndex = Root.WaveFiles.IndexOf(_waveSound.BinaryFile.Key);
            _envelopeAttack = (Byte)_waveSound.XmlData.AdsrEnvelope.Attack;
            _envelopeDecay = (Byte)_waveSound.XmlData.AdsrEnvelope.Decay;
            _envelopeSustain = (Byte)_waveSound.XmlData.AdsrEnvelope.Sustain;
            _envelopeRelease = (Byte)_waveSound.XmlData.AdsrEnvelope.Release;
            _envelopeHold = (Byte)_waveSound.XmlData.AdsrEnvelope.Hold;
        }

        #endregion
    }

    #endregion

    #endregion

    #endregion

    #region ** WaveSound 波形アーカイブファイル

    internal class Nw4rWsdWaveArchiveFile : Nw4rFileRoot
    {
        private static readonly ushort Version = 0x0100;

        public Nw4rWsdWaveArchiveFile(INw4rComponentCollection waveFiles) : base("WaveSoundWaveFileRoot")
        {
            if (null == waveFiles) { throw new ArgumentNullException("waveFiles"); }
            BuildNodes(waveFiles);
        }

        #region ** メソッド

        private void BuildNodes(INw4rComponentCollection waveFiles)
        {
            if (null == waveFiles) { throw new ArgumentNullException("waveFiles"); }

            Nw4rWavTableBlockBody _tableBlockBody = new Nw4rWavTableBlockBody();
            Nw4rWavDataBlockBody _dataBlockBody = new Nw4rWavDataBlockBody(_tableBlockBody, waveFiles);

            Nw4rFileHeader header = new Nw4rFileHeader("FileHeader", "RWAR", Version);
            Nw4rFileDataBlock tableBlock = new Nw4rFileDataBlock("TableBlock", new Nw4rFileDataBlockHeader("TABL"), _tableBlockBody);
            Nw4rFileDataBlock dataBlock = new Nw4rFileDataBlock("DataBlock", new Nw4rFileDataBlockHeader("DATA"), _dataBlockBody);

            header.AddBlockEntry(tableBlock);
            header.AddBlockEntry(dataBlock);

            AddNode(header);
            AddNode(tableBlock);
            AddNode(dataBlock);
        }

        #endregion
    }

    #region ** WaveFile TableBlock

    internal class Nw4rWavTableBlockBody : Nw4rFileNode
    {
        private Nw4rFileInnerLocationRefTable _table = new Nw4rFileInnerLocationRefTable();

        public Nw4rWavTableBlockBody() : base("Body") { }

        #region ** イベントハンドラ

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            base.OnWriteBinary(writer);

            // テーブル出力
            _table.Write(writer);
        }

        protected override void OnPostWriteBinary(Nw4rWriteFileEventArgs e)
        {
            // 32byte アライメント調整
            new Nw4rByteAligner(e.Writer.BaseStream).Pad(32);

            // Nw4rWavDataBlock からのオフセット基点を設定
            _table.OffsetOrigin = e.Writer.BaseStream.Position;

            base.OnPostWriteBinary(e);
        }

        #endregion

        #region ** メソッド

        public void AddEntry(Nw4rFileNode node)
        {
            if (null == node) { throw new ArgumentNullException("node"); }
            _table.AddItem(node);
        }

        #endregion
    }

    #endregion

    #region ** WaveFile DataBlock

    internal class Nw4rWavDataBlockBody : Nw4rFileDataBlockBody
    {
        private Nw4rWavTableBlockBody _tableBlockBody = null;

        public Nw4rWavDataBlockBody(Nw4rWavTableBlockBody tableBlockBody, INw4rComponentCollection waveFiles)
        {
            if (null == tableBlockBody) { throw new ArgumentNullException("tableBlockBody"); }
            if (null == waveFiles) { throw new ArgumentNullException("waveFiles"); }

            _tableBlockBody = tableBlockBody;

            BuildNodes(waveFiles);
        }

        #region ** イベントハンドラ

        protected override void WriteBinary(BinaryWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            // 32byte アライメント調整
            new Nw4rByteAligner(writer.BaseStream).Pad(32);

            FixOffset(writer.BaseStream.Position);

            Nw4rWriteFileEventArgs e = new Nw4rWriteFileEventArgs(writer);

            // 出力
            OnPreWriteBinary(e);
            OnWriteBinary(writer);
            OnWriteChildNodeBinaries(writer);
            OnPostWriteBinary(e);

            FixSize(writer.BaseStream.Position - Offset);
        }

        #endregion

        #region ** メソッド

        private void BuildNodes(INw4rComponentCollection waveFiles)
        {
            Debug.Assert(null != waveFiles);

            foreach (Nw4rSoundFile file in waveFiles)
            {

                Nw4rWavData newNode = new Nw4rWavData(file);

                _tableBlockBody.AddEntry(newNode);
                AddNode(newNode);

            }
        }

        #endregion
    }

    internal class Nw4rWavData : Nw4rFileNode
    {
        private Nw4rSoundFile _waveSoundFile = null;

        public Nw4rWavData(Nw4rSoundFile waveSoundFile) : base("WaveData")
        {
            if (null == waveSoundFile) { throw new ArgumentNullException("waveSoundFile"); }
            _waveSoundFile = waveSoundFile;
        }

        protected override void OnPreWriteBinary(Nw4rWriteFileEventArgs e)
        {
            base.OnPreWriteBinary(e);

            Nw4rWaveFile waveFileRoot = Nw4rWaveFile.FromFile(_waveSoundFile.FilePath);

            // 波形ファイルのバージョン確認
            if (waveFileRoot.Header.Version < 0x0100)
            {
                throw new Nw4rFileFormatException(
                    string.Format("Unsupported Version : Please reconvert all files. (\"{0}\"", _waveSoundFile.FilePath));
            }

            AddNode(waveFileRoot);
        }
    }

    #endregion

    #endregion

    #region ** WaveSound 波形ファイル

    internal class Nw4rWaveFile : Nw4rFileRoot
    {
        private Nw4rFileDataBlock _infoBlock = null;
        private Nw4rFileDataBlock _dataBlock = null;

        public Nw4rWaveFile() : base("WaveFileRoot") { }

        #region ** イベントハンドラ

        protected override void OnPostWriteBinary(Nw4rWriteFileEventArgs e)
        {
            base.OnPostWriteBinary(e);

            if (null == _infoBlock || null == _dataBlock) { throw new Nw4rFileFormatInternalException(); }

            long backupPostion = e.Writer.BaseStream.Position;

            e.Writer.BaseStream.Position = _infoBlock.Offset + 28;
            e.Writer.Write((UInt32)(_dataBlock.Offset - (_infoBlock.Offset + 8)));

            e.Writer.BaseStream.Position = backupPostion;
        }

        #endregion

        #region ** メソッド

        public static Nw4rWaveFile FromFile(string fileName)
        {
            if (null == fileName) { throw new ArgumentNullException("fileName"); }

            Nw4rWaveFile root = new Nw4rWaveFile();
            root.LoadFile(fileName);

            return root;
        }

        protected void LoadFile(string fileName)
        {
            if (null == fileName) { throw new ArgumentNullException("fileName"); }

            using (FileStream stream = File.OpenRead(fileName))
            {
                Read(BinaryReaderBigEndian.CreateInstance(stream));
            }
        }

        protected override Nw4rFileDataBlock ReadDataBlock(BinaryReader reader)
        {
            Nw4rFileDataBlockHeader header = new Nw4rFileDataBlockHeader();
            header.Read(reader);

            Nw4rFileWaveBinaryDataBlock newBlock = new Nw4rFileWaveBinaryDataBlock(header);
            newBlock.ReadBody(reader);

            if ("INFO" == header.Kind)
            {
                _infoBlock = newBlock;
            }
            else if ("DATA" == header.Kind)
            {
                _dataBlock = newBlock;
            }

            return newBlock;
        }

        #endregion
    }

    internal class Nw4rFileWaveBinaryDataBlock : Nw4rFileBinaryDataBlock
    {
        public static readonly string NodeName = "WaveBinaryDataBlock";

        public Nw4rFileWaveBinaryDataBlock(Nw4rFileDataBlockHeader header) : base(NodeName, header) { }

        protected override void OnPostWriteBinary(Nw4rWriteFileEventArgs e)
        {
            base.OnPostWriteBinary(e);

            // 32byte アライメント調整
            new Nw4rByteAligner(e.Writer.BaseStream).Pad(32);
        }
    }

    #endregion
}
