﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.IO;
using System.Text;

using NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Model;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{
    #region ** サウンドアーカイブファイル

    internal class Nw4rSoundArchiveFile : Nw4rFileRoot
    {
        #region ** 固定値

        private static readonly ushort Version = 0x0104;

        #endregion

        #region ** フィールド

        private Nw4rSoundArchive _archive = null;

        #endregion

        public Nw4rSoundArchiveFile(Nw4rSoundArchive archive) : base("SoundArchiveFileRoot")
        {
            if (null == archive) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("archive")); }
            _archive = archive;

            BuildNodes();
        }

        #region ** プロパティ

        public Nw4rSoundArchive SoundArchive
        {
            get { return _archive; }
        }

        public Nw4rSarcInformationBlock InformationBlock
        {
            get { return ChildNodes[Nw4rSarcInformationBlock.NodeName] as Nw4rSarcInformationBlock; }
        }

        #endregion

        #region ** メソッド

        private void BuildNodes()
        {
            Debug.Assert(null != _archive);

            // ヘッダを追加する
            Nw4rFileHeader header = new Nw4rFileHeader("FileHeader", "RSAR", Version);
            AddNode(header);

            // 各ブロックを追加する
            AddBlock(new Nw4rSarcSymbolBlock(_archive));
            AddBlock(new Nw4rSarcInformationBlock(_archive));
            AddBlock(new Nw4rSarcFileDataBlock(_archive, InformationBlock));
        }

        private void AddBlock(Nw4rFileDataBlock block)
        {
            Debug.Assert(null != block);

            // ヘッダにエントリを追加する
            Header.AddBlockEntry(block);

            // ブロックを追加する
            AddNode(block);
        }

        #endregion
    }

    #region ** 各ブロック

    #region ** シンボルブロック

    internal class Nw4rSarcSymbolBlock : Nw4rFileDataBlock
    {
        #region ** 固定値

        public static readonly string NodeName = "SymbolBlock";
        private static readonly string BlockKind = "SYMB";

        #endregion

        public Nw4rSarcSymbolBlock(Nw4rSoundArchive archive) : base(NodeName)
        {
            if (null == archive) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("archive")); }

            AddNode(new Nw4rFileDataBlockHeader(BlockKind));
            AddNode(new Nw4rSarcSymbolBlockBody(archive));
        }
    }

    internal class Nw4rSarcSymbolBlockBody : Nw4rFileDataBlockBody
    {
        private Nw4rSarcSymbolBlockInnerTable _nodeTable = new Nw4rSarcSymbolBlockInnerTable();

        public Nw4rSarcSymbolBlockBody(Nw4rSoundArchive archive)
        {
            if (null == archive) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("archive")); }

            Build(archive);
        }

        #region ** イベントハンドラ

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            // ノードテーブル
            _nodeTable.Write(writer);
        }

        protected override void OnPostWriteBinary(Nw4rWriteFileEventArgs e)
        {
            new Nw4rByteAligner(e.Writer.BaseStream).Pad(32);
        }

        #endregion

        #region ** メソッド

        private void Build(Nw4rSoundArchive archive)
        {
            Debug.Assert(null != archive);

            AddBlockNode(new Nw4rSarcSymbolTable(archive));
            AddBlockNode(new Nw4rSarcPatriciaTreeNode("SoundTree", archive.AllSounds));
            AddBlockNode(new Nw4rSarcPatriciaTreeNode("PlayerTree", archive.AllPlayers));
            AddBlockNode(new Nw4rSarcPatriciaTreeNode("GroupTree", archive.AllGroups));
            AddBlockNode(new Nw4rSarcPatriciaTreeNode("BankTree", archive.AllBanks));
        }

        private void AddBlockNode(Nw4rFileNode node)
        {
            Debug.Assert(null != node);

            _nodeTable.AddItem(node);
            AddNode(node);
        }

        #endregion
    }

    internal class Nw4rSarcSymbolBlockInnerTable : Nw4rFileInnerOffsetTable
    {
        #region ** イベントハンドラ

        protected override void OnWriteHeader(System.IO.BinaryWriter writer)
        {
            // 標準の処理を行わない（アイテム数を出力しない）
        }

        #endregion
    }

    internal class Nw4rSarcSymbolTable : Nw4rFileNode
    {
        #region ** 固定値

        public static readonly string NodeName = "SymbolTable";

        #endregion

        #region ** フィールド

        private Nw4rFileInnerOffsetTable _symbolTable = new Nw4rFileInnerOffsetTable();

        #endregion

        public Nw4rSarcSymbolTable(Nw4rSoundArchive archive) : base(NodeName)
        {
            if (null == archive) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("archive")); }

            Build(archive);
        }

        #region ** イベントハンドラ

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            // シンボルテーブル
            _symbolTable.OffsetOrigin = Parent.Offset;
            _symbolTable.Write(writer);
        }

        protected override void OnPostWriteBinary(Nw4rWriteFileEventArgs e)
        {
            new Nw4rByteAligner(e.Writer.BaseStream).Pad(4);

            base.OnPostWriteBinary(e);
        }

        #endregion

        #region ** メソッド

        private void Build(Nw4rSoundArchive archive)
        {
            Debug.Assert(null != archive);

            foreach (Nw4rSymbol symbol in archive.Symbols)
            {

                Nw4rSarcSymbolTableItem newSymbol = new Nw4rSarcSymbolTableItem(symbol);

                _symbolTable.AddItem(newSymbol);
                AddNode(newSymbol);

            }
        }

        #endregion
    }

    internal class Nw4rSarcSymbolTableItem : Nw4rFileNode
    {
        #region ** 固定値

        public static readonly string NodeName = "Symbol";

        #endregion

        private Nw4rSymbol _symbol = null;

        public Nw4rSarcSymbolTableItem(Nw4rSymbol symbol) : base(NodeName)
        {
            if (null == symbol) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("symbol")); }
            _symbol = symbol;
        }

        #region ** プロパティ

        public Nw4rSymbol Symbol
        {
            get { return _symbol; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            // ラベル名
            writer.Write(_symbol.Item.Label.ToCharArray());
            writer.Write('\0');
        }

        #endregion
    }

    internal class Nw4rSarcPatriciaTreeNode : Nw4rFileNode
    {
        #region ** フィールド

        // パトリシアツリー
        private PatriciaTree _patriciaTree = new PatriciaTree();
        private INw4rComponentCollection _components = null;

        #endregion

        protected Nw4rSarcPatriciaTreeNode(string name) : base(name) { }
        public Nw4rSarcPatriciaTreeNode(string name, INw4rComponentCollection components) : base(name)
        {
            if (null == components) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("components")); }

            _components = components;
            Build(components);
        }

        #region ** イベントハンドラ

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            WriteHeader(writer);
            WriteNodes(writer);

            new Nw4rByteAligner(writer.BaseStream).Pad(4);
        }

        #endregion

        #region ** メソッド

        protected void AddComponent(Model.Nw4rSoundSetItem item)
        {
            Debug.Assert(null != item);
            _patriciaTree.Add(item);
        }

        private void Build(INw4rComponentCollection components)
        {
            Debug.Assert(null != components);

            foreach (Model.Nw4rSoundSetItem component in components)
            {

                if (0 == component.Label.Length)
                {
                    if (!(component is Nw4rGroup)) { throw new Nw4rFileFormatInternalException(); }
                    continue;
                }

                AddComponent(component);

            }
        }

        private void WriteHeader(BinaryWriter writer)
        {
            Debug.Assert(null != writer);

            // ルートノードインデックス
            if (0 == _patriciaTree.Nodes.Count)
            {
                writer.Write((UInt32)0xFFFFFFFF);
            }
            else
            {
                writer.Write((UInt32)_patriciaTree.Root.Index);
            }

            // ノード数
            writer.Write((UInt32)_patriciaTree.Nodes.Count);
        }

        private void WriteNodes(BinaryWriter writer)
        {
            foreach (PatriciaTree.INode node in _patriciaTree.Nodes)
            {
                WriteNode(writer, node);
            }
        }

        private void WriteNode(BinaryWriter writer, PatriciaTree.INode node)
        {
            Debug.Assert(null != writer);
            if (null == node) { return; }

            PatriciaTree.ILeaf leaf = node as PatriciaTree.ILeaf;

            // Leaf フラグ
            writer.Write((UInt16)((null != leaf) ? 1 : 0));

            // ビット
            if (null != leaf)
            {
                writer.Write((UInt16)0xFFFF);
            }
            else
            {
                writer.Write((UInt16)node.Bit);
            }

            writer.Write((UInt32)((null != node.Left) ? (UInt32)node.Left.Index : 0xFFFFFFFF));	// 左ノードのインデックス
            writer.Write((UInt32)((null != node.Right) ? (UInt32)node.Right.Index : 0xFFFFFFFF));	// 右ノードのインデックス

            // シンボルインデックス
            if (null != leaf)
            {
                writer.Write((UInt32)(leaf.Item as Model.Nw4rSoundSetItem).Root.Symbols.IndexOf(leaf.Item.Key));
            }
            else
            {
                writer.Write((UInt32)0xFFFFFFFF);
            }

            // データインデックス
            if (null != leaf)
            {
                writer.Write((UInt32)_components.IndexOf(leaf.Item.Key));
            }
            else
            {
                writer.Write((UInt32)0xFFFFFFFF);
            }
        }

        #endregion
    }

    internal class Nw4rSarcSoundTreeNode : Nw4rSarcPatriciaTreeNode
    {
        #region ** 固定値

        public static readonly string NodeName = "SoundTree";

        #endregion

        public Nw4rSarcSoundTreeNode(Nw4rSoundArchive archive) : base(NodeName)
        {
            if (null == archive) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("archive")); }

            Build(archive);
        }

        #region ** メソッド

        private void Build(Nw4rSoundArchive archive)
        {
            Debug.Assert(null != archive);

            // サウンド
            foreach (Model.Nw4rSoundSet soundSet in archive.AllSoundSets)
            {
                foreach (Nw4rSound sound in soundSet.Sounds)
                {

                    if (0 == sound.Label.Length) { throw new Nw4rFileFormatInternalException(); }
                    AddComponent(sound);

                }
            }
        }

        #endregion
    }

    #endregion

    #region ** 情報ブロック

    internal class Nw4rSarcInformationBlock : Nw4rFileDataBlock
    {
        #region 固定値

        public static readonly string NodeName = "InformationBlock";
        private static readonly string BlockKind = "INFO";

        #endregion

        public Nw4rSarcInformationBlock(Nw4rSoundArchive archive) : base(NodeName)
        {
            if (null == archive) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("archive")); }

            AddNode(new Nw4rFileDataBlockHeader(BlockKind));
            AddNode(new Nw4rSarcInformationBlockBody(archive));
        }

        #region ** プロパティ

        public Nw4rSarcFileInformationSection FileInformationSection
        {
            get { return Body.ChildNodes[Nw4rSarcFileInformationSection.NodeName] as Nw4rSarcFileInformationSection; }
        }

        public Nw4rSarcGroupInformationSection GroupInformationSection
        {
            get { return Body.ChildNodes[Nw4rSarcGroupInformationSection.NodeName] as Nw4rSarcGroupInformationSection; }
        }

        #endregion
    }

    internal class Nw4rSarcInformationBlockBody : Nw4rFileDataBlockBody
    {
        private Nw4rSarcInformationBlockInnerTable _sectionTable = new Nw4rSarcInformationBlockInnerTable();

        public Nw4rSarcInformationBlockBody(Nw4rSoundArchive archive)
        {
            if (null == archive) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("archive")); }

            Build(archive);
        }

        #region ** イベントハンドラ

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            // セクションテーブル
            _sectionTable.Write(writer);

            new Nw4rByteAligner(writer.BaseStream).Pad(4);
        }

        protected override void OnPostWriteBinary(Nw4rWriteFileEventArgs e)
        {
            new Nw4rByteAligner(e.Writer.BaseStream).Pad(32);
        }

        #endregion

        #region ** メソッド

        private void Build(Nw4rSoundArchive archive)
        {
            Debug.Assert(null != archive);

            AddSection(new Nw4rSarcSoundInformationSection(archive));
            AddSection(new Nw4rSarcBankInformationSection(archive));
            AddSection(new Nw4rSarcPlayerInformationSection(archive));
            AddSection(new Nw4rSarcFileInformationSection(archive));
            AddSection(new Nw4rSarcGroupInformationSection(archive));
            AddSection(new Nw4rSarcSarcPlayerInformationSection(archive));
        }

        protected void AddSection(Nw4rFileNode node)
        {
            Debug.Assert(null != node);

            _sectionTable.AddItem(node);
            AddNode(node);
        }

        #endregion
    }

    internal class Nw4rSarcInformationBlockInnerTable : Nw4rFileInnerOffsetRefTable
    {
        #region ** イベントハンドラ

        protected override void OnWriteHeader(System.IO.BinaryWriter writer)
        {
            // 標準の処理を行わない（アイテム数を出力しない）
        }

        #endregion
    }

    internal abstract class Nw4rSarcInformationSection : Nw4rFileNode
    {
        // サウンド情報テーブル
        private Nw4rFileInnerOffsetRefTable _informationTable = new Nw4rFileInnerOffsetRefTable();

        protected Nw4rSarcInformationSection(string name, Nw4rSoundArchive archive) : base(name)
        {
            if (null == archive) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("archive")); }

            Build(archive);
        }

        #region ** イベントハンドラ

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            new Nw4rByteAligner(writer.BaseStream).Pad(4);

            // サウンド情報テーブル
            _informationTable.OffsetOrigin = Parent.Offset;
            _informationTable.Write(writer);
        }

        #endregion

        #region ** メソッド

        protected abstract void Build(Nw4rSoundArchive archive);

        protected void AddFileInformation(Nw4rFileNode node)
        {
            Debug.Assert(null != node);

            _informationTable.AddItem(node);
            AddNode(node);
        }

        #endregion
    }

    #region ** サウンド情報セクション

    internal class Nw4rSarcSoundInformationSection : Nw4rSarcInformationSection
    {
        #region ** 固定値

        public static readonly string NodeName = "SoundInformationSection";

        #endregion

        public Nw4rSarcSoundInformationSection(Nw4rSoundArchive archive) : base(NodeName, archive) { }

        #region ** メソッド

        protected override void Build(Nw4rSoundArchive archive)
        {
            foreach (Model.Nw4rSoundSet soundSet in archive.AllSoundSets)
            {

                foreach (Nw4rStreamSound sound in soundSet.StreamSounds)
                {
                    AddFileInformation(new Nw4rSarcStreamSoundInformation(sound));
                }

                foreach (Nw4rWaveSound sound in soundSet.WaveSounds)
                {
                    AddFileInformation(new Nw4rSarcWaveSoundInformation(sound));
                }

                foreach (Nw4rSequenceSound sound in soundSet.SequenceSounds)
                {
                    AddFileInformation(new Nw4rSarcSequenceSoundInformation(sound));
                }

            }
        }

        #endregion
    }

    internal abstract class Nw4rSarcSoundInformation : Nw4rFileNode
    {
        #region ** 固定値

        public static readonly string NodeName = "SoundInformation";

        #endregion

        #region ** 列挙値

        /// <summary>
        /// サウンド情報の種類
        /// </summary>
        protected enum Type
        {
            Invalid = 0,
            Sequence = 1,
            Stream = 2,
            Wave = 3,
        }

        /// <summary>
        /// パンモード
        /// </summary>
        protected enum PanMode
        {
            PAN_MODE_DUAL = 0,
            PAN_MODE_BALANCE = 1,
        }

        /// <summary>
        /// パンカーブ
        /// </summary>
        protected enum PanCurve
        {
            PAN_CURVE_SQRT = 0,
            PAN_CURVE_SQRT_0DB = 1,
            PAN_CURVE_SQRT_0DB_CLAMP = 2,
            PAN_CURVE_SINCOS = 3,
            PAN_CURVE_SINCOS_0DB = 4,
            PAN_CURVE_SINCOS_0DB_CLAMP = 5,
            PAN_CURVE_LINEAR = 6,
            PAN_CURVE_LINEAR_0DB = 7,
            PAN_CURVE_LINEAR_0DB_CLAMP = 8,
        }

        /// <summary>
        /// 3Dサウンド
        /// </summary>
        [Flags]
        private enum Flag3DSound
        {
            DisableControlVolume = 0x01,
            DisableControlPan = 0x02,
            DisableControlSurroundPan = 0x04,
            DisableControlPriority = 0x08,
        }

        /// <summary>
        /// 3Dカーブ
        /// </summary>
        private enum DecayCurve3DSound
        {
            Log = 1,
            Linear = 2,
        }

        #endregion

        #region ** フィールド

        private Nw4rSound _sound = null;

        #endregion

        protected Nw4rSarcSoundInformation(Nw4rSound sound) : base(NodeName)
        {
            if (null == sound) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("sound")); }
            _sound = sound;
        }

        #region ** プロパティ

        /// <summary>
        /// サウンドアイテムを取得します。
        /// </summary>
        public Nw4rSound Sound
        {
            get { return _sound; }
        }

        /// <summary>
        /// サウンドファイルを取得します。
        /// </summary>
        public virtual Nw4rSoundBinaryFile BinaryFile
        {
            get { return Sound.BinaryFile; }
        }

        protected Nw4rSoundArchive SoundArchive
        {
            get { return _sound.Root; }
        }

        /// <summary>
        /// サウンド情報の種類を取得します。
        /// </summary>
        protected abstract Type SoundType { get; }

        /// <summary>
        /// パンモードを取得します。
        /// </summary>
        protected abstract PanMode SoundPanMode { get; }

        /// <summary>
        /// パンカーブを取得します。
        /// </summary>
        protected abstract PanCurve SoundPanCurve { get; }

        /// <summary>
        /// 3Dサウンドフラグを取得します。
        /// </summary>
        private Flag3DSound Sound3DSoundFlag
        {
            get
            {
                Flag3DSound flag = 0;

                if (!Sound.XmlData.Sound3DParam.VolumeCtrl)
                {
                    flag |= Flag3DSound.DisableControlVolume;
                }

                if (!Sound.XmlData.Sound3DParam.PanCtrl)
                {
                    flag |= Flag3DSound.DisableControlPan;
                }

                if (!Sound.XmlData.Sound3DParam.SurroundPanCtrl)
                {
                    flag |= Flag3DSound.DisableControlSurroundPan;
                }

                if (!Sound.XmlData.Sound3DParam.PriorityCtrl)
                {
                    flag |= Flag3DSound.DisableControlPriority;
                }

                return flag;
            }
        }

        /// <summary>
        /// 3Dカーブを取得します。
        /// </summary>
        private DecayCurve3DSound Sound3DSoundDecayCurve
        {
            get
            {
                switch (Sound.XmlData.Sound3DParam.DecayCurve)
                {
                    case Nw4rDecayCurve.Log:
                        return DecayCurve3DSound.Log;

                    case Nw4rDecayCurve.Linear:
                        return DecayCurve3DSound.Linear;
                }

                throw new Nw4rFileFormatInternalException();
            }
        }

        #endregion

        #region ** イベントハンドラ

        protected sealed override void OnWriteBinary(BinaryWriter writer)
        {
            writer.Write((UInt32)SoundArchive.Symbols.IndexOf(_sound.Key));				// シンボルインデックス
            writer.Write((UInt32)SoundArchive.BinaryFiles.IndexOf(BinaryFile.Key));		// ファイルインデックス
            writer.Write((UInt32)SoundArchive.AllPlayers.IndexOf(Sound.Player.Key));	// プレイヤーインデックス

            // 3Dサウンド情報へのオフセット
            Nw4rFileDataOffsetRef referenceFor3DSoundInfo = new Nw4rFileDataOffsetRef();
            referenceFor3DSoundInfo.Origin = Parent.Parent.Offset;
            referenceFor3DSoundInfo.ReserveWrite(writer);

            writer.Write((Byte)Sound.XmlData.Volume);			// ボリューム
            writer.Write((Byte)Sound.XmlData.PlayerPrio);		// プレイヤープレイオリティ
            writer.Write((Byte)SoundType);					// サウンドの種類
            writer.Write((Byte)Sound.XmlData.RemoteFilter);	// リモコンフィルタ

            // 各種サウンド固有データへのオフセット
            Nw4rFileDataOffsetRef referenceForUniqueData = new Nw4rFileDataOffsetRef((Byte)SoundType);
            referenceForUniqueData.Origin = Parent.Parent.Offset;
            referenceForUniqueData.ReserveWrite(writer);

            writer.Write((UInt32)Sound.XmlData.UserParam);	// ユーザパラメータ1
            writer.Write((UInt32)0);							// ユーザパラメータ2（0を出力）
            writer.Write((Byte)SoundPanMode);					// パンモード
            writer.Write((Byte)SoundPanCurve);				// パンカーブ
            writer.Write((Byte)Sound.XmlData.ActorPlayer);	// アクタープレイヤー
            writer.Write((Byte)0);							// reserved

            // 各種サウンド固有のデータ
            referenceForUniqueData.Commit();
            OnWriteUniqueData(writer);

            // 3Dサウンド情報
            int decayRatio = (int)(Sound.XmlData.Sound3DParam.DecayRatio * 256);
            if (decayRatio > 255)
            {
                decayRatio = 255;
            }
            else if (decayRatio < 0)
            {
                decayRatio = 0;
            }

            referenceFor3DSoundInfo.Commit();
            writer.Write((UInt32)Sound3DSoundFlag);						// 3Dサウンドフラグ
            writer.Write((Byte)Sound3DSoundDecayCurve);					// 3Dカーブ
            writer.Write((Byte)decayRatio);								// 3D減衰率
            writer.Write((Byte)Sound.XmlData.Sound3DParam.DopplerFactor);	// 3Dドップラーファクター
            writer.Write((Byte)0);										// padding
            writer.Write((UInt32)0);										// reserved
        }

        protected abstract void OnWriteUniqueData(BinaryWriter writer);

        #endregion
    }

    internal class Nw4rSarcStreamSoundInformation : Nw4rSarcSoundInformation
    {
        public Nw4rSarcStreamSoundInformation(Nw4rStreamSound soundItem) : base(soundItem) { }

        #region ** プロパティ

        /// <summary>
        /// サウンドデータを取得します。
        /// </summary>
        public new Nw4rStreamSound Sound
        {
            get { return base.Sound as Nw4rStreamSound; }
        }

        /// <summary>
        /// サウンド情報の種類を取得します。
        /// </summary>
        protected override Nw4rSarcSoundInformation.Type SoundType
        {
            get { return Type.Stream; }
        }

        /// <summary>
        /// パンモードを取得します。
        /// </summary>
        protected override PanMode SoundPanMode
        {
            get { return (PanMode)(int)Sound.XmlData.PanMode; }
        }

        /// <summary>
        /// パンカーブを取得します。
        /// </summary>
        protected override PanCurve SoundPanCurve
        {
            get { return (PanCurve)(int)Sound.XmlData.PanCurve; }
        }

        #endregion

        #region ** イベントハンドラ

        /// <summary>
        /// ストリームサウンド固有データを出力します。
        /// </summary>
        /// <param name="writer">出力ストリーム</param>
        protected override void OnWriteUniqueData(BinaryWriter writer)
        {
            writer.Write((UInt32)0);					// 再生開始位置（0固定）
            writer.Write((UInt16)Sound.Channels);		// チャンネル数
            writer.Write((UInt16)Sound.AllocTracks);	// アロケートトラック
            writer.Write((UInt32)0);					// reserved
        }

        #endregion
    }

    internal class Nw4rSarcWaveSoundInformation : Nw4rSarcSoundInformation
    {
        public Nw4rSarcWaveSoundInformation(Nw4rWaveSound sound) : base(sound) { }

        #region ** プロパティ

        /// <summary>
        /// サウンドデータを取得します。
        /// </summary>
        public new Nw4rWaveSound Sound
        {
            get { return base.Sound as Nw4rWaveSound; }
        }

        /// <summary>
        /// サウンドファイルを取得します。
        /// </summary>
        public override Nw4rSoundBinaryFile BinaryFile
        {
            get { return Sound.Parent.BinaryFile; }
        }

        /// <summary>
        /// サウンド情報の種類を取得します。
        /// </summary>
        protected override Nw4rSarcSoundInformation.Type SoundType
        {
            get { return Type.Wave; }
        }

        /// <summary>
        /// パンモードを取得します。
        /// </summary>
        protected override PanMode SoundPanMode
        {
            get { return (PanMode)(int)Sound.XmlData.PanMode; }
        }

        /// <summary>
        /// パンカーブを取得します。
        /// </summary>
        protected override PanCurve SoundPanCurve
        {
            get { return (PanCurve)(int)Sound.XmlData.PanCurve; }
        }

        #endregion

        #region ** イベントハンドラ

        /// <summary>
        /// ウェーブサウンド固有データを出力します。
        /// </summary>
        /// <param name="writer">出力ストリーム</param>
        protected override void OnWriteUniqueData(BinaryWriter writer)
        {
            int index = 0;

            foreach (Nw4rSound sound in Sound.Parent.Components)
            {
                if (sound == Sound) { break; }
                index++;
            }

            writer.Write((UInt32)index);				                            // ウェーブサウンドセット上のインデックス
            writer.Write((UInt32)1);												// アロケートトラック（1固定）
            writer.Write((Byte)Sound.XmlData.ChannelPrio);						// チャンネルプライオリティ
            writer.Write((Byte)(Sound.XmlData.ReleasePrioFixFlag ? 1 : 0));		// リリース時優先度固定
            writer.Write((UInt16)0);												// padding
            writer.Write((UInt32)0);												// reserved
        }

        #endregion
    }

    internal class Nw4rSarcSequenceSoundInformation : Nw4rSarcSoundInformation
    {
        public Nw4rSarcSequenceSoundInformation(Nw4rSequenceSound sound) : base(sound) { }

        #region ** プロパティ

        /// <summary>
        /// サウンドデータを取得します。
        /// </summary>
        public new Nw4rSequenceSound Sound
        {
            get { return base.Sound as Nw4rSequenceSound; }
        }

        /// <summary>
        /// サウンド情報の種類を取得します。
        /// </summary>
        protected override Nw4rSarcSoundInformation.Type SoundType
        {
            get { return Type.Sequence; }
        }

        /// <summary>
        /// パンモードを取得します。
        /// </summary>
        protected override PanMode SoundPanMode
        {
            get { return PanMode.PAN_MODE_DUAL; }
        }

        /// <summary>
        /// パンカーブを取得します。
        /// </summary>
        protected override PanCurve SoundPanCurve
        {
            get { return PanCurve.PAN_CURVE_SQRT; }
        }

        #endregion

        #region ** イベントハンドラ

        /// <summary>
        /// シーケンスサウンド固有データを出力します。
        /// </summary>
        /// <param name="writer">出力ストリーム</param>
        protected override void OnWriteUniqueData(BinaryWriter writer)
        {
            writer.Write((UInt32)Sound.StartOffset);									// 再生開始位置
            writer.Write((Int32)SoundArchive.AllBanks.IndexOf(Sound.Bank.Key));		// バンクのインデックス
            writer.Write((UInt32)Sound.AllocTracks);									// アロケートトラック
            writer.Write((Byte)Sound.XmlData.ChannelPrio);							// チャンネルプライオリティ
            writer.Write((Byte)(Sound.XmlData.ReleasePrioFixFlag ? 1 : 0));			// リリース時優先度固定
            writer.Write((UInt16)0);													// padding
            writer.Write((UInt32)0);													// reserved
        }

        #endregion
    }

    #endregion

    #region ** バンク情報セクション

    internal class Nw4rSarcBankInformationSection : Nw4rSarcInformationSection
    {
        #region ** 固定値

        public static readonly string NodeName = "BankInformationSection";

        #endregion

        public Nw4rSarcBankInformationSection(Nw4rSoundArchive archive) : base(NodeName, archive) { }

        #region ** メソッド

        protected override void Build(Nw4rSoundArchive archive)
        {
            foreach (Nw4rBank bank in archive.AllBanks)
            {
                AddFileInformation(new Nw4rSarcBankInformation(bank));
            }
        }

        #endregion
    }

    internal class Nw4rSarcBankInformation : Nw4rFileNode
    {
        #region ** 固定値

        public static readonly string NodeName = "BankInformation";

        #endregion

        #region ** フィールド

        private Nw4rBank _bank = null;

        #endregion

        public Nw4rSarcBankInformation(Nw4rBank bank) : base(NodeName)
        {
            if (null == bank) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("bank")); }
            _bank = bank;
        }

        #region ** プロパティ

        /// <summary>
        /// サウンドアイテムを取得します。
        /// </summary>
        public Nw4rBank Bank
        {
            get { return _bank; }
        }

        private Nw4rSoundArchive SoundArchive
        {
            get { return _bank.Root; }
        }

        #endregion

        #region ** イベントハンドラ

        protected sealed override void OnWriteBinary(BinaryWriter writer)
        {
            writer.Write((UInt32)SoundArchive.Symbols.IndexOf(_bank.Key));					// シンボルインデックス
            writer.Write((UInt32)SoundArchive.BinaryFiles.IndexOf(Bank.BinaryFile.Key));	// ファイルインデックス
            writer.Write((UInt32)0);															// reserved
        }

        #endregion
    }

    #endregion

    #region ** プレイヤー情報セクション

    internal class Nw4rSarcPlayerInformationSection : Nw4rSarcInformationSection
    {
        #region ** 固定値

        public static readonly string NodeName = "PlayerInformationSection";

        #endregion

        public Nw4rSarcPlayerInformationSection(Nw4rSoundArchive archive) : base(NodeName, archive) { }

        #region ** メソッド

        protected override void Build(Nw4rSoundArchive archive)
        {
            foreach (Nw4rPlayer player in archive.AllPlayers)
            {
                AddFileInformation(new Nw4rSarcPlayerInformation(player));
            }
        }

        #endregion
    }

    internal class Nw4rSarcPlayerInformation : Nw4rFileNode
    {
        #region ** 固定値

        public static readonly string NodeName = "BankInformation";

        #endregion

        #region ** フィールド

        private Nw4rPlayer _player = null;

        #endregion

        public Nw4rSarcPlayerInformation(Nw4rPlayer player) : base(NodeName)
        {
            if (null == player) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("player")); }
            _player = player;
        }

        #region ** プロパティ

        /// <summary>
        /// サウンドデータを取得します。
        /// </summary>
        public Nw4rPlayer Player
        {
            get { return _player; }
        }

        private Nw4rSoundArchive SoundArchive
        {
            get { return _player.Root; }
        }

        #endregion

        #region ** イベントハンドラ

        protected sealed override void OnWriteBinary(BinaryWriter writer)
        {
            writer.Write((UInt32)SoundArchive.AllPlayers.IndexOf(_player.Key));	// シンボルインデックス
            writer.Write((Byte)Player.XmlData.SoundLimit);						// サウンドリミット
            writer.Write((Byte)0);												// padding
            writer.Write((UInt16)0);												// padding
            writer.Write((UInt32)Player.XmlData.HeapSize);						// プレイヤーヒープ
            writer.Write((UInt32)0);												// reserved
        }

        #endregion
    }

    #endregion

    #region ** ファイル情報セクション

    internal class Nw4rSarcFileInformationSection : Nw4rSarcInformationSection
    {
        #region ** 固定値

        public static readonly string NodeName = "FileInformationSection";

        #endregion

        public Nw4rSarcFileInformationSection(Nw4rSoundArchive archive) : base(NodeName, archive) { }

        #region ** メソッド

        protected override void Build(Nw4rSoundArchive archive)
        {
            foreach (Nw4rSoundBinaryFile file in archive.BinaryFiles)
            {
                AddFileInformation(new Nw4rSarcFileInformation(file));
            }
        }

        #endregion
    }

    internal class Nw4rSarcFileInformation : Nw4rFileNode
    {
        #region ** 固定値

        public static readonly string NodeName = "FileInformation";

        #endregion

        #region ** フィールド

        private Nw4rSoundFile _file = null;

        private Nw4rFileInnerOffsetRefTable _filePosTable = new Nw4rFileInnerOffsetRefTable();

        private Nw4rSize _aramFileSize = new Nw4rSize();

        #endregion

        public Nw4rSarcFileInformation(Nw4rSoundFile file) : base(NodeName)
        {
            if (null == file) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("file")); }
            _file = file;

            Build();
        }

        #region ** プロパティ

        /// <summary>
        /// ファイルデータを取得します。
        /// </summary>
        public Nw4rSoundBinaryFile File
        {
            get { return _file as Nw4rSoundBinaryFile; }
        }

        public Nw4rSize ARamFileSize
        {
            get { return _aramFileSize; }
        }

        private long FileSize
        {
            get
            {
                if (null == File.FilePath) { return 0; }
                if (0 == File.FilePath.Length) { return 0; }
                return new FileInfo(File.FilePath).Length;
            }
        }

        #endregion

        #region ** イベントハンドラ

        protected sealed override void OnWriteBinary(BinaryWriter writer)
        {
            _aramFileSize.Size = 0;

            writer.Write((UInt32)FileSize);		// ファイルサイズ
            _aramFileSize.ReserveWrite32(writer);	// ARAMファイルサイズの予約書き込み
            writer.Write((Int32)(-1));			// reserved（-1固定）

            // 外部ファイルパスへのオフセット
            Nw4rFileDataOffsetRef externalFilePathOffset = new Nw4rFileDataOffsetRef();
            if (0 < File.ExtensionFileRelativePath.Length)
            {
                externalFilePathOffset.Origin = Parent.Parent.Offset;
                externalFilePathOffset.ReserveWrite(writer);
            }
            else
            {
                Nw4rFileDataAddressRef.InvalidValue.Write(writer);
            }

            // ファイル位置テーブルへのオフセット
            Nw4rFileDataOffsetRef filePosTableOffset = new Nw4rFileDataOffsetRef();
            filePosTableOffset.Origin = Parent.Parent.Offset;
            filePosTableOffset.ReserveWrite(writer);

            // 外部ファイルパス
            if (0 < File.ExtensionFileRelativePath.Length)
            {
                externalFilePathOffset.Commit();
                writer.Write(File.ExtensionFileRelativePath.ToCharArray());
                new Nw4rByteAligner(writer.BaseStream).Pad(4);
            }

            // ファイル位置テーブル
            filePosTableOffset.Commit();

            _filePosTable.OffsetOrigin = Parent.Parent.Offset;
            _filePosTable.Write(writer);
        }

        #endregion

        #region ** メソッド

        private void Build()
        {
            foreach (Nw4rGroup item in File.Groups)
            {

                Nw4rSarcFilePosition newNode = new Nw4rSarcFilePosition(item);
                _filePosTable.AddItem(newNode);
                AddNode(newNode);

            }
        }

        #endregion
    }

    internal class Nw4rSarcFilePosition : Nw4rFileNode
    {
        #region ** 固定値

        public static readonly string NodeName = "FilePosition";

        #endregion

        private Nw4rGroup _group = null;

        public Nw4rSarcFilePosition(Nw4rGroup group) : base(NodeName)
        {
            Debug.Assert(null != group);
            _group = group;
        }

        #region ** プロパティ

        public Nw4rGroup Group
        {
            get { return _group; }
        }

        private Nw4rSoundBinaryFile File
        {
            get { return (Parent as Nw4rSarcFileInformation).File; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            Nw4rSoundArchive archive = (Root as Nw4rSoundArchiveFile).SoundArchive;
            Debug.Assert(null != archive);

            Nw4rGroup group = archive.AllGroups[Group.Label];
            Nw4rSoundBinaryFile file = Group.Files[File.Key] as Nw4rSoundBinaryFile;

            writer.Write((Int32)archive.AllGroups.IndexOf(group.Key));
            writer.Write((UInt32)Group.Files.IndexOf(file.Key));
        }

        #endregion
    }

    #endregion

    #region ** グループ情報セクション

    internal class Nw4rSarcGroupInformationSection : Nw4rSarcInformationSection
    {
        #region ** 固定値

        public static readonly string NodeName = "GroupInformationSection";

        #endregion

        public Nw4rSarcGroupInformationSection(Nw4rSoundArchive archive) : base(NodeName, archive) { }

        #region ** メソッド

        protected override void Build(Nw4rSoundArchive archive)
        {
            foreach (Nw4rGroup group in archive.AllGroups)
            {
                AddFileInformation(new Nw4rSarcGroupInformation(group));
            }
        }

        #endregion
    }

    internal class Nw4rSarcGroupInformation : Nw4rFileNode
    {
        #region ** 固定値

        public static readonly string NodeName = "GroupInformation";

        #endregion

        #region ** フィールド

        private Nw4rGroup _group = null;

        //
        private Nw4rFileInnerOffsetRefTable _itemInfoTable = new Nw4rFileInnerOffsetRefTable();

        private Nw4rLocation _imageLocation = new Nw4rLocation();
        private Nw4rLocation _aramImageLocation = new Nw4rLocation();

        #endregion

        public Nw4rSarcGroupInformation(Nw4rGroup group) : base(NodeName)
        {
            if (null == group) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("group")); }
            _group = group;

            Build();
        }

        #region ** プロパティ

        /// <summary>
        /// サウンドデータを取得します。
        /// </summary>
        public Nw4rGroup Group
        {
            get { return _group; }
        }

        public Nw4rLocation ImageLocation
        {
            get { return _imageLocation; }
        }

        public Nw4rLocation ARamImageLocation
        {
            get { return _aramImageLocation; }
        }

        private Nw4rSoundArchive SoundArchive
        {
            get { return _group.Root; }
        }

        private uint SymbolIndex
        {
            get
            {
                if (0 == Group.Key.Length) { return unchecked((uint)(-1)); }
                return (uint)SoundArchive.Symbols.IndexOf(Group.Key);
            }
        }

        #endregion

        #region ** イベントハンドラ

        protected sealed override void OnWriteBinary(BinaryWriter writer)
        {
            writer.Write((UInt32)SymbolIndex);						// シンボルインデックス
            writer.Write((Int32)(-1));								// reserved（-1固定）

            Nw4rFileDataAddressRef.InvalidValue.Write(writer);		// 外部ファイルパスへのオフセット（現在は未使用）

            _imageLocation.ReserveWrite32(writer);					// イメージへのオフセット
            _aramImageLocation.ReserveWrite32(writer);				// ARAMイメージへのオフセット

            // グループアイテム情報テーブルへのオフセット
            Nw4rFileDataOffsetRef itemInfoTableOffset = new Nw4rFileDataOffsetRef();
            itemInfoTableOffset.Origin = Parent.Parent.Offset;
            itemInfoTableOffset.ReserveWrite(writer);
            itemInfoTableOffset.Commit();

            // グループアイテム情報テーブル
            _itemInfoTable.OffsetOrigin = Parent.Parent.Offset;
            _itemInfoTable.Write(writer);
        }

        #endregion

        #region ** メソッド

        private void Build()
        {
            foreach (Nw4rSoundBinaryFile file in Group.Files)
            {

                Nw4rSarcGroupItemInformation newNode = new Nw4rSarcGroupItemInformation(file);

                _itemInfoTable.AddItem(newNode);
                AddNode(newNode);

            }
        }

        #endregion
    }

    internal class Nw4rSarcGroupItemInformation : Nw4rFileNode
    {
        #region ** 固定値

        public static readonly string NodeName = "GroupItemInformation";

        #endregion

        #region ** フィールド

        private Nw4rSoundFile _file = null;

        private Nw4rLocation _imageLocation = new Nw4rLocation();
        private Nw4rLocation _aramImageLocation = new Nw4rLocation();

        #endregion

        public Nw4rSarcGroupItemInformation(Nw4rSoundFile file) : base(NodeName)
        {
            if (null == file) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("file")); }
            _file = file;
        }

        #region ** プロパティ

        /// <summary>
        /// ファイルデータを取得します。
        /// </summary>
        public Nw4rSoundBinaryFile File
        {
            get { return _file as Nw4rSoundBinaryFile; }
        }

        public Nw4rLocation ImageLocation
        {
            get { return _imageLocation; }
        }

        public Nw4rLocation ARamImageLocation
        {
            get { return _aramImageLocation; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            Nw4rSoundArchive archive = (Root as Nw4rSoundArchiveFile).SoundArchive;
            Debug.Assert(null != archive);

            Nw4rSoundBinaryFile file = archive.BinaryFiles[File.Key];

            writer.Write((UInt32)archive.BinaryFiles.IndexOf(file.Key));	// ファイルインデックス

            _imageLocation.ReserveWrite32(writer);		// イメージへのオフセット
            _aramImageLocation.ReserveWrite32(writer);	// ARAMイメージへのオフセット

            writer.Write((UInt32)0);
        }

        #endregion
    }

    #endregion

    #region ** サウンドアーカイブプレイヤー情報セクション

    internal class Nw4rSarcSarcPlayerInformationSection : Nw4rFileNode
    {
        #region ** 固定値

        public static readonly string NodeName = "SarcPlayerInformationSection";

        #endregion

        private Nw4rSoundArchive _archive;

        public Nw4rSarcSarcPlayerInformationSection(Nw4rSoundArchive archive) : base(NodeName)
        {
            if (null == archive) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("archive")); }
            _archive = archive;
        }

        #region ** メソッド

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            writer.Write((UInt16)_archive.SoundArchivePlayerSettings.SeqSoundCount);		// 同時に再生可能なシーケンスサウンド数
            writer.Write((UInt16)_archive.SoundArchivePlayerSettings.SeqTrackCount);		// 同時に再生可能なシーケンスサウンドのトラック数
            writer.Write((UInt16)_archive.SoundArchivePlayerSettings.StrmSoundCount);		// 同時に再生可能なストリームサウンド数
            writer.Write((UInt16)_archive.SoundArchivePlayerSettings.StrmSoundCount);		// 同時に再生可能なストリームサウンドのトラック数
            writer.Write((UInt16)_archive.SoundArchivePlayerSettings.StrmChannelCount);	// 同時に再生可能なストリームサウンドのチャンネル数
            writer.Write((UInt16)_archive.SoundArchivePlayerSettings.WaveSoundCount);		// 同時に再生可能なウェーブサウンド数
            writer.Write((UInt16)_archive.SoundArchivePlayerSettings.WaveSoundCount);		// 同時に再生可能なウェーブサウンドのトラック数

            writer.Write((UInt16)0);	// padding
            writer.Write((UInt32)0);	// reserved
        }

        #endregion
    }

    #endregion

    #endregion

    #region ** ファイルデータブロック

    internal class Nw4rSarcFileDataBlock : Nw4rFileDataBlock
    {
        #region ** 固定値

        public static readonly string NodeName = "FileDataBlock";
        private static readonly string BlockKind = "FILE";

        #endregion

        public Nw4rSarcFileDataBlock(Nw4rSoundArchive archive, Nw4rSarcInformationBlock infoBlock) : base(NodeName)
        {
            if (null == archive) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("archive")); }

            AddNode(new Nw4rFileDataBlockHeader(BlockKind));
            AddNode(new Nw4rSarcFileDataBlockBody(archive, infoBlock));
        }

        #region ** プロパティ

        public new Nw4rSarcFileDataBlockBody Body
        {
            get { return base.Body as Nw4rSarcFileDataBlockBody; }
        }

        #endregion
    }

    internal class Nw4rSarcFileDataBlockBody : Nw4rFileDataBlockBody
    {
        public Nw4rSarcFileDataBlockBody(Nw4rSoundArchive archive, Nw4rSarcInformationBlock infoBlock)
        {
            if (null == archive) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("archive")); }
            if (null == infoBlock) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("infoBlock")); }

            Build(archive, infoBlock);
        }

        #region ** イベントハンドラ

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            new Nw4rByteAligner(writer.BaseStream).Pad(32);
        }

        #endregion

        #region ** メソッド

        private void Build(Nw4rSoundArchive archive, Nw4rSarcInformationBlock infoBlock)
        {
            foreach (Nw4rGroup group in archive.AllGroups)
            {
                AddNode(new Nw4rSarcGroupImageBlock(group, infoBlock));
            }
        }

        #endregion
    }

    internal class Nw4rSarcGroupImageBlock : Nw4rFileDataBlockBody
    {
        public Nw4rSarcGroupImageBlock(Nw4rGroup group, Nw4rSarcInformationBlock infoBlock)
        {
            if (null == group) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("group")); }
            if (null == infoBlock) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("infoBlock")); }

            Build(group, infoBlock);
        }

        #region ** プロパティ

        public Nw4rSarcGroupMainImage MainImages
        {
            get { return ChildNodes[0] as Nw4rSarcGroupMainImage; }
        }

        public Nw4rSarcGroupARamImage ARamImages
        {
            get { return ChildNodes[1] as Nw4rSarcGroupARamImage; }
        }

        #endregion

        #region ** メソッド

        private void Build(Nw4rGroup group, Nw4rSarcInformationBlock infoBlock)
        {
            Debug.Assert(null != group);
            Debug.Assert(null != group.Root);
            Debug.Assert(null != infoBlock);

            AddNode(new Nw4rSarcGroupMainImage(group, infoBlock));
            AddNode(new Nw4rSarcGroupARamImage(group, infoBlock));
        }

        #endregion
    }

    internal abstract class Nw4rSarcGroupRamImage : Nw4rFileNode
    {
        #region ** 固定値

        public static readonly string NodeName = "GroupImage";

        #endregion

        #region ** フィールド

        private Nw4rGroup _group;
        private Nw4rSarcInformationBlock _infoBlock;

        #endregion

        protected Nw4rSarcGroupRamImage(Nw4rGroup group, Nw4rSarcInformationBlock infoBlock)
            : base(NodeName)
        {
            if (null == group) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("group")); }
            if (null == infoBlock) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("infoBlock")); }

            _group = group;
            _infoBlock = infoBlock;

            Build(group.Root);
        }

        #region ** プロパティ

        public Nw4rGroup Group
        {
            get { return _group; }
        }

        public Nw4rSarcInformationBlock InformationBlock
        {
            get { return _infoBlock; }
        }

        public Nw4rSarcGroupInformation GroupInformation
        {
            get { return _infoBlock.GroupInformationSection.ChildNodes[GroupIndex] as Nw4rSarcGroupInformation; }
        }

        private Nw4rSoundArchive SoundArchive
        {
            get { return _group.Root; }
        }

        private int GroupIndex
        {
            get { return SoundArchive.AllGroups.IndexOf(_group.Key); }
        }

        #endregion

        #region ** メソッド

        protected abstract void Build(Nw4rSoundArchive archive);

        protected Nw4rSarcFileInformation GetFileInformation(Nw4rSoundBinaryFile file)
        {
            Debug.Assert(null != file);
            return _infoBlock.FileInformationSection.ChildNodes[SoundArchive.BinaryFiles.IndexOf(file.Key)] as Nw4rSarcFileInformation;
        }

        protected Nw4rSarcGroupItemInformation GetGroupItemInformation(Nw4rSoundBinaryFile file)
        {
            Debug.Assert(null != GroupInformation);
            Debug.Assert(null != file);
            return GroupInformation.ChildNodes[Group.Files.IndexOf(file.Key)] as Nw4rSarcGroupItemInformation;
        }

        #endregion
    }

    internal class Nw4rSarcGroupMainImage : Nw4rSarcGroupRamImage
    {
        public Nw4rSarcGroupMainImage(Nw4rGroup group, Nw4rSarcInformationBlock infoBlock)
            : base(group, infoBlock) { }

        #region ** イベントハンドラ

        protected override void OnPreWriteBinary(Nw4rWriteFileEventArgs e)
        {
            base.OnPreWriteBinary(e);

            GroupInformation.ImageLocation.Origin = 0;
            GroupInformation.ImageLocation.Head = e.Writer.BaseStream.Position;
        }

        protected override void OnPostWriteBinary(Nw4rWriteFileEventArgs e)
        {
            base.OnPostWriteBinary(e);

            GroupInformation.ImageLocation.Commit();
        }

        #endregion

        #region ** メソッド

        protected override void Build(Nw4rSoundArchive archive)
        {
            Debug.Assert(null != archive);
            Debug.Assert(null != Group);

            foreach (Nw4rSoundBinaryFile file in Group.Files)
            {

                Nw4rSarcGroupItemMainImage mainImage = new Nw4rSarcGroupItemMainImage(file);
                mainImage.GroupItemInformationNodes.Add(GetGroupItemInformation(file));

                AddNode(mainImage);

            }
        }

        #endregion
    }

    internal class Nw4rSarcGroupARamImage : Nw4rSarcGroupRamImage
    {
        public Nw4rSarcGroupARamImage(Nw4rGroup group, Nw4rSarcInformationBlock infoBlock)
            : base(group, infoBlock) { }

        #region ** イベントハンドラ

        protected override void OnPreWriteBinary(Nw4rWriteFileEventArgs e)
        {
            base.OnPreWriteBinary(e);

            GroupInformation.ARamImageLocation.Origin = 0;
            GroupInformation.ARamImageLocation.Head = e.Writer.BaseStream.Position;
        }

        protected override void OnPostWriteBinary(Nw4rWriteFileEventArgs e)
        {
            base.OnPostWriteBinary(e);

            GroupInformation.ARamImageLocation.Commit();
        }

        #endregion

        #region ** メソッド

        protected override void Build(Nw4rSoundArchive archive)
        {
            Debug.Assert(null != archive);
            Debug.Assert(null != Group);


            Nw4rSarcGroupItemARamImage groupARamImage = null;

            if (null != Group.BinaryFile)
            {
                groupARamImage = new Nw4rSarcGroupItemARamImage(Group.BinaryFile);
            }

            foreach (Nw4rSoundBinaryFile file in Group.Files)
            {

                if (file == Group.BinaryFile) { continue; }

                Nw4rSarcGroupItemARamImage groupItemARamImage = new Nw4rSarcGroupItemARamImage(file);

                if (null != groupARamImage &&
                    file is Nw4rSoundBinaryFileForGroup &&
                    (file as Nw4rSoundBinaryFileForGroup).OriginalFile.Groups.Contains(Group.Key))
                {
                    groupARamImage.FileInformationNodes.Add(GetFileInformation(file));
                    groupARamImage.GroupItemInformationNodes.Add(GetGroupItemInformation(file));
                }
                else
                {
                    groupItemARamImage.FileInformationNodes.Add(GetFileInformation(file));
                    groupItemARamImage.GroupItemInformationNodes.Add(GetGroupItemInformation(file));
                }

                AddNode(groupItemARamImage);

            }

            if (null != groupARamImage)
            {
                AddNode(groupARamImage);
            }
        }

        #endregion
    }

    internal abstract class Nw4rSarcGroupItemRamImage : Nw4rFileNode
    {
        #region ** 固定値

        public static readonly string NodeName = "GroupItemImage";

        private const int ReadBlockSize = 1024 * 4;	// ファイルイメージ 入力ブロックサイズ

        #endregion

        #region ** フィールド

        private Nw4rSoundFile _file;
        private Nw4rSarcFileInformationCollection _fileInfoNodes = new Nw4rSarcFileInformationCollection();
        private Nw4rSarcGroupItemInformationCollection _groupItemInfoNodes = new Nw4rSarcGroupItemInformationCollection();

        #endregion

        public Nw4rSarcGroupItemRamImage(Nw4rSoundFile file) : base(NodeName)
        {
            if (null == file) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("file")); }
            _file = file;
        }

        #region ** プロパティ

        public Nw4rSoundBinaryFile BinaryFile
        {
            get { return _file as Nw4rSoundBinaryFile; }
        }

        public Nw4rSarcFileInformationCollection FileInformationNodes
        {
            get { return _fileInfoNodes; }
        }

        public Nw4rSarcGroupItemInformationCollection GroupItemInformationNodes
        {
            get { return _groupItemInfoNodes; }
        }

        #endregion

        #region ** イベントハンドラ

        protected sealed override void OnWriteBinary(BinaryWriter writer)
        {
            Nw4rSoundArchive archive = (Root as Nw4rSoundArchiveFile).SoundArchive;
            Debug.Assert(null != archive);

            OnWriteImage(writer);

            new Nw4rByteAligner(writer.BaseStream).Pad(32);
        }

        protected abstract void OnWriteImage(BinaryWriter writer);

        #endregion

        #region ** メソッド

        protected void WriteFile(BinaryWriter writer, string filePath)
        {
            Debug.Assert(null != writer);
            Debug.Assert(null != filePath);

            if (!File.Exists(filePath)) { throw new FileNotFoundException(string.Empty, filePath); }

            try
            {

                using (FileStream stream = File.Open(filePath, FileMode.Open, FileAccess.Read, FileShare.Read))
                {

                    BinaryReader reader = new BinaryReader(stream);

                    while (reader.BaseStream.Position < reader.BaseStream.Length)
                    {
                        writer.Write(reader.ReadBytes(ReadBlockSize));
                    }

                }

            }
            catch (Exception exception)
            {
                throw new Nw4rFileFormatInternalException(exception);
            }
        }

        #endregion

        #region ** コレクションクラス

        public class Nw4rSarcFileInformationCollection : Collection<Nw4rSarcFileInformation> { }

        public class Nw4rSarcGroupItemInformationCollection : Collection<Nw4rSarcGroupItemInformation> { }

        #endregion
    }

    internal class Nw4rSarcGroupItemMainImage : Nw4rSarcGroupItemRamImage
    {
        public Nw4rSarcGroupItemMainImage(Nw4rSoundFile file) : base(file) { }

        #region ** メソッド

        protected override void OnWriteImage(BinaryWriter writer)
        {
            if (0 == BinaryFile.FilePath.Length) { return; }

            foreach (Nw4rSarcGroupItemInformation node in GroupItemInformationNodes)
            {
                node.ImageLocation.Origin = Parent.Offset;
                node.ImageLocation.Head = writer.BaseStream.Position - node.ImageLocation.Origin;
            }

            try
            {
                WriteFile(writer, BinaryFile.FilePath);
            }
            catch (FileNotFoundException exception)
            {
                if (0 == BinaryFile.Groups.Count) { throw exception; }
                throw new Nw4rFileNotFoundException(exception.FileName, BinaryFile.Groups[0].Label);
            }

            foreach (Nw4rSarcGroupItemInformation node in GroupItemInformationNodes)
            {
                node.ImageLocation.Size = writer.BaseStream.Position - node.ImageLocation.Begin;
                node.ImageLocation.Commit();
            }
        }

        #endregion
    }

    internal class Nw4rSarcGroupItemARamImage : Nw4rSarcGroupItemRamImage
    {
        public Nw4rSarcGroupItemARamImage(Nw4rSoundFile file) : base(file) { }

        #region ** メソッド

        protected override void OnWriteImage(BinaryWriter writer)
        {
            if (0 == BinaryFile.ARamFilePath.Length) { return; }

            long aramFileOffset = writer.BaseStream.Position - Parent.Offset;

            try
            {
                WriteFile(writer, BinaryFile.ARamFilePath);
            }
            catch (FileNotFoundException exception)
            {
                if (0 == BinaryFile.Groups.Count) { throw exception; }
                throw new Nw4rFileNotFoundException(exception.FileName, BinaryFile.Groups[0].Label);
            }


            long aramFileSize = writer.BaseStream.Position - (aramFileOffset + Parent.Offset);

            // ファイル情報セクションのファイル情報に確定したARAMデータサイズを出力する
            foreach (Nw4rSarcFileInformation node in FileInformationNodes)
            {

                // 最初のグループのARAMサイズを設定する
                if (0 != node.ARamFileSize.Size) { continue; }

                node.ARamFileSize.Size = aramFileSize;
                node.ARamFileSize.Commit();

            }

            // グループ情報セクションのグループアイテム情報に確定したARAMデータ位置情報を出力する
            foreach (Nw4rSarcGroupItemInformation node in GroupItemInformationNodes)
            {

                node.ARamImageLocation.Origin = Parent.Offset;
                node.ARamImageLocation.Head = aramFileOffset;
                node.ARamImageLocation.Size = aramFileSize;
                node.ARamImageLocation.Commit();

            }
        }

        #endregion
    }

    #endregion

    #endregion

    #endregion
}


