﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;
using System.IO;
using System.Text;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{
    public class Nw4rSpecialFolder
    {
        private static readonly string NW4R_ROOT = "NW4R_ROOT";

        public static string CommandLineTools
        {
            get
            {
                IDictionary envVariables = Environment.GetEnvironmentVariables();

                if (!envVariables.Contains(NW4R_ROOT))
                {
                    throw new Exception("Undefined environment variable \"" + NW4R_ROOT + "\"");
                }

                return Path.Combine(envVariables[NW4R_ROOT].ToString(), @"CommandLineTools\bin");
            }
        }
    }

    internal class Nw4rFileWriteBack
    {
        public const long InvalidValue = -1;

        private BinaryWriter _writer = null;
        private long _writePosition = InvalidValue;
        private int _reservedSize = 0;

        #region ** プロパティ

        public BinaryWriter Writer
        {
            get { return _writer; }
        }

        public long WritePosition
        {
            get { return _writePosition; }
        }

        public int ReservedSize
        {
            get { return _reservedSize; }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// ストリームの現在位置に値を書き込むために2byteを予約します。
        /// ReserveWrite では仮の値(2byte)が書き込まれ、Commit 実行時に正しい値が書き込まれます。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        public void ReserveWrite16(BinaryWriter writer)
        {
            ReserveWrite(writer, 2);
        }

        /// <summary>
        /// ストリームの現在位置に値を書き込むために4byteを予約します。
        /// ReserveWrite では仮の値(4byte)が書き込まれ、Commit 実行時に正しい値が書き込まれます。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        public void ReserveWrite32(BinaryWriter writer)
        {
            ReserveWrite(writer, 4);
        }

        /// <summary>
        /// ストリームの現在位置に値を書き込むために予約します。
        /// ReserveWrite では仮の値が書き込まれ、Commit 実行時に正しい値が書き込まれます。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <param name="size">予約サイズ</param>
        public void ReserveWrite(BinaryWriter writer, int size)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }
            if (0 >= size) { throw new ArgumentOutOfRangeException("size"); }

            _writePosition = writer.BaseStream.Position;
            writer.Write(new Byte[size]);

            _writer = writer;
            _reservedSize = size;
        }

        /// <summary>
        /// ReserveWrite にて予約した場所に 指定値を書き込みます。
        /// </summary>
        /// <param name="stream">対象ストリーム</param>
        public void Commit(UInt16 value)
        {
            if (null == _writer) { throw new Nw4rFileFormatInternalException(); }
            if (InvalidValue == _writePosition) { throw new Nw4rFileFormatInternalException(); }
            if (2 != _reservedSize) { throw new Nw4rFileFormatInternalException(); }

            long currentPosition = _writer.BaseStream.Position;
            _writer.BaseStream.Position = _writePosition;

            _writer.Write(value);

            _writer.BaseStream.Position = currentPosition;
        }

        /// <summary>
        /// ReserveWrite にて予約した場所に 指定値を書き込みます。
        /// </summary>
        /// <param name="stream">対象ストリーム</param>
        public void Commit(UInt32 value)
        {
            if (null == _writer) { throw new Nw4rFileFormatInternalException(); }
            if (InvalidValue == _writePosition) { throw new Nw4rFileFormatInternalException(); }
            if (4 != _reservedSize) { throw new Nw4rFileFormatInternalException(); }

            long currentPosition = _writer.BaseStream.Position;
            _writer.BaseStream.Position = _writePosition;

            _writer.Write(value);

            _writer.BaseStream.Position = currentPosition;
        }

        #endregion
    }

    internal class Nw4rSize : Nw4rFileWriteBack
    {
        private long _origin = 0;				// 基点
        private long _size = InvalidValue;	// 基点からのサイズ

        #region ** プロパティ

        public long Origin
        {
            get { return _origin; }
            set
            {
                if (0 > value) { throw new ArgumentOutOfRangeException("value"); }
                _origin = value;
            }
        }

        public long Size
        {
            get
            {
                if (null != Writer && InvalidValue == _size)
                {
                    if (InvalidValue == _origin) { throw new Nw4rFileFormatInternalException(); }
                    if (Writer.BaseStream.Position < _origin) { throw new Nw4rFileFormatInternalException(); }
                    return Writer.BaseStream.Position - _origin;
                }
                return _size;
            }
            set
            {
                if (0 > value) { throw new ArgumentOutOfRangeException("value"); }
                _size = value;
            }
        }

        public long Begin
        {
            get { return _origin; }
        }

        public long End
        {
            get
            {
                if (InvalidValue == Origin) { throw new Nw4rFileFormatInternalException(); }
                if (InvalidValue == Size) { throw new Nw4rFileFormatInternalException(); }
                if (Size < Origin) { throw new Nw4rFileFormatInternalException(); }
                return Origin + Size;
            }
        }

        #endregion

        #region ** メソッド

        public void Read32(BinaryReader reader)
        {
            if (null == reader) { throw new ArgumentNullException("reader"); }
            _size = reader.ReadUInt32();
        }

        /// <summary>
        /// ReserveWrite にて予約した場所にオフセット値(BeginPosition-EndPosition)を書き込みます。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        public void Commit()
        {
            switch (ReservedSize)
            {
                case 2:
                    Commit((UInt16)Size);
                    break;

                case 4:
                    Commit((UInt32)Size);
                    break;
            }
        }

        #endregion
    }

    internal class Nw4rLocation
    {
        #region ** パラメータ

        private Nw4rSize _offset = new Nw4rSize();	// オフセット
        private Nw4rSize _size = new Nw4rSize();	// サイズ

        #endregion

        #region ** プロパティ

        public long Origin
        {
            get { return _offset.Origin; }
            set { _offset.Origin = value; }
        }

        public long Head
        {
            get { return _offset.Size; }
            set
            {
                _offset.Size = value;
                _size.Origin = Origin + value;
            }
        }

        public long Size
        {
            get { return _size.Size; }
            set { _size.Size = value; }
        }

        public long Begin
        {
            get { return Origin + Head; }
        }

        public long End
        {
            get { return Begin + Size; }
        }

        #endregion

        #region ** メソッド

        static public Nw4rLocation FromStream(BinaryReader reader)
        {
            if (null == reader) { throw new ArgumentNullException("reader"); }

            Nw4rLocation newLocation = new Nw4rLocation();
            newLocation.Read32(reader);

            return newLocation;
        }

        public void Read32(BinaryReader reader)
        {
            if (null == reader) { throw new ArgumentNullException("reader"); }
            _offset.Read32(reader);
            _size.Read32(reader);
        }

        /// <summary>
        /// オフセット値を書き込みます。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <param name="offset">オフセット値</param>
        /// <param name="size">サイズ</param>
        public void Write32(BinaryWriter writer, UInt32 offset, UInt32 size)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            // 値の書き込み
            writer.Write(offset);	// オフセット
            writer.Write(size);	// サイズ
        }

        /// <summary>
        /// ストリームの現在位置に値を書き込むために予約します。
        /// ReserveWrite では仮の値が書き込まれ、Commit 実行時に正しい値が書き込まれます。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        public void ReserveWrite32(BinaryWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            // 値の書き込み予約
            _offset.ReserveWrite32(writer);	// オフセット
            _size.ReserveWrite32(writer);		// サイズ
        }

        /// <summary>
        /// ReserveWrite にて予約した場所にオフセット値(BeginPosition-EndPosition)を書き込みます。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        public void Commit()
        {
            _offset.Commit();
            _size.Commit();
        }

        #endregion
    }

    internal class Nw4rByteAligner
    {
        private Stream _stream = null;
        private long _origin = 0;

        public Nw4rByteAligner(Stream stream)
        {
            _stream = stream;
        }

        public Stream Stream
        {
            get { return _stream; }
        }

        public long Origin
        {
            get { return _origin; }
            set
            {
                if (0 > value) { throw new ArgumentOutOfRangeException("value"); }
                _origin = value;
            }
        }

        public void SeekForward(int alignment)
        {
            if (0 >= alignment) { throw new ArgumentOutOfRangeException("alignment"); }
            if (null == _stream) { throw new Exception("internal exception"); }
            if (-1 == _origin) { throw new Exception("internal exception"); }
            if (_stream.Position < _origin) { throw new Exception("internal exception"); }

            int remain = (int)(_stream.Position - _origin) % alignment;
            if (0 == remain) { return; }

            int paddingSize = alignment - remain;
            _stream.Seek(paddingSize, SeekOrigin.Current);

            _origin = -1;
        }

        public void Pad(int alignment)
        {
            if (0 >= alignment) { throw new ArgumentOutOfRangeException("alignment"); }
            if (null == _stream) { throw new Exception("internal exception"); }
            if (-1 == _origin) { throw new Exception("internal exception"); }
            if (_stream.Position < _origin) { throw new Exception("internal exception"); }

            int remain = (int)(_stream.Position - _origin) % alignment;
            if (0 == remain) { return; }

            int paddingSize = alignment - remain;
            _stream.Write(new Byte[paddingSize], 0, paddingSize);

            _origin = -1;
        }

    }
}
