﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.IO;
using System.Xml;
using System.Text;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{
    #region ** ファイルノード基本クラス

    /// <summary>
    /// Nw4rファイル ノード
    /// </summary>
    internal class Nw4rFileNode : IEnumerable<Nw4rFileNode>, IEnumerable
    {
        #region ** 定数

        public const int InvalidIndex = -1;
        public const long InvalidLength = -1;

        #endregion

        #region ** パラメータ

        // リンク
        private Nw4rFileNode _parent = null;
        private Nw4rFileNodeCollectionImpl _nodes = new Nw4rFileNodeCollectionImpl();

        // パラメータ
        private bool _numbered = false;
        private int _index = InvalidIndex;
        private string _name = string.Empty;
        private long _offset = InvalidLength;
        private long _size = InvalidLength;

        #endregion

        public Nw4rFileNode(string name)
        {
            if (null == name) { throw new ArgumentNullException("name"); }
            _name = name;
        }

        #region ** プロパティ

        public int Index
        {
            get
            {
                if (null == _parent) { return InvalidIndex; }

                if (!_parent._numbered)
                {
                    _parent.Number();
                }

                Debug.Assert(InvalidIndex != _index);
                return _index;
            }
        }

        public string Name
        {
            get { return _name; }
        }

        public long Offset
        {
            get { return _offset; }
        }

        public long Size
        {
            get { return _size; }
        }

        public Nw4rFileNode Root
        {
            get
            {
                if (null == _parent) { return this; }
                return _parent.Root;
            }
        }

        public Nw4rFileNode Parent
        {
            get { return _parent; }
        }

        public INw4rFileNodeCollection ChildNodes
        {
            get { return _nodes; }
        }

        #endregion

        #region ** イベント

        public event EventHandler OffsetFixed;
        public event EventHandler SizeFixed;

        public event Nw4rWriteFileEventHandler PreWriteBinary;
        public event Nw4rWriteFileEventHandler PostWriteBinary;

        #endregion

        #region ** イベントハンドラ

        #region ** 位置制御イベント

        protected virtual void OnOffsetFixed(EventArgs e)
        {
            if (null != OffsetFixed)
            {
                OffsetFixed(this, e);
            }
        }

        protected virtual void OnSizeFixed(EventArgs e)
        {
            if (null != SizeFixed)
            {
                SizeFixed(this, e);
            }
        }

        #endregion

        #region ** 入力イベント

        protected virtual void OnRead(BinaryReader reader)
        {
            ReadBinary(reader);
        }

        protected virtual void OnReadBinary(BinaryReader reader) { }

        #endregion

        #region ** 出力イベント

        protected virtual void OnWrite(BinaryWriter writer)
        {
            WriteBinary(writer);
        }

        protected virtual void OnPreWriteBinary(Nw4rWriteFileEventArgs e)
        {
            if (null != PreWriteBinary)
            {
                PreWriteBinary(this, e);
            }
        }

        protected virtual void OnPostWriteBinary(Nw4rWriteFileEventArgs e)
        {
            if (null != PostWriteBinary)
            {
                PostWriteBinary(this, e);
            }
        }

        protected virtual void OnWriteBinary(BinaryWriter writer) { }

        protected virtual void OnWriteChildNodeBinaries(BinaryWriter writer)
        {
            WriteChildNodeBinaries(writer);
        }

        #endregion

        #endregion

        #region ** メソッド

        #region ** リンク制御

        public Nw4rFileNode GetUpperNode<_Type>()
        {
            Nw4rFileNode parent = _parent;

            while (null != parent)
            {
                if (parent is _Type) { return parent; }
                parent = parent._parent;
            }

            return null;
        }

        public void AddNode(Nw4rFileNode node)
        {
            if (null == node) { throw new ArgumentNullException("node"); }
            if (null != node.Parent) { throw new ArgumentException("node still has parent"); }

            _nodes.Add(node);
            node.SetParent(this);

            ClearNumber();
        }

        protected virtual void SetParent(Nw4rFileNode node)
        {
            if (_parent == node) { return; }
            _parent = node;
        }

        #endregion

        #region ** 入力

        public virtual void Read(BinaryReader reader)
        {
            if (null == reader) { throw new ArgumentNullException("reader"); }
            OnRead(reader);
        }

        protected virtual void ReadBinary(BinaryReader reader)
        {
            if (null == reader) { throw new ArgumentNullException("reader"); }

            FixOffset(reader.BaseStream.Position);

            Nw4rReadFileEventArgs e = new Nw4rReadFileEventArgs(reader);

            // 入力
            OnReadBinary(reader);

            FixSize(reader.BaseStream.Position - Offset);
        }

        #endregion

        #region ** 出力

        public virtual void Write(BinaryWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }
            OnWrite(writer);
        }

        protected virtual void WriteBinary(BinaryWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            FixOffset(writer.BaseStream.Position);

            Nw4rWriteFileEventArgs e = new Nw4rWriteFileEventArgs(writer);

            // 出力
            OnPreWriteBinary(e);
            OnWriteBinary(writer);
            OnWriteChildNodeBinaries(writer);
            OnPostWriteBinary(e);

            FixSize(writer.BaseStream.Position - Offset);
        }

        protected virtual void WriteChildNodeBinaries(BinaryWriter writer)
        {
            foreach (Nw4rFileNode node in _nodes)
            {
                node.Write(writer);
            }
        }

        #endregion

        #region ** 位置制御

        protected virtual void FixOffset(long offset)
        {
            _offset = offset;
            OnOffsetFixed(new EventArgs());
        }

        protected virtual void FixSize(long size)
        {
            _size = size;
            OnSizeFixed(new EventArgs());
        }

        protected void Number()
        {
            if (_numbered) { return; }

            int index = 0;

            foreach (Nw4rFileNode node in _nodes)
            {
                node._index = index;
                index++;
            }

            _numbered = true;
        }

        protected void ClearNumber()
        {
            if (!_numbered) { return; }

            foreach (Nw4rFileNode node in _nodes)
            {
                node._index = InvalidIndex;
            }

            _numbered = false;
        }

        #endregion

        #endregion

        #region ** IEnumerable の実装

        /// <summary>
        /// ノードコレクションに対する反復子を取得します。
        /// </summary>
        /// <returns>ノードコレクションに対する反復子</returns>
        public IEnumerator<Nw4rFileNode> GetEnumerator()
        {
            return _nodes.GetEnumerator();
        }

        /// <summary>
        /// ノードコレクションに対する反復子を取得します。
        /// </summary>
        /// <returns>ノードコレクションに対する反復子</returns>
        IEnumerator IEnumerable.GetEnumerator()
        {
            return GetEnumerator();
        }

        #endregion

        #region ** 非公開コレクション

        protected class Nw4rFileNodeCollectionImpl : Collection<Nw4rFileNode>, INw4rFileNodeCollection
        {
            private Dictionary<string, Nw4rFileNode> _nameDictionary = new Dictionary<string, Nw4rFileNode>();

            public Nw4rFileNode this[string name]
            {
                get
                {
                    if (null == name) { throw new ArgumentNullException("name"); }
                    if (!_nameDictionary.ContainsKey(name)) { return null; }
                    return _nameDictionary[name];
                }
            }

            protected override void InsertItem(int index, Nw4rFileNode item)
            {
                base.InsertItem(index, item);

                if (_nameDictionary.ContainsKey(item.Name)) { return; }
                _nameDictionary.Add(item.Name, item);
            }

            protected override void RemoveItem(int index)
            {
                Nw4rFileNode item = this[index];

                base.RemoveItem(index);
                _nameDictionary.Remove(item.Name);
            }

            protected override void ClearItems()
            {
                base.ClearItems();
                _nameDictionary.Clear();
            }
        }

        #endregion
    }

    /// <summary>
    /// Nw4rファイル バイナリノード
    /// </summary>
    internal class Nw4rFileBinaryNode : Nw4rFileNode
    {
        private Byte[] _data = null;
        private int _byteAlignment = 0;

        public Nw4rFileBinaryNode(string name, long size) : this(name, size, 0) { }
        public Nw4rFileBinaryNode(string name, long size, int byteAlignment) : base(name)
        {
            if (size < 0) { throw new Nw4rFileFormatInternalException(new ArgumentOutOfRangeException("size")); }
            if (byteAlignment < 0) { throw new Nw4rFileFormatInternalException(new ArgumentOutOfRangeException("byteAlignment")); }

            FixSize(size);
            _byteAlignment = byteAlignment;
        }

        #region ** イベントハンドラ

        protected override void OnReadBinary(BinaryReader reader)
        {
            if (InvalidLength == Size) { throw new Nw4rFileFormatInternalException(); }

            _data = reader.ReadBytes((int)Size);
        }

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            if (null == _data) { throw new Nw4rFileFormatInternalException(); }

            writer.Write(_data);

            // バイトアライメント調整
            if (0 < _byteAlignment)
            {
                new Nw4rByteAligner(writer.BaseStream).Pad(_byteAlignment);
            }
        }

        #endregion

        #region ** メソッド

        public BinaryReader CreateDataReader()
        {
            return BinaryReaderBigEndian.CreateInstance(new MemoryStream(_data));
        }

        #endregion
    }

    #region ** コレクションインターフェイス

    /// <summary>
    /// ノードコレクション インターフェイス
    /// </summary>
    internal interface INw4rFileNodeCollection : IEnumerable<Nw4rFileNode>, IEnumerable
    {
        #region ** プロパティ

        /// <summary>
        /// ノードの数を取得します。
        /// </summary>
        int Count { get; }

        #endregion

        #region ** インデクサ

        /// <summary>
        /// 指定したインデックス位置にあるノードを取得します。
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <returns>コレクション内のノード。インデックスがノード数以上の場合、null を返します。</returns>
        Nw4rFileNode this[int index] { get; }

        /// <summary>
        /// 指定した名前をもつ最初のノードを取得します。
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <returns>コレクション内のノード。該当ノードが存在しない場合、null を返します。</returns>
        Nw4rFileNode this[string name] { get; }

        #endregion
    }

    #endregion

    #region ** イベント

    internal class Nw4rWriteFileEventArgs : EventArgs
    {
        private BinaryWriter _writer = null;

        public Nw4rWriteFileEventArgs(BinaryWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }
            _writer = writer;
        }

        public BinaryWriter Writer
        {
            get { return _writer; }
        }
    }

    internal class Nw4rReadFileEventArgs : EventArgs
    {
        private BinaryReader _reader = null;

        public Nw4rReadFileEventArgs(BinaryReader reader)
        {
            if (null == reader) { throw new ArgumentNullException("reader"); }
            _reader = reader;
        }

        public BinaryReader Reader
        {
            get { return _reader; }
        }
    }

    internal delegate void Nw4rWriteFileEventHandler(object sender, Nw4rWriteFileEventArgs e);
    internal delegate void Nw4rReadFileEventHandler(object sender, Nw4rReadFileEventArgs e);

    #endregion

    #endregion

    /// <summary>
    /// ファイルのルートノード
    /// </summary>
    internal class Nw4rFileRoot : Nw4rFileNode
    {
        public Nw4rFileRoot(string name) : base(name)
        {
            FixOffset(0);
        }

        protected override void OnReadBinary(BinaryReader reader)
        {
            Nw4rFileHeader header = new Nw4rFileHeader("FileHeader", 32);
            header.Read(reader);
            AddNode(header);

            ReadBinaryBlocks(reader, header);
        }

        #region ** プロパティ

        public Nw4rFileHeader Header
        {
            get
            {
                if (0 == ChildNodes.Count) { return null; }
                return ChildNodes[0] as Nw4rFileHeader;
            }
        }

        #endregion

        #region ** メソッド

        protected virtual void ReadBinaryBlocks(BinaryReader reader, Nw4rFileHeader header)
        {
            if (null == header) { throw new ArgumentNullException("header"); }
            if (this != header.Parent) { throw new ArgumentException("header"); }

            foreach (Nw4rFileHeader.BlockEntry entry in header.BlockEntries)
            {

                reader.BaseStream.Position = entry.Location.Head;

                entry.Block = ReadDataBlock(reader);
                AddNode(entry.Block);

            }
        }

        protected virtual Nw4rFileDataBlock ReadDataBlock(BinaryReader reader)
        {
            Nw4rFileDataBlockHeader header = new Nw4rFileDataBlockHeader();
            header.Read(reader);

            Nw4rFileBinaryDataBlock newBlock = new Nw4rFileBinaryDataBlock("BinaryBlock", header);
            newBlock.ReadBody(reader);

            return newBlock;
        }

        public void Write(string fileName)
        {
            if (null == fileName) { throw new ArgumentNullException("fileName"); }
            if (0 == fileName.Length) { throw new ArgumentException("fileName"); }
            if (fileName.IndexOfAny(Path.GetInvalidPathChars()) >= 0) { throw new ArgumentException("fileName"); }

            using (FileStream stream = File.Open(fileName, FileMode.CreateNew))
            {

                BinaryWriter writer = BinaryWriterBigEndian.CreateInstance(stream);

                Write(writer);
                writer.Flush();

            }
        }

        #endregion
    }

    internal class Nw4rFileHeader : Nw4rFileNode
    {
        #region ** パラメータ

        // パラメータ
        private string _signature = string.Empty;
        private UInt16 _version = 0;
        private int _byteAlignment = 32;

        private Nw4rSize _fileSize = new Nw4rSize();
        private Nw4rSize _headerSize = new Nw4rSize();

        private BlockLocationCollection _blockLocations = new BlockLocationCollection();	// ブロック位置情報コンテナ
        private UInt16 _blockCount = 0;								// ブロック数(Read 用に必要)

        #endregion

        public Nw4rFileHeader(string name, string signature, UInt16 version) : this(name, signature, version, 32) { }

        public Nw4rFileHeader(string name, string signature, UInt16 version, int byteAlignment) : base(name)
        {
            if (signature.Length != 4) { throw new Nw4rFileFormatInternalException("signature length must be 4."); }
            if (0 >= byteAlignment) { throw new ArgumentOutOfRangeException("byteAlignment"); }

            _signature = signature;
            _version = version;
            _byteAlignment = byteAlignment;
        }

        public Nw4rFileHeader(string name, int byteAlignment) : base(name)
        {
            if (0 >= byteAlignment) { throw new ArgumentOutOfRangeException("byteAlignment"); }

            _byteAlignment = byteAlignment;
        }

        #region ** プロパティ

        public UInt16 Version
        {
            get { return _version; }
        }

        public int MajorVersion
        {
            get { return GetMajorVersion(_version); }
        }

        public int MinorVersion
        {
            get { return GetMinorVersion(_version); }
        }

        public BlockLocationCollection BlockEntries
        {
            get { return _blockLocations; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnReadBinary(BinaryReader reader)
        {
            // ヘッダ入力
            _signature = new string(reader.ReadChars(4));			// シグネチャ
            reader.BaseStream.Seek(sizeof(UInt16), SeekOrigin.Current);	// バイトオーダー
            _version = reader.ReadUInt16();							// バージョン
            _fileSize.Size = reader.ReadUInt32();							// ファイルサイズの予約書き込み
            _headerSize.Size = reader.ReadUInt16();							// ヘッダサイズの予約書き込み
            _blockCount = reader.ReadUInt16();							// データブロック

            // ブロック位置情報
            for (int index = 0; index < _blockCount; index++)
            {
                Nw4rLocation newLocation = Nw4rLocation.FromStream(reader);
                _blockLocations.Add(new BlockEntry(newLocation));
            }
        }

        protected override void OnPreWriteBinary(Nw4rWriteFileEventArgs e)
        {
            base.OnPreWriteBinary(e);

            foreach (BlockEntry blockLocation in _blockLocations)
            {
                blockLocation.Block.SizeFixed += OnDataBlockSizeFixed;
            }
        }

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            // ヘッダ出力
            writer.Write(_signature.ToCharArray());		// シグネチャ
            writer.Write((UInt16)0xfeff);					// バイトオーダー
            writer.Write(_version);						// バージョン
            _fileSize.ReserveWrite32(writer);				// ファイルサイズの予約書き込み
            _headerSize.ReserveWrite16(writer);			// ヘッダサイズの予約書き込み
            writer.Write((UInt16)_blockLocations.Count);	// データブロック

            // ブロック位置情報の予約書き込み
            foreach (BlockEntry blockLocation in _blockLocations)
            {
                blockLocation.Location.ReserveWrite32(writer);
            }

            new Nw4rByteAligner(writer.BaseStream).Pad(32);
        }

        protected override void OnPostWriteBinary(Nw4rWriteFileEventArgs e)
        {
            base.OnPostWriteBinary(e);

            // アライメント調整
            new Nw4rByteAligner(e.Writer.BaseStream).Pad(_byteAlignment);

            // 確定したヘッダサイズ(現在値)を出力
            _headerSize.Origin = Offset;
            _headerSize.Commit();
        }

        private void OnDataBlockSizeFixed(object sender, EventArgs e)
        {
            BlockEntry blockLocation = _blockLocations[sender as Nw4rFileDataBlock];
            Debug.Assert(null != blockLocation);

            // 確定したデータブロック位置を出力
            blockLocation.Location.Origin = Parent.Offset;
            blockLocation.Location.Head = blockLocation.Block.Offset - blockLocation.Location.Origin;
            blockLocation.Location.Size = blockLocation.Block.Size;
            blockLocation.Location.Commit();
        }

        private void OnParentSizeFixed(object sender, EventArgs e)
        {
            // 確定したファイルサイズを出力
            _fileSize.Origin = Parent.Offset;
            _fileSize.Size = Parent.Size;
            _fileSize.Commit();
        }

        #endregion

        #region ** メソッド

        public static int GetMajorVersion(UInt16 version)
        {
            return ((version >> 8) & 0xFF);
        }

        public static int GetMinorVersion(UInt16 version)
        {
            return (version & 0xFF);
        }

        public void AddBlockEntry(Nw4rFileDataBlock block)
        {
            if (null == block) { throw new ArgumentNullException("block"); }

            _blockLocations.Add(new BlockEntry(block));
            _blockCount = (UInt16)_blockLocations.Count;
        }

        protected override void SetParent(Nw4rFileNode node)
        {
            if (this == Parent)
            {
                base.SetParent(node);
                return;
            }

            if (null != Parent)
            {
                Parent.SizeFixed -= OnParentSizeFixed;
            }

            base.SetParent(node);

            if (null != Parent)
            {
                Parent.SizeFixed += OnParentSizeFixed;
            }
        }

        #endregion

        #region ** ブロックの位置

        public class BlockEntry
        {
            private Nw4rFileDataBlock _block = null;
            private Nw4rLocation _location = new Nw4rLocation();

            public BlockEntry(Nw4rFileDataBlock block)
            {
                if (null == block) { throw new ArgumentNullException("block"); }
                _block = block;
            }
            public BlockEntry(Nw4rLocation location)
            {
                if (null == location) { throw new ArgumentNullException("location"); }
                _location = location;
            }

            public Nw4rFileDataBlock Block
            {
                get { return _block; }
                set
                {
                    if (null != _block) { throw new Nw4rFileFormatInternalException(); }
                    if (null == value) { throw new ArgumentNullException("value"); }

                    _block = value;
                }
            }
            public Nw4rLocation Location { get { return _location; } }
        }

        public class BlockLocationCollection : KeyedCollection<Nw4rFileDataBlock, BlockEntry>
        {
            protected override Nw4rFileDataBlock GetKeyForItem(BlockEntry item)
            {
                return item.Block;
            }
        }

        #endregion
    }

    internal class Nw4rFileDataBlock : Nw4rFileNode
    {
        public Nw4rFileDataBlock(string name) : base(name) { }

        public Nw4rFileDataBlock(string name, Nw4rFileDataBlockHeader header) : base(name)
        {
            if (null == header) { throw new ArgumentNullException("header"); }
            AddNode(header);
        }

        public Nw4rFileDataBlock(string name, Nw4rFileDataBlockHeader header, Nw4rFileNode body) : base(name)
        {
            if (null == header) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("header")); }
            if (null == body) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("body")); }

            AddNode(header);
            AddNode(body);
        }

        #region ** プロパティ

        public string Kind
        {
            get { return (ChildNodes[Nw4rFileDataBlockHeader.NodeName] as Nw4rFileDataBlockHeader).Kind; }
        }

        public Nw4rFileDataBlockHeader Header
        {
            get
            {
                if (0 == ChildNodes.Count) { return null; }
                return ChildNodes[0] as Nw4rFileDataBlockHeader;
            }
        }

        public Nw4rFileDataBlockBody Body
        {
            get
            {
                if (1 >= ChildNodes.Count) { return null; }
                return ChildNodes[1] as Nw4rFileDataBlockBody;
            }
        }

        #endregion

        #region ** イベントハンドラ

        protected sealed override void OnReadBinary(BinaryReader reader)
        {
            OnReadHeader(reader);
            OnReadBody(reader);
        }

        protected virtual void OnReadHeader(BinaryReader reader) { }
        protected virtual void OnReadBody(BinaryReader reader) { }

        #endregion

        #region ** メソッド

        public void ReadBody(BinaryReader reader)
        {
            if (null == reader) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("reader")); }
            if (null == Header) { throw new Nw4rFileFormatInternalException(); }

            FixOffset(Header.Offset);

            Nw4rReadFileEventArgs e = new Nw4rReadFileEventArgs(reader);

            // 入力
            OnReadBody(reader);

            FixSize(reader.BaseStream.Position - Offset);
        }

        #endregion
    }

    internal class Nw4rFileDataBlockT<_BodyType> : Nw4rFileDataBlock
        where _BodyType : Nw4rFileDataBlockBody, new()
    {
        protected Nw4rFileDataBlockT(string name) : base(name) { }
        protected Nw4rFileDataBlockT(string name, Nw4rFileDataBlockHeader header) : base(name, header) { }

        #region ** イベントハンドラ

        protected override void OnReadBody(BinaryReader reader)
        {
            _BodyType body = new _BodyType();
            body.Read(reader);

            AddNode(body);
        }

        #endregion
    }


    internal class Nw4rFileBinaryDataBlock : Nw4rFileDataBlock
    {
        private int _byteAlignment = 0;

        public Nw4rFileBinaryDataBlock(string name, Nw4rFileDataBlockHeader header) : base(name, header) { }
        public Nw4rFileBinaryDataBlock(string name, Nw4rFileDataBlockHeader header, int byteAlignment) : base(name, header)
        {
            if (byteAlignment < 0) { throw new Nw4rFileFormatInternalException(new ArgumentOutOfRangeException("byteAlignment")); }
            _byteAlignment = byteAlignment;
        }

        #region ** イベントハンドラ

        protected override void OnReadHeader(BinaryReader reader)
        {
            if (null != Header) { throw new Nw4rFileFormatInternalException(); }

            Nw4rFileDataBlockHeader header = new Nw4rFileDataBlockHeader();
            header.Read(reader);

            AddNode(header);
        }

        protected override void OnReadBody(BinaryReader reader)
        {
            Nw4rFileBinaryDataBlockBody body = new Nw4rFileBinaryDataBlockBody(Header.BodySize, _byteAlignment);
            body.Read(reader);

            AddNode(body);
        }

        #endregion

        #region ** メソッド

        public BinaryReader CreateDataReader()
        {
            return (ChildNodes[Nw4rFileBinaryDataBlockBody.NodeName] as Nw4rFileBinaryDataBlockBody).CreateDataReader();
        }

        #endregion
    }

    internal class Nw4rFileDataBlockHeader : Nw4rFileNode
    {
        #region 固定値

        public static readonly string NodeName = "BlockHeader";
        public const long DataSize = 4 + 4;			// kind + blockSize

        #endregion

        #region ** フィールド

        private string _kind = string.Empty;
        private Nw4rSize _blockSize = new Nw4rSize();

        #endregion

        public Nw4rFileDataBlockHeader(string kind) : base(NodeName)
        {
            if (null == kind) { throw new ArgumentNullException("kind"); }
            if (4 != kind.Length) { throw new Nw4rFileFormatException("kind.Length must be 4."); }
            _kind = kind;
        }
        public Nw4rFileDataBlockHeader() : base(NodeName) { }

        #region ** プロパティ

        public string Kind
        {
            get { return _kind; }
        }

        public long BlockSize
        {
            get { return _blockSize.Size; }
        }

        public long BodySize
        {
            get { return _blockSize.Size - Size; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnReadBinary(BinaryReader reader)
        {
            // ヘッダ入力
            _kind = new string(reader.ReadChars(4));	// 種類
            _blockSize.Size = reader.ReadUInt32();					// サイズ
        }

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            // ヘッダ出力
            writer.Write(_kind.ToCharArray());	// 種類
            _blockSize.ReserveWrite32(writer);	// サイズの出力予約
        }

        private void OnParentFixedSize(object sender, EventArgs e)
        {
            Nw4rFileNode target = sender as Nw4rFileNode;

            // 確定したサイズを出力
            _blockSize.Origin = target.Offset;
            _blockSize.Size = target.Size;
            _blockSize.Commit();
        }

        #endregion

        #region ** メソッド

        protected override void SetParent(Nw4rFileNode node)
        {
            if (null != Parent)
            {
                Parent.SizeFixed -= OnParentFixedSize;
            }

            base.SetParent(node);

            if (null != Parent)
            {
                Parent.SizeFixed += OnParentFixedSize;
            }
        }

        #endregion
    }

    internal class Nw4rFileDataBlockBody : Nw4rFileNode
    {
        public static readonly string NodeName = "BlockBody";

        public Nw4rFileDataBlockBody() : base(NodeName) { }
    }

    internal class Nw4rFileBinaryDataBlockBody : Nw4rFileDataBlockBody
    {
        private int _byteAlignment = 0;

        public Nw4rFileBinaryDataBlockBody(long size, int byteAlignment)
        {
            if (size < 0) { throw new Nw4rFileFormatInternalException(new ArgumentOutOfRangeException("size")); }
            if (byteAlignment < 0) { throw new Nw4rFileFormatInternalException(new ArgumentOutOfRangeException("byteAlignment")); }

            _byteAlignment = byteAlignment;
            FixSize(size);
        }

        #region ** イベントハンドラ

        protected override void OnReadBinary(BinaryReader reader)
        {
            Nw4rFileBinaryNode binaryNode = new Nw4rFileBinaryNode("Binary", Size, _byteAlignment);
            binaryNode.Read(reader);

            AddNode(binaryNode);
        }

        #endregion

        #region ** メソッド

        public BinaryReader CreateDataReader()
        {
            return (ChildNodes["Binary"] as Nw4rFileBinaryNode).CreateDataReader();
        }

        #endregion
    }

    #region ** 内部テーブルクラス

    #region ** 基本クラス

    internal abstract class Nw4rFileInnerTableItem
    {
        private Nw4rFileInnerTable _parent = null;
        private Nw4rFileNode _data = null;

        public Nw4rFileInnerTableItem(Nw4rFileInnerTable parent, Nw4rFileNode data)
        {
            if (null == parent) { throw new ArgumentNullException("parent"); }
            //			if( null == data ) { throw new ArgumentNullException( "data" ); }

            _parent = parent;
            _data = data;
        }

        #region ** プロパティ

        public Nw4rFileInnerTable Parent
        {
            get { return _parent; }
        }

        public Nw4rFileNode Data
        {
            get { return _data; }
            set
            {
                if (null != _data) { throw new Nw4rFileFormatInternalException(); }
                if (null == value) { throw new Nw4rFileFormatInternalException(); }

                _data = value;
            }
        }

        #endregion

        #region ** イベントハンドラ

        protected abstract void OnReadItem(BinaryReader reader);
        protected abstract void OnWriteItem(BinaryWriter writer);

        #endregion

        #region ** メソッド

        public void ReadItem(BinaryReader reader)
        {
            if (null == reader) { throw new ArgumentNullException("reader"); }
            OnReadItem(reader);
        }

        public void WriteItem(BinaryWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }
            OnWriteItem(writer);
        }

        #endregion
    }

    internal abstract class Nw4rFileInnerTable
    {
        private long _offset = Nw4rFileNode.InvalidLength;
        private Nw4rFileInnerTableItemCollection _items = new Nw4rFileInnerTableItemCollection();

        #region ** プロパティ

        public long Offset
        {
            get { return _offset; }
        }

        public INw4rFileInnerTableItemCollection Items
        {
            get { return _items; }
        }

        #endregion

        #region ** インデクサ

        public Nw4rFileInnerTableItem this[int index]
        {
            get { return _items[index]; }
        }

        #endregion

        #region ** イベントハンドラ

        protected virtual void OnRead(BinaryReader reader)
        {
            ReadBinary(reader);
        }

        protected virtual void OnReadBinary(BinaryReader reader)
        {
            // アイテム数のロード
            int itemCount = 0;
            OnReadHeader(reader, out itemCount);

            // アイテムのロード
            for (int index = 0; index < itemCount; index++)
            {

                Nw4rFileInnerTableItem newItem = CreateItem(null);
                newItem.ReadItem(reader);

                _items.Add(newItem);

            }
        }

        protected virtual void OnReadHeader(BinaryReader reader, out int itemCount)
        {
            itemCount = (int)reader.ReadUInt32();
        }

        protected virtual void OnWrite(BinaryWriter writer)
        {
            WriteBinary(writer);
        }

        protected virtual void OnPreWriteBinary(BinaryWriter writer) { }

        protected virtual void OnWriteBinary(BinaryWriter writer)
        {
            // ヘッダ出力
            OnWriteHeader(writer);

            // アイテム出力
            foreach (Nw4rFileInnerTableItem item in _items)
            {
                item.WriteItem(writer);
            }
        }

        protected virtual void OnPostWriteBinary(BinaryWriter writer) { }

        protected virtual void OnWriteHeader(BinaryWriter writer)
        {
            writer.Write((UInt32)_items.Count);	// アイテム数
        }

        #endregion

        #region ** メソッド

        public void AddItem(Nw4rFileNode data)
        {
            if (null == data) { throw new ArgumentNullException("data"); }
            _items.Add(CreateItem(data));
        }

        public void Read(BinaryReader reader)
        {
            if (null == reader) { throw new ArgumentNullException("reader"); }
            OnRead(reader);
        }

        protected void ReadBinary(BinaryReader reader)
        {
            if (null == reader) { throw new ArgumentNullException("reader"); }

            // オフセットを保持する
            _offset = reader.BaseStream.Position;

            OnReadBinary(reader);
        }

        public void Write(BinaryWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }
            OnWrite(writer);
        }

        protected void WriteBinary(BinaryWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            // オフセットを保持する
            _offset = writer.BaseStream.Position;

            OnPreWriteBinary(writer);
            OnWriteBinary(writer);
            OnPostWriteBinary(writer);
        }

        protected abstract Nw4rFileInnerTableItem CreateItem(Nw4rFileNode data);

        #endregion

        #region ** コレクション

        private class Nw4rFileInnerTableItemCollection : Collection<Nw4rFileInnerTableItem>, INw4rFileInnerTableItemCollection { }

        #endregion
    }

    internal interface INw4rFileInnerTableItemCollection : IEnumerable<Nw4rFileInnerTableItem>
    {
        #region ** プロパティ

        /// <summary>
        /// アイテムの数を取得します。
        /// </summary>
        int Count { get; }

        #endregion

        #region ** インデクサ

        /// <summary>
        /// 指定したインデックス位置にあるアイテムを取得します。
        /// </summary>
        /// <param name="index">インデックス</param>
        /// <returns>コレクション内のノード。インデックスがノード数以上の場合、null を返します。</returns>
        Nw4rFileInnerTableItem this[int index] { get; }

        #endregion
    }

    internal interface INw4rFileInnerTableItemData
    {
        #region ** プロパティ

        byte DataType { get; }

        #endregion
    }

    #endregion

    #region ** オフセットテーブル

    internal class Nw4rFileInnerOffsetTableItem : Nw4rFileInnerTableItem
    {
        private Nw4rSize _dataOffset = new Nw4rSize();

        public Nw4rFileInnerOffsetTableItem(Nw4rFileInnerTable parent, Nw4rFileNode data) : base(parent, data)
        {
            if (null != Data)
            {
                Data.OffsetFixed += OnDataOffsetFixed;
            }
        }

        #region ** プロパティ

        public new Nw4rFileInnerOffsetTable Parent
        {
            get { return base.Parent as Nw4rFileInnerOffsetTable; }
        }

        public long Offset
        {
            get { return _dataOffset.Size; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnReadItem(BinaryReader reader)
        {
            if (null == reader) { throw new ArgumentNullException("reader"); }

            // データオフセットの読み込み
            _dataOffset.Read32(reader);
        }

        protected override void OnWriteItem(BinaryWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            // データオフセットの書き込み予約
            _dataOffset.ReserveWrite32(writer);
        }

        private void OnDataOffsetFixed(object sender, EventArgs e)
        {
            // オフセットの書き込み
            _dataOffset.Origin = Parent.OffsetOrigin;
            _dataOffset.Size = Data.Offset - _dataOffset.Origin;
            _dataOffset.Commit();
        }

        #endregion
    }

    internal class Nw4rFileInnerOffsetTable : Nw4rFileInnerTable
    {
        private long _offsetOrigin = Nw4rFileNode.InvalidLength;

        public long OffsetOrigin
        {
            get { return _offsetOrigin; }
            set { _offsetOrigin = value; }
        }

        protected override void OnPreWriteBinary(BinaryWriter writer)
        {
            if (Nw4rFileNode.InvalidLength == _offsetOrigin)
            {
                _offsetOrigin = Offset;
            }
        }

        protected override Nw4rFileInnerTableItem CreateItem(Nw4rFileNode data)
        {
            return new Nw4rFileInnerOffsetTableItem(this, data);
        }
    }

    #endregion

    #region ** オフセット参照テーブル

    internal class Nw4rFileInnerOffsetRefTableItem : Nw4rFileInnerTableItem
    {
        private Nw4rFileDataOffsetRef _dataReference = new Nw4rFileDataOffsetRef();

        public Nw4rFileInnerOffsetRefTableItem(Nw4rFileInnerTable parent, Nw4rFileNode data) : base(parent, data)
        {
            if (null == Data) { return; }

            if (Data is INw4rFileInnerTableItemData)
            {
                _dataReference = new Nw4rFileDataOffsetRef((Data as INw4rFileInnerTableItemData).DataType);
            }

            Data.OffsetFixed += OnDataOffsetFixed;
        }

        #region ** プロパティ

        public new Nw4rFileInnerOffsetRefTable Parent
        {
            get { return base.Parent as Nw4rFileInnerOffsetRefTable; }
        }

        public Nw4rFileDataOffsetRef Reference
        {
            get { return _dataReference; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnReadItem(BinaryReader reader)
        {
            _dataReference.Read(reader);
        }

        protected override void OnWriteItem(BinaryWriter writer)
        {
            // データ参照タイプを出力
            _dataReference.ReserveWrite(writer);
        }

        private void OnDataOffsetFixed(object sender, EventArgs e)
        {
            // オフセットの書き込み
            _dataReference.Origin = Parent.OffsetOrigin;
            _dataReference.Size = Data.Offset - _dataReference.Origin;
            _dataReference.Commit();
        }

        #endregion
    }

    internal class Nw4rFileInnerOffsetRefTable : Nw4rFileInnerTable
    {
        private long _offsetOrigin = Nw4rFileNode.InvalidLength;

        public long OffsetOrigin
        {
            get { return _offsetOrigin; }
            set { _offsetOrigin = value; }
        }

        protected override void OnPreWriteBinary(BinaryWriter writer)
        {
            if (Nw4rFileNode.InvalidLength == _offsetOrigin)
            {
                _offsetOrigin = Offset;
            }
        }

        protected override Nw4rFileInnerTableItem CreateItem(Nw4rFileNode data)
        {
            return new Nw4rFileInnerOffsetRefTableItem(this, data);
        }
    }

    #endregion

    #region ** ロケーション参照テーブル

    internal class Nw4rFileInnerLocationRefTableItem : Nw4rFileInnerTableItem
    {
        private Nw4rFileDataLocationRef _dataReference = new Nw4rFileDataLocationRef();

        public Nw4rFileInnerLocationRefTableItem(Nw4rFileInnerTable parent, Nw4rFileNode data) : base(parent, data)
        {
            if (null == Data) { return; }

            if (Data is INw4rFileInnerTableItemData)
            {
                _dataReference = new Nw4rFileDataLocationRef((Data as INw4rFileInnerTableItemData).DataType);
            }

            Data.SizeFixed += OnDataSizeFixed;
        }

        #region ** プロパティ

        public new Nw4rFileInnerLocationRefTable Parent
        {
            get { return base.Parent as Nw4rFileInnerLocationRefTable; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnReadItem(BinaryReader reader)
        {
            if (null == reader) { throw new ArgumentNullException("reader"); }

            // データ参照タイプを読み込み
            _dataReference.Read(reader);
        }

        protected override void OnWriteItem(BinaryWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            // データ参照タイプを書き込み
            _dataReference.ReserveWrite(writer);
        }

        private void OnDataSizeFixed(object sender, EventArgs e)
        {
            // 確定したオフセットとサイズを出力
            _dataReference.Origin = Parent.OffsetOrigin;
            _dataReference.Head = Data.Offset - _dataReference.Origin;
            _dataReference.Size = Data.Size;
            _dataReference.Commit();
        }

        #endregion
    }

    internal class Nw4rFileInnerLocationRefTable : Nw4rFileInnerTable
    {
        private long _offsetOrigin = Nw4rFileNode.InvalidLength;

        public long OffsetOrigin
        {
            get { return _offsetOrigin; }
            set { _offsetOrigin = value; }
        }

        protected override void OnPreWriteBinary(BinaryWriter writer)
        {
            if (Nw4rFileNode.InvalidLength == _offsetOrigin)
            {
                _offsetOrigin = Offset;
            }
        }

        protected override Nw4rFileInnerTableItem CreateItem(Nw4rFileNode data)
        {
            return new Nw4rFileInnerLocationRefTableItem(this, data);
        }
    }

    #endregion

    #endregion

    #region ** データ参照クラス

    internal enum Nw4rFileDataReferenceType
    {
        Address = 0,
        Offset = 1,
    }

    internal abstract class Nw4rFileDataReference
    {
        private const long InvalidValue = -1;

        #region ** パラメータ

        private Nw4rFileDataReferenceType _referenceType = Nw4rFileDataReferenceType.Address;	// 参照の種類
        private Byte _dataType = 0;									// データの種類（各箇所で固有の値）

        private BinaryWriter _writer = null;
        private long _writePosition = InvalidValue;

        #endregion

        protected Nw4rFileDataReference(Nw4rFileDataReferenceType referenceType, Byte dataType)
        {
            _referenceType = referenceType;
            _dataType = dataType;
        }

        #region ** プロパティ

        public Nw4rFileDataReferenceType ReferenceType
        {
            get { return _referenceType; }
        }

        public int DataType
        {
            get { return _dataType; }
        }

        #endregion

        #region ** メソッド

        public virtual void Read(BinaryReader reader)
        {
            if (null == reader) { throw new ArgumentNullException("reader"); }

            _referenceType = (Nw4rFileDataReferenceType)reader.ReadByte();	// 参照の種類
            _dataType = reader.ReadByte();								// データの種類

            reader.BaseStream.Seek(2, SeekOrigin.Current);				// 0 Padding
        }

        public virtual void Write(BinaryWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            writer.Write((Byte)_referenceType);	// reference type
            writer.Write(_dataType);				// data type

            // 0 Padding
            writer.Write((Byte)0);
            writer.Write((Byte)0);
        }

        /// <summary>
        /// ストリームの現在位置に値を書き込むために予約します。
        /// ReserveWrite では仮の値が書き込まれ、Commit 実行時に正しい値が書き込まれます。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        public virtual void ReserveWrite(BinaryWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            _writePosition = writer.BaseStream.Position;

            writer.Write((Byte)0);	// reference type
            writer.Write((Byte)0);	// data type

            // 0 Padding
            writer.Write((Byte)0);
            writer.Write((Byte)0);

            _writer = writer;
        }

        /// <summary>
        /// ReserveWrite にて予約した場所にオフセット値(BeginPosition-EndPosition)を書き込みます。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        public virtual void Commit()
        {
            if (null == _writer) { throw new Nw4rFileFormatInternalException(); }
            if (InvalidValue == _writePosition) { throw new Nw4rFileFormatInternalException(); }

            long currentPosition = _writer.BaseStream.Position;
            _writer.BaseStream.Position = _writePosition;

            Write(_writer);

            _writer.BaseStream.Position = currentPosition;
        }

        #endregion
    }

    internal class Nw4rFileDataOffsetRef : Nw4rFileDataReference
    {
        #region ** 無効値

        public static readonly Nw4rFileDataOffsetRef InvalidValue = new Nw4rFileDataOffsetRef(Nw4rFileDataReferenceType.Address, 0);

        #endregion

        #region ** パラメータ

        private Nw4rSize _offset = new Nw4rSize();	// オフセット

        #endregion

        public Nw4rFileDataOffsetRef() : this(0) { }
        public Nw4rFileDataOffsetRef(Byte dataType) : base(Nw4rFileDataReferenceType.Offset, dataType) { }

        /// <summary>
        /// 無効値専用コンストラクタ
        /// </summary>
        private Nw4rFileDataOffsetRef(Nw4rFileDataReferenceType type, Byte dataType) : base(type, dataType) { }

        #region ** プロパティ

        public long Origin
        {
            get { return _offset.Origin; }
            set { _offset.Origin = value; }
        }

        public long Size
        {
            get { return _offset.Size; }
            set { _offset.Size = value; }
        }

        #endregion

        #region ** メソッド

        public override void Read(BinaryReader reader)
        {
            if (null == reader) { throw new ArgumentNullException("reader"); }

            base.Read(reader);

            // オフセット値の読み込み
            _offset.Read32(reader);
        }

        /// <summary>
        /// オフセット値を書き込みます。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <param name="offset">オフセット値</param>
        public void Write(BinaryWriter writer, UInt32 offset)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            base.Write(writer);

            // オフセット値の書き込み
            writer.Write(offset);
        }

        /// <summary>
        /// ストリームの現在位置に値を書き込むために予約します。
        /// ReserveWrite では仮の値が書き込まれ、Commit 実行時に正しい値が書き込まれます。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        public override void ReserveWrite(BinaryWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            base.ReserveWrite(writer);

            // オフセット値の書き込み予約
            _offset.ReserveWrite32(writer);
        }

        /// <summary>
        /// ReserveWrite にて予約した場所にオフセット値(BeginPosition-EndPosition)を書き込みます。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        public override void Commit()
        {
            base.Commit();
            _offset.Commit();
        }

        #endregion
    }

    internal class Nw4rFileDataLocationRef : Nw4rFileDataReference
    {
        #region ** パラメータ

        private Nw4rLocation _location = new Nw4rLocation();	// 位置

        #endregion

        public Nw4rFileDataLocationRef() : this(0) { }
        public Nw4rFileDataLocationRef(Byte dataType) : base(Nw4rFileDataReferenceType.Offset, dataType) { }

        #region ** プロパティ

        public long Origin
        {
            get { return _location.Origin; }
            set { _location.Origin = value; }
        }

        public long Head
        {
            get { return _location.Head; }
            set { _location.Head = value; }
        }

        public long Size
        {
            get { return _location.Size; }
            set { _location.Size = value; }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// オフセット値を書き込みます。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <param name="offset">オフセット値</param>
        /// <param name="size">サイズ</param>
        public void Write(BinaryWriter writer, UInt32 offset, UInt32 size)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            base.Write(writer);

            // 値の書き込み
            _location.Write32(writer, offset, size);
        }

        /// <summary>
        /// ストリームの現在位置に値を書き込むために予約します。
        /// ReserveWrite では仮の値が書き込まれ、Commit 実行時に正しい値が書き込まれます。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        public override void ReserveWrite(BinaryWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            base.ReserveWrite(writer);

            // 値の書き込み予約
            _location.ReserveWrite32(writer);
        }

        /// <summary>
        /// ReserveWrite にて予約した場所にオフセット値(BeginPosition-EndPosition)を書き込みます。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        public override void Commit()
        {
            base.Commit();
            _location.Commit();
        }

        #endregion
    }

    internal class Nw4rFileDataAddressRef : Nw4rFileDataReference
    {
        #region ** 無効値

        public static readonly Nw4rFileDataAddressRef InvalidValue = new Nw4rFileDataAddressRef();

        #endregion

        private Nw4rFileDataAddressRef() : base(Nw4rFileDataReferenceType.Address, 0) { }

        #region ** メソッド

        /// <summary>
        /// アドレス値を書き込みます。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <param name="offset">オフセット値</param>
        /// <param name="size">サイズ</param>
        public override void Write(BinaryWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            base.Write(writer);

            // 空アドレスの出力
            writer.Write((UInt32)0);
        }

        /// <summary>
        /// ストリームの現在位置に値を書き込むために予約します。
        /// ReserveWrite では仮の値が書き込まれ、Commit 実行時に正しい値が書き込まれます。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        public override void ReserveWrite(BinaryWriter writer)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            base.ReserveWrite(writer);

            // 空アドレスの出力
            writer.Write((UInt32)0);
        }

        #endregion
    }

    #endregion
}
