﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;

using NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Model;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat
{
    class Nw4rBankFile : Nw4rFileRoot
    {
        #region ** 固定値

        private const ushort Version = 0x0102;

        public const int KeyRegionRangeLimit = 11;
        public const int VelocityRegionRangeLimit = 11;

        #endregion

        #region ** フィールド

        private Nw4rBank _bank = null;
        private INw4rComponentCollection _waveFiles = null;

        #endregion

        public Nw4rBankFile(Nw4rBank bank) : this(bank, null) { }
        public Nw4rBankFile(Nw4rBank bank, INw4rComponentCollection waveFiles) : base("BankFileRoot")
        {
            if (null == bank) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("bank")); }

            _bank = bank;
            _waveFiles = (null == waveFiles) ? bank.BinaryFile.Components : waveFiles;

            Build();
        }

        #region ** プロパティ

        public Nw4rBank Bank
        {
            get { return _bank; }
        }

        public INw4rComponentCollection WaveFiles
        {
            get { return _waveFiles; }
        }

        #endregion

        #region ** メソッド

        private void Build()
        {
            Debug.Assert(null != _bank);

            // ヘッダを追加する
            Nw4rFileHeader header = new Nw4rFileHeader("FileHeader", "RBNK", Version);
            AddNode(header);

            // 各ブロックを追加する
            AddBlock(new Nw4rBankDataBlock(_bank));		// データブロック
        }

        private void AddBlock(Nw4rFileDataBlock block)
        {
            Debug.Assert(null != block);

            // ヘッダにエントリを追加する
            Header.AddBlockEntry(block);

            // ブロックを追加する
            AddNode(block);
        }

        #endregion
    }

    #region ** 各ブロック

    #region ** データブロック

    internal class Nw4rBankDataBlock : Nw4rFileDataBlock
    {
        #region ** 固定値

        public static readonly string NodeName = "DataBlock";
        private static readonly string BlockKind = "DATA";

        #endregion

        public Nw4rBankDataBlock(Nw4rBank bank) : base(NodeName)
        {
            if (null == bank) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("bank")); }

            AddNode(new Nw4rFileDataBlockHeader(BlockKind));
            AddNode(new Nw4rBankDataBlockBody(bank));
        }
    }

    internal class Nw4rBankDataBlockBody : Nw4rFileDataBlockBody
    {
        private Nw4rFileInnerOffsetRefTable _dataTable = new Nw4rFileInnerOffsetRefTable();

        public Nw4rBankDataBlockBody(Nw4rBank bank)
        {
            if (null == bank) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("bank")); }

            Build(bank);
        }

        #region ** イベントハンドラ

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            // データテーブル
            _dataTable.Write(writer);
        }

        protected override void OnWriteChildNodeBinaries(BinaryWriter writer)
        {
            // エラー時にラベルを表示するために、base.OnWriteChildNodeBinaries をオーバーライド
            NW4rBankInstrumentData instrumentNode = null;

            try
            {

                foreach (Nw4rFileNode node in ChildNodes)
                {
                    instrumentNode = node as NW4rBankInstrumentData;
                    node.Write(writer);
                }

            }
            catch (Nw4rFileFormatException exception)
            {
                throw exception;
            }
            catch (Exception exception)
            {
                if (null == instrumentNode) { throw exception; }
                throw new Nw4rFileFormatException(exception.Message, instrumentNode.Instrument.Label, exception);
            }
        }

        protected override void OnPostWriteBinary(Nw4rWriteFileEventArgs e)
        {
            new Nw4rByteAligner(e.Writer.BaseStream).Pad(32);
        }

        #endregion

        #region ** メソッド

        private void Build(Nw4rBank bank)
        {
            Debug.Assert(null != bank);

            for (int programNo = 0; programNo < bank.InstrumentMap.Count; programNo++)
            {

                Nw4rInstrument instrument = bank.InstrumentMap[programNo];

                if (null == instrument)
                {
                    AddDataTableEntry(new NW4rBankInstrumentNull());
                    continue;
                }

                if (0 == instrument.Components.Count)
                {
                    AddBlockNode(new NW4rBankInstrumentNull(instrument));
                    continue;
                }

                int splitCount = GetKeyRegionSplitCount(instrument);

                if (1 == splitCount && 1 == instrument.Components[0].Components.Count)
                {
                    AddBlockNode(new NW4rBankInstrumentDirectData(instrument));
                }
                else if (Nw4rBankFile.KeyRegionRangeLimit >= splitCount)
                {
                    AddBlockNode(new NW4rBankInstrumentRangeData(instrument));
                }
                else
                {
                    AddBlockNode(new NW4rBankInstrumentIndexData(instrument));
                }

            }
        }

        private void AddBlockNode(Nw4rFileNode node)
        {
            Debug.Assert(null != node);

            AddDataTableEntry(node);
            AddNode(node);
        }

        private void AddDataTableEntry(Nw4rFileNode node)
        {
            Debug.Assert(null != node);

            _dataTable.AddItem(node);
        }

        /// <summary>
        /// キーリージョン数を計算します。（NULLリージョンも数に含めます）
        /// </summary>
        /// <param name="instrument">対象インストルメント</param>
        /// <returns>キーリージョン数</returns>
        private int GetKeyRegionSplitCount(Nw4rInstrument instrument)
        {
            Debug.Assert(null != instrument);

            if (0 == instrument.Components.Count) { return 1; }

            int splitCount = 0;
            Nw4rInstrumentKeyRegion prevKeyRegion = null;

            foreach (Nw4rInstrumentKeyRegion keyRegion in instrument.Components)
            {

                // キーリージョンの間にNULLリージョンが含まれる場合は、数に含める
                if (null != prevKeyRegion &&
                    prevKeyRegion.XmlData.RangeMax + 1 < keyRegion.XmlData.RangeMin)
                {
                    splitCount++;
                }

                prevKeyRegion = keyRegion;
                splitCount++;

            }

            return splitCount;
        }

        #endregion
    }

    #region ** インストルメントデータ

    #region ** テーブルクラス

    internal class Nw4rBankKeyRegionRangeTable : Nw4rFileInnerOffsetRefTable
    {
        #region ** イベントハンドラ

        protected override void OnWriteHeader(BinaryWriter writer)
        {
            // アイテム数の出力
            writer.Write((Byte)Items.Count);

            // キーマップの出力
            foreach (Nw4rFileInnerTableItem item in Items)
            {
                writer.Write((Byte)(item.Data as NW4rBankKeyRegionData).RangeMax);
            }

            new Nw4rByteAligner(writer.BaseStream).Pad(4);
        }

        #endregion
    }

    internal class Nw4rBankKeyRegionIndexTable : Nw4rFileInnerOffsetRefTable
    {
        #region ** イベントハンドラ

        protected override void OnWriteHeader(BinaryWriter writer)
        {
            // キー範囲の出力
            writer.Write((Byte)(Items[0].Data as NW4rBankKeyRegionData).RangeMin);
            writer.Write((Byte)(Items[Items.Count - 1].Data as NW4rBankKeyRegionData).RangeMax);

            new Nw4rByteAligner(writer.BaseStream).Pad(4);
        }

        #endregion
    }

    #endregion

    internal abstract class NW4rBankInstrumentData : Nw4rFileNode, INw4rFileInnerTableItemData
    {
        #region ** 固定値

        public static readonly string NodeName = "InstrumentData";

        #endregion

        private Nw4rInstrument _instrument = null;

        public NW4rBankInstrumentData(Nw4rInstrument instrument) : base(NodeName)
        {
            if (null == instrument) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("instrument")); }
            _instrument = instrument;

            Build(_instrument);
        }
        protected NW4rBankInstrumentData() : base(NodeName) { }

        #region ** プロパティ

        public Nw4rInstrument Instrument
        {
            get { return _instrument; }
        }

        protected abstract byte DataType { get; }

        byte INw4rFileInnerTableItemData.DataType
        {
            get { return DataType; }
        }

        #endregion

        #region ** イベントハンドラ

        protected virtual void OnAddDataNode(Nw4rFileNode node) { }

        #endregion

        #region ** メソッド

        protected virtual void Build(Nw4rInstrument instrument)
        {
            Debug.Assert(null != instrument);

            Nw4rInstrumentKeyRegion prevKeyRegion = null;

            foreach (Nw4rInstrumentKeyRegion keyRegion in instrument.Components)
            {

                // リージョン間にスペースがある場合、NULLリージョンを追加する
                if (null != prevKeyRegion &&
                    prevKeyRegion.XmlData.RangeMax + 1 < keyRegion.XmlData.RangeMin)
                {
                    AddKeyRegionNull(prevKeyRegion.XmlData.RangeMax + 1, keyRegion.XmlData.RangeMin - 1);
                }

                AddKeyRegion(keyRegion);

                prevKeyRegion = keyRegion;

            }

            // 最後にスペースが残っている場合、NULLリージョンを追加する
            if (null != prevKeyRegion &&
                prevKeyRegion.XmlData.RangeMax < Nw4rInstrument.KeyMax)
            {
                AddKeyRegionNull(prevKeyRegion.XmlData.RangeMax + 1, Nw4rInstrument.KeyMax);
            }
        }

        protected void AddKeyRegionNull(int rangeMin, int rangeMax)
        {
            AddDataNode(new NW4rBankKeyRegionNull(rangeMin, rangeMax));
        }

        protected void AddKeyRegion(Nw4rInstrumentKeyRegion keyRegion)
        {
            if (null == keyRegion)
            {
                throw new Nw4rFileFormatInternalException("keyRegion is null.");
            }

            if (0 == keyRegion.Components.Count)
            {
                throw new Nw4rFileFormatInternalException("velocity region not found.");
            }
            else if (1 == keyRegion.Components.Count)
            {
                AddDataNode(new NW4rBankKeyRegionDirect(keyRegion));
            }
            else if (Nw4rBankFile.KeyRegionRangeLimit >= keyRegion.Components.Count)
            {
                AddDataNode(new NW4rBankKeyRegionRangeTable(keyRegion));
            }
            else
            {
                AddDataNode(new NW4rBankKeyRegionIndexTable(keyRegion));
            }
        }

        private void AddDataNode(Nw4rFileNode node)
        {
            AddNode(node);
            OnAddDataNode(node);
        }

        #endregion
    }

    /// <summary>
    /// インストルメントデータクラス（キーリージョンデータを直接出力）
    /// </summary>
    internal class NW4rBankInstrumentDirectData : NW4rBankInstrumentData
    {
        #region ** 固定値

        private const byte Type = 1;	// REGION_SET_DIRECT

        #endregion

        public NW4rBankInstrumentDirectData(Nw4rInstrument instrument) : base(instrument) { }

        #region ** プロパティ

        protected override byte DataType
        {
            get { return Type; }
        }

        #endregion
    }

    /// <summary>
    /// インストルメントデータクラス（キーリージョンデータを範囲で出力）
    /// </summary>
    internal class NW4rBankInstrumentRangeData : NW4rBankInstrumentData
    {
        #region ** 固定値

        private const byte Type = 2;	// REGION_SET_RANGE_TABLE

        #endregion

        private Nw4rBankKeyRegionRangeTable _table = new Nw4rBankKeyRegionRangeTable();

        public NW4rBankInstrumentRangeData(Nw4rInstrument instrument) : base(instrument) { }

        #region ** プロパティ

        protected override byte DataType
        {
            get { return Type; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnAddDataNode(Nw4rFileNode node)
        {
            _table.AddItem(node);
        }

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            // テーブル出力
            _table.OffsetOrigin = GetUpperNode<Nw4rBankDataBlockBody>().Offset;
            _table.Write(writer);
        }

        #endregion
    }

    /// <summary>
    /// インストルメントデータクラス（キーリージョンデータをインデックスで出力）
    /// </summary>
    internal class NW4rBankInstrumentIndexData : NW4rBankInstrumentData
    {
        #region ** 固定値

        private const byte Type = 3;	// REGION_SET_INDEX_TABLE

        #endregion

        private Nw4rBankKeyRegionIndexTable _table = new Nw4rBankKeyRegionIndexTable();

        public NW4rBankInstrumentIndexData(Nw4rInstrument instrument) : base(instrument) { }

        #region ** プロパティ

        protected override byte DataType
        {
            get { return Type; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnAddDataNode(Nw4rFileNode node)
        {
            _table.AddItem(node);
        }

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            // テーブル出力
            _table.OffsetOrigin = GetUpperNode<Nw4rBankDataBlockBody>().Offset;
            _table.Write(writer);
        }

        #endregion

        #region ** メソッド

        protected override void Build(Nw4rInstrument instrument)
        {
            Debug.Assert(null != instrument);

            int rangeMin = (instrument.Components[0] as Nw4rInstrumentKeyRegion).XmlData.RangeMin;
            int rangeMax = (instrument.Components[instrument.Components.Count - 1] as Nw4rInstrumentKeyRegion).XmlData.RangeMax;
            int currentKey = rangeMin;

            foreach (Nw4rInstrumentKeyRegion keyRegion in instrument.Components)
            {

                // NULLリージョンの追加
                for (int key = currentKey; key < keyRegion.XmlData.RangeMin; key++)
                {
                    AddKeyRegionNull(key, key);
                }

                for (int key = keyRegion.XmlData.RangeMin; key <= keyRegion.XmlData.RangeMax; key++)
                {
                    AddKeyRegion(keyRegion);
                }

                currentKey = keyRegion.XmlData.RangeMax + 1;

            }
        }

        #endregion
    }

    /// <summary>
    /// インストルメントデータクラス（キーリージョンデータがない場合）
    /// </summary>
    internal class NW4rBankInstrumentNull : NW4rBankInstrumentData
    {
        #region ** 固定値

        private const byte Type = 4;	// REGION_SET_NULL

        #endregion

        public NW4rBankInstrumentNull() : base() { }
        public NW4rBankInstrumentNull(Nw4rInstrument instrument) : base(instrument) { }

        #region ** プロパティ

        protected override byte DataType
        {
            get { return Type; }
        }

        #endregion
    }

    #endregion

    #region ** キーリージョンデータ

    #region ** テーブルクラス

    internal class Nw4rBankVelocityRegionRangeTable : Nw4rFileInnerOffsetRefTable
    {
        #region ** イベントハンドラ

        protected override void OnWriteHeader(BinaryWriter writer)
        {
            // アイテム数の出力
            writer.Write((Byte)Items.Count);

            // キーマップの出力
            foreach (Nw4rFileInnerTableItem item in Items)
            {
                writer.Write((Byte)(item.Data as NW4rBankVelocityRegionData).VelocityRegion.XmlData.RangeMax);
            }

            new Nw4rByteAligner(writer.BaseStream).Pad(4);
        }

        #endregion
    }

    internal class Nw4rBankVelocityRegionIndexTable : Nw4rFileInnerOffsetRefTable
    {
        #region ** イベントハンドラ

        protected override void OnWriteHeader(BinaryWriter writer)
        {
            // キー範囲の出力
            writer.Write((Byte)(Items[0].Data as NW4rBankVelocityRegionData).VelocityRegion.XmlData.RangeMin);
            writer.Write((Byte)(Items[Items.Count - 1].Data as NW4rBankVelocityRegionData).VelocityRegion.XmlData.RangeMax);

            new Nw4rByteAligner(writer.BaseStream).Pad(4);
        }

        #endregion
    }

    #endregion

    internal abstract class NW4rBankKeyRegionData : Nw4rFileNode, INw4rFileInnerTableItemData
    {
        #region ** 固定値

        public const string NodeName = "KeyRegionData";

        #endregion

        private Nw4rInstrumentKeyRegion _keyRegion = null;

        public NW4rBankKeyRegionData(Nw4rInstrumentKeyRegion keyRegion) : base(NodeName)
        {
            if (null == keyRegion)
            {
                throw new Nw4rFileFormatInternalException(new ArgumentNullException("keyRegion"));
            }
            _keyRegion = keyRegion;

            Build(keyRegion);
        }
        public NW4rBankKeyRegionData() : base(NodeName) { }

        #region ** プロパティ

        public Nw4rInstrumentKeyRegion KeyRegion
        {
            get { return _keyRegion; }
        }

        public virtual int RangeMin
        {
            get
            {
                if (null == _keyRegion) { throw new Nw4rFileFormatInternalException("keyRegion is null."); }
                return _keyRegion.XmlData.RangeMin;
            }
        }

        public virtual int RangeMax
        {
            get
            {
                if (null == _keyRegion) { throw new Nw4rFileFormatInternalException("keyRegion is null."); }
                return _keyRegion.XmlData.RangeMax;
            }
        }

        protected abstract byte DataType { get; }

        byte INw4rFileInnerTableItemData.DataType
        {
            get { return DataType; }
        }

        #endregion

        #region ** イベントハンドラ

        protected virtual void OnAddDataNode(Nw4rFileNode node) { }

        #endregion

        #region ** メソッド

        protected virtual void Build(Nw4rInstrumentKeyRegion keyRegion)
        {
            Debug.Assert(null != keyRegion);

            Nw4rInstrumentVelocityRegion prevVelocityRegion = null;

            foreach (Nw4rInstrumentVelocityRegion velocityRegion in keyRegion.Components)
            {

                // NULLリージョンの追加
                if (null != prevVelocityRegion &&
                    prevVelocityRegion.XmlData.RangeMax + 1 < velocityRegion.XmlData.RangeMin)
                {
                    AddVelocityRegion(null);
                }

                AddVelocityRegion(velocityRegion);

                prevVelocityRegion = velocityRegion;

            }

            // NULLリージョンの追加
            if (null != prevVelocityRegion &&
                prevVelocityRegion.XmlData.RangeMax + 1 < Nw4rInstrumentKeyRegion.VelocityMax)
            {
                AddVelocityRegion(null);
            }
        }

        protected void AddVelocityRegion(Nw4rInstrumentVelocityRegion velocityRegion)
        {
            if (null == velocityRegion)
            {
                AddDataNode(new NW4rBankVelocityRegionNull());
                return;
            }

            AddDataNode(new NW4rBankVelocityRegionData(velocityRegion));
        }

        private void AddDataNode(Nw4rFileNode node)
        {
            AddNode(node);
            OnAddDataNode(node);
        }

        #endregion
    }

    /// <summary>
    /// キーリージョンデータクラス（ベロシティリージョンデータを直接出力）
    /// </summary>
    internal class NW4rBankKeyRegionDirect : NW4rBankKeyRegionData
    {
        #region ** 固定値

        private const byte Type = 1;	// REGION_SET_DIRECT

        #endregion

        public NW4rBankKeyRegionDirect(Nw4rInstrumentKeyRegion keyRegion) : base(keyRegion) { }

        #region ** プロパティ

        protected override byte DataType
        {
            get { return Type; }
        }

        #endregion
    }

    /// <summary>
    /// キーリージョンデータクラス（ベロシティリージョンデータを範囲で出力）
    /// </summary>
    internal class NW4rBankKeyRegionRangeTable : NW4rBankKeyRegionData
    {
        #region ** 固定値

        private const byte Type = 2;	// REGION_SET_RANGE_TABLE

        #endregion

        private Nw4rBankVelocityRegionRangeTable _table = new Nw4rBankVelocityRegionRangeTable();

        public NW4rBankKeyRegionRangeTable(Nw4rInstrumentKeyRegion keyRegion) : base(keyRegion) { }

        #region ** プロパティ

        protected override byte DataType
        {
            get { return Type; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnAddDataNode(Nw4rFileNode node)
        {
            _table.AddItem(node);
        }

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            // テーブル出力
            _table.OffsetOrigin = GetUpperNode<Nw4rBankDataBlockBody>().Offset;
            _table.Write(writer);
        }

        #endregion
    }

    /// <summary>
    /// キーリージョンデータクラス（ベロシティリージョンデータをインデックスで出力）
    /// </summary>
    internal class NW4rBankKeyRegionIndexTable : NW4rBankKeyRegionData
    {
        #region ** 固定値

        private const byte Type = 3;	// REGION_SET_INDEX_TABLE

        #endregion

        private Nw4rBankVelocityRegionIndexTable _table = new Nw4rBankVelocityRegionIndexTable();

        public NW4rBankKeyRegionIndexTable(Nw4rInstrumentKeyRegion keyRegion) : base(keyRegion) { }

        #region ** プロパティ

        protected override byte DataType
        {
            get { return Type; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnAddDataNode(Nw4rFileNode node)
        {
            _table.AddItem(node);
        }

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            // テーブル出力
            _table.OffsetOrigin = GetUpperNode<Nw4rBankDataBlockBody>().Offset;
            _table.Write(writer);
        }

        #endregion

        #region ** メソッド

        protected override void Build(Nw4rInstrumentKeyRegion keyRegion)
        {
            Debug.Assert(null != keyRegion);

            int rangeMin = (keyRegion.Components[0] as Nw4rInstrumentVelocityRegion).XmlData.RangeMin;
            int rangeMax = (keyRegion.Components[keyRegion.Components.Count - 1] as Nw4rInstrumentVelocityRegion).XmlData.RangeMax;
            int currentKey = rangeMin;

            foreach (Nw4rInstrumentVelocityRegion velocityRegion in keyRegion.Components)
            {

                // NULLリージョンの追加
                for (int key = currentKey; key < velocityRegion.XmlData.RangeMin; key++)
                {
                    AddVelocityRegion(null);
                }

                for (int key = velocityRegion.XmlData.RangeMin; key <= velocityRegion.XmlData.RangeMax; key++)
                {
                    AddVelocityRegion(velocityRegion);
                }

                currentKey = velocityRegion.XmlData.RangeMax + 1;

            }
        }

        #endregion
    }

    /// <summary>
    /// キーリージョンデータクラス（NULLリージョン）
    /// </summary>
    internal class NW4rBankKeyRegionNull : NW4rBankKeyRegionData
    {
        #region ** 固定値

        private const byte Type = 4;	// REGION_SET_NULL

        #endregion

        #region ** フィールド

        private int _rangeMin = 0;	// キーリージョンの開始位置
        private int _rangeMax = 0;	// キーリージョンの終了位置

        #endregion

        public NW4rBankKeyRegionNull(int rangeMin, int rangeMax)
        {
            _rangeMin = (rangeMin < Nw4rInstrument.KeyMin) ? Nw4rInstrument.KeyMin : rangeMin;
            _rangeMax = (rangeMax > Nw4rInstrument.KeyMax) ? Nw4rInstrument.KeyMax : rangeMax;
        }

        #region ** プロパティ

        public override int RangeMin
        {
            get { return _rangeMin; }
        }

        public override int RangeMax
        {
            get { return _rangeMax; }
        }

        protected override byte DataType
        {
            get { return Type; }
        }

        #endregion
    }

    #endregion

    #region ** ベロシティリージョンデータ

    internal class NW4rBankVelocityRegionData : Nw4rFileNode, INw4rFileInnerTableItemData
    {
        #region ** 固定値

        public const string NodeName = "VelocityRegionData";
        private const byte Type = 1;						// REGION_SET_DIRECT

        #endregion

        private Nw4rInstrumentVelocityRegion _velocityRegion = null;

        public NW4rBankVelocityRegionData(Nw4rInstrumentVelocityRegion velocityRegion) : base(NodeName)
        {
            if (null == velocityRegion) { throw new Nw4rFileFormatInternalException(new ArgumentNullException("velocityRegion")); }
            _velocityRegion = velocityRegion;
        }

        #region ** プロパティ

        public new Nw4rBankFile Root
        {
            get { return base.Root as Nw4rBankFile; }
        }

        public Nw4rInstrumentVelocityRegion VelocityRegion
        {
            get { return _velocityRegion; }
        }

        private Nw4rBank Bank
        {
            get
            {
                Debug.Assert(null != _velocityRegion.Bank);
                return _velocityRegion.Bank;
            }
        }

        private Nw4rInstrument Instrument
        {
            get
            {
                Debug.Assert(null != _velocityRegion.Instrument);
                return _velocityRegion.Instrument;
            }
        }

        private Nw4rInstrumentKeyRegion KeyRegion
        {
            get
            {
                Debug.Assert(null != (_velocityRegion.Parent as Nw4rInstrumentKeyRegion));
                return _velocityRegion.Parent as Nw4rInstrumentKeyRegion;
            }
        }

        private Nw4rAdsrEnvelope SelectedEnvelope
        {
            get
            {
                Nw4rAdsrEnvelope selectedEnvelope = null;

                switch (Instrument.XmlData.AdsrEnvelopeSelect)
                {
                    case Nw4rXmlBankAdsrEnvelopeSelect.Inst:
                        selectedEnvelope = Instrument.XmlData.AdsrEnvelope;
                        break;

                    case Nw4rXmlBankAdsrEnvelopeSelect.KeyRegion:
                        selectedEnvelope = KeyRegion.XmlData.AdsrEnvelope;
                        break;

                    case Nw4rXmlBankAdsrEnvelopeSelect.VelRegion:
                        selectedEnvelope = _velocityRegion.XmlData.AdsrEnvelope;
                        break;

                    case Nw4rXmlBankAdsrEnvelopeSelect.LowerPriority:
                    default:
                        if (null != _velocityRegion.XmlData.AdsrEnvelope)
                        {
                            selectedEnvelope = _velocityRegion.XmlData.AdsrEnvelope;
                        }
                        else if (null != KeyRegion.XmlData.AdsrEnvelope)
                        {
                            selectedEnvelope = KeyRegion.XmlData.AdsrEnvelope;
                        }
                        else
                        {
                            selectedEnvelope = Instrument.XmlData.AdsrEnvelope;
                        }
                        break;
                }

                if (null == selectedEnvelope)
                {
                    throw new Nw4rFileFormatInternalException("invalid instrument (envelope not selected)");
                }

                return selectedEnvelope;
            }
        }

        private int Pan
        {
            get
            {
                int pan = (KeyRegion.XmlData.Pan - 64) + (_velocityRegion.XmlData.Pan - 64) + 64;

                if (pan < 0) { return 0; }
                if (pan > 127) { return 127; }

                return pan;
            }
        }

        private int Volume
        {
            get
            {
                double volume = ((float)Instrument.XmlData.Volume / 127.0f) * ((float)_velocityRegion.XmlData.Volume / 127.0f);
                int volumeInt = (int)(volume * 127.0f);

                if (volumeInt < 0) { return 0; }
                if (volumeInt > 127) { return 127; }

                return volumeInt;
            }
        }

        private float Tune
        {
            get
            {
                double tune = 1.0f;

                tune *= Math.Pow(2.0f, ((double)Instrument.XmlData.FineTune) / 1200.0f);
                tune *= Math.Pow(2.0f, ((double)_velocityRegion.XmlData.FineTune) / 1200.0f);
                tune *= Math.Pow(2.0f, ((double)Instrument.XmlData.CoarseTune) / 12.0f);
                tune *= Math.Pow(2.0f, ((double)_velocityRegion.XmlData.CoarseTune) / 12.0f);

                return Convert.ToSingle(tune);
            }
        }

        byte INw4rFileInnerTableItemData.DataType
        {
            get { return Type; }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnWriteBinary(BinaryWriter writer)
        {
            base.OnWriteBinary(writer);

            writer.Write((Int32)Root.WaveFiles.IndexOf(_velocityRegion.BinaryFile.Key));	// wave index
            writer.Write((Byte)SelectedEnvelope.Attack);					// attack
            writer.Write((Byte)SelectedEnvelope.Decay);					// decay
            writer.Write((Byte)SelectedEnvelope.Sustain);					// sustain
            writer.Write((Byte)SelectedEnvelope.Release);					// release
            writer.Write((Byte)SelectedEnvelope.Hold);					// hold
            writer.Write((Byte)0);										// padding
            writer.Write((Byte)_velocityRegion.XmlData.NoteOffType);		// note off type
            writer.Write((Byte)_velocityRegion.XmlData.AlternateAssign);	// alternateAssign
            writer.Write((Byte)_velocityRegion.XmlData.OriginalKey);		// original key
            writer.Write((Byte)Volume);									// volume
            writer.Write((Byte)Pan);										// pan
            writer.Write((Byte)0);										// padding
            writer.Write(Tune);											// tune

            Nw4rFileDataAddressRef.InvalidValue.Write(writer);			// lfoTable
            Nw4rFileDataAddressRef.InvalidValue.Write(writer);			// graphEnvTable
            Nw4rFileDataAddressRef.InvalidValue.Write(writer);			// randomizerTable

            writer.Write((UInt32)0);										// reserved
        }

        #endregion
    }

    /// <summary>
    /// キーリージョンデータクラス（NULLリージョン）
    /// </summary>
    internal class NW4rBankVelocityRegionNull : Nw4rFileNode, INw4rFileInnerTableItemData
    {
        #region ** 固定値

        public const string NodeName = "VelocityRegionData";
        private const byte Type = 4;						// REGION_SET_NULL

        #endregion

        public NW4rBankVelocityRegionNull() : base(NodeName) { }

        #region ** プロパティ

        byte INw4rFileInnerTableItemData.DataType
        {
            get { return Type; }
        }

        #endregion
    }

    #endregion

    #endregion

    #endregion
}
