﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;
using System.IO;
using System.Text;
using System.Threading;

using NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Model;

namespace NintendoWare.SoundFoundation.Legacies.FileFormat.Nw4rFileFormat.Converter
{
    internal class Nw4rConvert2WaveCommand : Nw4rConvertCommandT<Nw4rWaveSound, Nw4rSoundOutput>
    {
        #region ** 固定値

        private static readonly string ConverterFileName = "nw4r_waveconv.exe";
        private static readonly string EncodingOption_PCM16 = "--pcm16 ";
        private static readonly string EncodingOption_PCM8 = "--pcm8 ";
        private static readonly string EncodingOption_ADPCM = "--adpcm ";
        private static readonly string ConvertOption_UPDATE = "--update ";
        private static readonly string ConvertOption_OUTPUT = "-o ";

        #endregion

        public Nw4rConvert2WaveCommand(Nw4rWaveSound input, Nw4rSoundOutput output) : base(input, output) { }

        #region ** プロパティ

        protected override bool ShouldConvert
        {
            get
            {
                try
                {
                    return Output.Dirty;
                }
                catch (DependencyFileNotFoundException exception)
                {
                    throw new Nw4rFileNotFoundException(exception.FileName, Input.Label);
                }
            }
        }

        private string AlignLoopOption
        {
            get
            {
                if (null == Input) { return string.Empty; }
                if (null == Input.Parent) { return string.Empty; }

                Model.Nw4rSoundSet soundSet = Input.Parent.Parent as Model.Nw4rSoundSet;
                if (null == soundSet) { return string.Empty; }

                if ("Nitro" != soundSet.XmlData.Platform) { return string.Empty; }

                switch (Input.XmlData.FileFormat)
                {
                    case Nw4rSoundSetWaveSoundFileFormat.Pcm16:
                        return "--alignloop 2 ";

                    case Nw4rSoundSetWaveSoundFileFormat.Pcm8:
                        return "--alignloop 4 ";

                    case Nw4rSoundSetWaveSoundFileFormat.Adpcm:
                        return "--alignloop 8 ";
                }

                return string.Empty;
            }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnInvoke()
        {
            // 出力ディレクトリの生成
            CreateDirectory(Path.GetDirectoryName(Output.File.FilePath), Input.Label);

            string commandArgs = string.Empty;

            switch (Input.XmlData.FileFormat)
            {
                case Nw4rSoundSetWaveSoundFileFormat.Pcm16:
                    commandArgs += EncodingOption_PCM16;
                    break;

                case Nw4rSoundSetWaveSoundFileFormat.Pcm8:
                    commandArgs += EncodingOption_PCM8;
                    break;

                case Nw4rSoundSetWaveSoundFileFormat.Adpcm:
                    commandArgs += EncodingOption_ADPCM;
                    break;

                default:
                    throw new Nw4rFileFormatInternalException();
            }

            commandArgs += AlignLoopOption;

            if (!ForcedConvert)
            {
                commandArgs += ConvertOption_UPDATE;
            }

            // 出力ファイルパス
            commandArgs += ConvertOption_OUTPUT + "\"" + Output.File.FilePath + "\" ";

            // 入力ファイルパス
            commandArgs += "\"" + Input.XmlData.FullPath + "\"";

            // テキスト出力
            string converterFilePath = Path.Combine(Nw4rSpecialFolder.CommandLineTools, ConverterFileName);
            OutputLineInternal(new InformationLine(converterFilePath + " " + commandArgs));

            // コンバート処理（処理完了までブロッキング）
            if (!Util.StartProcess(converterFilePath, commandArgs, string.Empty, OnCommandOutput, OnCommandError, ProcessHandler))
            {
                throw new Nw4rCommandException(string.Empty);
            }
        }

        protected override void OnClean()
        {
            Output.OutputItem.Clean();
        }

        private void OnCommandOutput(string message)
        {
            OutputLineInternal(new InformationLine(message, Input));
        }

        private void OnCommandError(string message)
        {
            OutputLineInternal(new ErrorLine(message, Input));
            //throw new Nw4rFileFormatException( message, Input.Label );
        }

        #endregion
    }

    internal class Nw4rConvertBankWave2WaveCommand : Nw4rConvertCommandT<Nw4rInstrumentVelocityRegion, Nw4rSoundOutput>
    {
        #region ** 固定値

        private static readonly string ConverterFileName = "nw4r_waveconv.exe";
        private static readonly string EncodingOption_PCM16 = "--pcm16 ";
        private static readonly string EncodingOption_PCM8 = "--pcm8 ";
        private static readonly string EncodingOption_ADPCM = "--adpcm ";
        private static readonly string ConvertOption_UPDATE = "--update ";
        private static readonly string ConvertOption_OUTPUT = "-o ";

        #endregion

        public Nw4rConvertBankWave2WaveCommand(Nw4rInstrumentVelocityRegion input, Nw4rSoundOutput output) : base(input, output) { }

        #region ** プロパティ

        protected override bool ShouldConvert
        {
            get
            {
                try
                {
                    return Output.Dirty;
                }
                catch (DependencyFileNotFoundException exception)
                {
                    throw new Nw4rFileNotFoundException(exception.FileName, Input.Label);
                }
            }
        }

        private string AlignLoopOption
        {
            get
            {
                if (null == Input) { return string.Empty; }
                if (null == Input.Bank) { return string.Empty; }

                if ("Nitro" != Input.Bank.XmlBankFileData.Platform) { return string.Empty; }

                switch (Input.XmlData.FileFormat)
                {
                    case Nw4rXmlBankPcmFileFormat.Pcm16:
                        return "--alignloop 2 ";

                    case Nw4rXmlBankPcmFileFormat.Pcm8:
                        return "--alignloop 4 ";

                    case Nw4rXmlBankPcmFileFormat.Adpcm:
                        return "--alignloop 8 ";
                }

                return string.Empty;
            }
        }

        #endregion

        #region ** イベントハンドラ

        protected override void OnInvoke()
        {
            // 出力ディレクトリの生成
            CreateDirectory(Path.GetDirectoryName(Output.File.FilePath), Input.Label);

            string commandArgs = string.Empty;

            switch (Input.XmlData.FileFormat)
            {
                case Nw4rXmlBankPcmFileFormat.Pcm16:
                    commandArgs += EncodingOption_PCM16;
                    break;

                case Nw4rXmlBankPcmFileFormat.Pcm8:
                    commandArgs += EncodingOption_PCM8;
                    break;

                case Nw4rXmlBankPcmFileFormat.Adpcm:
                    commandArgs += EncodingOption_ADPCM;
                    break;

                default:
                    throw new Nw4rFileFormatInternalException();
            }

            commandArgs += AlignLoopOption;

            if (!ForcedConvert)
            {
                commandArgs += ConvertOption_UPDATE;
            }

            // 出力ファイルパス
            commandArgs += ConvertOption_OUTPUT + "\"" + Output.File.FilePath + "\" ";

            // 入力ファイルパス
            commandArgs += "\"" + Input.FilePath + "\"";

            // テキスト出力
            string converterFilePath = Path.Combine(Nw4rSpecialFolder.CommandLineTools, ConverterFileName);
            OutputLineInternal(new InformationLine(converterFilePath + " " + commandArgs));

            // コンバート処理（処理完了までブロッキング）
            if (!Util.StartProcess(converterFilePath, commandArgs, string.Empty, OnCommandOutput, OnCommandError, ProcessHandler))
            {
                throw new Nw4rCommandException(string.Empty);
            }
        }

        protected override void OnClean()
        {
            Output.OutputItem.Clean();
        }

        private void OnCommandOutput(string message)
        {
            OutputLineInternal(new InformationLine(message));
        }

        private void OnCommandError(string message)
        {
            OutputLineInternal(new ErrorLine(message, new Nw4rItem[] { Input.Bank, Input.Instrument }));
            //throw new Nw4rFileFormatException( message, Input.Bank.Label + "." + Input.Instrument.Label );
        }

        #endregion
    }
}
