﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace Nintendo.AudioToolkit.WaveConverter
{
    using Nintendo.Foundation.IO;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareBinary;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Resources;
    using NintendoWare.ToolDevelopmentKit;
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;

    /// <summary>
    /// WaveConverter アプリケーションクラスです。
    /// </summary>
    public class ConsoleApplication
    {
        private const string WaveBinaryFileExtension = "bfwav";
        private const string StreamBinaryFileExtension = "bfstm";
        private const string StreamPrefetchBinaryFileExtension = "bfstp";
        private const int DefaultStreamBufferScale = 1;
        private const int DefaultStreamPrefetchBlockCount = 5;

        private Logger logger;

        /// <summary>
        /// アプリケーションを実行します。
        /// </summary>
        /// <param name="args">コマンドライン引数を指定します。</param>
        /// <returns>成功したら true、失敗したら false を返します。</returns>
        public bool Run(string[] args)
        {
            CommandLineParameters commandLineParameters = null;

            try
            {
                Nintendo.Foundation.IO.CommandLineParserSettings settings = new CommandLineParserSettings()
                {
                    ApplicationDescription = Resources.CommandLineHelpDescriptions.ApplicationDescription,
                };

                if (!new Nintendo.Foundation.IO.CommandLineParser(settings).ParseArgs(args, out commandLineParameters))
                {
                    return true;
                }
            }
            catch
            {
                return false;
            }

            this.logger = new Logger(commandLineParameters.IsVerbose);

            try
            {
                var parameters = this.ValidateParameters(commandLineParameters);

                return this.Run(parameters);
            }
            catch (Exception exception)
            {
                this.logger.WriteErrorLine(exception.Message);
                return false;
            }
        }

        private static string GetBinaryFileExtention(WaveEncoding encoding, WaveBinaryFormat format)
        {
            string result = string.Empty;

            switch (encoding)
            {
                case WaveEncoding.Pcm8:
                    result = ".pcm8.";
                    break;

                case WaveEncoding.Pcm16:
                    result = ".pcm16.";
                    break;

                case WaveEncoding.Adpcm:
                    result = ".adpcm.";
                    break;

                default:
                    throw new InvalidOperationException(string.Format(Resources.ErrorMessages.NotSupportedEncoding, encoding));
            }

            switch (format)
            {
                case WaveBinaryFormat.Bfstm:
                    return result + StreamBinaryFileExtension;

                case WaveBinaryFormat.Bfwav:
                    return result + WaveBinaryFileExtension;

                default:
                    throw new InvalidOperationException(string.Format(Resources.ErrorMessages.NotSupportedBinaryFormat, format));
            }
        }

        private bool Run(ConvertParameters parameters)
        {
            Ensure.Argument.NotNull(parameters);
            Ensure.Operation.ObjectNotNull(this.logger);

            try
            {
                Directory.CreateDirectory(
                    Path.GetDirectoryName(parameters.Output));

                switch (parameters.BinaryFormat)
                {
                    case WaveBinaryFormat.Bfstm:
                        this.ConvertToStreamBinary(parameters);
                        break;

                    case WaveBinaryFormat.Bfwav:
                        this.ConvertToWaveBinary(parameters);
                        break;

                    default:
                        // ここにはこないはず
                        throw new InvalidOperationException("invalid WaveBinaryFormat.");
                }
            }
            catch (Exception e)
            {
                this.logger.WriteErrorLine(e.Message);
                return false;
            }

            return true;
        }

        private void ConvertToWaveBinary(ConvertParameters parameters)
        {
            this.logger.WriteLine(
                string.Format("convert to {0}: {1} -> {2}", WaveBinaryFileExtension, parameters.Input, parameters.Output));

            var waveConverter = new WaveConverter("FWAV", new BinaryVersion(0, 1, 2, 0), true);

            int loopEnd = -1;

            if (parameters.LoopStart.HasValue)
            {
                // loopEnd が null（指定なし）の場合は、終端までループを示す int.MaxValue
                loopEnd = parameters.LoopEnd ?? int.MaxValue;
            }

            waveConverter.Convert(
                parameters.Input,
                parameters.Output,
                parameters.Encoding,
                parameters.LoopStart ?? -1,
                loopEnd);
        }

        private void ConvertToStreamBinary(ConvertParameters parameters)
        {
            this.logger.WriteLine(
                string.Format("convert to {0}: {1} -> {2}", StreamBinaryFileExtension, parameters.Input, parameters.Output));

            var converter = new StreamConverter();

            try
            {
                converter.Convert(parameters.Input, parameters.Encoding, parameters.LoopStart, parameters.LoopEnd);

                if (parameters.WithStreamPrefetch)
                {
                    ValidatePrefetchDataLength(converter, parameters.StreamPrefetchBlockCount);
                }

                converter.WriteBinary(parameters.Output);

                if (parameters.WithStreamPrefetch)
                {
                    converter.WritePrefetchBinary(
                        Path.ChangeExtension(parameters.Output, StreamPrefetchBinaryFileExtension),
                        parameters.StreamPrefetchBlockCount);
                }
            }
            catch (ApplicationException)
            {
                // ApplicationException は想定する例外なので、↓の catch に通らないようにする
                throw;
            }
            catch (Exception exception)
            {
                // この catch 文は、想定外のエラーハンドリングなので、StackTrace も含め、出力する
                var errorTextBuilder = new StringBuilder();
                errorTextBuilder.AppendLine(exception.ToString());

                var exceptions = new List<Exception>();
                var currentException = exception.InnerException;

                while (currentException != null)
                {
                    exceptions.Add(currentException);
                    currentException = currentException.InnerException;
                }

                foreach (var targetException in exceptions.Reverse<Exception>())
                {
                    errorTextBuilder.AppendLine("-- InnerException");
                    errorTextBuilder.AppendLine(targetException.ToString());
                }

                Console.Error.WriteLine("[error] " + errorTextBuilder.ToString());

                throw exception;
            }
            finally
            {
                converter.Reset();
            }
        }

        public void ValidatePrefetchDataLength(StreamConverter converter, int prefetchBlockCount)
        {
            // ストリームデータがストリームバッファサイズ（プリフェッチに必要なサイズ）より大きいか確認する
            var prefetchDataMinLengthPerChannel = prefetchBlockCount * converter.GetOutputWaveStreamData().BlockByte;
            var prefetchDataMinLength = prefetchDataMinLengthPerChannel * converter.GetOutputWaveFormat().ChannelCount;

            if (prefetchDataMinLength > converter.GetOutputWaveDataLength())
            {
                throw new ApplicationException(
                    string.Format(MessageResourceCommon.Message_PrefetchStreamDataIsTooShort, prefetchDataMinLengthPerChannel));
            }
        }

        private ConvertParameters ValidateParameters(CommandLineParameters parameters)
        {
            var encoding = this.ValidateEncoding(parameters.Encoding);
            var format = this.ValidateFormat(parameters.BinaryFormat);
            var loop = this.ValidateLoop(parameters);

            return new ConvertParameters()
            {
                Input = this.ValidateInputPath(parameters.Input),
                Output = this.ValidateOutputPath(parameters.Input, parameters.Output, encoding, format),
                Encoding = encoding,
                LoopStart = loop.Item1,
                LoopEnd = loop.Item2,
                BinaryFormat = format,
                WithStreamPrefetch = parameters.WithStreamPrefetch,
                StreamPrefetchBlockCount = this.ValidateStreamPrefetchBlockCount(parameters),
            };
        }

        private string ValidateInputPath(string filePath)
        {
            if (string.IsNullOrEmpty(filePath))
            {
                throw new InvalidOperationException(Resources.ErrorMessages.InputIsEmpty);
            }

            if (!File.Exists(filePath))
            {
                throw new InvalidOperationException(
                    string.Format(Resources.ErrorMessages.InputFileNotFound, filePath));
            }

            try
            {
                return Path.GetFullPath(filePath);
            }
            catch
            {
                throw new InvalidOperationException(
                    string.Format(Resources.ErrorMessages.InputFileNotFound, filePath));
            }
        }

        private string ValidateOutputPath(string inputFilePath, string outputFilePath, WaveEncoding encoding, WaveBinaryFormat format)
        {
            string filePath = outputFilePath;

            if (string.IsNullOrEmpty(outputFilePath))
            {
                filePath = Path.ChangeExtension(inputFilePath, GetBinaryFileExtention(encoding, format));
            }

            return Path.GetFullPath(filePath);
        }

        private WaveEncoding ValidateEncoding(string encoding)
        {
            Ensure.Argument.NotNull(encoding);

            WaveEncoding result;
            if (!Enum.TryParse(encoding, true, out result))
            {
                throw new InvalidOperationException(string.Format(Resources.ErrorMessages.NotSupportedEncoding, encoding));
            }

            switch (result)
            {
                case WaveEncoding.Pcm16:
                case WaveEncoding.Adpcm:
                    break;

                default:
                    throw new InvalidOperationException(
                        string.Format(Resources.ErrorMessages.NotSupportedEncoding, encoding));
            }

            return result;
        }

        private WaveBinaryFormat ValidateFormat(string format)
        {
            Ensure.Argument.NotNull(format);

            WaveBinaryFormat result;
            if (!Enum.TryParse(format, true, out result))
            {
                throw new InvalidOperationException(string.Format(Resources.ErrorMessages.NotSupportedBinaryFormat, format));
            }

            return result;
        }

        private Tuple<int?, int?> ValidateLoop(CommandLineParameters parameters)
        {
            // 以下の場合は失敗させる
            //  - LoopStart が無効値
            //  - LoopEnd が無効値
            //  - LoopEnd <= LoopStart
            //  - LoopEnd だけが指定されている（LoopStart が指定されていない）
            if (parameters.LoopStart != CommandLineParameters.UnsetValue && parameters.LoopStart < 0)
            {
                throw new InvalidOperationException(Resources.ErrorMessages.LoopStartMustBeLargerThan0);
            }

            if (parameters.LoopEnd != CommandLineParameters.UnsetValue && parameters.LoopEnd < 0)
            {
                throw new InvalidOperationException(Resources.ErrorMessages.LoopEndMustBeLargerThan0);
            }

            if (parameters.LoopStart >= 0 && parameters.LoopStart >= parameters.LoopEnd)
            {
                throw new InvalidOperationException(Resources.ErrorMessages.LoopEndMustBeLargerThanLoopStart);
            }

            if (parameters.LoopStart == CommandLineParameters.UnsetValue && parameters.LoopEnd >= 0)
            {
                throw new InvalidOperationException(Resources.ErrorMessages.LoopEndMustBeSetWithLoopStart);
            }

            // LoopStart が設定されていない場合は、ループなしを示す null を LoopStart に設定する
            // LoopEnd が設定されていない場合は、終端までループを示す null を LoopEnd に設定する
            int? loopStart = null;
            int? loopEnd = null;

            if (parameters.LoopStart != CommandLineParameters.UnsetValue)
            {
                loopStart = parameters.LoopStart;

                if (parameters.LoopEnd != CommandLineParameters.UnsetValue)
                {
                    loopEnd = parameters.LoopEnd;
                }
            }

            return new Tuple<int?, int?>(loopStart, loopEnd);
        }

        private int ValidateStreamPrefetchBlockCount(CommandLineParameters parameters)
        {
            if (parameters.StreamBufferScale.HasValue && parameters.StreamPrefetchBlockCount.HasValue)
            {
                throw new InvalidOperationException(Resources.ErrorMessages.StreamBufferScaleAndStreamPrefetchBlockSizeHaveValues);
            }

            if (parameters.StreamBufferScale.HasValue)
            {
                if (parameters.StreamBufferScale.Value < 1)
                {
                    throw new InvalidOperationException(Resources.ErrorMessages.StreamBufferScaleMustBeLargerThan1);
                }

                return DefaultStreamPrefetchBlockCount * parameters.StreamBufferScale.Value;
            }

            if (parameters.StreamPrefetchBlockCount.HasValue)
            {
                if (parameters.StreamPrefetchBlockCount.Value < 1)
                {
                    throw new InvalidOperationException(Resources.ErrorMessages.StreamPrefetchBlockCountMustBeLargerThan1);
                }

                return parameters.StreamPrefetchBlockCount.Value;
            }

            return DefaultStreamPrefetchBlockCount * DefaultStreamBufferScale;
        }

        private class Logger
        {
            private bool isVerbose = true;

            public Logger(bool isVerbose)
            {
                this.isVerbose = isVerbose;
            }

            public void WriteLine(string text)
            {
                if (this.isVerbose)
                {
                    Console.WriteLine(text);
                }
            }

            public void WriteErrorLine(string text)
            {
                Console.Error.WriteLine(text);
            }
        }

        private class CommandLineParameters
        {
            public const int UnsetValue = int.MinValue;

            public CommandLineParameters()
            {
                this.WithStreamPrefetch = false;
            }

            [CommandLineValue(0, IsRequired = true, ValueName = "input-filepath", Description = "Input", DescriptionConverterName = "LocalizeDescription")]
            public string Input { get; set; }

            [CommandLineOption("format", IsRequired = true, ValueName = "format-type", Description = "Format", DescriptionConverterName = "LocalizeDescription")]
            public string BinaryFormat { get; set; }

            [CommandLineOption("encoding", IsRequired = true, ValueName = "encoding-type", Description = "Encoding", DescriptionConverterName = "LocalizeDescription")]
            public string Encoding { get; set; }

            [CommandLineOption('v', "verbose", Description = "Verbose", DescriptionConverterName = "LocalizeDescription")]
            public bool IsVerbose { get; set; }

            [CommandLineOption('o', "output", ValueName = "output-filepath", Description = "Output", DescriptionConverterName = "LocalizeDescription")]
            public string Output { get; set; }

            [CommandLineOption("loop-start", ValueName = "frame", Description = "LoopStart", DescriptionConverterName = "LocalizeDescription")]
            public int? LoopStart { get; set; }

            [CommandLineOption("loop-end", ValueName = "frame", Description = "LoopEnd", DescriptionConverterName = "LocalizeDescription")]
            public int? LoopEnd { get; set; }

            [CommandLineOption("stream-prefetch", Description = "StreamPrefetch", DescriptionConverterName = "LocalizeDescription")]
            public bool WithStreamPrefetch { get; set; }

            [CommandLineOption("stream-buffer-scale", Description = "StreamBufferScale", DescriptionConverterName = "LocalizeDescription")]
            public int? StreamBufferScale { get; set; }

            [CommandLineOption("stream-prefetch-block-count", Description = "StreamPrefetchBlockCount", DescriptionConverterName = "LocalizeDescription")]
            public int? StreamPrefetchBlockCount { get; set; }

            public static string LocalizeDescription(string description, string argName)
            {
                return Resources.CommandLineHelpDescriptions.ResourceManager.GetString(description);
            }
        }

        private class ConvertParameters
        {
            public string Input { get; set; }

            public string Output { get; set; }

            public WaveEncoding Encoding { get; set; }

            public int? LoopStart { get; set; }

            public int? LoopEnd { get; set; }

            public WaveBinaryFormat BinaryFormat { get; set; }

            public bool WithStreamPrefetch { get; set; }

            public int StreamPrefetchBlockCount { get; set; }
        }
    }
}
