﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.Foundation.Contracts;
using Nintendo.Foundation.IO;
using NintendoWare.SoundFoundation.Core.IO;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundMaker.Framework;
using NintendoWare.SoundProjectUtility.Resources;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using NintendoWare.SoundFoundation.Documents;
using System;

namespace NintendoWare.SoundProjectUtility
{
    /// <summary>
    /// サウンドプロジェクトが参照しているファイルの一覧を出力します。
    /// </summary>
    public class PrintFilesStrategy
    {
        public class Settings : LocalizableSettings
        {
            [CommandLineOption(
                'o',
                "output",
                ValueName = "filepath",
                Description = nameof(MessageResources.CommandLine_UsageMessage_OutputFilePath),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public string OutputFilePath { get; set; }

            [CommandLineOption(
                "relative",
                ValueName = "basepath",
                Description = nameof(MessageResources.CommandLine_UsageMessage_BasePath),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public string BasePath { get; set; }

            [CommandLineOption(
                "backslash",
                Description = nameof(MessageResources.CommandLine_UsageMessage_UseBackslash),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public bool UseBackslash { get; set; }

            [CommandLineValues(
                ValueName = "input-filepath",
                CountMin = 1,
                Description = nameof(MessageResources.CommandLine_UsageMessage_InputFilePaths),
                DescriptionConverterName = nameof(LocalizeDescription))]
            public string[] InputFilePaths { get; set; }
        }

        private class RequiredServices : RequiredServicesBase
        {
            internal RequiredServices()
            {
                var intermediateOutputTraits = ServiceUtility.CreateIntermediateOutputTraits();
                this.DocumentService = new DocumentService(ServiceUtility.CreateReadOnlyDocumentServiceTraits());
                this.ProjectService = new SoundProjectServiceCommon(this.DocumentService, intermediateOutputTraits);
                this.BankServices = new BankServiceManager(this.DocumentService, intermediateOutputTraits);
            }
        }

        private readonly Settings _settings;
        private readonly SoundProjectService _soundProjectService;
        private readonly BankServiceManager _bankServices;
        private readonly HashSet<string> _filePaths = new HashSet<string>();
        private readonly string _basePath;
        private readonly bool _useBackslash;

        internal PrintFilesStrategy(Settings settings)
        {
            Ensure.Argument.NotNull(settings, nameof(settings));

            _settings = settings;
            _basePath = (settings.BasePath != null) ? Path.GetFullPath(settings.BasePath) : null;
            _useBackslash = settings.UseBackslash;

            ComponentConfigurationCommon.Instance.Initialize();
            CoreServicesProvider.SetInstance(new RequiredServices());
            _soundProjectService = CoreServicesProvider.Instance.ProjectService;
            _bankServices = CoreServicesProvider.Instance.BankServices;
        }

        internal void Execute(IResultWriter resultWriter)
        {
            TextWriter outWriter = null;
            try
            {
                if (_settings.OutputFilePath != null)
                {
                    outWriter = ResultWriter.OpenOutputFile(_settings.OutputFilePath);

                    // ResultWriter を差し替え。
                    resultWriter = new ResultWriter(outWriter);
                }

                foreach (var filePath in _settings.InputFilePaths)
                {
                    this.AccumulateSoundProject(filePath);
                }

                this.Print(resultWriter);
            }
            finally
            {
                outWriter?.Close();
            }
        }

        /// <summary>
        /// サウンドプロジェクトを開き、参照しているファイルパスの情報を蓄積します。
        /// </summary>
        /// <param name="projectFilePath"></param>
        private void AccumulateSoundProject(string projectFilePath)
        {
            string fullFilePath = Path.GetFullPath(projectFilePath);
            _soundProjectService.Open(fullFilePath);

            _soundProjectService.Documents.ForEach(it =>
            {
                this.AddFilePath(it.Key);
            });

            // プロジェクトの標準で読み込まれるコンポーネントからファイルパスを収集します。
            _soundProjectService.Project
                .EnumerateComponentTree()
                .Select(component => component.GetFilePath())
                .Where(path => !string.IsNullOrEmpty(path))
                .ForEach(path => this.AddFilePath(path));

            // バンク内のファイルパスを収集します。
            _soundProjectService.Project
                .EnumerateComponentTree()
                .OfType<SoundSetBankBase>()
                .Select(component => component.GetFilePath())
                .Where(path => !string.IsNullOrEmpty(path))
                .SelectMany(path => _bankServices.OpenItem(path).Target.Bank.EnumerateComponentTree())
                .Select(component => component.GetFilePath())
                .Where(path => !string.IsNullOrEmpty(path))
                .ForEach(path => this.AddFilePath(path));

            _bankServices.Clear();
            _soundProjectService.Close();
        }

        /// <summary>
        /// 蓄積したファイルパスの情報を出力します。
        /// </summary>
        private void Print(IResultWriter resultWriter)
        {
            _filePaths.OrderBy(it => it).ForEach(it => resultWriter.Out.WriteLine(it));
        }

        private void AddFilePath(string filePath)
        {
            // 仕様：ユーザーによる指定がなければフルパスで出力します。
            _filePaths.Add(PathUtility.FormatFilePath(filePath, _basePath, _useBackslash));
        }
    }
}
