﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;
    using System.Xml;
    using System.Xml.XPath;

    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using NintendoWare.SoundMaker.Framework.Windows;
    using NintendoWare.ToolDevelopmentKit.Collections;
    using Resources;
    using Windows.Forms;
    using Preview = NintendoWare.SoundMaker.Preview.Service;

    /// <summary>
    ///
    /// </summary>
    public class ClipboardService
    {
        private static XmlParameterSerializer _Serializer = null;
        private static XmlParameterDeserializer _Deserializer = null;

        private static string _DataName = "NintendoWareSoundMakerData";
        private static string _TypeName = "NintendoWareSoundMakerClipboardData";

        /// データの情報です。
        /// データを解析しないで情報を取得する為に利用されます。
        private static string HintDataName = "NintendoWareSoundMakerHintData";
        private static string HintTypeName = "NintendoWareSoundMakerClipboardHintData";

        /// <summary>
        ///
        /// </summary>
        static ClipboardService()
        {
            IParameterValueFactory factory = null;

            _Serializer = new XmlParameterSerializer();
            _Deserializer = new XmlParameterDeserializer();

            factory = new ParameterValueFactory<SequenceSoundFileTypeParameterValue>();
            _Serializer.Factorys.Add("FileType", factory);
            _Deserializer.Factorys.Add("FileType", factory);

            factory = new ParameterValueFactory<DecayCurve3DParameterValue>();
            _Serializer.Factorys.Add("DecayCurve3D", factory);
            _Deserializer.Factorys.Add("DecayCurve3D", factory);

            factory = new ParameterValueFactory<WaveEncodingParameterValue>();
            _Serializer.Factorys.Add("WaveEncoding", factory);
            _Deserializer.Factorys.Add("WaveEncoding", factory);

            factory = new ParameterValueFactory<PanModeParameterValue>();
            _Serializer.Factorys.Add("PanMode", factory);
            _Deserializer.Factorys.Add("PanMode", factory);

            factory = new ParameterValueFactory<PanCurveParameterValue>();
            _Serializer.Factorys.Add("PanCurve", factory);
            _Deserializer.Factorys.Add("PanCurve", factory);

            ///
            factory = new ParameterValueFactory<EnvelopeParameterValue>();
            _Serializer.Factorys.Add("Envelope", factory);
            _Deserializer.Factorys.Add("Envelope", factory);

            factory = new ParameterValueFactory<ComponentReferenceCollection>();
            _Serializer.Factorys.Add("ComponentReferenceCollection", factory);
            _Deserializer.Factorys.Add("ComponentReferenceCollection", factory);

            GetParameterFilter = delegate (Component component)
                {
                    if (component is StreamSoundBase == true)
                    {
                        return new StreamSoundFilter();
                    }
                    else if (component is WaveSoundBase == true)
                    {
                        return new WaveSoundFilter();
                    }
                    else if (component is SequenceSoundBase == true)
                    {
                        return new SequenceSoundFilter();
                    }
                    else if (component is SoundSetBankBase == true)
                    {
                        return new SoundSetBankFilter();
                    }

                    return null;
                };
        }

        private class StreamSoundFilter : IParameterFilter
        {
            bool IParameterFilter.Filter(string name)
            {
                switch (name)
                {
                    case ProjectParameterNames.SndEdit:
                    case Preview.ParameterNames.StreamSound.TrackAllocationFlags:
                    case Preview.ParameterNames.StreamSound.TotalChannelCount:
                    case Preview.ParameterNames.StreamSound.ContainerType:
                    case Preview.ParameterNames.StreamSound.BinaryFilePathForPartsConvert:
                    case Preview.ParameterNames.StreamSound.PrefetchBinaryFilePathForPartsConvert:
                    case Preview.ParameterNames.StreamSound.HasLoop:
                    case Preview.ParameterNames.StreamSound.LoopStartFrame:
                    case Preview.ParameterNames.StreamSound.LoopEndFrame:
                        return false;
                }

                return true;
            }
        }

        private class WaveSoundFilter : IParameterFilter
        {
            bool IParameterFilter.Filter(string name)
            {
                switch (name)
                {
                    case ProjectParameterNames.SndEdit:
                    case Preview.ParameterNames.WaveSound.Index:
                    case Preview.ParameterNames.WaveSound.BinaryFilePathForPartsConvert:
                    case Preview.ParameterNames.WaveSound.WaveArchiveBinaryFilePathForPartsConvert:
                        return false;
                }

                return true;
            }
        }

        private class SequenceSoundFilter : IParameterFilter
        {
            bool IParameterFilter.Filter(string name)
            {
                switch (name)
                {
                    case ProjectParameterNames.SndEdit:
                    case Preview.ParameterNames.SequenceSound.StartOffset:
                    case Preview.ParameterNames.SequenceSound.AllocateTrackFlags:
                    case Preview.ParameterNames.SequenceSound.BinaryFilePathForPartsConvert:
                    case Preview.ParameterNames.SequenceSound.BankBinaryFilePathsForPartsConvert:
                    case Preview.ParameterNames.SequenceSound.WaveArchiveBinaryFilePathsForPartsConvert:
                    case Preview.ParameterNames.SoundSetBank.BinaryFilePathForPartsConvert:
                    case Preview.ParameterNames.SoundSetBank.WaveArchiveBinaryFilePathForPartsConvert:
                        return false;
                }

                return true;
            }
        }

        private class SoundSetBankFilter : IParameterFilter
        {
            bool IParameterFilter.Filter(string name)
            {
                switch (name)
                {
                    case ProjectParameterNames.SndEdit:
                    case Preview.ParameterNames.SoundSetBank.BinaryFilePathForPartsConvert:
                    case Preview.ParameterNames.SoundSetBank.WaveArchiveBinaryFilePathForPartsConvert:
                        return false;
                }

                return true;
            }
        }

        public static Dictionary<string, Type[]> TextTypeTable
        {
            get { return _TextTypeTable; }
        }

        public static Dictionary<Type, Type> TypeChildrenTypeTable
        {
            get { return _TypeChildrenTypeTable; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public static DataObject SerializeComponents(Component[] components)
        {
            XmlDocument xmlDoc = new XmlDocument();
            XmlElement xmlDocElement = null;
            XmlElement xmlItemsElement = null;
            XmlElement xmlItemElement = null;

            xmlDocElement = xmlDoc.CreateElement(_TypeName);
            xmlDoc.AppendChild(xmlDocElement);

            //
            xmlItemsElement = xmlDoc.CreateElement("Items");
            xmlDocElement.AppendChild(xmlItemsElement);

            foreach (Component component in components)
            {
                xmlItemElement = Serialize(xmlDoc, component);
                xmlItemsElement.AppendChild(xmlItemElement);
            }

            //
            DataObject dataObject = null;
            MemoryStream stream = new MemoryStream();
            StreamWriter writer = new StreamWriter(stream);

            xmlDoc.Save(writer);

            dataObject = new DataObject();
            dataObject.SetData(_DataName, stream.GetBuffer());
            dataObject.SetData(HintDataName, GetHintData(components));
            return dataObject;
        }

        /// <summary>
        ///
        /// </summary>
        public static void Copy(Component[] components, string[] columnNames)
        {
            DataObject dataObject = SerializeComponents(components);
            dataObject.SetData(GetText(columnNames, components));
            Clipboard.SetDataObject(dataObject, true);
        }

        ///--------------------------------
        /// <summary>
        /// "セルのコピー"が可能なのかを調べる
        /// </summary>
        public static bool CanCopyCell(string[] orderedItemsName, ListItemSelectedDictionary selectedItems)
        {
            string[] itemNames = GetCommonItemsName(orderedItemsName, selectedItems);
            if (itemNames == null)
            {
                return false;
            }

            return !itemNames.Contains(ProjectParameterNames.SoundSetItem.PreviewPlay);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public static void CopyCell(string[] orderedItemsName, ListItemSelectedDictionary selectedItems)
        {
            if (selectedItems.Keys.Count <= 0)
            {
                return;
            }

            XmlDocument xmlDoc = new XmlDocument();
            XmlElement xmlDocElement = null;
            XmlElement xmlItemsElement = null;
            XmlElement xmlItemElement = null;
            string[] copyItemsName = null;

            if ((copyItemsName = GetCommonItemsName(orderedItemsName, selectedItems)) == null)
            {
                return;
            }

            xmlDocElement = xmlDoc.CreateElement(_TypeName);
            xmlDoc.AppendChild(xmlDocElement);

            xmlItemsElement = xmlDoc.CreateElement("SubItems");
            xmlDocElement.AppendChild(xmlItemsElement);

            StringBuilder sb = new StringBuilder();
            Component component = null;

            foreach (CommonListItem item in selectedItems.Keys)
            {
                XmlElement xmlParametersElement = null;
                XmlElement xmlElement = null;

                xmlItemElement = xmlDoc.CreateElement("SubItem");
                xmlParametersElement = xmlDoc.CreateElement("Parameters");
                xmlItemElement.AppendChild(xmlParametersElement);

                bool firstItemName = true;
                foreach (string itemName in copyItemsName)
                {
                    if ((component = item.GetTargetByName(itemName)) != null)
                    {
                        IParameterValue value =
                            (IParameterValue)GetConstParameterValue(itemName, component);

                        //
                        xmlElement = _Serializer.CreateParameterElement
                            (xmlDoc, itemName, value);

                        xmlParametersElement.AppendChild(xmlElement);

                        //
                        string valueText = GetText(itemName, component);
                        if (valueText != null)
                        {
                            if (firstItemName == true)
                            {
                                firstItemName = false;
                            }
                            else
                            {
                                sb.Append("\t");
                            }

                            sb.Append(valueText);
                        }
                    }
                }

                xmlItemsElement.AppendChild(xmlItemElement);

                //
                sb.AppendLine(String.Empty);
            }

            DataObject dataObject = null;
            MemoryStream stream = new MemoryStream();
            StreamWriter writer = new StreamWriter(stream);

            xmlDoc.Save(writer);

            dataObject = new DataObject();
            dataObject.SetData(_DataName, stream.GetBuffer());
            dataObject.SetData(sb.ToString());

            Clipboard.SetDataObject(dataObject, true);
        }

        ///--------------------------------
        /// <summary>
        /// "貼り付け"が可能なのかを調べる
        /// </summary>
        public static bool CanPaste(Component component)
        {
            //Type        type = null;

            if (component == null)
            {
                return false;
            }

            //type = GetChildrenType( component.GetType());
            //return CanDeserialize( Clipboard.GetDataObject() as DataObject, type);
            return CanPaste(component.GetType());
        }

        ///--------------------------------
        /// <summary>
        /// "貼り付け"が可能なのかを調べる
        /// </summary>
        public static bool CanPaste(Type[] parentTypes)
        {
            Type type = null;

            foreach (Type parentType in parentTypes)
            {
                type = GetChildrenType(parentType);
                if (CanDeserialize(Clipboard.GetDataObject() as DataObject, type) == true)
                {
                    return true;
                }
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        /// "貼り付け"が可能なのかを調べる
        /// </summary>
        public static bool CanPaste(Type parentType)
        {
            Type type = GetChildrenType(parentType);
            return CanDeserialize(Clipboard.GetDataObject() as DataObject, type);
        }

        /// <summary>
        /// 貼り付けを行ないます。
        /// </summary>
        public static Component[] Paste(ComponentService componentService, Document document, Component parentComponent, Component prevSiblingComponent, Component nextSiblingComponent, ComponentReplaceHandler replacer, bool pasteToItemLower)
        {
            SoundSetDocument targetDocument = document as SoundSetDocument;
            OperationHistory operationHistory = document.OperationHistory;
            DataObject dataObject = Clipboard.GetDataObject() as DataObject;
            List<Component> addComponents = new List<Component>();
            List<Component> removeComponents = new List<Component>();
            ProgramNoCreator programNoCreator = null;
            Instrument instrument = null;
            Component[] components = null;
            Operation operation = null;

            if ((components = Deserialize(dataObject, componentService, replacer)) != null)
            {
                try
                {
                    operationHistory.BeginTransaction();

                    //
                    foreach (Component component in components)
                    {

                        //
                        if (component is Instrument &&
                            componentService is BankService)
                        {
                            if (programNoCreator == null)
                            {
                                programNoCreator = new ProgramNoCreator
                                    (componentService as BankService);
                            }

                            instrument = component as Instrument;
                            instrument.ProgramNo = programNoCreator.Create();
                        }

                        // グループアイテムのグループへの登録です。
                        if (component is GroupItemBase)
                        {
                            GroupBase targetGroup = parentComponent as GroupBase;
                            GroupItemBase groupItem = component as GroupItemBase;

                            if (AlreadyEntryOnGroup(targetGroup, groupItem) == true)
                            {
                                continue;
                            }

                            //
                            string groupItemTargetName = groupItem.TargetItemReference;
                            Component[] targetComponents =
                                ApplicationBase.Instance.ProjectService.SequenceSoundSets
                                .Where(c => c.Name == groupItemTargetName)
                                .ToArray();

                            if (targetComponents.Length > 0)
                            {
                                Component targetComponent = targetComponents[0];
                                groupItem.Target = targetComponent as SequenceSoundSetBase;

                                GroupItemBase[] removeGroupItems =
                                    GroupUtility.GetRemoveGroupItem(targetGroup, targetComponent);
                                foreach (GroupItemBase removeGroupItem in removeGroupItems)
                                {
                                    removeComponents.Add(removeGroupItem);
                                }
                            }
                        }

                        //
                        if (component is StreamSoundTrackBase)
                        {
                            StreamSoundTrackBase track = component as StreamSoundTrackBase;

                            if (ComponentConfiguration.Instance.StreamSoundTrackNumberMaximum <
                                parentComponent.Children.Count + addComponents.Count() + 1)
                            {
                                continue;
                            }

                            if (AppConfiguration.EnabledMultiChannelAAC == true)
                            {
                                StreamSoundTrackBase firstTrack = parentComponent.Children[0] as StreamSoundTrackBase;
                                if (AACUtil.IsAACFile(firstTrack.FilePath) !=
                                    AACUtil.IsAACFile(track.FilePath))
                                {
                                    continue;
                                }
                            }
                        }

                        //
                        addComponents.Add(component);
                    }

                    //
                    if (addComponents.Count > 0)
                    {
                        int insertIndex = parentComponent.Children.Count;
                        if (pasteToItemLower == true)
                        {
                            if (nextSiblingComponent != null)
                            {
                                insertIndex = parentComponent.Children.IndexOf
                                    (nextSiblingComponent) + 1;
                            }
                        }
                        else
                        {
                            if (prevSiblingComponent != null)
                            {
                                insertIndex = parentComponent.Children.IndexOf
                                    (prevSiblingComponent);
                            }
                        }

                        operation = new InsertComponentOperation
                            (parentComponent, insertIndex, addComponents.ToArray());
                        operation.Execute();
                        operationHistory.AddOperation(operation);
                    }

                    //
                    if (removeComponents.Count > 0)
                    {
                        operation = new RemoveComponentOperation(removeComponents.ToArray());
                        operation.Execute();
                        operationHistory.AddOperation(operation);
                    }

                    operationHistory.EndTransaction();
                }
                catch
                {
                    operationHistory.CancelTransaction();
                    return null;
                }
            }

            return addComponents.ToArray();
        }

        /// <summary>
        /// 貼り付け
        /// </summary>
        public static Component[] Paste(ComponentService componentService, Document document, Type[] types, Component parentComponent, Component prevSiblingComponent, Component nextSiblingComponent, ComponentReplaceHandler replacer, bool pasteToItemLower)
        {
            //貼り付けなのか？
            if (CanPaste(types) == true)
            {
                return Paste(componentService, document,
                              parentComponent, prevSiblingComponent, nextSiblingComponent, replacer, pasteToItemLower);
            }

            return null;
        }

        /// <summary>
        /// 貼り付け
        /// </summary>
        public static Component[] Paste(ComponentService componentService, Document document, Component parentComponent, Component prevSiblingComponent, Component nextSiblingComponent, bool pasteToItemLower)
        {
            DataObject dataObject = Clipboard.GetDataObject() as DataObject;

            //貼り付けなのか？
            if (CanPaste(parentComponent) == true)
            {
                return Paste(componentService, document,
                              parentComponent, prevSiblingComponent, nextSiblingComponent, null, pasteToItemLower);
            }

            //テキストデータなのか？
            if (dataObject.GetDataPresent(DataFormats.Text) == true)
            {
                string[] texts = null;
                string text = null;
                string[] separators = new string[] {
                    "\t",
                    "\r\n"
                };

                if ((text = dataObject.GetData(DataFormats.Text) as string) == null)
                {
                    return null;
                }

                texts = text.Split(separators, StringSplitOptions.None);
                // 何も処理を行なっていない？
                // 未実装なのか？
                // 2013/7/23 aoyagi
            }

            return null;
        }

        /// <summary>
        /// リージョンの貼り付け
        /// </summary>
        public static void PasteRegion(ComponentService componentService, OperationHistory operationHistory, Component parent, Component[] targetComponents)
        {
            DataObject dataObject = Clipboard.GetDataObject() as DataObject;
            Component[] components = Deserialize(dataObject, componentService, null);
            Component component = null;
            Component target = null;
            Operation operation = null;

            if (components != null && components.Length == 1)
            {
                component = components[0];

                operationHistory.BeginTransaction();
                if (targetComponents != null && 0 < targetComponents.Length)
                {
                    foreach (Component targetComponent in targetComponents)
                    {
                        if (component is KeyRegion)
                        {
                            if (targetComponent is KeyRegion)
                            {
                                target = targetComponent;
                            }
                            else if (targetComponent is VelocityRegion &&
                                      targetComponent.Parent.Children.Count == 1)
                            {
                                target = targetComponent.Parent;
                            }
                            else
                            {
                                continue;
                            }
                            KeyRegion keyRegionTarget = (KeyRegion)target;
                            KeyRegion keyRegion = (KeyRegion)component;
                            keyRegion.KeyMin = keyRegionTarget.KeyMin;
                            keyRegion.KeyMax = keyRegionTarget.KeyMax;
                            int index = parent.Children.IndexOf(target);
                            operation = new RemoveComponentOperation(target);
                            operation.Execute();
                            operationHistory.AddOperation(operation);
                            operation = new InsertComponentOperation(parent, index, component);
                            operation.Execute();
                            operationHistory.AddOperation(operation);
                        }
                        else if (component is VelocityRegion)
                        {
                            if (targetComponent is VelocityRegion)
                            {
                                target = targetComponent;
                            }
                            else if (targetComponent is KeyRegion &&
                                      targetComponent.Children.Count == 1)
                            {
                                target = targetComponent.Children[0];
                            }
                            else
                            {
                                continue;
                            }
                            VelocityRegion velRegionTarget = (VelocityRegion)target;
                            VelocityRegion velRegion = (VelocityRegion)component;
                            IParameterValue value =
                                target.Parameters[ProjectParameterNames.FilePath];
                            IParameterValue newValue =
                                component.Parameters[ProjectParameterNames.FilePath];
                            if (value.ValidateValue(newValue.Value) == ValidationResult.NoError)
                            {
                                operation = new SetParameterOperation
                                    (target.Parameters,
                                      ProjectParameterNames.FilePath,
                                      newValue.Value);
                                operation.Execute();
                                operationHistory.AddOperation(operation);
                            }
                        }
                    }
                }
                else
                {
                    if (component is VelocityRegion)
                    {
                        KeyRegion newKeyRegion = new KeyRegion();
                        newKeyRegion.KeyMin = 0;
                        newKeyRegion.KeyMax = 127;
                        newKeyRegion.Children.Add(component);
                        component = newKeyRegion;
                    }

                    if (parent.Children == null || parent.Children.Count == 0)
                    {
                        KeyRegion keyRegion = (KeyRegion)component;
                        keyRegion.KeyMin = 0;
                        keyRegion.KeyMax = 127;
                        operation = new InsertComponentOperation(parent, 0, component);
                        operation.Execute();
                        operationHistory.AddOperation(operation);
                    }
                    else
                    {
                        int min = 0;
                        KeyRegion keyRegion = (KeyRegion)component;
                        KeyRegion targetKeyRegion = null;
                        foreach (KeyRegion child in parent.Children)
                        {
                            targetKeyRegion = child;
                            if (targetKeyRegion.KeyMin != min)
                            {
                                keyRegion.KeyMin = min;
                                keyRegion.KeyMax = targetKeyRegion.KeyMin - 1;
                                goto Next;
                            }
                            min = targetKeyRegion.KeyMax + 1;
                        }
                        if (min != 128)
                        {
                            keyRegion.KeyMin = min;
                            keyRegion.KeyMax = 127;
                            targetKeyRegion = null;
                        }
                        Next:
                        operation =
                            new InsertComponentOperation(parent, targetKeyRegion, component);
                        operation.Execute();
                        operationHistory.AddOperation(operation);
                    }
                }
                operationHistory.EndTransaction();
            }
        }

        /// <summary>
        /// "項目を選択して貼り付け"が可能なのかを調べる
        /// </summary>
        public static bool CanPasteSpecial(Component[] components)
        {
            if (components == null || components.Length == 0)
            {
                return false;
            }

            //
            DataObject dataObject = Clipboard.GetDataObject() as DataObject;
            if (dataObject == null)
            {
                return false;
            }

            //
            if (dataObject.GetDataPresent(HintDataName) == false)
            {
                return false;
            }

            //
            byte[] data = dataObject.GetData(HintDataName) as byte[];
            if (data == null)
            {
                return false;
            }

            //
            XmlDocument xmlDoc = new XmlDocument();
            MemoryStream stream = new MemoryStream(data);
            xmlDoc.Load(stream);

            string xPath = "ItemTypes";
            XmlElement xmlElement = xmlDoc.DocumentElement;
            XmlElement xmlTypesElement = xmlElement.SelectSingleNode(xPath) as XmlElement;
            if (xmlTypesElement == null)
            {
                return false;
            }

            if (xmlTypesElement.ChildNodes.Count != 1)
            {
                return false;
            }

            string targetTypeText = ToString(components[0].GetType());
            foreach (XmlElement xmlTypeElement in xmlTypesElement.ChildNodes)
            {
                string typeText = xmlTypeElement.GetAttribute("Name");
                //string typeCount = xmlTypeElement.GetAttribute( "Count");

                if (targetTypeText == typeText)
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        public class HeaderItem
        {
            private string name = String.Empty;
            private string text = String.Empty;
            private bool initialCheck = false;

            public string Name { get { return this.name; } }
            public string Text { get { return this.text; } }
            public bool Checked { get { return this.initialCheck; } }

            public HeaderItem(string name, string text, bool initialCheck)
            {
                this.name = name;
                this.text = text;
                this.initialCheck = initialCheck;
            }
        }

        /// <summary>
        /// 選択可能な項目を取得します。
        /// </summary>
        public static HeaderItem[] GetHeaderItems(ComponentHeaderAdapter adapter, ListItemSelectedDictionary selectedItems)
        {
            DataObject dataObject = Clipboard.GetDataObject() as DataObject;
            Component[] components = null;
            Component component = null;

            if ((components = Deserialize(dataObject, null, null)) == null ||
               components.Length != 1)
            {
                return null;
            }

            //
            List<HeaderItem> list = new List<HeaderItem>();
            foreach (ComponentHeaderItem item in adapter.Items)
            {
                string text = item.Text.Replace("\r\n", String.Empty);
                bool initialChecked = item.Name != "Comment" ? true : false;

                list.Add(new HeaderItem(item.Name, text, initialChecked));
            }

            //
            Instrument[] instruments = null;

            try
            {
                instruments = selectedItems.Keys
                    .Cast<CommonListItem>()
                    .Select(i => i.Target)
                    .Cast<Instrument>()
                    .ToArray<Instrument>();

            }
            catch
            {
            }

            //
            component = components[0];

            if (component is Instrument)
            {
                //
                bool changeToRegion = false;

                Instrument instrument = component as Instrument;
                if (IsSingleRegionInstrument(instrument) == true)
                {
                    if (IsSingleRegionInstruments(instruments) == false)
                    {
                        changeToRegion = true;
                    }
                }
                else
                {
                    changeToRegion = true;
                }

                if (changeToRegion == true)
                {
                    RemoveHeaderItem(list, new String[]
                    {
                        "WaveEncoding",
                        "FilePath",
                        "OriginalKey",
                        "InterpolationType",
                    });

                    list.Insert(0, new HeaderItem("Region",
                                                    MessageResource.HeaderText_Region
                                                    , true));
                }

                if (MixedRegionInstruments(instruments) == true)
                {
                    RemoveHeaderItem(list, new String[]
                    {
                        "WaveEncoding",
                        "FilePath",
                        "OriginalKey",
                        "InterpolationType",
                        "Region",
                    });
                }

                // エンベローブ関係
                if (ContainsRegionEnvelopeModeInstruments(instruments) == true ||
                    instrument.IsVelocityRegionEnvelope == true)
                {
                    RemoveHeaderItem(list, new String[]
                    {
                        "Envelope/Attack",
                        "Envelope/Decay",
                        "Envelope/Hold",
                        "Envelope/Sustain",
                        "Envelope/Release",
                    });
                }
            }

            ///
            RemoveHeaderItem(list, new String[]
            {
                "RowHeader",
                "Key",
                "Name",
                "ProgramNo",
                "PreviewPlay",
                "IntegratedLoudness",
                "WaveTime",
                "WaveTick",
                "SampleRate",
                "WaveBitRate",
                "WaveSampleBit",
                "WaveChannel",
                "TrackNo",
                "ChannelNo",
                "PreviewMute",
                "PreviewSoloPlay",
                "DataSize",
                "TargetItemReference",
                "GroupItemTargetSoundSet",
                ListTraits.ColumnName_MaximumVoiceCount,
            });

            return list.ToArray();
        }

        /// <summary>
        /// データについての情報を取得します。
        /// </summary>
        private static object GetHintData(Component[] components)
        {
            XmlDocument xmlDoc = new XmlDocument();
            XmlElement xmlDocElement = xmlDoc.CreateElement(HintTypeName);
            xmlDoc.AppendChild(xmlDocElement);

            //
            XmlElement xmlElement = GetItemTypes(xmlDoc, components);
            xmlDocElement.AppendChild(xmlElement);

            //
            MemoryStream stream = new MemoryStream();
            StreamWriter writer = new StreamWriter(stream);

            xmlDoc.Save(writer);

            return stream.GetBuffer();
        }

        ///
        private static XmlElement GetItemTypes(XmlDocument xmlDoc, Component[] components)
        {
            Dictionary<Type, int> dictionary = new Dictionary<Type, int>();
            foreach (Component component in components)
            {
                Type type = component.GetType();
                if (dictionary.ContainsKey(type) == false)
                {
                    dictionary.Add(type, 0);
                }
                dictionary[type]++;
            }

            //
            XmlElement xmlElement = xmlDoc.CreateElement("ItemTypes");

            foreach (KeyValuePair<Type, int> pair in dictionary)
            {
                string typeText = ToString(pair.Key);
                string countText = pair.Value.ToString();

                XmlElement xmlElm = xmlDoc.CreateElement("ItemType");
                xmlElm.SetAttribute("Name", typeText);
                xmlElm.SetAttribute("Count", countText);

                xmlElement.AppendChild(xmlElm);
            }

            return xmlElement;
        }

        /// <summary>
        /// リストから指定した名前のアイテムを削除します。
        /// </summary>
        private static void RemoveHeaderItem(List<HeaderItem> list, string[] names)
        {
            foreach (string name in names)
            {
                int index = list.FindIndex(delegate (HeaderItem headerItem)
                       {
                           return headerItem.Name == name ? true : false;
                       });

                if (index >= 0)
                {
                    list.RemoveAt(index);
                }
            }
        }

        /// <summary>
        /// シングルリージョンなのか調べます。
        /// </summary>
        private static bool IsSingleRegionInstrument(Instrument instrument)
        {
            return IsSingleRegionInstruments(new Instrument[] { instrument });
        }

        /// <summary>
        /// シングルリージョンだけなのか調べます。
        /// </summary>
        private static bool IsSingleRegionInstruments(Instrument[] instruments)
        {
            int count = 0;

            foreach (Instrument instrument in instruments)
            {
                if (instrument.Children.Count == 1 &&
                    instrument.Children[0].Children.Count == 1)
                {
                    count++;
                }
            }
            return count == instruments.Length ? true : false;
        }

        /// <summary>
        /// シングル、マルチリージョンが混ざっているのか調べます。
        /// </summary>
        private static bool MixedRegionInstruments(Instrument[] instruments)
        {
            int sCount = 0;
            int mCount = 0;

            foreach (Instrument instrument in instruments)
            {
                if (instrument.Children.Count == 1 &&
                    instrument.Children[0].Children.Count == 1)
                {
                    sCount++;
                }
                else
                {
                    mCount++;
                }

                if (sCount > 0 && mCount > 0)
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// リージョン単位でエンベローブを指定するインストルメントが含まれているのか調べます。
        /// </summary>
        private static bool ContainsRegionEnvelopeModeInstruments(Instrument[] instruments)
        {
            foreach (Instrument instrument in instruments)
            {
                if (instrument.IsVelocityRegionEnvelope == true)
                {
                    return true;
                }
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        /// 項目を選択して貼り付け
        /// </summary>
        public static bool PasteSpecial(OperationHistory operationHistory, string[] itemsName, ListItemSelectedDictionary selectedItems)
        {
            DataObject dataObject = Clipboard.GetDataObject() as DataObject;
            Operation operation = null;
            Component component = null;
            Component[] newComponents = null;
            Component newComponent = null;

            if ((newComponents = Deserialize(dataObject, null, null)) == null ||
               newComponents.Length != 1)
            {
                return false;
            }

            newComponent = newComponents[0];

            //キーリージョンから貼り付けることはないので、子供のベロシティリージョンにする
            if (newComponent is KeyRegion)
            {
                if (newComponent.Children.Count <= 0)
                {
                    return false;
                }
                newComponent = newComponent.Children[0];
            }

            //
            operationHistory.BeginTransaction();

            foreach (CommonListItem item in selectedItems.Keys)
            {

                foreach (string itemName in itemsName)
                {

                    //
                    component = item.Target;

                    //
                    if (DoNeedReplaceChildren(component, itemName) == true)
                    {
                        Component[] childComponents = null;

                        childComponents = component.Children.Cast<Component>().ToArray();
                        operation = new RemoveComponentOperation(childComponents);
                        operation.Execute();
                        operationHistory.AddOperation(operation);

                        //
                        childComponents = newComponent.Children.Cast<Component>().ToArray();

                        foreach (Component childComponent in childComponents)
                        {
                            Component clonedComponent = null;
                            ICloneable cloner = childComponent as ICloneable;
                            if (cloner != null)
                            {
                                clonedComponent = cloner.Clone() as Component;
                                operation = new InsertComponentOperation
                                    (component, null, clonedComponent);

                                operation.Execute();
                                operationHistory.AddOperation(operation);
                            }
                        }

                        continue;
                    }

                    //
                    if ((component = item.GetTargetByName(itemName)) == null)
                    {
                        continue;
                    }

                    if (component is SequenceSoundBase == true)
                    {
                        SequenceSoundBase oldSeq = component as SequenceSoundBase;
                        SequenceSoundBase newSeq = newComponent as SequenceSoundBase;

                        switch (itemName)
                        {
                            case ProjectParameterNames.SequenceSound.SoundSetBankReference0:
                                PasteCellOnBankReferences(operationHistory, oldSeq, newSeq, itemName, 0);
                                continue;

                            case ProjectParameterNames.SequenceSound.SoundSetBankReference1:
                                PasteCellOnBankReferences(operationHistory, oldSeq, newSeq, itemName, 1);
                                continue;

                            case ProjectParameterNames.SequenceSound.SoundSetBankReference2:
                                PasteCellOnBankReferences(operationHistory, oldSeq, newSeq, itemName, 2);
                                continue;

                            case ProjectParameterNames.SequenceSound.SoundSetBankReference3:
                                PasteCellOnBankReferences(operationHistory, oldSeq, newSeq, itemName, 3);
                                continue;
                        }
                    }

                    //
                    if (component is StreamSoundTrackBase == true)
                    {
                        if (itemName == ProjectParameterNames.FilePath)
                        {
                            string chCount = ProjectParameterNames.StreamSoundTrack.ChannelCount;
                            IParameterValue newValue = GetTargetByName(newComponent, itemName).Parameters[chCount];
                            operation = new SetParameterOperation(component.Parameters, chCount, newValue.Value);
                            operation.Execute();
                            operationHistory.AddOperation(operation);
                        }
                    }

                    if (ListTraits.IsUserParameters(itemName) == true)
                    {
                        UserParameterStructure structure = UserParameterSettingWatcher.GetStructure(itemName);
                        string userParameterName = UserParameterSettingWatcher.GetUserParameterName(itemName);
                        ulong oldValue = (ulong)component.Parameters.GetValue(userParameterName).Value;
                        ulong newValue = (ulong)newComponent.Parameters.GetValue(userParameterName).Value;
                        object value = UserParameterUtility.GetValue(structure, newValue);
                        newValue = UserParameterUtility.SetValue(structure, oldValue, value);

                        operation = new SetParameterOperation(component.Parameters, userParameterName, newValue);
                        operation.Execute();
                        operationHistory.AddOperation(operation);
                        continue;
                    }

                    //
                    {
                        IParameterValue value = component.Parameters[itemName];
                        IParameterValue newValue = GetTargetByName(newComponent, itemName).Parameters[itemName];

                        if (value.ValidateValue(newValue.Value) == ValidationResult.NoError)
                        {
                            operation = new SetParameterOperation(component.Parameters, itemName, newValue.Value);
                            operation.Execute();
                            operationHistory.AddOperation(operation);
                        }
                    }
                }
            }

            operationHistory.EndTransaction();
            return true;
        }

        /// <summary>
        /// 子供の差し替えが必要なのか調べます。
        /// </summary>
        private static bool DoNeedReplaceChildren(Component component, string name)
        {
            // ストリームサウンド、項目名が"FilePath"、"FrontByPass"の時の処理
            if (component is StreamSoundBase)
            {
                if (name == ProjectParameterNames.FilePath ||
                    name == ProjectParameterNames.FrontBypass)
                {
                    return true;
                }
            }

            // インストルメント、項目名が"Region"の時の処理
            if (component is Instrument)
            {
                if (name == "Region")
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        private static Component GetTargetByName(Component component, string itemName)
        {
            if (component is Instrument)
            {
                if (itemName == "WaveEncoding" ||
                    itemName == "FilePath" ||
                    itemName == "OriginalKey" ||
                    itemName == "InterpolationType")
                {
                    Instrument instrument = component as Instrument;
                    return instrument.Children[0].Children[0];
                }
            }

            return component;
        }

        ///--------------------------------
        /// <summary>
        /// "セルの貼り付け"が可能なのかを調べる
        /// </summary>
        public static bool CanPasteCell(string[] orderedItemsName, ListItemSelectedDictionary selectedItems)
        {
            DataObject dataObject = Clipboard.GetDataObject() as DataObject;
            string[] values = null;
            int width = 0;
            int height = 0;

            //テキストデータなのか？
            if (dataObject.GetDataPresent(DataFormats.Text) == true)
            {
                string[] texts = null;
                string text = null;

                if ((text = dataObject.GetData
                     (DataFormats.Text) as String) == null)
                {
                    return false;
                }

                texts = SplitString(text);
                return text.Length > 0 ? true : false;
            }

            if ((values = DeserializeText(dataObject, ref width, ref height)) == null)
            {
                return false;
            }

            foreach (ListItemSelectedState state in selectedItems.Values)
            {
                if (state.SubSelected.Count > 0)
                {
                    return true;
                }
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public static void PasteCell(OperationHistory operationHistory, Component parentComponent, string[] orderedItemsName, ListItemSelectedDictionary selectedItems, ComponentListItemCollection itemsCollection)
        {
            DataObject dataObject = Clipboard.GetDataObject() as DataObject;
            ListItemSelectedState state = null;
            Component component = null;
            ComponentListItem[] items = null;
            string[] values = null;
            string[] itemsName = null;
            int width = 0;
            int height = 0;

            //
            if (dataObject.GetDataPresent(DataFormats.Text) == true &&
                dataObject.GetDataPresent(_DataName) == false)
            {
                string text = null;
                if ((text = dataObject.GetData(DataFormats.Text) as String) == null)
                {
                    return;
                }

                //
                HashSet<string> errors = new HashSet<string>();
                int index = 0;
                values = SplitString(text);

                operationHistory.BeginTransaction();

                foreach (ComponentListItem item in selectedItems.Keys)
                {
                    component = item.Target;
                    state = selectedItems[item];

                    itemsName = GetOrderedItemsName(orderedItemsName, state);

                    foreach (string itemName in itemsName)
                    {
                        ValidationResult result = PasteCellForText
                            (operationHistory, orderedItemsName,
                              item as CommonListItem, itemName, values[index]);
                        if (result.IsValid == false)
                        {
                            errors.Add(result.ToString());
                        }

                        if (++index >= values.Length)
                        {
                            index = 0;
                        }
                    }
                }

                operationHistory.EndTransaction();

                // 貼り付けの処理中に発生したエラーメッセージを表示します。
                foreach (string error in errors)
                {
                    ApplicationBase.Instance.UIService.ShowMessageBox
                        (error, AppMessageBoxButton.OK, AppMessageBoxImage.Warning);
                }

                return;
            }

            //
            if ((values = DeserializeText(dataObject, ref width, ref height)) == null)
            {
                return;
            }

            //
            PasteResult pasteResult = new PasteResult();

            operationHistory.BeginTransaction();

            foreach (ComponentListItem item in selectedItems.Keys)
            {
                component = item.Target;
                state = selectedItems[item];

                itemsName = GetOrderedItemsName(orderedItemsName, state);
                items = GetItems(itemsCollection, item, height);

                foreach (string itemName in itemsName)
                {
                    PasteCell(pasteResult, operationHistory, orderedItemsName, items,
                               itemName, values, width);
                }
            }

            operationHistory.EndTransaction();

            //
            if (pasteResult.Successful == false)
            {
                ApplicationBase.Instance.UIService.ShowMessageBox
                    (Resources.MessageResource.Message_FaildPasteCellContains,
                      AppMessageBoxButton.OK,
                      AppMessageBoxImage.Warning);
            }
        }

        /// <summary>
        /// 検索結果パネル用のセルの貼り付け
        /// </summary>
        public static TransactionDictionary PasteCellForFind(string[] orderedItemsName, ComponentListItem[] allItems, ListItemSelectedDictionary selectedItems)
        {
            DataObject dataObject = Clipboard.GetDataObject() as DataObject;
            string[] values = null;
            int width = 0;
            int height = 0;

            //
            if ((values = DeserializeText(dataObject, ref width, ref height)) == null)
            {
                return null;
            }

            //
            PasteResult pasteResult = new PasteResult();
            TransactionDictionary dictionary = new TransactionDictionary();

            foreach (ComponentListItem item in selectedItems.Keys)
            {
                Component component = item.Target;
                ListItemSelectedState state = selectedItems[item];
                string[] itemsName = GetOrderedItemsName(orderedItemsName, state);

                // 貼り付けの対象アイテムを取得します。
                // 対象アイテムの Transactionも作成します。
                ComponentListItem[] targetItems = GetItems(allItems, item, height);
                foreach (ComponentListItem targetItem in targetItems)
                {
                    Document document = GetDocument(targetItem.Target);
                    if (dictionary.ContainsKey(document) == false)
                    {
                        dictionary.Add(document, new UserTransaction2(String.Empty));
                    }
                }

                //
                foreach (string itemName in itemsName)
                {
                    PasteCell(pasteResult, dictionary, orderedItemsName, targetItems,
                               itemName, values, width);
                }
            }

            // 不要な Transactionを削除します。
            Document[] removeCandidates = dictionary
                .Where(p => p.Value.Operations.Count <= 0)
                .Select(p => p.Key)
                .ToArray();

            foreach (Document document in removeCandidates)
            {
                dictionary.Remove(document);
            }

            //
            if (pasteResult.Successful == false)
            {
                ApplicationBase.Instance.UIService.ShowMessageBox
                    (Resources.MessageResource.Message_FaildPasteCellContains,
                      AppMessageBoxButton.OK,
                      AppMessageBoxImage.Warning);
            }

            return dictionary;
        }

        /// <summary>
        ///
        /// </summary>
        private class PasteResult
        {
            public class ErrorItem
            {
                private Component target = null;
                private string message = null;

                public ErrorItem(Component target, string message)
                {
                    this.target = target;
                    this.message = message;
                }

                public Component Target
                {
                    get
                    {
                        return this.target;
                    }
                }

                public string Message
                {
                    get
                    {
                        return this.message;
                    }
                }
            }

            private List<ErrorItem> errorItems = new List<ErrorItem>();

            public List<ErrorItem> ErrorItems
            {
                get
                {
                    return this.errorItems;
                }
            }

            public bool Successful
            {
                get
                {
                    return this.errorItems.Count == 0 ? true : false;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private static void PasteCell(PasteResult pasteResult, TransactionDictionary dictionary, string[] orderedItemsName, ComponentListItem[] items, string targetName, string[] values, int width)
        {
            Component component = null;
            int valueIndex = 0;
            int targetIndex = ToIndex(orderedItemsName, targetName);
            string name = null;
            IParameterValue value = null;
            ValidationResult result = null;
            object parsedValue = null;
            string text = null;

            foreach (CommonListItem item in items)
            {
                for (int index = targetIndex; index < targetIndex + width; index++)
                {
                    if ((name = ToString(orderedItemsName, index)) != null)
                    {
                        try
                        {
                            component = item.GetTargetByName(name);
                            if ((value = item.GetValue(name, true)) == null)
                            {
                                //continue;
                                goto Done;
                            }
                            text = values[valueIndex];

                            //
                            Document document = GetDocument(component);
                            UserTransaction2 transaction = dictionary[document];

                            //
                            if (component is GroupItemBase)
                            {
                                GroupBase group = component.Parent as GroupBase;
                                if (AlreadyEntryOnGroup(group, text) == true)
                                {
                                    //continue;
                                    goto Done;
                                }
                            }

                            //
                            switch (name)
                            {
                                case ProjectParameterNames.VelocityRegion.OriginalKey:
                                    parsedValue = KeyNoteConverter.ToNote(int.Parse(text));
                                    break;

                                default:
                                    parsedValue = value.ParseValue(text);
                                    break;
                            }

                            //
                            result = value.ValidateValue(parsedValue);
                            if (result.IsValid == true)
                            {
                                switch (name)
                                {
                                    case ProjectParameterNames.SequenceSound.SoundSetBankReference0:
                                    case ProjectParameterNames.SequenceSound.SoundSetBankReference1:
                                    case ProjectParameterNames.SequenceSound.SoundSetBankReference2:
                                    case ProjectParameterNames.SequenceSound.SoundSetBankReference3:
                                        SequenceSoundBase sequenceSound = component as SequenceSoundBase;
                                        Operation[] operations = PasteCellOnBankReferences
                                            (sequenceSound, name, parsedValue.ToString());
                                        foreach (Operation operation in operations)
                                        {
                                            transaction.Operations.Add(operation);
                                        }
                                        break;

                                    case ProjectParameterNames.SoundSetItem.WaveArchiveReference:
                                        {
                                            SoundSetItem soundSetItem = component as SoundSetItem;
                                            Operation operation = PasteCellOnWaveArchiveReference
                                                (soundSetItem, parsedValue.ToString());
                                            transaction.Operations.Add(operation);
                                        }
                                        break;

                                    default:
                                        {
                                            Operation operation = new SetParameterOperation
                                                (component.Parameters, name, values[valueIndex]);
                                            operation.Execute();
                                            transaction.Operations.Add(operation);
                                        }
                                        break;
                                }
                            }
                            else
                            {
                                pasteResult.ErrorItems.Add
                                    (new PasteResult.ErrorItem(item.Target, result.ToString()));
                            }
                        }
                        catch
                        {
                        }
                    }

                    Done:
                    valueIndex++;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private static ComponentListItem[] GetItems(ComponentListItem[] items, ComponentListItem item, int count)
        {
            List<ComponentListItem> list = new List<ComponentListItem>();
            int beginIndex = Array.IndexOf(items, item);
            int endIndex = 0;

            if ((endIndex = beginIndex + count - 1) >= items.Length)
            {
                endIndex = items.Length - 1;
            }

            for (int index = beginIndex; index <= endIndex; index++)
            {
                list.Add(items[index] as ComponentListItem);
            }
            return list.ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        private static Document GetDocument(Component component)
        {
            return GetSoundSetDocument(GetSoundSet(component));
        }

        /// <summary>
        ///
        /// </summary>
        private static SoundSet GetSoundSet(Component component)
        {
            while (component.Parent != null)
            {
                if (component.Parent is SoundSet)
                {
                    return component.Parent as SoundSet;
                }
                component = component.Parent;
            }
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        private static SoundSetDocument GetSoundSetDocument(SoundSet soundSet)
        {
            SoundSetDocument soundSetDocument =
                FormsApplication.Instance.DocumentService.Documents
                .Where(d => d is SoundSetDocument)
                .Cast<SoundSetDocument>()
                .Where(d => d.SoundSet == soundSet)
                .First();
            return soundSetDocument;
        }

        /// <summary>
        ///
        /// </summary>
        private static ValidationResult PasteCellForText(OperationHistory operationHistory, string[] orderedItemsName, CommonListItem item, string targetName, string text)
        {
            int targetIndex = ToIndex(orderedItemsName, targetName);
            string name = null;

            if ((name = ToString(orderedItemsName, targetIndex)) != null)
            {
                try
                {
                    Component component = item.GetTargetByName(name);
                    IParameterValue value = item.GetValue(name, true);

                    //
                    if (component is GroupItemBase)
                    {
                        if (AlreadyEntryOnGroup(component.Parent as GroupBase, text) == true)
                        {
                            return new ValidationResult
                                (false, Resources.MessageResource.Message_AlreadyRegistGroupItem);
                        }
                    }

                    //
                    object parsedValue = null;
                    switch (name)
                    {
                        case ProjectParameterNames.VelocityRegion.OriginalKey:
                            parsedValue = text;
                            text = KeyNoteConverter.ToKey(text).ToString();
                            break;

                        default:
                            parsedValue = value.ParseValue(text);
                            break;
                    }

                    //
                    ValidationResult result = value.ValidateValue(parsedValue);
                    if (result.IsValid == false)
                    {
                        return result;
                    }

                    switch (name)
                    {
                        case ProjectParameterNames.SequenceSound.SoundSetBankReference0:
                        case ProjectParameterNames.SequenceSound.SoundSetBankReference1:
                        case ProjectParameterNames.SequenceSound.SoundSetBankReference2:
                        case ProjectParameterNames.SequenceSound.SoundSetBankReference3:
                            PasteCellOnBankReferences(operationHistory,
                                                       component as SequenceSoundBase,
                                                       name,
                                                       parsedValue.ToString());
                            break;

                        default:
                            Operation operation = new SetParameterOperation
                                (component.Parameters, name, text);
                            operation.Execute();
                            operationHistory.AddOperation(operation);
                            break;
                    }
                }
                catch
                {
                    return new ValidationResult
                        (false, Resources.MessageResource.Message_UnknownError);
                }
            }

            return ValidationResult.NoError;
        }

        /// <summary>
        /// 文字列の分割
        /// </summary>
        private static string[] SplitString(string text)
        {
            string[] separators = new string[] {
                "\t",
                "\r\n"
            };

            return text.Split(separators, StringSplitOptions.None);
        }

        /// <summary>
        ///
        /// </summary>
        private static void PasteCell(PasteResult pasteResult, OperationHistory operationHistory, string[] orderedItemsName, ComponentListItem[] items, string targetName, string[] values, int width)
        {
            Operation operation = null;
            Component component = null;
            int valueIndex = 0;
            int targetIndex = ToIndex(orderedItemsName, targetName);
            string name = null;
            IParameterValue value = null;
            ValidationResult result = null;
            object parsedValue = null;
            string text = null;

            foreach (CommonListItem item in items)
            {

                for (int index = targetIndex; index < targetIndex + width; index++)
                {
                    if ((name = ToString(orderedItemsName, index)) != null)
                    {

                        try
                        {
                            component = item.GetTargetByName(name);
                            value = item.GetValue(name, true) as IParameterValue;
                            text = values[valueIndex];

                            //
                            if (component is GroupItemBase)
                            {
                                if (AlreadyEntryOnGroup
                                    (component.Parent as GroupBase, text) == true)
                                {
                                    continue;
                                }
                            }

                            //
                            switch (name)
                            {
                                case ProjectParameterNames.VelocityRegion.OriginalKey:
                                    parsedValue = KeyNoteConverter.ToNote(int.Parse(text));
                                    break;

                                default:
                                    parsedValue = value.ParseValue(text);
                                    break;
                            }

                            //
                            result = value.ValidateValue(parsedValue);
                            if (result.IsValid == true)
                            {
                                switch (name)
                                {
                                    case ProjectParameterNames.SequenceSound.SoundSetBankReference0:
                                    case ProjectParameterNames.SequenceSound.SoundSetBankReference1:
                                    case ProjectParameterNames.SequenceSound.SoundSetBankReference2:
                                    case ProjectParameterNames.SequenceSound.SoundSetBankReference3:
                                        PasteCellOnBankReferences(operationHistory,
                                                                   component as SequenceSoundBase,
                                                                   name,
                                                                   parsedValue.ToString());
                                        break;

                                    case ProjectParameterNames.SoundSetItem.WaveArchiveReference:
                                        PasteCellOnWaveArchiveReference(operationHistory,
                                                                         component as SoundSetItem,
                                                                         parsedValue.ToString());
                                        break;

                                    case ProjectParameterNames.SampleRate:
                                        operation = new SetParameterOperation(component.Parameters, name, parsedValue);
                                        operation.Execute();
                                        operationHistory.AddOperation(operation);
                                        if ((bool)component.Parameters[ProjectParameterNames.IsResampleEnabled].Value == false)
                                        {
                                            operation = new SetParameterOperation(component.Parameters, ProjectParameterNames.IsResampleEnabled, true);
                                            operation.Execute();
                                            operationHistory.AddOperation(operation);
                                        }
                                        break;

                                    default:
                                        if (ListTraits.IsUserParameters(name) == true)
                                        {
                                            UserParameterStructure structure = UserParameterSettingWatcher.GetStructure(name);
                                            string userParameterName = UserParameterSettingWatcher.GetUserParameterName(name);
                                            ulong oldValue = (ulong)component.Parameters.GetValue(userParameterName).Value;
                                            parsedValue = UserParameterUtility.SetValue(structure, oldValue, parsedValue);
                                            name = userParameterName;
                                        }

                                        operation = new SetParameterOperation(component.Parameters, name, parsedValue);
                                        operation.Execute();
                                        operationHistory.AddOperation(operation);
                                        break;
                                }
                            }
                            else
                            {
                                pasteResult.ErrorItems.Add
                                    (new PasteResult.ErrorItem(item.Target, result.ToString()));
                            }
                        }

                        catch { }
                    }
                    valueIndex++;
                }
            }
        }

        /// <summary>
        /// シーケンスサウンドの参照サウンドセットバンクの項目への貼り付けを行います。
        /// </summary>
        private static Operation[] PasteCellOnBankReferences(SequenceSoundBase sequenceSound, string name, string value)
        {
            int index = -1;
            switch (name)
            {
                case ProjectParameterNames.SequenceSound.SoundSetBankReference0: index = 0; break;
                case ProjectParameterNames.SequenceSound.SoundSetBankReference1: index = 1; break;
                case ProjectParameterNames.SequenceSound.SoundSetBankReference2: index = 2; break;
                case ProjectParameterNames.SequenceSound.SoundSetBankReference3: index = 3; break;
                default: return new Operation[0];
            }

            string parameterName = ProjectParameterNames.SequenceSound.SoundSetBankReferences;
            ObservableList<ComponentReference> reference =
                (ObservableList<ComponentReference>)(sequenceSound.Parameters[parameterName].Value);
            SetParameterOperation operation = new SetParameterOperation
                (reference[index].Parameters,
                  ProjectParameterNames.TargetName,
                  value);
            operation.Execute();
            return new Operation[] { operation };
        }

        /// <summary>
        /// シーケンスサウンドの参照サウンドセットバンクの項目への貼り付けを行います。
        /// </summary>
        private static void PasteCellOnBankReferences(OperationHistory operationHistory, SequenceSoundBase oldSeq, SequenceSoundBase newSeq, string itemName, int itemIndex)
        {
            Debug.Assert(0 <= itemIndex && itemIndex <= 3);

            string paramName = ProjectParameterNames.SequenceSound.SoundSetBankReferences;
            ObservableList<ComponentReference> reference = newSeq.Parameters[paramName].Value as ObservableList<ComponentReference>;
            string newValue = reference[itemIndex].Parameters[ProjectParameterNames.TargetName].Value as string;
            PasteCellOnBankReferences(operationHistory, oldSeq, itemName, newValue);
        }

        /// <summary>
        /// シーケンスサウンドの参照サウンドセットバンクの項目への貼り付けを行います。
        /// </summary>
        private static bool PasteCellOnBankReferences(OperationHistory operationHistory, SequenceSoundBase sequenceSound, string name, string value)
        {
            Operation[] operations = PasteCellOnBankReferences(sequenceSound, name, value);
            foreach (Operation operation in operations)
            {
                operationHistory.AddOperation(operation);
            }
            return true;
        }

        /// <summary>
        /// 参照する波形アーカイブの項目への貼り付けを行います。
        /// </summary>
        private static Operation PasteCellOnWaveArchiveReference(SoundSetItem soundSetItem, string text)
        {
            object value = text;

            if (text == MessageResource.Label_WaveArchiveReference_Shared)
            {
                value = WaveArchiveConsts.AutoShared;
            }

            if (text == MessageResource.Label_WaveArchiveReference_Individual)
            {
                value = WaveArchiveConsts.AutoIndividual;
            }

            Operation operation = new SetParameterOperation
                (soundSetItem.Parameters,
                  ProjectParameterNames.SoundSetItem.WaveArchiveReference, value);
            operation.Execute();

            return operation;
        }

        /// <summary>
        /// 参照する波形アーカイブの項目への貼り付けを行います。
        /// </summary>
        private static bool PasteCellOnWaveArchiveReference(OperationHistory operationHistory, SoundSetItem soundSetItem, string text)
        {
            Operation operation = PasteCellOnWaveArchiveReference(soundSetItem, text);
            operationHistory.AddOperation(operation);
            return true;
        }

        /// <summary>
        /// グループ内に指定グループアイテムが登録されているのか調べます。
        /// </summary>
        private static bool AlreadyEntryOnGroup(GroupBase group, GroupItemBase groupItem)
        {
            return AlreadyEntryOnGroup(group, groupItem.TargetItemReference);
        }

        /// <summary>
        /// 指定した名前のグループか、グループアイテムが登録されているのか調べます。
        /// </summary>
        private static bool AlreadyEntryOnGroup(GroupBase group, string targetName)
        {
            //自分自身なのか？
            if (group.Name == targetName)
            {
                return true;
            }

            //
            foreach (GroupItemBase groupItem in group.Children)
            {
                if (groupItem.TargetItemReference == targetName)
                {
                    return true;
                }
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static ComponentListItem[] GetItems(ComponentListItemCollection items, ComponentListItem item, int count)
        {
            List<ComponentListItem> list = new List<ComponentListItem>();
            int beginIndex = items.IndexOf(item);
            int endIndex = 0;

            if ((endIndex = beginIndex + count - 1) >= items.Count)
            {
                endIndex = items.Count - 1;
            }

            for (int index = beginIndex; index <= endIndex; index++)
            {
                list.Add(items[index] as ComponentListItem);
            }
            return list.ToArray();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static int ToIndex(string[] names, string name)
        {
            for (int index = 0; index < names.Length; index++)
            {
                if (names[index] == name)
                {
                    return index;
                }
            }
            return -1;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static string ToString(string[] names, int index)
        {
            if (index < 0 || index >= names.Length)
            {
                return null;
            }
            return names[index];
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static string[] GetOrderedItemsName(string[] orderedItemsName, ListItemSelectedState state)
        {
            List<string> list = new List<string>();

            foreach (string itemName in orderedItemsName)
            {
                if (state.SubSelected.Contains(itemName) == true)
                {
                    list.Add(itemName);
                }
            }
            return list.ToArray();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static string[] GetCommonItemsName(string[] orderedItemsName, ListItemSelectedDictionary selectedItems)
        {
            ListItemSelectedState cardinalState = null;
            List<string> list = new List<string>();

            foreach (ListItemSelectedState state in selectedItems.Values)
            {

                if (cardinalState == null)
                {
                    cardinalState = state;

                    foreach (string itemName in orderedItemsName)
                    {
                        if (state.SubSelected.Contains(itemName) == true)
                        {
                            list.Add(itemName);
                        }
                    }
                }
                else
                {

                    if (state.SubSelected.Count != list.Count)
                    {
                        return null;
                    }

                    foreach (string itemName in list)
                    {
                        if (state.SubSelected.Contains(itemName) == false)
                        {
                            return null;
                        }
                    }
                }
            }
            return list.Count > 0 ? list.ToArray() : null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static XmlElement Serialize(XmlDocument xmlDoc, Component component)
        {
            string name = ToString(component.GetType());
            XmlElement xmlElement = xmlDoc.CreateElement(name);
            XmlElement xmlParametersElement = null;

            xmlParametersElement = _Serializer.Execute(xmlDoc, component.Parameters,
                                                       key => !key.EndsWith(ParameterNames.TemporaryPostfix));

            //
            if (component is Instrument)
            {
                Instrument instrument = component as Instrument;
                VelocityRegion velRegion = null;

                if (instrument.Children.Count > 0 &&
                    instrument.Children[0].Children.Count > 0)
                {
                    velRegion = instrument.Children[0].Children[0] as VelocityRegion;

                    foreach (XmlElement xmlParameterElement in
                             xmlParametersElement.SelectNodes("Parameter"))
                    {

                        switch (xmlParameterElement.GetAttribute("Name"))
                        {
                            case "WaveEncoding":
                                xmlParameterElement.SetAttribute("Value",
                                                                  velRegion.Encoding.ToString());
                                break;

                            case "FilePath":
                                xmlParameterElement.SetAttribute("Value", velRegion.FilePath);
                                break;

                            case "OriginalKey":
                                xmlParameterElement.SetAttribute("Value",
                                                                  velRegion.OriginalKey.ToString());
                                break;
                        }
                    }
                }
            }

            //
            xmlElement.AppendChild(xmlParametersElement);
            xmlElement.AppendChild(SerializeChildren(xmlDoc, component));

            return xmlElement;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static XmlElement SerializeChildren(XmlDocument xmlDoc, Component component)
        {
            XmlElement xmlCardinalElement = xmlDoc.CreateElement("Items");
            XmlElement xmlElement = null;

            foreach (Component childComponent in component.Children)
            {
                xmlElement = Serialize(xmlDoc, childComponent);
                xmlCardinalElement.AppendChild(xmlElement);
            }
            return xmlCardinalElement;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static Component[] Deserialize(DataObject dataObject, ComponentService componentService, ComponentReplaceHandler replacer)
        {
            List<Component> list = new List<Component>();
            XmlElement xmlItemsElement = null;
            Component component = null;

            if ((xmlItemsElement = GetXmlElement(dataObject, "Items")) == null)
            {
                return null;
            }

            foreach (XmlElement xmlItemElement in xmlItemsElement.ChildNodes)
            {
                if ((component = Deserialize
                     (componentService, xmlItemElement, replacer)) != null)
                {
                    list.Add(component);
                }
            }
            return list.ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        public delegate Component ComponentReplaceHandler(Component component);

        /// <summary>
        ///
        /// </summary>
        private delegate IParameterFilter GetParameterFilterHandler(Component component);
        private static GetParameterFilterHandler GetParameterFilter { get; set; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static Component Deserialize(ComponentService componentService, XmlElement xmlCardinalElement, ComponentReplaceHandler replacer)
        {
            Component cardinalComponent = null;
            Component component = null;
            XmlElement xmlItemsElement = null;
            XmlElement xmlParametersElement = null;
            Type type = null;
            string newName = null;
            int value = 0;

            if ((type = ToType(xmlCardinalElement.Name)) == null)
            {
                return null;
            }

            cardinalComponent = Activator.CreateInstance(type) as Component;

            if ((xmlParametersElement =
                 xmlCardinalElement.SelectSingleNode("Parameters") as XmlElement) != null)
            {
                IParameterFilter filter = null;
                if (GetParameterFilter != null)
                {
                    filter = GetParameterFilter(cardinalComponent);
                }

                _Deserializer.Execute
                    (xmlParametersElement, cardinalComponent.Parameters, filter);
            }

            //Componentの変換((例) WaveSoundSet -> GroupItem)
            if (replacer != null)
            {
                if ((cardinalComponent = replacer(cardinalComponent)) == null)
                {
                    return null;
                }
            }

            //
            if (cardinalComponent.Name.Length > 0 && componentService != null)
            {
                if (componentService.ComponentDictionary.Contains
                     (cardinalComponent.Name) == true)
                {
                    string postfix =
                        ApplicationBase.Instance.ProjectService.Project.ItemPastePostfix;
                    string interimName = cardinalComponent.Name + postfix;

                    ValidationResult result = ItemNameValidator.ValidateSoundProjectItemName
                        (interimName);
                    if (result.IsValid == false)
                    {
                        throw new InvalidPasteNameException();
                    }

                    //
                    value = 0;
                    do
                    {
                        newName = interimName + value.ToString();
                        value++;
                    } while (componentService.ComponentDictionary.Contains
                             (newName) == true);

                    cardinalComponent.Name = newName;
                }
            }

            //
            xmlItemsElement = xmlCardinalElement.SelectSingleNode("Items") as XmlElement;
            if (xmlItemsElement != null)
            {
                foreach (XmlElement xmlItemElement in xmlItemsElement.ChildNodes)
                {
                    if ((component = Deserialize
                         (componentService, xmlItemElement, replacer)) != null)
                    {
                        cardinalComponent.Children.Add(component);
                    }
                }
            }
            return cardinalComponent;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static string[] DeserializeText(DataObject dataObject, ref int width, ref int height)
        {
            List<string> list = new List<string>();
            XmlElement xmlItemsElement = null;

            if ((xmlItemsElement = GetXmlElement(dataObject, "SubItems")) == null)
            {
                return null;
            }

            foreach (XmlElement xmlItemElement in xmlItemsElement.ChildNodes)
            {

                foreach (XmlElement xmlParameterElement in
                         xmlItemElement.SelectNodes("Parameters/Parameter"))
                {
                    if (xmlParameterElement.HasAttribute("Value") == false)
                    {
                        return null;
                    }
                    list.Add(xmlParameterElement.GetAttribute("Value").ToString());
                }
            }

            if (list.Count <= 0)
            {
                return new string[0];
            }

            width = list.Count / xmlItemsElement.ChildNodes.Count;
            if (width <= 0)
            {
                return new string[0];
            }

            height = list.Count / width;
            return list.ToArray();
        }



        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static bool CanDeserialize(DataObject dataObject, Type type)
        {
            XmlElement xmlItemsElement = null;

            if (dataObject == null)
            {
                return false;
            }

            if ((xmlItemsElement = GetXmlElement(dataObject, "Items")) == null)
            {
                return false;
            }

            if (xmlItemsElement.ChildNodes.Count <= 0)
            {
                return false;
            }

            foreach (XmlElement xmlItemElement in xmlItemsElement.ChildNodes)
            {
                if (CanDeserialize(type, xmlItemElement) == false)
                {
                    return false;
                }
            }
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static bool CanDeserialize(Type targetType, XmlElement xmlCardinalElement)
        {
            Type type = null;

            if ((type = ToType(xmlCardinalElement.Name)) == null)
            {
                return false;
            }

            if (type != targetType)
            {
                return false;
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        /// DataObjectから XPathで指定された XmlElementを取得する
        /// </summary>
        private static XmlElement GetXmlElement(DataObject dataObject, string xPath)
        {
            XmlDocument xmlDoc = new XmlDocument();
            XmlElement xmlElement = null;
            byte[] data = null;

            if (dataObject.GetDataPresent(_DataName) == false)
            {
                return null;
            }

            //
            if ((data = dataObject.GetData(_DataName) as byte[]) == null)
            {
                return null;
            }

            MemoryStream stream = new MemoryStream(data);
            xmlDoc.Load(stream);

            xmlElement = xmlDoc.DocumentElement;
            return xmlElement.SelectSingleNode(xPath) as XmlElement;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static Dictionary<string, Type[]> _TextTypeTable = new Dictionary<string, Type[]>()
        {
            {"StreamSound", new Type[] {typeof(StreamSoundBase)}},
            {"WaveSoundSet", new Type[] {typeof(WaveSoundSetBase)}},
            {"WaveSound", new Type[] {typeof(WaveSoundBase)}},
            {"SequenceSoundSet", new Type[] {typeof(SequenceSoundSetBase)}},
            {"SequenceSound", new Type[] {typeof(SequenceSoundBase)}},
            {"Bank", new Type[] {typeof(SoundSetBankBase)}},
            {"WaveArchive", new Type[] {typeof(WaveArchiveBase)}},
            {"Player", new Type[] {typeof(PlayerBase)}},
            {"Group", new Type[] {typeof(GroupBase)}},
            {"GroupItem", new Type[] {typeof(GroupItemBase)}},
            {"StreamSoundTrack", new Type[] {typeof(StreamSoundTrackBase)}},

            {"Instrument", new Type[] {typeof(Instrument), typeof(ImaginaryInstrument)}},
            {"KeyRegion", new Type[] {typeof(KeyRegion)}},
            {"VelocityRegion", new Type[] {typeof(VelocityRegion)}},
        };

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static Dictionary<Type, Type> _TypeChildrenTypeTable = new Dictionary<Type, Type> {
            {typeof(StreamSoundPack), typeof(StreamSoundBase)},
            {typeof(WaveSoundSetPack), typeof(WaveSoundSetBase)},
            {typeof(SequenceSoundSetPack), typeof(SequenceSoundSetBase)},
            {typeof(SequenceSoundPack), typeof(SequenceSoundBase)},
            {typeof(SoundSetBankPack), typeof(SoundSetBankBase)},
            {typeof(WaveArchivePack), typeof(WaveArchiveBase)},
            {typeof(PlayerPack), typeof(PlayerBase)},
            {typeof(GroupPack), typeof(GroupBase)},
            {typeof(GroupBase), typeof(GroupItemBase)},
            {typeof(SequenceSoundSetBase), typeof(SequenceSoundBase)},
            {typeof(WaveSoundSetBase), typeof(WaveSoundBase)},
            {typeof(StreamSoundBase), typeof(StreamSoundTrackBase)},

            {typeof(Bank), typeof(Instrument)},
            {typeof(Instrument), typeof(KeyRegion)},
            {typeof(KeyRegion), typeof(VelocityRegion)},
        };

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static Type GetChildrenType(Type type)
        {
            if (_TypeChildrenTypeTable.ContainsKey(type) == true)
            {
                return _TypeChildrenTypeTable[type];
            }

            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static string ToString(Type type)
        {
            Array values = _TextTypeTable.Values.ToArray();
            int i = 0;
            foreach (Type[] types in values)
            {
                foreach (Type t in types)
                {
                    if (t == type)
                    {
                        return _TextTypeTable.Keys.ToArray()[i];
                    }
                }
                ++i;
            }
            return String.Empty;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static Type ToType(string text)
        {
            var data = _TextTypeTable[text];
            return data.Length > 0 ? data[0] : null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static string GetText(string[] columnNames, Component[] components)
        {
            StringBuilder sb = new StringBuilder();
            string value = null;

            if (columnNames == null)
            {
                return String.Empty;
            }

            foreach (Component component in components)
            {
                foreach (string name in columnNames)
                {
                    if ((value = GetText(name, component)) != null)
                    {
                        sb.Append(value);
                        sb.Append("\t");
                    }
                }
                sb.AppendLine(String.Empty);
            }
            return sb.ToString();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private static string GetText(string name, Component component)
        {
            IConstParameterValue value = GetConstParameterValue(name, component);
            return value != null ? value.ToString() : String.Empty;
        }

        /// <summary>
        ///
        /// </summary>
        private static IConstParameterValue GetConstParameterValue(string name, Component component)
        {
            ComponentListItem item = null;

            if (component is KeyRegion)
            {
                KeyRegion keyRegion = component as KeyRegion;
                item = new PercussionListItem
                    (0, keyRegion.Parent as Instrument, keyRegion, false);
            }
            else
            {
                if (component is Instrument)
                {
                    item = new InstrumentListItem(component);
                }
                else
                {

                    item = new CommonListItem(component);
                }
            }

            //
            switch (name)
            {
                case ProjectParameterNames.SequenceSound.SoundSetBankReferences:
                    IList<ComponentReference> list =
                        component.Parameters[name].Value as IList<ComponentReference>;
                    if (list == null)
                    {
                        return null;
                    }

                    string text = ValueTextUtility.CreateBankReferenceText(list);
                    if (text == null)
                    {
                        return null;
                    }
                    return new TextParameterValue(text);

                case ProjectParameterNames.SampleRate:
                    string sampleRate = item.GetConstValue(name).Value.ToString().Replace(" Hz", string.Empty);

                    return new TextParameterValue(sampleRate);

                case ProjectParameterNames.Sound.SinglePlayEffectiveDuration:
                    string effectiveDuration = item.GetConstValue(name).Value.ToString().Replace(" msec", string.Empty);

                    return new TextParameterValue(effectiveDuration);

                default:
                    return item.GetConstValue(name);
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class InvalidPasteNameException : ApplicationException
    {
    }
}
