﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Preset;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers;

    public partial class HistoryPanel : CommonPanel, IQueryCommandParameter
    {
        private ListHeaderAdapterDictionary headerAdapters = new ListHeaderAdapterDictionary();
        private CommonListAdapter historyListAdapter = new CommonListAdapter(decorationEvaluator: null);

        public HistoryPanel()
        {
            InitializeComponent();

            historyListAdapter.ListItemCreator = delegate (Component component)
                {
                    CommonListItem item = new CommonListItem(component);
                    item.ConstValueGetters.Add("DateTime", GetConstValueGetter);
                    item.ValueGetters.Add("Name", GetValueGetter);

                    return item;
                };

            this.historyListAdapter.DecorationEvaluator = this.ListDecorationEvaluator;
            this.historyListCtrl.HeaderSource = FormsApplication.Instance.UIService.DefaultHeaderAdapters[ListTraits.ListName_History];
            this.historyListCtrl.ReadOnly = true; // D&D を無効
            this.historyListCtrl.ItemsSource = this.historyListAdapter;
            this.historyListCtrl.ItemDoubleClicked += OnItemDoubleClicked;
            this.historyListCtrl.Hide();

            this.ListDecorationEvaluator.AddDoShowErrorIconDelegate(CommonListDecorationEvaluator.RowHeaderName, CheckBookmarkTarget);

            this.historyListCtrl.PartDrawers.Replace(new ListPartHistoryRowHeaderDrawer());

        }

        public HistoryService HistoryService { get; set; }

        /// <summary>
        /// セルの装飾に関する情報を提供します。
        /// </summary>
        protected CommonListDecorationEvaluator ListDecorationEvaluator { get; } =
            new CommonListDecorationEvaluator();

        public void AttachHistory(HistoryService historyService)
        {
            this.HistoryService = historyService;
            this.HistoryService.Changed += OnChanged;
            this.AttachHistory(historyService.BookmarkRoot);
            this.historyListCtrl.ItemsSource = this.historyListAdapter;
            this.historyListCtrl.Show();
            this.button_Clear.Visible = true;
        }

        public void DetachHistory()
        {
            if (this.HistoryService != null)
            {
                this.HistoryService.Clear();
                this.HistoryService.Changed -= OnChanged;
                this.HistoryService = null;
            }
            this.historyListAdapter.Items.Clear();
            this.historyListCtrl.ItemsSource = null;
            this.historyListCtrl.Hide();
            this.button_Clear.Visible = false;
        }

        public void UpdateHistoryList()
        {
            this.AttachHistory(this.HistoryService.BookmarkRoot);
        }

        object IQueryCommandParameter.GetParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponents:
                    return TargetComponents;
            }

            throw new KeyNotFoundException();
        }

        bool IQueryCommandParameter.ContainsParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponents:
                    return true;
            }

            return false;
        }

        private void AttachHistory(BookmarkFolder historyRoot)
        {
            this.historyListAdapter.Items.Clear();
            if (historyRoot != null)
            {
                this.historyListAdapter.AddItems(historyRoot.Children);
            }
        }

        private IConstParameterValue GetConstValueGetter(Component component, string name, ref bool cancel)
        {
            if (name == "DateTime" && component is Bookmark)
            {
                Bookmark bookmark = component as Bookmark;
                cancel = false;
                return new TextParameterValue(bookmark.DateTime);
            }

            cancel = true;
            return null;
        }

        private IParameterValue GetValueGetter(Component component, string name, ref bool cancel)
        {
            if (name == ProjectParameterNames.Name || name == "DateTime")
            {
                //ApplyHistory();
                cancel = false;
                return null;
            }

            cancel = true;
            return null;
        }

        private void OnItemDoubleClicked(object sender, ListItemDoubleClickedEventArgs e)
        {
            ApplyHistory();
        }

        private void ApplyHistory()
        {
            IListItem[] items = this.historyListCtrl.SelectedItems;
            if (items.Length == 1)
            {
                CommonListItem item = items[0] as CommonListItem;
                Bookmark bookmark = item.Target as Bookmark;
                SoundSet soundSet = bookmark.SoundSet;

                if (bookmark.Target == null)
                {
                    return;
                }
                if (bookmark.Target is SoundSetBankBase)
                {
                    SoundSetBankBase bank = bookmark.Target as SoundSetBankBase;

                    string filePath = new ReferenceFileFinder()
                    {
                        Text = bank.FilePath,
                    }.Validate(bank.FilePath);

                    if (null == filePath) { return; }

                    CommonTabPage page0 = FormsApplication.Instance.UIService.MainWindow.ActivePage;
                    FormsApplication.Instance.UIService.MainWindow.OpenFile(bank);
                    CommonTabPage page1 = FormsApplication.Instance.UIService.MainWindow.ActivePage;
                    if (page0 == page1)
                    {
                        FormsApplication.Instance.UIService.MainWindow.HistoryAdd(bookmark.Target, true);
                    }
                }
                else if (soundSet != null)
                {
                    CommonTabPage page0 = FormsApplication.Instance.UIService.MainWindow.ActivePage;
                    CommonTabPage page = FormsApplication.Instance.UIService.MainWindow.AddPage(soundSet);
                    SoundSetPanel soundSetPanel = page.Panel as SoundSetPanel;

                    SoundSetItem soundSetItem = bookmark.Target as SoundSetItem;
                    soundSetPanel.Show(soundSetItem);
                    soundSetPanel.ActivatePanel(true, true);
                    CommonTabPage page1 = FormsApplication.Instance.UIService.MainWindow.ActivePage;
                    bool forceAdd = (page0 == null || page0 == page1);
                    FormsApplication.Instance.UIService.MainWindow.HistoryAdd(bookmark.Target, forceAdd);
                }
            }
        }

        private void OnClick_button_Clear(object sender, EventArgs e)
        {
            this.HistoryService.Clear();
            this.historyListAdapter.Items.Clear();
        }

        private void OnChanged(object sender, EventArgs e)
        {
            this.AttachHistory(this.HistoryService.BookmarkRoot);
        }

        private static string CheckBookmarkTarget(Component component)
        {
            var bookmark = component as Bookmark;
            if (bookmark != null && bookmark.Target == null)
            {
                // ブックマークのターゲットがありません。
                return MessageResource.ErrorMessage_BookmarkTargetNotExist;
            }
            return null;
        }
    }

    /// <summary>
    /// 行ヘッダ
    /// </summary>
    public class ListPartHistoryRowHeaderDrawer : ListPartCommonRowHeaderDrawer
    {
        /// <summary>
        ///
        /// </summary>
        protected override Image GetImage(ListDrawDescriptor desc, Component component)
        {
            Image image = GetImageStatic(component);

            if (image == null)
            {
                Bookmark bookmark = GetBookmark(desc);
                if (bookmark != null)
                {
                    if (bookmark.Target == null)
                    {
                        switch (bookmark.Type)
                        {
                            case Bookmark.TypeStringWaveSoundSet:
                                return ImageResource.BitmapIconWaveSoundSet;
                            case Bookmark.TypeStringSequenceSoundSet:
                                return ImageResource.BitmapIconSequenceSoundSet;
                            case Bookmark.TypeStringSoundSetBank:
                                return ImageResource.BitmapIconBank;

                            case Bookmark.TypeStringStreamSoundPack:
                                return ImageResource.BitmapIconStreamSoundFolder;
                            case Bookmark.TypeStringWaveSoundSetPack:
                                return ImageResource.BitmapIconWaveSoundSetFolder;
                            case Bookmark.TypeStringSequenceSoundPack:
                                return ImageResource.BitmapIconSequenceSoundFolder;
                            case Bookmark.TypeStringSequenceSoundSetPack:
                                return ImageResource.BitmapIconSequenceSoundSetFolder;
                            case Bookmark.TypeStringSoundSetBankPack:
                                return ImageResource.BitmapIconBankFolder;
                            case Bookmark.TypeStringWaveArchivePack:
                                return ImageResource.BitmapIconWaveArchiveFolder;
                            case Bookmark.TypeStringGroupPack:
                                return ImageResource.BitmapIconGroupFolder;
                            case Bookmark.TypeStringPlayerPack:
                                return ImageResource.BitmapIconPlayerFolder;

                            default:
                                Debug.Assert(false, "Invalid bookmark type.");
                                break;
                        }
                    }
                }
            }

            return image;
        }

        /// <summary>
        ///
        /// </summary>
        protected override Component GetTarget(ListDrawDescriptor desc)
        {
            Bookmark bookmark = GetBookmark(desc);
            if (bookmark != null &&
                bookmark.Target != null)
            {
                return bookmark.Target;
            }
            return base.GetTarget(desc);
        }

        /// <summary>
        ///
        /// </summary>
        private Bookmark GetBookmark(ListDrawDescriptor desc)
        {
            CommonListItem item = desc.Item as CommonListItem;

            if (item != null &&
                item.Target is Bookmark)
            {
                return item.Target as Bookmark;
            }
            return null;
        }
    }
}
