﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core.Drawing;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Resources;

    /// <summary>
    ///
    /// </summary>
    public partial class FloatParameterEditPanel : UserControl, IParameterEditPanel
    {
        private Component[] targetComponents = null;
        private string name = null;

        public event ValueChangingEventHandler ValueChanging;
        public event ValueChangedEventHandler ValueChanged;

        private bool suspendEvent = false;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FloatParameterEditPanel()
        {
            InitializeComponent();

            this.textBox.ValueChanged += OnFinished;
            this.textBox.Leave += OnLeave;
        }

        /// <summary>
        ///
        /// </summary>
        public Component[] Components
        {
            get
            {
                return this.targetComponents;
            }
            set
            {
                this.targetComponents = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string ParameterName
        {
            set
            {
                this.name = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string Label
        {
            set
            {
                this.label.Text = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public float Minimum
        {
            get
            {
                return Convert.ToSingle(this.trackBar.Minimum);
            }
            set
            {
                Decimal decimalValue = Convert.ToDecimal(value);
                this.trackBar.Minimum = decimalValue;
                this.textBox.Minimum = decimalValue;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public float Maximum
        {
            get
            {
                return Convert.ToSingle(this.trackBar.Maximum);
            }
            set
            {
                Decimal decimalValue = Convert.ToDecimal(value);
                this.trackBar.Maximum = decimalValue;
                this.textBox.Maximum = decimalValue;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public float Value
        {
            get
            {
                return Convert.ToSingle(this.trackBar.Value);
            }
            set
            {
                SetValueToUI(value);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdateUI()
        {
            if (this.targetComponents == null)
            {
                return;
            }

            if (this.targetComponents.Length == 1)
            {
                Component component = this.targetComponents[0];
                IParameterValue value = component.Parameters[this.name];
                SetValueToUI(Convert.ToSingle(value.Value));
            }
            else
            {
                // 未実装
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValueToUI(float value)
        {
            try
            {
                this.suspendEvent = true;

                this.trackBar.Value = (Decimal)value;
                this.textBox.Text = ((Decimal)value).ToString();
            }
            finally
            {
                this.suspendEvent = false;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private bool IsValueChanged(float value)
        {
            if (this.targetComponents.Length == 1)
            {
                Component component = this.targetComponents[0];
                float origValue = (float)component.Parameters[this.name].Value;
                return origValue != value ? true : false;
            }
            else
            {
                // 未実装
                return false;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void SetValue(float value)
        {
            // 同じ値ならば再設定を行いません。
            if (IsValueChanged(value) == false)
            {
                return;
            }

            //
            SetValueToUI(value);

            if (ValueChanged != null)
            {
                ValueChanged(this, new FloatValueChangedEventArgs
                              (this.targetComponents, this.name, value));
            }
        }

#if false
        /// <summary>
        ///
        /// </summary>
        private void SetValueByText( string text)
        {
            if( String.IsNullOrEmpty( text) != false )
            {
                return;
            }

            try
            {
                float value = float.Parse( text);
                if( value < Minimum ||
                    value > Maximum )
                {
                    return;
                }

                SetValue( value);
            }
            catch
            {
            }
        }
#endif

        /// <summary>
        ///
        /// </summary>
        private void OnFinished(object sender, ValueChangedEventArgs e)
        {
            if (this.suspendEvent != false)
            {
                return;
            }

            NNumericUpDown control = sender as NNumericUpDown;
            Debug.Assert(control != null, "Sender is not NNumericUpDown control");
            SetValue((float)control.Value);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnLeave(object sender, EventArgs e)
        {
            if (this.suspendEvent != false)
            {
                return;
            }

            NNumericUpDown control = sender as NNumericUpDown;
            Debug.Assert(control != null, "Sender is not NNumericUpDown control");
            SetValue((float)control.Value);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnValueChanged(object sender, ValueChangedEventArgs e)
        {
            if (this.suspendEvent != false)
            {
                return;
            }

            NTrackBar trackBar = sender as NTrackBar;
            float value = Convert.ToSingle(trackBar.Value);

            if (trackBar.IsDragging != false)
            {
                SetValueToUI(value);

                if (ValueChanging != null)
                {
                    ValueChanging(this, new FloatValueChangingEventArgs
                                   (this.targetComponents, this.name, value));
                }
            }
            else
            {
                SetValue(value);
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class FloatValueChangeEventArgs : ValueChangeEventArgs
    {
        public FloatValueChangeEventArgs(Component[] components, string name, object value) :
            base(components, name, value)
        {
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class FloatValueChangingEventArgs : FloatValueChangeEventArgs
    {
        /// <summary>
        ///
        /// </summary>
        public FloatValueChangingEventArgs(Component[] components, string name, float value) :
            base(components, name, value)
        {
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class FloatValueChangedEventArgs : FloatValueChangeEventArgs
    {
        /// <summary>
        ///
        /// </summary>
        public FloatValueChangedEventArgs(Component[] components, string name, float value) :
            base(components, name, value)
        {
        }
    }
}
