﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Collections;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Utilities;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Preset;
    using NintendoWare.SoundMaker.Framework.Projects;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers;
    using ToolDevelopmentKit;

    /// <summary>
    ///
    /// </summary>
    public partial class BankPanel : CommonListPanel, IQueryCommandParameter, IParameterPanelOperationExecutedListener, IParameterPanelKeyPreviewListener, IEditableControlContainer
    {
        private SampleMapPanel _SampleMapPanel = null;
        private PercussionListPanel _PercussionListPanel = null;
        private ParameterPanel _ParameterPanel;

        private string filePath;
        private SoundSetBankBase _SoundSetBank = null;
        private BankServiceReference bankServiceRef = null;

        private CommonListAdapter _ListAdapter = null;

        private ListHeaderAdapterDictionary _headerAdapters = new ListHeaderAdapterDictionary();
        private IDictionary<string, XmlList> _xmlLists;

        private double _InstrumentViewSplitterRatio = 0.5;

        private ToolStripAdapter _instrumentListMenuAdapter;

        private ToolStripMenuItem _HeaderContextMenuItem_AddPresetListColumns = null;
        private ToolStripMenuItem _HeaderContextMenuItem_ApplyPresetListColumns = null;

        private bool _isPlaying = false;
        private bool _isEdit = false;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public BankPanel(string filePath)
        {
            Ensure.Argument.NotNull(filePath);
            Ensure.Argument.StringNotEmpty(filePath);

            InitializeComponent();

            this.filePath = filePath;
            this.bankServiceRef = ApplicationBase.Instance.BankServices.OpenItem(filePath);
            this.bankServiceRef.Target.Closing += OnBankServiceClosing;
            this.BankDocument.DirtyChanged += OnDirtyChanged;

            UpdateTitle();

            //
            InnerBankPage page = null;
            Panel panel = null;

            _SampleMapPanel = new SampleMapPanel(this);
            _SampleMapPanel.Dock = DockStyle.Fill;
            _SampleMapPanel.AutoScroll = false;

            panel = new Panel();
            panel.Bounds = new Rectangle(0, 0, 767, 299);
            panel.AutoScroll = false;

            page = new ScrollTabPage();
            page.Text = Resources.MessageResource.TabText_SampleMap;
            page.AutoScroll = true;
            page.Controls.Add(_SampleMapPanel);
            page.Controls.Add(panel);

            _TabCtrl.TabPages.Add(page);

            //
            _PercussionListPanel = new PercussionListPanel(this, _percussionListMenu);
            _PercussionListPanel.Dock = DockStyle.Fill;

            page = new InnerBankPage();
            page.Text = Resources.MessageResource.TabText_PercussionList;
            page.Controls.Add(_PercussionListPanel);
            _TabCtrl.TabPages.Add(page);

            _TabCtrl.SelectedTabChanged += OnSelectedTabChanged;

            //
            _ListAdapter = CreateListAdapter();
            _ListAdapter.OperationHistory = this.BankDocument.OperationHistory;
            _ListAdapter.AddItems(Bank.Children);

            _ListCtrl.HeaderHeight = 28;
            _ListCtrl.SelectChanged += OnSelectChanged;
            _ListCtrl.QueryFileDropped += OnQueryFileDropped;
            _ListCtrl.FileDropped += OnFileDropped;
            _ListCtrl.EditBegan += OnEditBegan;
            _ListCtrl.EditEnded += OnEditEnded;
            _ListCtrl.Enter += OnEnter;
            _ListCtrl.Leave += OnLeave;

            _HeaderContextMenuItem_AddPresetListColumns = new ToolStripMenuItem
                (Resources.MessageResource.HeaderMenuItem_AddPresetListColumns);
            _HeaderContextMenuItem_AddPresetListColumns.Click +=
                OnHeaderMenuItemAddPresetListColumnsClick;
            _HeaderContextMenuItem_ApplyPresetListColumns = new ToolStripMenuItem
                (Resources.MessageResource.HeaderMenuItem_ApplyPresetListColumns);
            _ListCtrl.HeaderCtrl.InsertContextMenuItems =
                new ToolStripMenuItem[] { _HeaderContextMenuItem_AddPresetListColumns,
                                          _HeaderContextMenuItem_ApplyPresetListColumns };
            _ListCtrl.HeaderCtrl.ContextMenuOpening += OnHeaderCtrlContextMenuOpening;

            _ParameterPanel = FormsApplication.Instance.UIService.
                MainWindow.ToolPages[ParameterPanel.PageName] as ParameterPanel;

            InitializeCommandBindings();

            ApplyBankConfiguration();
            UpdatedOptions();
            UpdateReadOnly();

            //
            _ListCtrl.HeaderSource = _headerAdapters[ApplicationBase.Instance.Traits.ListTraits.ListNames[typeof(Bank)]];
            _ListCtrl.ItemsSource = _ListAdapter;

            ApplyOptionSettings();
        }

        /// <summary>
        ///
        /// </summary>
        void IEditableControlContainer.CancelEdit()
        {
            this._ListCtrl.CancelEdit();
            this._PercussionListPanel.PercussionList.CancelEdit();
        }

        /// <summary>
        ///
        /// </summary>
        public string FilePath
        {
            get
            {
                return this.filePath;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public SoundSetBankBase SoundSetBank
        {
            get { return _SoundSetBank; }
            set
            {
                if (value == _SoundSetBank) { return; }

                if (null != _SoundSetBank)
                {
                    _SoundSetBank.NameChanged -= OnSoundSetBankNameChanged;
                }

                _SoundSetBank = ValidateSoundSetBank(value);

                if (null != _SoundSetBank)
                {
                    _SoundSetBank.NameChanged += OnSoundSetBankNameChanged;
                }

                UpdateTitle();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public BankServiceReference BankServiceReference
        {
            get { return this.bankServiceRef; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public BankDocument BankDocument
        {
            get
            {
                if (this.bankServiceRef.IsClosed)
                {
                    return null;
                }

                return this.bankServiceRef.Target.BankDocument;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override Document Document
        {
            get { return this.BankDocument; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public Bank Bank
        {
            get { return this.BankDocument.Bank; }
        }

        /// <summary>
        /// リストヘッダアダプタディクショナリを取得します。
        /// </summary>
        public IReadOnlyDictionary<string, ListHeaderAdapterBase> HeaderAdapters
        {
            get { return _headerAdapters; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override CommonListCtrl CurrentListCtrl
        {
            get { return _ListCtrl; }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsSelectedSampleMapPanel
        {
            get
            {
                return SelectedPanel == _SampleMapPanel ? true : false;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override SoundDocument TargetDocument
        {
            get { return this.BankDocument; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override Component[] TargetComponents
        {
            get
            {
                if (_ListCtrl.ContainsFocus == true)
                {
                    Component[] components = _ListCtrl.GetComponentsBySelectedItem();
                    return _ListCtrl.SortToDisplayedOrder(components);
                }

                if (_PercussionListPanel.ContainsFocus == true)
                {
                    CommonListCtrl listCtrl = null;

                    if ((listCtrl = _PercussionListPanel.ActiveListCtrl) != null)
                    {
                        Component[] components = null;
                        components = listCtrl.GetComponentsBySelectedItem()
                            .Where(c => c != null).ToArray();
                        return components;
                    }
                }

                if (_SampleMapPanel.ContainsFocus == true)
                {
                    return _SampleMapPanel.SampleMapCtrl.SelectedComponents;
                }

                return new Component[0];
            }
        }

        /// <summary>
        /// 現在選択中のキーを取得または設定します。
        /// </summary>
        public int SelectedKeyboardKey
        {
            get { return _SampleMapPanel.KeyboardCtrl.SelectedKey; }
            set { _SampleMapPanel.KeyboardCtrl.SelectedKey = value; }
        }

        /// <summary>
        /// 現在のベロシティ値を取得または設定します。
        /// </summary>
        public int SelectedVelocity
        {
            get { return _SampleMapPanel.GetVelocityValue(); }
            set { _SampleMapPanel.SetVelocityValue(value); }
        }

        /// <summary>
        /// パネル内のリストコントロールを取得します。
        /// </summary>
        public override CommonListCtrl[] ListControls
        {
            get
            {
                return new CommonListCtrl[] {
                    this._ListCtrl,
                    this._PercussionListPanel.PercussionList,
                };
            }
        }

        /// <summary>
        /// セルの装飾に関する情報を提供します。
        /// </summary>
        protected CommonListDecorationEvaluator ListDecorationEvaluator { get; } =
            new CommonListDecorationEvaluator();

        /// <summary>
        /// アクティブになった時に呼ばれます。
        /// </summary>
        public override void Activated()
        {
            SelectChangedInstList();

            SoundSetBank = ValidateSoundSetBank(_SoundSetBank);
            ParameterPanel.SetBankPanel(this);
        }

        /// <summary>
        /// ディアクティブになった時に呼ばれます。
        /// </summary>
        public override void Deactivated()
        {
        }

        /// <summary>
        /// クリップボードが更新された時に呼ばれる
        /// </summary>
        public override void UpdatedClipboard()
        {
            _ListCtrl.ClearDashLine();
            _PercussionListPanel.UpdatedClipboard();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void OnClosed()
        {
            ProgramChange(null, null);
            ExtractBankConfiguration();

            ApplicationBase.Instance.ProjectConfiguration.Saving -= OnProjectConfigurationSaving;

            if (_SampleMapPanel != null)
            {
                _SampleMapPanel.OnClosed();
            }

            base.OnClosed();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override CommonListCtrl NextCandidateListCtrl(CommonListCtrl ctrl, int direction, bool repeat)
        {
            if (repeat == false) { return null; }
            return CurrentListCtrl;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void CommandExecuted(Command command)
        {
            IInnerBankPanel panel = SelectedPanel;
            if (panel != null)
            {
                panel.CommandExecuted(command);
            }

            //
            if (command == EditCommands.CopyCell)
            {
                if (_ListCtrl != null &&
                    _ListCtrl.ContainsFocus == true)
                {
                    _ListCtrl.SetDashLineBySelectedSubItem();
                }
            }

            //
            if (command == ToolCommands.ShowOptions ||
                command == ProjectCommands.ShowProjectSettings)
            {
                MainWindow.UpdatedOptions();
            }

            //
            RedrawControls();
            MainWindow.RedrawPanels();
            BuildListCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void RefreshPanel()
        {
            if (null == Bank)
            {
                _ListAdapter.Items.Clear();
                return;
            }

            _ListAdapter.AddItems(Bank.Children);
            _ListCtrl.GetItemSelecteds().Clear();
            _ListCtrl.Sort();

            this.UpdateReadOnly();
            this.UpdateTitle();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void UpdateTitle()
        {
            if (this.InvokeRequired == true)
            {
                this.Invoke(new MethodInvoker(UpdateTitleInternal));
            }
            else
            {
                this.UpdateTitleInternal();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void UpdateInfos()
        {
            _ListCtrl.UpdateInfos();
            _PercussionListPanel.UpdateInfos();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void RedrawPanel()
        {
            RedrawControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override void ActivatePanel(bool resetSelection, bool mainList)
        {
            ActivateList(resetSelection);
        }

        /// <summary>
        ///
        /// </summary>
        public override void KeyDown(KeyEventArgs e)
        {
            if (DoPreviewPlayKey(e.KeyCode) == true && _isEdit == false)
            {
                PlayInstrument();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public override void KeyUp(KeyEventArgs e)
        {
            if (DoPreviewPlayKey(e.KeyCode) == true && _isEdit == false)
            {
                StopInstrument();
            }
        }

        ///
        public void PlayByInstrumentList()
        {
            int key = SelectedKeyboardKey;
            PlayInstrument(key, _SampleMapPanel.GetVelocityValue());
        }

        ///
        public void StopByInstrumentList()
        {
            StopInstrument();
        }

        ///
        public void FinalizeEdit()
        {
            _ListCtrl.EndEdit();
        }

        /// <summary>
        ///
        /// </summary>
        public override void ApplyProjectConfigurations()
        {
            ApplyBankConfiguration();
        }

        /// <summary>
        /// プロジェクト設定によりカラム名が変更された時に呼ばれます。
        /// </summary>
        public override void ApplyColumnsText()
        {
            CommentColumnTextApplier.Apply(this._headerAdapters.Values.ToArray());

            this._ListCtrl.InvalidateHeader();
            this._PercussionListPanel.PercussionList.InvalidateHeader();
        }

        /// <summary>
        /// オプション設定が変更された時に呼ばれます。
        /// </summary>
        public override void ApplyOptionSettings()
        {
            this._ListCtrl.ApplyOptionConfigurations();
            this._PercussionListPanel.PercussionList.ApplyOptionConfigurations();
        }

        /// <summary>
        ///
        /// </summary>
        public override void UpdatedOptions()
        {
            _ListCtrl.UpdatePartDrawerOptions();
            _PercussionListPanel.UpdatedOptions();
        }

        /// <summary>
        ///
        /// </summary>
        public override void ApplySettings(SettingKinds kind)
        {
            switch (kind)
            {
                case SettingKinds.ColorComment:
                    if (this._instrumentListMenuAdapter == null)
                    {
                        return;
                    }

                    CommandTextReplacer.ReplaceColorCommentCommandTexts
                        (ProjectService, this._instrumentListMenuAdapter);
                    break;
            }
        }

        /// <summary>
        /// プリセット名からをプリセットを適用します。
        /// </summary>
        public override void ApplyPresetListColumns(string presetName)
        {
            CommonListCtrl listCtrl = this.GetListCtrl();
            string listName = string.Empty;
            if (_ListCtrl == listCtrl)
            {
                listName = ListTraits.ListName_Instrument;
            }
            else
            {
                listName = ListTraits.ListName_Percussion;
            }

            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(_headerAdapters);

            ListHeaderHelper.ApplyPresetListColumns(applier, listName, presetName);

            listCtrl.UpdateInfos();

            MainWindow.PresetListColumnsPresetCurrentNameOnSplitButton = presetName;
            PresetListColumnsService preset = ApplicationBase.Instance.PresetListColumnsService;
            preset.SetCurrentPresetName(listName, presetName);
        }

        /// <summary>
        /// プリセットを更新します。
        /// </summary>
        public override void UpdatePresetListColumns()
        {
            this.UpdatePresetListColumnsSplitButton();
        }

        /// <summary>
        ///
        /// </summary>
        bool IQueryCommandParameter.ContainsParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponentServiceReference:
                case CommandParameterNames.TargetComponentService:
                case CommandParameterNames.TargetDocuments:
                case CommandParameterNames.TargetComponents:
                case CommandParameterNames.InsertParentComponent:
                case CommandParameterNames.InsertTargetComponent:
                case "ColumnNames":
                case CaretItemSetter.CommandParameterOfListCtrl:
                    return true;
            }

            return false;
        }

        object IQueryCommandParameter.GetParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponentServiceReference:
                    return this.bankServiceRef;

                case CommandParameterNames.TargetComponentService:
                    return this.BankService;

                case CommandParameterNames.TargetDocuments:
                    if (null == TargetDocument) { return new SoundDocument[0]; }
                    return new SoundDocument[] { TargetDocument };

                case CommandParameterNames.TargetComponents:
                    return TargetComponents;

                case CommandParameterNames.InsertParentComponent:
                    return InsertParentComponent;

                case CommandParameterNames.InsertTargetComponent:
                    Component[] components = TargetComponents;
                    return (0 == components.Length) ? null : TargetComponents[0];

                case "ColumnNames":
                    if (ActiveListCtrl == null) { return null; }
                    return ActiveListCtrl.ItemsName
                        .Where(s => s != "Key" && s != "RowHeader").ToArray();

                case CaretItemSetter.CommandParameterOfListCtrl:
                    return ActiveListCtrl;
            }

            throw new KeyNotFoundException();
        }

        /// <summary>
        ///
        /// </summary>
        void IParameterPanelOperationExecutedListener.OperationExecuted(OperationExecutedEventArgs e)
        {
            _ListCtrl.UpdateInfos();
        }

        /// <summary>
        ///
        /// </summary>
        void IParameterPanelKeyPreviewListener.KeyDown(KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Space)
            {
                PlayByInstrumentList();
            }

            //
            if (e.KeyCode == Keys.F9)
            {
                StopByInstrumentList();
                PlayByInstrumentList();
            }
        }

        /// <summary>
        ///
        /// </summary>
        void IParameterPanelKeyPreviewListener.KeyUp(KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Space)
            {
                StopByInstrumentList();
            }

            //
            if (e.KeyCode == Keys.F10)
            {
                StopByInstrumentList();
            }
        }

        /// <summary>
        /// <see cref="CommonListAdapter"/>  を生成します。
        /// <para>
        /// 本メソッドをオーバーライドし生成方法を変更する場合は <see cref="ListDecorationEvaluator"/> を与えるようにしてください。
        /// </para>
        /// </summary>
        protected virtual CommonListAdapter CreateListAdapter()
        {
            CommonListAdapter listAdapter = new CommonListAdapter(ListDecorationEvaluator);
            listAdapter.ListItemCreator = delegate (Component component)
                {
                    InstrumentListItem item = new InstrumentListItem(component);
                    item.ValueGetters.Add(ProjectParameterNames.VelocityRegion.OriginalKey, GetValueGetter);

                    return item;
                };

            return listAdapter;
        }

        /// <summary>
        /// インストルメントリストを取得します。
        /// </summary>
        protected CommonListCtrl InstrumentList
        {
            get { return _ListCtrl; }
        }

        /// <summary>
        /// サンプルマップパネルを取得します。
        /// </summary>
        protected SampleMapPanel SampleMapPanel
        {
            get { return _SampleMapPanel; }
        }

        /// <summary>
        /// パーカッションリストパネルを取得します。
        /// </summary>
        protected PercussionListPanel PercussionListPanel
        {
            get { return _PercussionListPanel; }
        }

        ///--------------------------------
        /// <summary>
        /// アクティブなリストコントロールを取得します。
        /// </summary>
        protected override CommonListCtrl ActiveListCtrl
        {
            get
            {
                if (_ListCtrl.ContainsFocus) { return _ListCtrl; }

                if (_PercussionListPanel.ContainsFocus == true)
                {
                    return _PercussionListPanel.ActiveListCtrl;
                }

                return null;
            }
        }

        ///--------------------------------
        /// <summary>
        /// アクティブなリストアダプタを取得します。
        /// </summary>
        protected override ComponentListAdapter ActiveListAdapter
        {
            get
            {
                if (_ListCtrl.ContainsFocus) { return _ListAdapter; }

                if (_PercussionListPanel.ContainsFocus == true)
                {
                    return _PercussionListPanel.ActiveListAdapter;
                }

                return null;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override ComponentService ComponentService
        {
            get { return this.BankService; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected SoundProjectConfiguration ProjectConfiguration
        {
            get { return FormsApplication.Instance.ProjectConfiguration; }
        }

        protected XmlBankDocumentView BankConfiguration
        {
            get
            {
                if (null == ProjectConfiguration) { return null; }
                if (!ProjectConfiguration.DocumentViews.ContainsKey(this.filePath))
                {
                    return null;
                }

                return ProjectConfiguration.DocumentViews[this.filePath] as XmlBankDocumentView;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected SoundProjectService ProjectService
        {
            get { return FormsApplication.Instance.ProjectService; }
        }

        ///--------------------------------
        /// <summary>
        /// コンポーネント作成サービスを取得します。
        /// </summary>
        protected CreateComponentService CreateComponentService
        {
            get { return FormsApplication.Instance.CreateComponentService; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected ParameterPanel ParameterPanel
        {
            get
            {
                return FormsApplication.Instance.UIService.
                    MainWindow.ToolPages[ParameterPanel.PageName] as ParameterPanel;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected bool ContainsInProject
        {
            get
            {
                foreach (SoundSetBankBase bank in ProjectService.SoundSetBanks)
                {
                    if (bank.FilePath == this.BankDocument.Resource.Key) { return true; }
                }

                return false;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Component InsertParentComponent
        {
            get { return Bank; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private BankService BankService
        {
            get { return this.bankServiceRef.Target; }
        }

        private int SelectedKey
        {
            get
            {
                if (InstrumentList.ContainsFocus)
                {
                    if (0 == InstrumentList.SelectedItems.Length) { return -1; }
                    return SelectedKeyboardKey;
                }

                if (SampleMapPanel.ContainsFocus)
                {

                    VelocityRegion[] velocityRegions = SampleMapPanel.SelectedVelocityRegions;
                    if (1 != velocityRegions.Length) { return -1; }

                    return velocityRegions[0].OriginalKey;

                }

                if (PercussionListPanel.ContainsFocus)
                {

                    if (null == PercussionListPanel.ActiveListCtrl) { return -1; }

                    IListItem[] items = PercussionListPanel.ActiveListCtrl.SelectedItems;
                    if (1 != items.Length) { return -1; }

                    return (items[0] as PercussionListItem).Index;

                }

                return -1;
            }
        }

        ///--------------------------------
        /// <summary>
        /// コマンドの関連付けを初期化します。
        /// </summary>
        protected virtual void InitializeCommandBindings()
        {
            //編集
            BindCommand(new CommandBinding(this, EditCommands.SelectAll.ID,
                                             QueryStatusSelectAll, ExecuteSelectAll));
            BindCommand(new CommandBinding(this, EditCommands.Undo.ID, QueryStatusUndo, ExecuteUndo));
            BindCommand(new CommandBinding(this, EditCommands.Redo.ID, QueryStatusRedo, ExecuteRedo));
            BindCommand(new CommandBinding(this, new CutBankPanelHandler(this)));
            BindCommand(new CommandBinding(this, new CopyBankPanelHandler(this)));
            BindCommand(new CommandBinding(this, EditCommands.Paste.ID, QueryStatusPaste, ExecutePaste));
            BindCommand(new CommandBinding(this, new DeleteBankPanelHandler(this)));
            BindCommand(new CommandBinding(this, EditCommands.CopyCell.ID,
                                           QueryStatusCopyCell, ExecuteCopyCell));
            BindCommand(new CommandBinding(this, EditCommands.PasteSpecial.ID,
                                             QueryStatusPasteSpecial, ExecutePasteSpecial));
            BindCommand(new CommandBinding(this, EditCommands.BatchEdit.ID,
                                             QueryStatusBatchEdit, ExecuteBatchEdit));
            BindCommand(new CommandBinding(this, EditCommands.RenumberProgramNo.ID,
                                           QueryStatusRenumberProgramNo,
                                           ExecuteRenumberProgramNo));
            BindCommand(new CommandBinding(this, new SetEnableBankPanelHandler(this, true)));
            BindCommand(new CommandBinding(this, new SetEnableBankPanelHandler(this, false)));
            BindCommand(new CommandBinding(this, new SetColorBankPanelHandler(this)));

            // プレビュー関連コマンドハンドラ
            BindCommand(new CommandBinding(this, PreviewCommands.Play.ID,
                                            QueryStatusPlayInstrument, ExecutePlayInstrument));
            BindCommand(new CommandBinding(this, PreviewCommands.StopAll.ID,
                                            QueryStatusStopInstrument, ExecuteStopInstrument));

            // プロジェクト関連コマンドハンドラ
            BindCommand(new CommandBinding(this, ProjectCommands.AddItem.ID,
                                             QueryStatusAddItem, ExecuteAddItem));
            BindCommand(new CommandBinding(this, ProjectCommands.InsertItem.ID,
                                             QueryStatusInsertItem, ExecuteInsertItem));
            BindCommand(new CommandBinding(this,
                        new QueryAndAddInstrumentHandler(true, this)));
            BindCommand(new CommandBinding(this,
                        new QueryAndAddInstrumentHandler(false, this)));
            BindCommand(new CommandBinding(this, new OpenItemFolderHandler(this)));
            BindCommand(new CommandBinding(this, new ExecuteItemFolderHandler(this)));

            // インストルメントリスト関連コマンドハンドラ
            BindCommand(new CommandBinding(this,
                                             new InstrumentListQueryAndAddInstrumentHandler(true, this)));
            BindCommand(new CommandBinding(this,
                                             new InstrumentListQueryAndAddInstrumentHandler(false, this)));
            BindCommand(new CommandBinding(this,
                                             InstrumentListCommands.SetEnvelopeToRegion.ID,
                                             QueryStatusSetEnvelopeToRegion,
                                             ExecuteSetEnvelopeToRegion));
            BindCommand(new CommandBinding(this,
                                             InstrumentListCommands.SetEnvelopeToInstrument.ID,
                                             QueryStatusSetEnvelopeToInstrument,
                                             ExecuteSetEnvelopeToInstrument));

            //
            BindCommand(new CommandBinding(this, PercussionListCommands.SplitKeyRegion.ID,
                                             QueryStatusSplitKeyRegion, ExecuteSplitKeyRegion));
        }

        protected virtual void ProgramChange(BankDocument bankDocument, Instrument instrument)
        {
        }

        protected virtual void NoteOn(int programNo, int key, int velocity)
        {
        }

        protected virtual void NoteOff(int key)
        {
        }

        protected virtual void ProgramChange(Instrument instrument)
        {
        }

        protected virtual void Play(int key, int velocity)
        {
        }

        protected virtual void Stop()
        {
        }

        /// <summary>
        ///
        /// </summary>
        protected bool ExecutePlayInstrument(Command command)
        {
            StopInstrument();
            PlayInstrument();
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        protected bool ExecuteStopInstrument(Command command)
        {
            StopInstrument();
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        protected CommandStatus QueryStatusPlayInstrument(Command command)
        {
            if (-1 == SelectedKey)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        ///
        /// </summary>
        protected CommandStatus QueryStatusStopInstrument(Command command)
        {
            if (this._isPlaying == false)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }


        ///--------------------------------
        /// <summary>
        /// コントロールがロードされると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            //
            InitializeMenu();

            _ListCtrl.Sort();

            _SampleMapPanel.NoteOn += OnNoteOnInternal;
            _SampleMapPanel.NoteOff += OnNoteOffInternal;

            ProjectService.ComponentsRemoved += OnComponentRemoved;

            ApplicationBase.Instance.ProjectConfiguration.Saving += OnProjectConfigurationSaving;

            _ListCtrl.GetItemSelecteds().Clear();
            _ListCtrl.SetCaret();

            // Load直後のリストにフォーカスが渡るタイミングを検知する術がないので、
            // BeginInvoke でタイミングを遅らせる
            BeginInvoke(new MethodInvoker(BuildCommandUIs));
        }

        /// <summary>
        /// コントロールが入力されると発生します。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected override void OnEnter(EventArgs e)
        {
            base.OnEnter(e);

            BuildListCommandUI();
        }

        protected override string ConvertText(string name, string text)
        {
            if (name == ProjectParameterNames.VelocityRegion.OriginalKey)
            {
                int key = KeyNoteConverter.ToKey(text);
                if (0 <= key)
                {
                    return key.ToString();
                }
            }

            return base.ConvertText(name, text);
        }

        protected override IParameterValue ConvertValue(string name, string text, IParameterValue value)
        {
            if (name == ProjectParameterNames.VelocityRegion.OriginalKey)
            {
                return new IntParameterValue((int)value.Value);
            }

            return base.ConvertValue(name, text, value);
        }

        protected override bool ValidateValueForBatchEdit(Component component, string name, string text, object value)
        {
            if (name == ProjectParameterNames.VelocityRegion.OriginalKey && value is int == true)
            {
                try
                {
                    KeyNoteConverter.ToNote((int)value);
                    return true;
                }
                catch
                {
                    return false;
                }
            }

            return base.ValidateValueForBatchEdit(component, name, text, value);
        }

        ///--------------------------------
        /// <summary>
        /// リスト項目関連
        /// </summary>
        private void OnHeaderMenuItemAddPresetListColumnsClick(object sender, EventArgs e)
        {
            string listName = ListTraits.ListName_Instrument;
            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(_headerAdapters);

            ListHeaderHelper.AddPresetListColumns(listName, applier);
            this.UpdatePresetListColumnsSplitButton();
        }

        private void OnHeaderMenuItemApplyPresetListColumnsClick(object sender, EventArgs e)
        {
            string listName = ListTraits.ListName_Instrument;
            ToolStripMenuItem menuItem = sender as ToolStripMenuItem;
            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(_headerAdapters);
            ListHeaderHelper.ApplyPresetListColumns(applier, listName, menuItem.Text);
        }

        private void OnHeaderCtrlContextMenuOpening(object sender, System.ComponentModel.CancelEventArgs e)
        {
            string listName = ListTraits.ListName_Instrument;

            ListHeaderHelper.CreateMeunApplyPresetListColumns(_HeaderContextMenuItem_ApplyPresetListColumns, listName, OnHeaderMenuItemApplyPresetListColumnsClick);
        }

        private void OnEnter(object sender, EventArgs e)
        {
            this.UpdatePresetListColumnsSplitButton();
        }

        private void OnLeave(object sender, EventArgs e)
        {
            MainWindow.ClearPresetListColumnsOnSplitButton();
        }

        private void UpdatePresetListColumnsSplitButton()
        {
            if (_PercussionListPanel.ContainsFocus == true)
            {
                _PercussionListPanel.UpdatePresetListColumns();
            }
            else
            {
                string listName = ListTraits.ListName_Instrument;
                PresetListColumnsService preset = ApplicationBase.Instance.PresetListColumnsService;
                MainWindow.PresetListColumnsPresetNamesOnSplitButton = preset.GetPresetNames(listName);
                MainWindow.PresetListColumnsPresetCurrentNameOnSplitButton = preset.GetCurrentPresetName(listName);
            }
        }

        private CommonListCtrl GetListCtrl()
        {
            CommonListCtrl listCtrl = ActiveListCtrl;

            if (listCtrl == null)
            {
                if (this.SelectedPanel == _PercussionListPanel)
                {
                    listCtrl = _PercussionListPanel.ActiveListCtrl;
                }
                else
                {
                    listCtrl = _ListCtrl;
                }
            }

            return listCtrl;
        }

        private string GetListName()
        {
            string listName = string.Empty;

            if (ActiveListCtrl == _ListCtrl)
            {
                listName = ListTraits.ListName_Instrument;
            }
            else if (ActiveListCtrl != null)
            {
                listName = ListTraits.ListName_Percussion;
            }

            return listName;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateTitleInternal()
        {
            if (null == this.BankDocument)
            {
                Title = string.Empty;
                return;
            }

            string title = string.Empty;

            if (null == _SoundSetBank)
            {
                title = Path.GetFileName(this.BankDocument.Resource.Key);
            }
            else
            {
                title = string.Format("{0} - {1}", _SoundSetBank.Name,
                                                    Path.GetFileName(this.BankDocument.Resource.Key));
            }

            if (this.BankDocument.IsDirty == true)
            {
                title += " *";
            }

            if (this.BankDocument.IsReadOnly == true)
            {
                title += String.Format(" [{0}]", MessageResource.Message_ReadOnly);
            }

            Title = title;
        }

        /// <summary>
        ///
        /// </summary>
        private bool DoPreviewPlayKey(Keys code)
        {
            if (code == Keys.Space)
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// コマンドバーを再構築します。
        /// </summary>
        private void BuildCommandUIs()
        {
            BuildListCommandUI();
            MainWindow.BuildCommandUI();
        }

        /// <summary>
        /// コマンドバーを再構築します。
        /// </summary>
        private void BuildListCommandUI()
        {
            if (null != _instrumentListMenuAdapter)
            {
                _instrumentListMenuAdapter.BuildUI();
            }
        }

        private void ActivateList(bool resetSelection)
        {
            if (resetSelection && 0 < _ListCtrl.ItemCount)
            {
                _ListCtrl.GetItemSelecteds().Clear();
                _ListCtrl.SetCaret();
            }

            _ListCtrl.Select();
            BuildListCommandUI();
        }

        private SoundSetBankBase ValidateSoundSetBank(SoundSetBankBase soundSetBank)
        {
            if (null == this.BankDocument) { return null; }
            if (null == soundSetBank) { return null; }

            SoundSetBankBase firstFindBank = null;

            foreach (SoundSetBankBase soundSetBankWork in ProjectService.SoundSetBanks)
            {
                if (soundSetBankWork == soundSetBank) { return soundSetBank; }
                if (soundSetBankWork.Name != soundSetBank.Name) { continue; }

                if (null != firstFindBank)
                {
                    firstFindBank = soundSetBankWork;
                }
            }

            if (null != firstFindBank) { return firstFindBank; }

            return (from SoundSetBankBase otherSoundSetBank in ProjectService.SoundSetBanks
                    where otherSoundSetBank.FilePath == this.BankDocument.Resource.Key
                    select otherSoundSetBank).FirstOrDefault();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateReadOnly()
        {
            bool readOnly = this.BankDocument.IsReadOnly;

            _ListCtrl.ReadOnly = readOnly;

            ((IInnerBankPanel)_SampleMapPanel).UpdateReadOnly(readOnly);
            ((IInnerBankPanel)_PercussionListPanel).UpdateReadOnly(readOnly);
        }

        /// <summary>
        /// バンクサービスが閉じられる前に発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元を指定します。</param>
        /// <param name="e">イベントパラメータを指定します。</param>
        private void OnBankServiceClosing(object sender, EventArgs e)
        {
            if (this.BankDocument == null)
            {
                return;
            }

            this.BankDocument.DirtyChanged -= OnDirtyChanged;
        }

        /// <summary>
        /// プロジェクト設定が保存される前に発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントデータ。</param>
        private void OnProjectConfigurationSaving(object sender, EventArgs e)
        {
            ExtractBankConfiguration();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnDirtyChanged(object sender, EventArgs e)
        {
            UpdateTitle();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnSoundSetBankNameChanged(object sender, NameChangedEventArgs e)
        {
            UpdateTitle();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnComponentRemoved(object sender, ComponentEventArgs e)
        {
            foreach (Component component in e.Components)
            {
                if (component != _SoundSetBank) { continue; }

                SoundSetBank = ValidateSoundSetBank(_SoundSetBank);
                break;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnNoteOnInternal(object sender, SampleMapPanel.NoteEventArgs e)
        {
            NoteOn(_SampleMapPanel.GetProgramNo(), e.Key, _SampleMapPanel.GetVelocityValue());
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnNoteOffInternal(object sender, SampleMapPanel.NoteEventArgs e)
        {
            NoteOff(e.Key);
        }

        ///--------------------------------
        /// <summary>
        /// ファイルがドロップできるかどうか判断する時に呼ばれる
        /// </summary>
        private void OnQueryFileDropped(object sender, FileDroppedEventArgs e)
        {
            if (this.BankDocument.IsReadOnly == true)
            {
                e.Cancel = true;
                return;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Instrument NextSiblingInstrument(ListCtrl listCtrl, ComponentListItem item, DropPosition dropPosition)
        {
            CommonListAdapter adapter = listCtrl.ItemsSource as CommonListAdapter;
            int index = -1;

            index = adapter.Items.IndexOf(item);

            switch (dropPosition)
            {
                case DropPosition.Upper:
                    return ((ComponentListItem)adapter.Items[index]).Target as Instrument;

                case DropPosition.Lower:
                    if (index + 1 >= adapter.Items.Count) { break; }
                    return ((ComponentListItem)adapter.Items[index + 1]).Target as Instrument;

                case DropPosition.None:
                case DropPosition.Center:
                    break;
            }
            return null;
        }

        ///--------------------------------
        /// <summary>
        /// ファイルがドロップされた時に呼ばれる
        /// </summary>
        private void OnFileDropped(object sender, FileDroppedEventArgs e)
        {
            BeginInvoke(new ListCtrl.FileDroppedEventHandler(OnFileDroppedInternal),
                        new object[] { sender, e });
        }

        ///--------------------------------
        /// <summary>
        /// ファイルがドロップされた時に呼ばれる
        /// </summary>
        private void OnFileDroppedInternal(object sender, FileDroppedEventArgs e)
        {
            MainWindow.Activate();

            //
            Instrument instrument = null;
            Instrument nextSiblingInstrument = null;
            int addCount = 0;

            StringBuilder unsupportedFileNames = new StringBuilder();

            try
            {
                nextSiblingInstrument = NextSiblingInstrument
                    ((ListCtrl)sender, (ComponentListItem)e.DropTarget, e.DropPosition);

                //
                this.BankDocument.OperationHistory.BeginTransaction();

                foreach (string filePath in e.FilePaths)
                {
                    if ((instrument = CreateInstrument(filePath)) != null)
                    {
                        AddBankItemHandler.Execute
                            (this.BankDocument, this.Bank,
                              nextSiblingInstrument, instrument);
                        addCount++;
                    }
                    else
                    {

                        unsupportedFileNames.AppendLine(filePath);
                    }
                }
            }

            finally
            {
                if (addCount > 0)
                {
                    this.BankDocument.OperationHistory.EndTransaction();
                }
                else
                {

                    this.BankDocument.OperationHistory.CancelTransaction();
                }
            }

            if (0 < unsupportedFileNames.Length)
            {
                string message = Resources.MessageResource.Message_CanNotUseUnsupportedFileWithAAC;

                TextDisplayMessageBox dialog = new TextDisplayMessageBox
                    (message, unsupportedFileNames.ToString(),
                      TextDisplayMessageBoxStyle.OKButton);
                dialog.ShowDialog();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnEditBegan(object sender, EventArgs e)
        {
            ApplicationBase.Instance.DisableCommandKeyProcess();
            MainWindow.EnableAllCommands(false);
            BuildListCommandUI();
            _isEdit = true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnEditEnded(object sender, EventArgs e)
        {
            ApplicationBase.Instance.EnableCommandKeyProcess();
            MainWindow.EnableAllCommands(true);
            MainWindow.RedrawPanels();
            BuildListCommandUI();
            _isEdit = false;
        }

        ///--------------------------------
        /// <summary>
        /// インストルメントリストコマンドが実行されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">コマンドイベントデータ。</param>
        private void OnInstrumentListCommandExecuted(object sender, CommandEventArgs e)
        {
            CommandExecuted(e.Command);
            MainWindow.BuildCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnSelectedTabChanged(object sender, EventArgs e)
        {
            IInnerBankPanel panel = null;

            if ((panel = SelectedPanel) != null)
            {
                panel.Shown();
            }

            if (panel != _PercussionListPanel)
            {
                MainWindow.ClearPresetListColumnsOnSplitButton();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void SelectChangedInstList()
        {
            ComponentListItem[] items = _ListCtrl.GetSelectedItems();
            Instrument item = null;
            string filePath = string.Empty;

            if (items.Length == 1)
            {
                item = items[0].Target as Instrument;
                if (item.Children.Count == 1)
                {
                    if (item.Children[0].Children.Count == 1)
                    {
                        VelocityRegion velRegion =
                            item.Children[0].Children[0] as VelocityRegion;
                        filePath = velRegion.FilePath;
                    }
                }
            }

            //
            _PercussionListPanel.Instrument = item;
            _SampleMapPanel.Instrument = item;
            _ParameterPanel.Instrument = item;

            UpdatePreviewTarget();

            VelocityRegion[] velRegions = _SampleMapPanel.SelectedVelocityRegions;
            if (velRegions.Length == 1)
            {
                _ParameterPanel.VelocityRegion = velRegions[0];
            }
            else
            {
                _ParameterPanel.VelocityRegion = null;
            }

            ParameterPanel.SetBankPanel(this);
            MainWindow.UpdateStatusText(filePath);
            BuildCommandUIs();
        }

        ///--------------------------------
        /// <summary>
        /// インストリストの選択が変更された時に呼ばれる
        /// </summary>
        private void OnSelectChanged(object sender, EventArgs e)
        {
            SelectChangedInstList();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnSplitterMoved(object sender, SplitterEventArgs e)
        {
            UpdateSplitterRatio();
        }

        private void OnSplitContainerResize(object sender, EventArgs e)
        {
            UpdateSplitterRatio();
        }

        private void UpdatePreviewTarget()
        {
            ComponentListItem[] items = _ListCtrl.GetSelectedItems();

            if (0 < items.Length)
            {
                ProgramChange(this.BankDocument, items[0].Target as Instrument);
            }
            else
            {
                ProgramChange(this.BankDocument, null);
            }
        }

        private void UpdateSplitterRatio()
        {
            _InstrumentViewSplitterRatio = (double)_splitContainer.SplitterDistance / _splitContainer.Height;

            double value = _InstrumentViewSplitterRatio;
            if (value <= 0.0 || 1.0 <= value)
            {
                throw new ArgumentOutOfRangeException("value");
            }

        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void RedrawControls()
        {
            _ListCtrl.Invalidate();

            IInnerBankPanel panel = SelectedPanel;

            if (panel != null)
            {
                panel.RedrawControls();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private IInnerBankPanel SelectedPanel
        {
            get
            {
                InnerBankPage page = _TabCtrl.SelectedTab as InnerBankPage;
                return page.Panel;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void ApplyBankConfiguration()
        {
            if (null == BankConfiguration) { return; }

            XmlBankDocumentView configuration = BankConfiguration;
            _xmlLists = configuration.Lists.ExportDictionary();

            _InstrumentViewSplitterRatio = configuration.InstrumentListView.SplitterPositionRatio;

            try
            {
                _splitContainer.SplitterDistance = (int)(_splitContainer.Height * _InstrumentViewSplitterRatio);
            }
            catch
            {
            }

            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(_headerAdapters);

            foreach (XmlList xmlList in _xmlLists.Values)
            {
                applier.Apply(xmlList);
            }
        }

        private void ExtractBankConfiguration()
        {
            if (null == BankConfiguration) { return; }
            if (null == _xmlLists) { return; }

            XmlBankDocumentView configuration = BankConfiguration;

            configuration.InstrumentListView.SplitterPositionRatio = _InstrumentViewSplitterRatio;

            ListConfigurationApplier applier = FormsApplication.Instance.UIService.CreateListConfigurationApplier(_headerAdapters);

            foreach (XmlList xmlList in _xmlLists.Values)
            {
                applier.Extract(xmlList);
            }

            configuration.Lists.List = _xmlLists.Values.ToArray();
        }

        ///--------------------------------
        /// <summary>
        /// メニューを初期化します。
        /// </summary>
        private void InitializeMenu()
        {
            if (null != _instrumentListMenuAdapter) { return; }

            _instrumentListMenuAdapter = ToolStripAdapter.FromToolStrip(FormsApplication.Instance.CommandService,
                                                                         _instrumentListMenu);
            _instrumentListMenuAdapter.CommandTarget = FormsApplication.Instance;
            _instrumentListMenuAdapter.CommandExecuted += OnInstrumentListCommandExecuted;

            ApplySettings(SettingKinds.ColorComment);
            BuildListCommandUI();
        }

        /// <summary>
        ///
        /// </summary>
        private void PlayInstrument()
        {
            if (-1 == SelectedKey) { return; }
            PlayInstrument(SelectedKey, _SampleMapPanel.GetVelocityValue());
        }

        /// <summary>
        ///
        /// </summary>
        private void PlayInstrument(int key, int velocity)
        {
            if (_isPlaying) { return; }

            try
            {
                if (-1 == key) { return; }

                Play(key, velocity);
                _isPlaying = true;

                BuildCommandUIs();
            }
            catch (FileNotFoundException exception)
            {
                FormsApplication.Instance.UIService.ShowMessageBox(
                    string.Format(MessageResource.Message_TargetFileNotFound,
                                   exception.FileName));
            }
            catch (ApplicationException exception)
            {
                FormsApplication.Instance.UIService.ShowMessageBox(exception.Message);
            }
            catch
            {
                FormsApplication.Instance.UIService.ShowMessageBox(
                    MessageResource.Message_FailedToPlaySound);
            }
        }

        private void StopInstrument()
        {
            Stop();
            _isPlaying = false;

            BuildCommandUIs();
        }

        ///--------------------------------------------------------------------------
        /// <summary>
        /// Bank関連"コピー"ハンドラ
        /// </summary>
        private class CopyBankPanelHandler : CopyHandler
        {
            ///--------------------------------
            /// <summary>
            /// </summary>
            public CopyBankPanelHandler(IQueryCommandParameter queryParameter)
                : base(queryParameter)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
            {
                Component[] components = GetTargetComponents(parameters);

                if (null == GetTargetDocument(parameters)) { return CommandStatus.SupportedAndVisible; }
                if (null == components) { return CommandStatus.SupportedAndVisible; }

                if (components.Length == 1 && (components[0] is KeyRegion ||
                                                 components[0] is VelocityRegion))
                {
                    return CommandStatus.SupportedAndEnabledAndVisible;
                }
                else
                {
                    foreach (Component comp in components)
                    {
                        if (comp is KeyRegion || comp is VelocityRegion)
                        {
                            return CommandStatus.SupportedAndVisible;
                        }
                    }
                }

                return components.Length > 0 ?
                    CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
            }

            ///--------------------------------
            /// <summary>
            /// </summary>
            protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
            {
                return base.ExecuteInternal(command, parameters);
            }
        }

        ///--------------------------------------------------------------------------
        /// <summary>
        /// Bank関連"アイテムを有効/無効にする"ハンドラ
        /// </summary>
        private class SetEnableBankPanelHandler : EnableItemHandler
        {
            ///--------------------------------
            /// <summary>
            /// </summary>
            public SetEnableBankPanelHandler(IQueryCommandParameter queryParameter, bool value)
                : base(queryParameter, value)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
            {
                SoundDocument document = GetTargetDocument(parameters);
                Component[] components = GetTargetComponents(parameters);

                if (null == document) { return CommandStatus.SupportedAndVisible; }
                if (null == components) { return CommandStatus.SupportedAndVisible; }
                if (document.IsReadOnly == true)
                {
                    return CommandStatus.SupportedAndVisible;
                }

                if (components.Length > 0 && (components[0] is KeyRegion ||
                                                components[0] is VelocityRegion))
                {
                    return CommandStatus.SupportedAndVisible;
                }

                return components.Length > 0 ?
                    CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
            }

            ///--------------------------------
            /// <summary>
            /// </summary>
            protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
            {
                return base.ExecuteInternal(command, parameters);
            }
        }

        private class SetColorBankPanelHandler : SetItemColorHandler
        {
            ///--------------------------------
            /// <summary>
            /// </summary>
            public SetColorBankPanelHandler(IQueryCommandParameter queryParameter)
                : base(queryParameter)
            {
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
            {
                SoundDocument document = GetTargetDocument(parameters);
                Component[] components = GetTargetComponents(parameters);

                if (null == document) { return CommandStatus.SupportedAndVisible; }
                if (null == components) { return CommandStatus.SupportedAndVisible; }
                if (document.IsReadOnly == true)
                {
                    return CommandStatus.SupportedAndVisible;
                }

                if (components.Length > 0 && (components[0] is KeyRegion ||
                                                components[0] is VelocityRegion))
                {
                    return CommandStatus.SupportedAndVisible;
                }

                return components.Length > 0 ?
                    CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
            }

            ///--------------------------------
            /// <summary>
            /// </summary>
            protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
            {
                return base.ExecuteInternal(command, parameters);
            }
        }

        ///--------------------------------
        /// <summary>
        /// "元に戻す"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusUndo(Command command)
        {
            if (null == this.BankDocument) { return CommandStatus.SupportedAndVisible; }
            return this.BankDocument.OperationHistory.CanUndo() ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool ExecuteUndo(Command command)
        {
            if (this.BankDocument.OperationHistory.Undo() == false)
            {
                return false;
            }

            ((IInnerBankPanel)_SampleMapPanel).UndoExecuted();
            ((IInnerBankPanel)_PercussionListPanel).UndoExecuted();
            SelectChangedInstList();
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// "やり直す"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusRedo(Command command)
        {
            if (null == this.BankDocument) { return CommandStatus.SupportedAndVisible; }
            return this.BankDocument.OperationHistory.CanRedo() ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool ExecuteRedo(Command command)
        {
            if (this.BankDocument.OperationHistory.Redo() == false)
            {
                return false;
            }

            ((IInnerBankPanel)_SampleMapPanel).RedoExecuted();
            ((IInnerBankPanel)_PercussionListPanel).RedoExecuted();
            SelectChangedInstList();
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// "すべて選択"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusSelectAll(Command command)
        {
            return CanSelectAll() ? CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// "すべて選択"の実行
        /// </summary>
        private bool ExecuteSelectAll(Command command)
        {
            return SelectAll();
        }

        ///--------------------------------
        /// <summary>
        /// "貼り付け"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusPaste(Command command)
        {
            if (TargetDocument == null) { return CommandStatus.SupportedAndVisible; }
            if (TargetDocument.IsReadOnly == true) { return CommandStatus.SupportedAndVisible; }

            if (_SampleMapPanel.ContainsFocus == true)
            {
                return QueryStatusRegionPaste();
            }

            return (CanPaste() ?
                     CommandStatus.SupportedAndEnabledAndVisible :
                     CommandStatus.SupportedAndVisible);
        }

        private CommandStatus QueryStatusRegionPaste()
        {
            bool isKeyRegion = false;
            bool isVelcityRegion = false;

            if (ClipboardService.CanPaste(typeof(Instrument)))
            {
                isKeyRegion = true;
            }
            else if (ClipboardService.CanPaste(typeof(KeyRegion)))
            {
                isVelcityRegion = true;
            }

            if (isKeyRegion == true || isVelcityRegion == true)
            {
                if (TargetComponents.Length == 0)
                {
                    Component[] instruments = _ListCtrl.GetComponentsBySelectedItem();
                    if (instruments != null && instruments.Length == 1)
                    {
                        Instrument instrument = (Instrument)instruments[0];
                        if (_SampleMapPanel.ExistBlankKeyRegion == true)
                        {
                            return CommandStatus.SupportedAndEnabledAndVisible;
                        }
                    }
                    return CommandStatus.SupportedAndVisible;
                }
                else
                {
                    foreach (Component component in TargetComponents)
                    {
                        if (isKeyRegion == true)
                        {
                            if (component is KeyRegion == true)
                            {
                            }
                            else if (component is VelocityRegion == true)
                            {
                                VelocityRegion velRegion = (VelocityRegion)component;
                                if (velRegion.Parent.Children.Count != 1)
                                {
                                    return CommandStatus.SupportedAndVisible;
                                }
                            }
                            else
                            {
                                return CommandStatus.SupportedAndVisible;
                            }
                        }
                        else if (isVelcityRegion == true)
                        {
                            if (component is KeyRegion == true)
                            {
                                KeyRegion keyRegion = (KeyRegion)component;
                                if (keyRegion.Children.Count != 1)
                                {
                                    return CommandStatus.SupportedAndVisible;
                                }
                            }
                            else if (component is VelocityRegion == true)
                            {
                            }
                            else
                            {
                                return CommandStatus.SupportedAndVisible;
                            }
                        }
                    }
                    return CommandStatus.SupportedAndEnabledAndVisible;
                }
            }
            else
            {
                return CommandStatus.SupportedAndVisible;
            }
        }

        ///--------------------------------
        /// <summary>
        /// "貼り付け"の実行
        /// </summary>
        private bool ExecutePaste(Command command)
        {
            if (ClipboardService.CanPaste(typeof(Instrument)) ||
                 ClipboardService.CanPaste(typeof(KeyRegion)))
            {
                Component[] instruments = _ListCtrl.GetComponentsBySelectedItem();
                if (instruments != null && instruments.Length == 1)
                {
                    Instrument instrument = (Instrument)instruments[0];
                    ClipboardService.PasteRegion(ComponentService,
                                                  TargetDocument.OperationHistory,
                                                  instrument,
                                                  TargetComponents);
                }
            }

            if (Paste() == false)
            {
                return false;
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        /// "セルのコピー"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusCopyCell(Command command)
        {
            if (_SampleMapPanel.ContainsFocus == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CanCopyCell() ? CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// "セルのコピー"の実行
        /// </summary>
        private bool ExecuteCopyCell(Command command)
        {
            return CopyCell();
        }

        ///--------------------------------
        /// <summary>
        /// "項目を選択して貼り付け"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusPasteSpecial(Command command)
        {
            if (TargetDocument == null) { return CommandStatus.SupportedAndVisible; }
            if (TargetDocument.IsReadOnly == true) { return CommandStatus.SupportedAndVisible; }

            if (_SampleMapPanel.ContainsFocus == true)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CanPasteSpecial() ? CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// "項目を選択して貼り付け"の実行
        /// </summary>
        private bool ExecutePasteSpecial(Command command)
        {
            if (PasteSpecial() == false)
            {
                return false;
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        /// "一括編集"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusBatchEdit(Command command)
        {
            if (TargetDocument == null) { return CommandStatus.SupportedAndVisible; }
            if (TargetDocument.IsReadOnly == true) { return CommandStatus.SupportedAndVisible; }

            return CanBatchEdit() ? CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// "一括編集"の実行
        /// </summary>
        private bool ExecuteBatchEdit(Command command)
        {
            if (BatchEdit() == false)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// "プログラムナンバーの振り直し"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusRenumberProgramNo(Command command)
        {
            if (TargetDocument == null) { return CommandStatus.SupportedAndVisible; }
            if (TargetDocument.IsReadOnly == true) { return CommandStatus.SupportedAndVisible; }

            return CanRenumberProgramNo() == true ?
                CommandStatus.SupportedAndEnabledAndVisible :
                CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// "プログラムナンバーの振り直し"の実行
        /// </summary>
        private bool ExecuteRenumberProgramNo(Command command)
        {
            RenumberProgramNoPrepareDialog dialog = new RenumberProgramNoPrepareDialog();
            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return true;
            }

            RenumberProgramNo(dialog.No);
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private bool CanRenumberProgramNo()
        {
            if (ActiveListCtrl == null)
            {
                return false;
            }

            ListItemSelectedDictionary selectedItems = ActiveListCtrl.GetItemSelecteds();
            return selectedItems.Values.Count > 0 ? true : false;
        }

        /// <summary>
        ///
        /// </summary>
        private void RenumberProgramNo(int cardinalNo)
        {
            ListItemSelectedDictionary selectedItems = ActiveListCtrl.GetItemSelecteds();
            List<CommonListItem> targetList = new List<CommonListItem>();
            List<CommonListItem> excludeList = new List<CommonListItem>();

            foreach (IListItemInfo info in ActiveListCtrl.ItemInfos)
            {
                CommonListItem item = info.Item as CommonListItem;
                Debug.Assert(item != null, "Item is null");

                if (selectedItems.ContainsKey(item) == true)
                {
                    targetList.Add(item);
                }
                else
                {
                    excludeList.Add(item);
                }
            }

            // 番号が重複するアイテムを収集します。
            RenumberProgramNoDialog dialog = new RenumberProgramNoDialog();
            string name = ProjectParameterNames.Instrument.ProgramNo;
            int overlapCount = 0;
            int no = cardinalNo;

            foreach (CommonListItem item in targetList)
            {
                int programNo = (int)item.Target.Parameters[name].Value;

                CommonListItem overlapItem = excludeList.Find
                    (delegate (CommonListItem i)
                           {
                               return (int)i.Target.Parameters[name].Value == no;
                           });
                if (overlapItem != null)
                {
                    string message = String.Format
                        (MessageResource.Message_OverlapProgramNo,
                          overlapItem.Target.Name, no);
                    dialog.AppendMessage(message);
                    overlapCount++;
                }

                no++;
            }

            // 番号の重複するアイテムが存在したのか？
            if (overlapCount > 0)
            {
                dialog.ShowDialog();
                return;
            }

            // 番号の振り直しを実行します。
            OperationHistory operationHistory = TargetDocument.OperationHistory;
            Operation operation = null;
            no = cardinalNo;

            operationHistory.BeginTransaction();

            foreach (CommonListItem item in targetList)
            {
                if ((operation = SetProgramNo(item, no)) == null)
                {
                    return;
                }

                operationHistory.AddOperation(operation);
                no++;
            }

            operationHistory.EndTransaction();
        }

        /// <summary>
        ///
        /// </summary>
        private Operation SetProgramNo(ComponentListItem item, int no)
        {
            Component component = item.Target;
            Operation operation = null;
            IParameterValue value = null;
            ValidationResult result = null;
            string name = ProjectParameterNames.Instrument.ProgramNo;

            try
            {
                value = item.GetValue(name);

                //
                result = value.ValidateValue(no);
                if (result.IsValid == false)
                {
                    return null;
                }

                operation = new SetParameterOperation(component.Parameters, name, no);
                operation.Execute();
            }

            catch { operation = null; }
            return operation;
        }

        ///--------------------------------
        /// <summary>
        /// "キーリージョンの分割"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusSplitKeyRegion(Command command)
        {
            if (TargetDocument == null) { return CommandStatus.SupportedAndVisible; }
            if (TargetDocument.IsReadOnly == true) { return CommandStatus.SupportedAndVisible; }

            return _PercussionListPanel.CanSplitKeyRegion() ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// "キーリージョンの分割"の実行
        /// </summary>
        private bool ExecuteSplitKeyRegion(Command command)
        {
            return _PercussionListPanel.SplitKeyRegion();
        }

        ///--------------------------------
        /// <summary>
        /// アイテムの追加コマンドが実行できるかどうかを取得します。
        /// </summary>
        /// <param name="command">アイテムの追加コマンド</param>
        /// <returns>実行できる場合は true、できない場合は false。</returns>
        private CommandStatus QueryStatusAddItem(Command command)
        {
            return ApplicationBase.Instance.QueryCommandStatus(ProjectCommands.AddInstrument);
        }

        /// <summary>
        /// アイテムの追加コマンドを実行します。
        /// </summary>
        /// <param name="command"></param>
        /// <returns></returns>
        private bool ExecuteAddItem(Command command)
        {
            return ApplicationBase.Instance.ExecuteCommand(ProjectCommands.AddInstrument);
        }

        ///--------------------------------
        /// <summary>
        /// アイテムの挿入コマンドが実行できるかどうかを取得します。
        /// </summary>
        /// <param name="command">アイテムの挿入コマンド</param>
        /// <returns>実行できる場合は true、できない場合は false。</returns>
        private CommandStatus QueryStatusInsertItem(Command command)
        {
            return ApplicationBase.Instance.QueryCommandStatus(ProjectCommands.InsertInstrument);
        }

        /// <summary>
        /// アイテムの挿入コマンドを実行します。
        /// </summary>
        /// <param name="command"></param>
        /// <returns></returns>
        private bool ExecuteInsertItem(Command command)
        {
            return ApplicationBase.Instance.ExecuteCommand(ProjectCommands.InsertInstrument);
        }

        private CommandStatus QueryStatusSetEnvelopeToRegion(Command command)
        {
            if (TargetDocument == null) { return CommandStatus.SupportedAndVisible; }
            if (TargetDocument.IsReadOnly == true) { return CommandStatus.SupportedAndVisible; }

            bool ret = false;

            if (_ListCtrl.SelectedItems.Length == 1)
            {
                ret = ((Instrument)(_ListCtrl.GetSelectedItems()[0].Target)).IsInstListEnvelope;
            }

            return ret ? CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        private bool ExecuteSetEnvelopeToRegion(Command command)
        {
            bool ret = false;

            if (_ListCtrl.SelectedItems.Length == 1)
            {
                if (FormsApplication.Instance.UIService.
                    ShowMessageBox(MessageResource.Inquire_CopyEnvelopeToVelocityRegion,
                                   MessageResource.DialogTitle_InstrumentList_SetEnvelopeToRegion_Text,
                                   AppMessageBoxButton.OKCancel,
                                   AppMessageBoxImage.Warning) == AppMessageBoxResult.OK)
                {
                    _ListAdapter.BeginTransaction();
                    {
                        Component target = _ListCtrl.GetSelectedItems()[0].Target;
                        _ListAdapter.
                            SetValue(target, ProjectParameterNames.Instrument.EnvelopeMode, InstrumentEnvelopeMode.VelocityRegion);
                        CopyEnvelopeToVelocityRegions(target);
                        ret = true;
                    }
                    _ListAdapter.EndTransaction();

                    BuildCommandUIs();
                    _ParameterPanel.UpdateUI();
                }
            }

            return ret;
        }

        private CommandStatus QueryStatusSetEnvelopeToInstrument(Command command)
        {
            if (TargetDocument == null) { return CommandStatus.SupportedAndVisible; }
            if (TargetDocument.IsReadOnly == true) { return CommandStatus.SupportedAndVisible; }

            bool ret = false;

            if (_ListCtrl.SelectedItems.Length == 1)
            {
                ret =
                    ((Instrument)(_ListCtrl.GetSelectedItems()[0].Target)).IsVelocityRegionEnvelope;
            }

            return ret ? CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        private bool ExecuteSetEnvelopeToInstrument(Command command)
        {
            bool ret = false;

            if (_ListCtrl.SelectedItems.Length == 1)
            {
                if (FormsApplication.Instance.UIService.
                    ShowMessageBox(MessageResource.Inquire_CopyEnvelopeToInstList,
                                   MessageResource.DialogTitle_InstrumentList_SetEnvelopeToInstrument_Text,
                                   AppMessageBoxButton.OKCancel,
                                   AppMessageBoxImage.Warning) == AppMessageBoxResult.OK)
                {
                    Component target = _ListCtrl.GetSelectedItems()[0].Target;
                    _ListAdapter.SetValue(target, ProjectParameterNames.Instrument.EnvelopeMode, InstrumentEnvelopeMode.Instrument);
                    ret = true;

                    BuildCommandUIs();
                    _ParameterPanel.UpdateUI();
                }
            }

            return ret;
        }

        private void CopyEnvelopeToVelocityRegions(Component target)
        {
            Instrument instrument = (Instrument)target;
            ComponentList keyRegions = target.Children;

            foreach (Component keyRegion in keyRegions)
            {
                foreach (Component velocityRegion in keyRegion.Children)
                {
                    _ListAdapter.SetValue(
                        velocityRegion,
                        ProjectParameterNames.Envelope.Attack,
                        instrument.Envelope.Attack);
                    _ListAdapter.SetValue(
                        velocityRegion,
                        ProjectParameterNames.Envelope.Decay,
                        instrument.Envelope.Decay);
                    _ListAdapter.SetValue(
                        velocityRegion,
                        ProjectParameterNames.Envelope.Hold,
                        instrument.Envelope.Hold);
                    _ListAdapter.SetValue(
                        velocityRegion,
                        ProjectParameterNames.Envelope.Release,
                        instrument.Envelope.Release);
                    _ListAdapter.SetValue(
                        velocityRegion,
                        ProjectParameterNames.Envelope.Sustain,
                        instrument.Envelope.Sustain);
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Instrument CreateInstrument(string filePath)
        {
            try
            {
                WaveFileReader.CreateInstance(filePath);
            }
            catch
            {
                return null;
            }

            Instrument instrument = null;
            KeyRegion keyRegion = null;
            VelocityRegion velRegion = null;

            keyRegion = new KeyRegion()
            {
                KeyMin = 0,
                KeyMax = 127,
            };

            velRegion = new VelocityRegion()
            {
                VelocityMin = 0,
                VelocityMax = 127,
            };

            instrument = ApplicationBase.Instance.CreateComponentService.Create<Instrument>();
            instrument.ProgramNo = ProgramNoCreator.Create(BankService);
            instrument.Name = ItemNamer.CreateUniqueNameFromFileName
                (BankService.ComponentDictionary,
                 new ItemNamingSettings(ProjectService.Project),
                 ProjectService.Project.InstrumentNamePrefix, String.Empty, filePath);

            velRegion.OriginalKey = OriginalKeyCreator.FromFile(filePath);
            velRegion.FilePath = filePath;

            keyRegion.Children.Add(velRegion);
            instrument.Children.Add(keyRegion);
            return instrument;
        }

        private IParameterValue GetValueGetter(Component component, string name, ref bool cancel)
        {
            if (name == ProjectParameterNames.VelocityRegion.OriginalKey)
            {
                if (component == null || component is Instrument == false)
                {
                    return null;
                }
                Instrument inst = component as Instrument;
                if (inst.Children.Count <= 0)
                {
                    return null;
                }
                KeyRegion key = inst.Children[0] as KeyRegion;
                if (key.Children.Count <= 0)
                {
                    return null;
                }
                VelocityRegion vel = key.Children[0] as VelocityRegion;

                return new OriginalKeyParameterEditor(name, vel.Parameters[name]);
            }

            return null;
        }
    }


    /// <summary>
    /// インストルメントのリストアイテム
    /// </summary>
    public class InstrumentListItem : CommonListItem
    {
        private VelocityRegion _TargetVelocityRegion = null;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public InstrumentListItem(Component component)
            : base(component)
        {
            Instrument instrument = component as Instrument;
            if (instrument != null)
            {
                instrument.UpdateLinkedRegion();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public int OriginalKey
        {
            get
            {
                VelocityRegion velRegion = TargetVelocityRegion;

                if (velRegion == null ||
                    velRegion.Parameters.ContainsKey(ProjectParameterNames.VelocityRegion.OriginalKey) == false)
                {
                    return -1;
                }

                return velRegion.OriginalKey;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override Component GetTargetByName(string name)
        {
            if (name == ProjectParameterNames.WaveEncoding ||
                name == ProjectParameterNames.FilePath ||
                name == ProjectParameterNames.VelocityRegion.OriginalKey ||
                name == ProjectParameterNames.VelocityRegion.InterpolationType)
            {
                return TargetVelocityRegion;
            }

            return base.GetTargetByName(name);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public override IParameterValue GetValue(string name)
        {
            ParameterEditor parameterEditor = null;
            VelocityRegion velRegion = TargetVelocityRegion;

            _TargetVelocityRegion = velRegion;

            //
            switch (name)
            {
                case ProjectParameterNames.Name:
                    return CreateParameterEditor
                        (name, new NameParameterValue
                          ((string)(((Instrument)Target).Parameters[name].Value),
                            ((Instrument)Target)));

                case ProjectParameterNames.VelocityRegion.OriginalKey:
                    if (_TargetVelocityRegion == null ||
                        _TargetVelocityRegion.Parameters.ContainsKey(name) == false)
                    {
                        return null;
                    }

                    parameterEditor = new OriginalKeyParameterEditor
                        (name, _TargetVelocityRegion.Parameters[name]);
                    parameterEditor.ValueChanged += OnValueChanged;
                    return parameterEditor;

                //
                case ProjectParameterNames.WaveEncoding:
                case ProjectParameterNames.FilePath:
                case ProjectParameterNames.VelocityRegion.InterpolationType:
                    if (_TargetVelocityRegion == null ||
                        _TargetVelocityRegion.Parameters.ContainsKey(name) == false)
                    {
                        return null;
                    }

                    parameterEditor = CreateParameterEditor
                        (name, _TargetVelocityRegion.Parameters[name], OnValueChanged) as
                        ParameterEditor;

                    return parameterEditor;

                //
                case ProjectParameterNames.Envelope.Attack:
                case ProjectParameterNames.Envelope.Decay:
                case ProjectParameterNames.Envelope.Hold:
                case ProjectParameterNames.Envelope.Sustain:
                case ProjectParameterNames.Envelope.Release:
                    if (((Instrument)Target).IsVelocityRegionEnvelope == true)
                    {
                        return null;
                    }
                    break;
            }

            return base.GetValue(name);
        }

        /// <summary>
        ///
        /// </summary>
        public override IConstParameterValue GetConstValue(string name)
        {
            IConstParameterValue ret = null;
            VelocityRegion velRegion = null;

            if (Target.Children.Count == 1 && Target.Children[0].Children.Count == 1)
            {
                velRegion = (VelocityRegion)(Target.Children[0].Children[0]);
            }

            switch (name)
            {
                case ProjectParameterNames.WaveEncoding:
                    if (velRegion != null)
                    {
                        ret = new WaveEncodingParameterValue(velRegion.Encoding);
                    }
                    else
                    {
                        ret = new TextParameterValue("*");
                    }
                    break;

                case ProjectParameterNames.FilePath:
                    if (velRegion != null)
                    {
                        ret = new FilePathParameterValue(velRegion.FilePath);
                    }
                    else
                    {
                        ret = new TextParameterValue("*");
                    }
                    break;

                case ProjectParameterNames.WaveTick:
                    if (velRegion != null)
                    {
                        velRegion.UpdateWaveFile();
                        string tick = WaveFileUtility.GetWaveTickString(velRegion.WaveFile);
                        ret = GetNATextParameterValue(tick);
                    }
                    else
                    {
                        ret = new TextParameterValue("*");
                    }
                    break;

                case ProjectParameterNames.WaveTime:
                    if (velRegion != null)
                    {
                        velRegion.UpdateWaveFile();
                        string times = WaveFileUtility.GetWaveTimeStringWithLoop(velRegion.WaveFile);
                        ret = GetNATextParameterValue(times);
                    }
                    else
                    {
                        ret = new TextParameterValue("*");
                    }
                    break;

                case ProjectParameterNames.SampleRate:
                    // リサンプルが有効ならサンプルレートを返す。
                    if ((bool)Target.Parameters[ProjectParameterNames.IsResampleEnabled].Value == true)
                    {
                        SampleRateParameterValue param = Target.Parameters[ProjectParameterNames.SampleRate] as SampleRateParameterValue;
                        // サンプルレートが設定されていれば（未設定でなければ）
                        if (param.IsUnsettingValue == false)
                        {
                            // サンプルレートを返す。
                            ret = new TextParameterValue(WaveFileUtility.GetSampleRateString(param.Value));
                            break;
                        }
                    }

                    if (velRegion != null)
                    {
                        // 波形ファイルのサンプルレートを返す。
                        WaveFile waveFile = GetComponentReferenceWaveFile(velRegion);
                        string text = WaveFileUtility.GetSampleRateString(waveFile);
                        ret = GetNATextParameterValue(text);
                    }
                    else
                    {
                        ret = new TextParameterValue("*");
                    }
                    break;

                case ProjectParameterNames.WaveBitRate:
                    if (velRegion != null)
                    {
                        WaveFile waveFile = GetComponentReferenceWaveFile(velRegion);

                        // リサンプルが有効なら設定されたサンプルレートからビットレートを求める。
                        if ((bool)Target.Parameters[ProjectParameterNames.IsResampleEnabled].Value == true)
                        {
                            SampleRateParameterValue param = Target.Parameters[ProjectParameterNames.SampleRate] as SampleRateParameterValue;
                            // サンプルレートが設定されていれば（未設定でなければ）
                            if (param.IsUnsettingValue == false)
                            {
                                ret = GetNATextParameterValue(WaveFileUtility.GetWaveBitRateString(waveFile, param.Value));
                                break;
                            }
                        }

                        // 波形ファイルのサンプルレートからビットレートを求める。
                        ret = GetNATextParameterValue(WaveFileUtility.GetWaveBitRateString(waveFile));
                    }
                    else
                    {
                        ret = new TextParameterValue("*");
                    }
                    break;

                case ProjectParameterNames.WaveSampleBit:
                    if (velRegion != null)
                    {
                        WaveFile waveFile = GetComponentReferenceWaveFile(velRegion);
                        string text = WaveFileUtility.GetWaveSampleBitString(waveFile);
                        ret = GetNATextParameterValue(text);
                    }
                    else
                    {
                        ret = new TextParameterValue("*");
                    }
                    break;

                case ProjectParameterNames.WaveChannel:
                    // ダウンミックスが有効ならモノラルを表示する。
                    if ((bool)Target.Parameters[ProjectParameterNames.IsDownMixEnabled].Value == true)
                    {
                        var str = WaveFileUtility.GetChannelString(1); // モノラル

                        return new TextWithNAParameterValue(str, str != NotAvailable.Text);
                    }

                    if (velRegion != null)
                    {
                        WaveFile waveFile = GetComponentReferenceWaveFile(velRegion);
                        string text = WaveFileUtility.GetChannelString(waveFile);
                        ret = GetNATextParameterValue(text);
                    }
                    else
                    {
                        ret = new TextParameterValue("*");
                    }
                    break;

                case ProjectParameterNames.VelocityRegion.OriginalKey:
                    if (velRegion != null)
                    {
                        ret = new TextParameterValue(KeyNoteConverter.ToNote(velRegion.OriginalKey));
                    }
                    else
                    {
                        ret = new TextParameterValue("*");
                    }
                    break;

                case ProjectParameterNames.VelocityRegion.InterpolationType:
                    if (velRegion != null)
                    {
                        ret = new InterpolationTypeParameterValue(velRegion.InterpolationType);
                    }
                    else
                    {
                        ret = new TextParameterValue("*");
                    }
                    break;

                case ProjectParameterNames.Envelope.Attack:
                case ProjectParameterNames.Envelope.Decay:
                case ProjectParameterNames.Envelope.Hold:
                case ProjectParameterNames.Envelope.Sustain:
                case ProjectParameterNames.Envelope.Release:
                    if (((Instrument)Target).IsVelocityRegionEnvelope == true)
                    {
                        ret = new TextParameterValue("*");
                    }
                    else
                    {
                        ret = base.GetConstValue(name);
                    }
                    break;

                default:
                    ret = base.GetConstValue(name);
                    break;
            }

            return ret;
        }

        /// <summary>
        ///
        /// </summary>
        public class NameParameterValue : TextParameterValue
        {
            private Instrument _Instrument = null;

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            public NameParameterValue(string text, Instrument instrument)
            {
                _Instrument = instrument;
                Value = text;
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            private BankService BankService
            {
                get
                {
                    BankService bankService = null;
                    Bank bank = _Instrument.Bank;
                    IEnumerable<BankService> values =
                        FormsApplication.Instance.BankServices.Values;
                    foreach (BankService service in values)
                    {
                        if (service.Bank == bank)
                        {
                            bankService = service;
                            break;
                        }
                    }

                    return bankService;
                }
            }

            ///--------------------------------
            /// <summary>
            ///
            /// </summary>
            protected override ValidationResult ValidateInternal(string value)
            {
                ValidationResult result = null;

                result = ItemNameValidator.ValidateSoundProjectItemName(value);
                if (result.IsValid == false)
                {
                    return result;
                }

                if (BankService.ComponentDictionary.Contains(value) == true)
                {
                    return new ValidationResult
                        (false, MessageResource.Message_ItemNameAlreadyExisting);
                }

                return ValidationResult.NoError;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private TextParameterValue GetNATextParameterValue(string text)
        {
            // 引数の文字列が NotAvailable.Textであることで判断しています。
            bool available = text == NotAvailable.Text ? false : true;
            return new TextWithNAParameterValue(text, available);
        }

        /// <summary>
        ///
        /// </summary>
        private VelocityRegion TargetVelocityRegion
        {
            get
            {
                if (Target.Children.Count != 1 ||
                     Target.Children[0].Children.Count != 1)
                {
                    return null;
                }
                return Target.Children[0].Children[0] as VelocityRegion;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Bank TargetBank
        {
            get { return ((Bank)(Target.Parent)); }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected void OnValueChanged(object sender, EventArgs e)
        {
            ParameterEditor parameterEditor = sender as ParameterEditor;
            string name = parameterEditor.Name;

            if (name == ProjectParameterNames.SampleRate)
            {
                Adapter.SetValue(Target, parameterEditor.Name, parameterEditor.Value);
                return;
            }

            if (name == ProjectParameterNames.FilePath && Target != null)
            {
                if (Target is Instrument)
                {
                    if (Target.Children.Count == 1 &&
                        Target.Children[0].Children.Count == 1 &&
                        Target.Children[0].Children[0] is VelocityRegion)
                    {
                        VelocityRegion velocityRegion =
                            Target.Children[0].Children[0] as VelocityRegion;
                        velocityRegion.WaveFile = null;
                    }
                }
            }

            Adapter.SetValue(_TargetVelocityRegion, parameterEditor.Name, parameterEditor.Value);
        }

        protected override WaveFile GetComponentReferenceWaveFile(Component component)
        {
            VelocityRegion velRegion = null;

            // component がリージョンだった場合
            if (component is VelocityRegion == true)
            {
                velRegion = component as VelocityRegion;
            }

            // Instrument 下のリージョンが１つの場合
            if (component is Instrument == true &&
                component.Children.Count == 1 &&
                component.Children[0].Children.Count == 1)
            {
                velRegion = component.Children[0].Children[0] as VelocityRegion;
            }

            if (velRegion != null)
            {
                if (velRegion.WaveFile == null)
                {
                    velRegion.UpdateWaveFile();
                }

                return velRegion.WaveFile;
            }

            return null;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class BankTabCtrl : NTabControl
    {
        ///
        protected override void OnClosing(NTabControlCancelEventArgs e)
        {
            e.Cancel = true;
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public interface IInnerBankPanel
    {
        void Shown();

        void UndoExecuted();
        void RedoExecuted();
        void RedrawControls();

        void CommandExecuted(Command command);
        void UpdateReadOnly(bool readOnly);
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class InnerBankPage : NTabPage
    {
        public InnerBankPage()
        {
            SelectOnVisible = false;
        }

        public IInnerBankPanel Panel
        {
            get { return Controls[0] as IInnerBankPanel; }
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class ScrollTabPage : InnerBankPage
    {
        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnMouseWheel(MouseEventArgs e)
        {
            int relativeMove = e.Delta * SystemInformation.MouseWheelScrollLines / 120;
            relativeMove *= -16;

            if ((Control.ModifierKeys & Keys.Shift) != 0)
            {
                if (HorizontalScroll.Visible == true)
                {
                    if (HorizontalScroll.Value + relativeMove > HorizontalScroll.Maximum)
                    {
                        HorizontalScroll.Value = HorizontalScroll.Maximum;
                    }
                    else if (HorizontalScroll.Value + relativeMove < HorizontalScroll.Minimum)
                    {
                        HorizontalScroll.Value = HorizontalScroll.Minimum;
                    }
                    else
                    {
                        HorizontalScroll.Value += relativeMove;
                    }
                    Invalidate();
                }
            }
            else
            {

                if (VerticalScroll.Visible == true)
                {
                    if (VerticalScroll.Value + relativeMove > VerticalScroll.Maximum)
                    {
                        VerticalScroll.Value = VerticalScroll.Maximum;
                    }
                    else if (VerticalScroll.Value + relativeMove < VerticalScroll.Minimum)
                    {
                        VerticalScroll.Value = VerticalScroll.Minimum;
                    }
                    else
                    {
                        VerticalScroll.Value += relativeMove;
                    }
                    Invalidate();
                }
            }
        }
    }
}
