﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Data;
    using System.Diagnostics;
    using System.Drawing;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;

    public partial class CreateStreamSoundTrackDialog : CreateSoundDialog
    {
        private readonly StreamSoundTrackBase _track = ApplicationBase.Instance.CreateComponentService.Create<StreamSoundTrackBase>();

        public CreateStreamSoundTrackDialog()
            : base()
        {
            InitializeComponent();

            this.Volume = _track.Volume;
            this.Pan = _track.Pan;
            this.SurroundPan = _track.SurroundPan;
            this.checkBox_FrontBypass.Checked = false;
        }

        public CreateStreamSoundTrackDialog(string filePath)
            : this()
        {
            if (AppConfiguration.EnabledMultiChannelAAC == true &&
                AACUtil.IsAACFile(filePath) == true)
            {
                this.FilePath = filePath;
                this.radioButton_Channel1.Checked = true;

                this.label_ChannelCount.Visible = true;
                this.label_ChannelCount.Enabled = true;
                this.radioButton_Channel1.Visible = true;
                this.radioButton_Channel1.Enabled = true;
                this.radioButton_Channel2.Visible = true;
                this.radioButton_Channel2.Enabled = true;
                this.button_Reference.Visible = false;
                this.button_Reference.Enabled = false;
                this.textBox_FilePath.ReadOnly = true;
            }
        }

        ///--------------------------------
        /// <summary>
        /// ファイルパスの取得
        /// </summary>
        private string FilePath
        {
            get
            {
                return textBox_FilePath.Text;
            }
            set
            {
                textBox_FilePath.Text = value;
            }
        }

        ///--------------------------------
        /// <summary>
        /// 音量の取得
        /// </summary>
        private int Volume
        {
            get
            {
                int value = 0;
                try
                {
                    value = int.Parse(textBox_Volume.Text);
                }
                catch
                {
                    Debug.Assert(false, "Failure int Parse");
                }

                return value;
            }
            set
            {
                textBox_Volume.Text = value.ToString();
            }
        }

        ///--------------------------------
        /// <summary>
        /// パンの取得
        /// </summary>
        private int Pan
        {
            get
            {
                int value = 0;
                try
                {
                    value = int.Parse(textBox_Pan.Text);
                }
                catch
                {
                    Debug.Assert(false, "Failure int Parse");
                }

                return value;
            }
            set
            {
                textBox_Pan.Text = value.ToString();
            }
        }

        ///--------------------------------
        /// <summary>
        /// サラウンドパンの取得
        /// </summary>
        private int SurroundPan
        {
            get
            {
                int value = 0;
                try
                {
                    value = int.Parse(textBox_SurroundPan.Text);
                }
                catch
                {
                    Debug.Assert(false, "Failure int Parse");
                }

                return value;
            }
            set
            {
                textBox_SurroundPan.Text = value.ToString();
            }
        }

        ///--------------------------------
        /// <summary>
        /// フロントバイパスの取得
        /// </summary>
        private bool FrontBypass
        {
            get { return checkBox_FrontBypass.Checked; }
        }

        ///--------------------------------
        /// <summary>
        /// １トラックあたりのチャンネル数(AAC)の取得
        /// </summary>
        private int ChannelCount
        {
            get
            {
                return (this.radioButton_Channel1.Checked == true ? 1 : 2);
            }
        }

        /// <summary>
        /// 作成したトラックを取得します。
        /// </summary>
        public StreamSoundTrackBase NewTrack
        {
            get
            {
                _track.FilePath = this.FilePath;
                _track.Volume = this.Volume;
                _track.Pan = this.Pan;
                _track.SurroundPan = this.SurroundPan;
                _track.FrontBypass = this.FrontBypass;
                _track.ChannelCount = this.ChannelCount;

                return _track;
            }
        }

        ///--------------------------------
        /// <summary>
        /// 確定に使用するボタンの取得
        /// </summary>
        protected override Control GetAcceptButton()
        {
            return button_OK;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override bool CanCreate
        {
            get
            {
                try
                {
                    bool filePath = InspectFilePath();
                    bool volume = InspectVolume();
                    bool pan = InspectPan();
                    bool surroundPan = InspectSurroundPan();

                    if (filePath == true &&
                        volume == true &&
                        pan == true &&
                        surroundPan == true)
                    {
                        return true;
                    }
                }
                catch { }
                return false;
            }
        }

        ///--------------------------------
        /// <summary>
        /// ファイルパスの検査
        /// </summary>
        private bool InspectFilePath()
        {
            if (File.Exists(textBox_FilePath.Text) == false)
            {
                ShowError(textBox_FilePath, MessageResource.Message_FileNotFound);
                return false;
            }

            HideError(textBox_FilePath);

            return true;
        }

        ///--------------------------------
        /// <summary>
        /// ボリュームの検査
        /// </summary>
        private bool InspectVolume()
        {
            return ValidateIntParameter(_track, ProjectParameterNames.Volume,
                                        textBox_Volume.Text, textBox_Volume);
        }

        ///--------------------------------
        /// <summary>
        /// パンの検査
        /// </summary>
        private bool InspectPan()
        {
            return ValidateIntParameter(_track, ProjectParameterNames.Pan,
                                        textBox_Pan.Text, textBox_Pan);
        }

        ///--------------------------------
        /// <summary>
        /// サラウンドパンの検査
        /// </summary>
        private bool InspectSurroundPan()
        {
            return ValidateIntParameter(_track, ProjectParameterNames.SurroundPan,
                                        textBox_SurroundPan.Text, textBox_SurroundPan);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnClickReference(object sender, System.EventArgs e)
        {
            string filePath = null;

            if ((filePath = WaveFileQuester.QueryLoad(this, ApplicationBase.Instance.AppConfiguration.StreamWaveFileFilter)) != null)
            {
                this.FilePath = filePath;
            }

            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnTextChanged(object sender, System.EventArgs e)
        {
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedFilePath(object sender, System.EventArgs e)
        {
            InspectFilePath();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedVolume(object sender, System.EventArgs e)
        {
            InspectVolume();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedPan(object sender, System.EventArgs e)
        {
            InspectPan();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnValidatedSurroundPan(object sender, System.EventArgs e)
        {
            InspectSurroundPan();
            UpdateControls();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnClosing(object sender, FormClosingEventArgs e)
        {
            if (DialogResult == DialogResult.OK)
            {
                if (ProjectFilePathUtility.ConfirmFilePath(FilePath) == false)
                {
                    e.Cancel = true;
                    return;
                }
            }
        }
    }
}
