﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.Drawing.Imaging;
    using System.IO;
    using System.Linq;
    using System.Security.Cryptography;
    using System.Windows.Forms;
    using System.Windows.Forms.VisualStyles;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.ToolDevelopmentKit.Collections;
    using SFresrc = NintendoWare.SoundFoundation.Resources;
    using Utilities;

    /// <summary>
    ///
    /// </summary>
    public interface IPanel
    {
        void Activated();
        void Deactivated();
    }

    /// <summary>
    ///
    /// </summary>
    public interface IEditableControlContainer
    {
        void CancelEdit();
    }

    /// <summary>
    ///
    /// </summary>
    public class CommonPanel : UserControl, IPanel, ICommandTarget
    {
        private CommandBindingCollection _CommandBindings = new CommandBindingCollection();
        private string _Title = String.Empty;

        public event EventHandler Closed;

        ///--------------------------------
        /// <summary>
        /// タイトルの取得
        /// </summary>
        public string Title
        {
            get { return _Title; }
            set
            {
                _Title = value;

                if (TabPage != null)
                {
                    TabPage.Text = _Title;
                }
            }
        }

        ///--------------------------------
        /// <summary>
        /// パネルに関連付けられたドキュメントを取得します。
        /// </summary>
        public virtual Document Document
        {
            get { return null; }
        }

        ///--------------------------------
        /// <summary>
        /// アイコンの種類を取得または設定
        /// </summary>
        public TabCtrl.IconType ImageIndex
        {
            get
            {
                if (TabPage == null) { return TabCtrl.IconType.None; }
                return (TabCtrl.IconType)TabPage.ImageIndex;
            }
            set
            {
                if (TabPage == null) { return; }
                TabPage.ImageIndex = (int)value;
            }
        }

        ///--------------------------------
        /// <summary>
        /// クリップボードが更新された時に呼ばれる
        /// </summary>
        public virtual void UpdatedClipboard()
        {
        }

        ///--------------------------------
        /// <summary>
        /// タイトルの更新
        /// </summary>
        public virtual void UpdateTitle()
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual void RefreshPanel()
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual void RedrawPanel()
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual void ActivatePanel(bool resetSelection, bool mainList)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual SoundDocument TargetDocument
        {
            get { return null; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual Component[] TargetComponents
        {
            get { return null; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual void Activated() { }
        public virtual void Deactivated() { }

        ///--------------------------------
        /// <summary>
        ///
        /// 戻り値が falseの時には閉じる処理がキャンセルされる
        /// </summary>
        public virtual bool OnClosing() { return true; }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual void OnClosed()
        {
            if (Closed != null)
            {
                Closed(this, EventArgs.Empty);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public new virtual void KeyDown(KeyEventArgs e) { }
        public new virtual void KeyUp(KeyEventArgs e) { }
        public new virtual void KeyPress(KeyPressEventArgs e) { }

        /// <summary>
        ///
        /// </summary>
        public virtual void CommandExecuted(Command command)
        {
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual CommonListCtrl CurrentListCtrl
        {
            get { return null; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public virtual CommonListCtrl NextCandidateListCtrl(CommonListCtrl ctrl, int direction, bool repeat)
        {
            return null;
        }

        /// <summary>
        /// PanelBase クラスにコマンドを関連付けます。
        /// </summary>
        /// <param name="binding">コマンドバインディング。</param>
        public void BindCommand(CommandBinding binding)
        {
            if (null == binding) { throw new ArgumentNullException("binding"); }
            if (this._CommandBindings.Contains(binding.CommandID) == true)
            {
                this._CommandBindings.Remove(binding.CommandID);
            }
            _CommandBindings.Add(binding);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private CommonTabPage TabPage
        {
            get { return Parent as CommonTabPage; }
        }

        ///--------------------------------
        /// <summary>
        /// 指定コマンドを実行できるコマンドターゲットを検索します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>他のコマンドターゲット。</returns>
        ICommandTarget ICommandTarget.FindTarget(Command command)
        {
            return (_CommandBindings.Contains(command.ID)) ? this : FindOtherTarget(command);
        }

        /// <summary>
        /// 指定コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <param name="command">実行するコマンド。</param>
        /// <returns>コマンドの状態。</returns>
        CommandStatus ICommandTarget.QueryStatus(Command command)
        {
            if (!_CommandBindings.Contains(command.ID)) { return CommandStatus.Unsupported; }
            return _CommandBindings[command.ID].QueryStatus(command);
        }

        /// <summary>
        /// 指定コマンドを実行します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドを実行した場合は true、キャンセルされた場合は false。</returns>
        bool ICommandTarget.Execute(Command command)
        {
            if (!_CommandBindings.Contains(command.ID)) { return false; }
            return _CommandBindings[command.ID].Execute(command);
        }

        /// <summary>
        /// PanelBase クラスにコマンドを関連付けます。
        /// </summary>
        /// <param name="binding">コマンドバインディング。</param>
        protected void BindCommand(string commandId, CommandBinding.QueryStatusHandler queryStatusHandler, CommandBinding.ExecuteHandler executeHandler)
        {
            CommandBinding binding = new CommandBinding
                (this, commandId, queryStatusHandler, executeHandler);
            if (null == binding) { throw new ArgumentNullException("binding"); }
            if (this._CommandBindings.Contains(commandId) == true)
            {
                this._CommandBindings.Remove(commandId);
            }
            this._CommandBindings.Add(binding);
        }

        /// <summary>
        ///
        /// </summary>
        protected void BindCommand(CommandHandler commandHandler)
        {
            CommandBinding binding = new CommandBinding(this, commandHandler);
            if (null == binding) { throw new ArgumentNullException("binding"); }
            if (this._CommandBindings.Contains(binding.CommandID) == true)
            {
                this._CommandBindings.Remove(binding.CommandID);
            }
            this._CommandBindings.Add(binding);
        }

        /// <summary>
        /// 指定コマンドを実行できる他のコマンドターゲットを検索します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>他のコマンドターゲット。</returns>
        protected virtual ICommandTarget FindOtherTarget(Command command)
        {
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        protected override void OnParentChanged(EventArgs e)
        {
            if (null != TabPage)
            {
                TabPage.Text = _Title;
            }

            base.OnParentChanged(e);
        }
    }

    /// <summary>
    /// リストを含むパネルの基本クラス
    /// </summary>
    public class CommonListPanel : CommonPanel
    {
        /// <summary>
        ///
        /// </summary>
        public enum SettingKinds
        {
            ColorComment,
        }

        /// <summary>
        ///
        /// </summary>
        public virtual CommonListCtrl[] ListControls
        {
            get
            {
                return new CommonListCtrl[0];
            }
        }

        /// <summary>
        ///
        /// </summary>
        public virtual void UpdatedOptions()
        {
        }

        /// <summary>
        ///
        /// </summary>
        public virtual void ApplyProjectConfigurations()
        {
        }

        /// <summary>
        /// プロジェクト設定によりユーザーパラメータが変更された時に呼ばれます。
        /// </summary>
        public virtual void ApplyUserParameterColumns()
        {
        }

        /// <summary>
        /// プロジェクト設定によりカラム名が変更された時に呼ばれます。
        /// </summary>
        public virtual void ApplyColumnsText()
        {
        }

        /// <summary>
        /// オプション設定が変更された時に呼ばれます。
        /// </summary>
        public virtual void ApplyOptionSettings()
        {
        }

        /// <summary>
        /// 設定が変更された時に呼ばれます。
        /// </summary>
        public virtual void ApplySettings(SettingKinds kind)
        {
        }

        /// <summary>
        /// プリセット名からをプリセットを適用します。
        /// </summary>
        public virtual void ApplyPresetListColumns(string presetName)
        {
        }

        /// <summary>
        /// プリセットを更新します。
        /// </summary>
        public virtual void UpdatePresetListColumns()
        {
        }

        /// <summary>
        ///
        /// </summary>
        protected XmlOptions Options
        {
            get
            {
                return ApplicationBase.Instance.AppConfiguration.Options;
            }
        }

        /// <summary>
        /// アクティブリストコントロールの取得
        /// </summary>
        protected virtual CommonListCtrl ActiveListCtrl
        {
            get { return null; }
        }

        /// <summary>
        /// アクティブリストアダプタの取得
        /// </summary>
        protected virtual ComponentListAdapter ActiveListAdapter
        {
            get { return null; }
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual ComponentService ComponentService
        {
            get { return null; }
        }

        ///--------------------------------
        /// <summary>
        /// "すべて選択"が実行可能なのか調べる
        /// </summary>
        protected bool CanSelectAll()
        {
            return ActiveListCtrl != null ? true : false;
        }

        ///--------------------------------
        /// <summary>
        /// "すべて選択"の実行
        /// </summary>
        protected bool SelectAll()
        {
            if (ActiveListCtrl == null)
            {
                return false;
            }

            ActiveListCtrl.SelectAll();
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// "貼り付け"が実行可能なのか調べる
        /// </summary>
        protected bool CanPaste()
        {
            ComponentListAdapter activeListAdapter = ActiveListAdapter;

            if (activeListAdapter == null ||
                ActiveListCtrl == null)
            {
                return false;
            }

            if (ActiveListCtrl.ReadOnly == true)
            {
                return false;
            }

            var targetComponent = activeListAdapter.ItemsOwner;
            if (targetComponent is PlayerPack ||
                targetComponent is GroupPack ||
                targetComponent is GroupBase)
            {
                if (AddonSoundArchiveUtility.IsItemInAddonSoundArchive(targetComponent) == true)
                {
                    return false;
                }
            }

            //セルの貼り付けが可能なのか？
            if (ClipboardService.CanPasteCell(ActiveListCtrl.ItemsName,
                                               ActiveListCtrl.GetItemSelecteds()) != false)
            {
                return true;
            }

            //貼り付けが可能なのか？
            return ClipboardService.CanPaste(activeListAdapter.ItemsOwner);
        }

        /// <summary>
        /// "貼り付け"の実行
        /// </summary>
        protected virtual bool Paste()
        {
            if (ActiveListAdapter == null ||
                ActiveListCtrl == null)
            {
                return false;
            }

            if (ActiveListCtrl.ReadOnly == true)
            {
                return false;
            }

            //
            if (ClipboardService.CanPaste(ActiveListAdapter.ItemsOwner) != false)
            {
                try
                {
                    Component[] targetComponents = TargetComponents;
                    Component prevSiblingComponent = null;
                    Component nextSiblingComponent = null;

                    if (targetComponents.Length > 0)
                    {
                        prevSiblingComponent = targetComponents[0];
                        nextSiblingComponent = targetComponents[targetComponents.Length - 1];
                    }

                    Component[] components = ClipboardService.Paste
                        (ComponentService, TargetDocument, ActiveListAdapter.ItemsOwner,
                          //targetComponents.Length > 0 ? targetComponents[0] : null,
                          prevSiblingComponent,
                          nextSiblingComponent,
                          this.Options.Application.General.PasteToItemLower);

                    ActiveListCtrl.SelectItemsByComponent(components);
                }
                catch (InvalidPasteNameException)
                {
                    ApplicationBase.Instance.UIService.ShowMessageBox
                        (Resources.MessageResource.Message_InvalidPasteName,
                          AppMessageBoxButton.OK, AppMessageBoxImage.Warning);
                }
                return true;
            }

            //
            if (ClipboardService.CanPasteCell(ActiveListCtrl.ItemsName,
                                               ActiveListCtrl.GetItemSelecteds()) != false)
            {

                ClipboardService.PasteCell(TargetDocument.OperationHistory,
                                            ActiveListAdapter.ItemsOwner,
                                            ActiveListCtrl.ItemsName,
                                            ActiveListCtrl.GetItemSelecteds(),
                                            ActiveListAdapter.Items);
                return true;
            }

            return false;
        }

        /// <summary>
        ///
        /// </summary>
        protected enum PasteKinds
        {
            Nothing,    // 行うことのできる貼り付けが無い
            Paste,      // 貼り付け
            PasteCell,  // セルの貼り付け
        }

        /// <summary>
        /// 実行可能な貼り付けの種類を取得します。
        /// </summary>
        protected PasteKinds GetNowExecutePasteKind()
        {
            // 行の貼り付けが可能なのか？
            if (ClipboardService.CanPaste(ActiveListAdapter.ItemsOwner) != false)
            {
                return PasteKinds.Paste;
            }

            // セルの貼り付けが可能なのか？
            if (ClipboardService.CanPasteCell(ActiveListCtrl.ItemsName,
                                              ActiveListCtrl.GetItemSelecteds()) != false)
            {
                return PasteKinds.PasteCell;
            }

            return PasteKinds.Nothing;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Type[] _CanPasteToGroupItemTypes = new[] {
            typeof(WaveSoundSetPack),
            typeof(SequenceSoundPack),
            typeof(SequenceSoundSetPack),
            typeof(SoundSetBankPack),
            typeof(WaveArchivePack),
            typeof(GroupPack),
        };

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected Component ReplaceComponent(Component component)
        {
            GroupItemBase groupItem = null;

            groupItem = new GroupItemBase();
            groupItem.Target = component as SoundSetItem;
            groupItem.TargetItemReference = component.Name;
            return groupItem;
        }

        /// <summary>
        /// "貼り付け"の実行
        /// </summary>
        protected bool PasteToGroupItemByDragDrop(Component parentComponent, Component prevSiblingComponent)
        {
            Component[] components = ClipboardService.Paste
                (ComponentService, TargetDocument, _CanPasteToGroupItemTypes,
                  parentComponent, prevSiblingComponent, null, ReplaceComponent,
                  false);
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        protected bool CanPasteToGroupItemByDragDrop()
        {
            return ClipboardService.CanPaste(_CanPasteToGroupItemTypes);
        }

        ///--------------------------------
        /// <summary>
        /// "セルのコピー"が実行可能なのか調べる
        /// </summary>
        protected bool CanCopyCell()
        {
            if (ActiveListCtrl == null)
            {
                return false;
            }

            return ClipboardService.CanCopyCell(ActiveListCtrl.ItemsName,
                                                 ActiveListCtrl.GetItemSelecteds());
        }

        ///--------------------------------
        /// <summary>
        /// "セルのコピー"の実行
        /// </summary>
        protected bool CopyCell()
        {
            if (ActiveListCtrl == null)
            {
                return false;
            }

            ClipboardService.CopyCell(ActiveListCtrl.ItemsName,
                                       ActiveListCtrl.GetItemSelecteds());
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// "項目を選択して貼り付け"が実行可能なのか調べる
        /// </summary>
        protected bool CanPasteSpecial()
        {
            return ClipboardService.CanPasteSpecial(TargetComponents);
        }

        ///--------------------------------
        /// <summary>
        /// "項目を選択して貼り付け"の実行
        /// </summary>
        protected bool PasteSpecial()
        {
            ComponentHeaderAdapter adapter = null;
            PasteColumnChooseDialog dialog = null;

            if (ActiveListCtrl == null)
            {
                return false;
            }

            Component[] components = TargetComponents;

            adapter = ActiveListCtrl.HeaderSource as ComponentHeaderAdapter;
            ClipboardService.HeaderItem[] headerItems =
                ClipboardService.GetHeaderItems(adapter, ActiveListCtrl.GetItemSelecteds());
            if (headerItems == null)
            {
                return false;
            }

            dialog = new PasteColumnChooseDialog(headerItems);
            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return false;
            }

            return ClipboardService.PasteSpecial(TargetDocument.OperationHistory,
                                                  dialog.CheckedNames,
                                                  ActiveListCtrl.GetItemSelecteds());
        }

        /// <summary>
        /// "一括編集"が実行可能なのか調べる
        /// </summary>
        protected bool CanBatchEdit()
        {
            if (ActiveListCtrl == null)
            {
                return false;
            }

            ListItemSelectedDictionary selectedItems = ActiveListCtrl.GetItemSelecteds();
            ComponentListItem[] items = null;

            foreach (ListItemSelectedState state in selectedItems.Values)
            {
                if (state.SubSelected.Count > 0)
                {
                    items = selectedItems.Keys
                        .Take(1).Cast<ComponentListItem>().ToArray();

                    if (items.Length <= 0 ||
                        items[0].Target is GroupItemBase)
                    {
                        return false;
                    }
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// "一括編集"の実行
        /// </summary>
        protected bool BatchEdit()
        {
            BatchEditDialog dialog = null;

            dialog = new BatchEditDialog();
            if (dialog.ShowDialog() != DialogResult.OK)
            {
                return false;
            }

            if (dialog.ValueText == String.Empty)
            {
                return false;
            }

            //
            OperationHistory operationHistory = TargetDocument.OperationHistory;
            Operation[] operations = null;
            ListItemSelectedDictionary selectedItems = ActiveListCtrl.GetItemSelecteds();
            ListItemSelectedState state = null;

            operationHistory.BeginTransaction();

            foreach (ComponentListItem item in selectedItems.Keys)
            {
                state = selectedItems[item];

                foreach (string name in state.SubSelected)
                {
                    if ((operations = BatchEditUtility.SetEffect
                         (item, name, dialog.EffectKind, dialog.ValueText,
                           ConvertText, ConvertValue, ValidateValueForBatchEdit)) != null)
                    {
                        foreach (Operation operation in operations)
                        {
                            operationHistory.AddOperation(operation);
                        }
                    }
                }
            }

            operationHistory.EndTransaction();
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual string ConvertText(string name, string text)
        {
            return text;
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual IParameterValue ConvertValue(string name, string text, IParameterValue value)
        {
            return value;
        }

        /// <summary>
        /// 一括編集で最終的な値が決まった後に呼び出されます。
        /// 編集を許容するのであれば true, キャンセルするのであれば falseを返す必要があります。
        /// </summary>
        protected virtual bool ValidateValueForBatchEdit(Component component, string name, string text, object value)
        {
            if (component is StreamSoundBase &&
                name == ProjectParameterNames.WaveEncoding)
            {
                Debug.Assert(component.Children.Count > 0, "StreamSoundBase not has children.");
                StreamSoundTrackBase track = component.Children[0] as StreamSoundTrackBase;

                if (AACUtil.IsAACFile(track.FilePath) == true || FileUtil.IsOpusFile(track.FilePath) == true)
                {
                    return false;
                }
            }

            return true;
        }
    }
}
