﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation;
    using NintendoWare.SoundFoundation.Conversion;
    using NintendoWare.SoundFoundation.Core.Collections;
    using NintendoWare.SoundFoundation.Core.ComInterop;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Logs;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using MsgRsrc = NintendoWare.SoundFoundation.Resources;
    using Schemas = NintendoWare.SoundMaker.Framework.Configurations.Schemas;

    /// <summary>
    /// WindowsForms ベースの UI 機能を提供します。
    /// </summary>
    public class FormsUIService : UIServiceBase
    {
        private MainWindow _mainWindow;
        private ConvertDialog _convertDialog;

        private ITaskbarList3 taskbarList3;

        private ListHeaderAdapterDictionary _defaultHeaderAdapters = new ListHeaderAdapterDictionary();
        private IDictionary<string, XmlList> _xmlListDefaults;

        /// <summary>
        /// メインウィンドウを取得します。
        /// </summary>
        public MainWindow MainWindow
        {
            get { return _mainWindow; }
        }

        /// <summary>
        /// メッセージボックスをTopMostで表示するか？のフラグ
        /// </summary>
        public bool TopMostMessageBox { get; set; }

        /// <summary>
        /// デフォルトリストヘッダのヘッダアダプタディクショナリを取得します。
        /// </summary>
        public IReadOnlyDictionary<string, ListHeaderAdapterBase> DefaultHeaderAdapters
        {
            get { return _defaultHeaderAdapters; }
        }

        protected SoundProjectConfiguration ProjectConfiguration
        {
            get { return FormsApplication.Instance.ProjectConfiguration; }
        }

        protected XmlSoundProjectDocumentView ProjectDocumentConfiguration
        {
            get
            {
                if (null == ProjectConfiguration) { return null; }
                if (!ProjectConfiguration.DocumentViews.
                      ContainsKey(ProjectService.ProjectDocument.Resource.Key)) { return null; }

                return ProjectConfiguration.DocumentViews[ProjectService.ProjectDocument.Resource.Key]
                            as XmlSoundProjectDocumentView;
            }
        }

        protected SoundProjectService ProjectService
        {
            get { return FormsApplication.Instance.ProjectService; }
        }

        /// <summary>
        /// UIサービスを初期化します。
        /// </summary>
        public override void Initialize()
        {
#if DEBUG
            // WindowsForm デザイナ対応
            if (null == ApplicationBase.Instance.ProjectConfiguration) { return; }
#endif

            // プロジェクトが読み込まれるまでの間、デフォルト設定を読み込んでおく。
            _xmlListDefaults = ProjectConfiguration.LoadListsDefault().ExportDictionary();

            //ListConfigurationApplier applier = new ListConfigurationApplier( _defaultHeaderAdapters );
            ListConfigurationApplier applier = CreateListConfigurationApplier(_defaultHeaderAdapters);

            foreach (Schemas.XmlList xmlList in _xmlListDefaults.Values)
            {
                applier.Apply(xmlList);
            }


            _mainWindow = CreateMainWindow();

            ApplicationBase.Instance.ProjectService.Opened += OnProjectOpended;
            ApplicationBase.Instance.ProjectConfiguration.Saving += OnProjectConfigurationSaving;
            ApplicationBase.Instance.ConvertService.BeginConvert += OnBeginConvert;
            ApplicationBase.Instance.ConvertService.EndConvert += OnEndConvert;
            ApplicationBase.Instance.ConvertService.OutputLine += OnConvertOutputLine;
            ApplicationBase.Instance.DocumentImportService.Importing += OnDocumentImporting;
            ApplicationBase.Instance.DocumentImportService.Imported += OnDocumentImported;
            ApplicationBase.Instance.DocumentImportService.ImportError += OnDocumentImportError;
        }

        /// <summary>
        /// メッセージボックスを表示します。
        /// </summary>
        /// <param name="text">テキスト。</param>
        /// <param name="caption">キャプション。</param>
        /// <param name="button">ボタンの種類。</param>
        /// <param name="image">アイコン。</param>
        /// <param name="defaultResult">デフォルトの結果。</param>
        /// <returns>結果。</returns>
        public override AppMessageBoxResult ShowMessageBox(string text, string caption, AppMessageBoxButton button,
                                                            AppMessageBoxImage image, AppMessageBoxResult defaultResult)
        {
            if (this.TopMostMessageBox != false)
            {
                Form dummyForm = new Form();
                dummyForm.Owner = this._mainWindow;
                dummyForm.TopMost = true;

                try
                {
                    return FormsMessageBox.Show(dummyForm, text, caption, button, image, defaultResult);
                }

                finally
                {
                    dummyForm.TopMost = false;
                    dummyForm.Dispose();
                }
            }
            else
            {
                return FormsMessageBox.Show(this._mainWindow, text, caption, button, image, defaultResult);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public virtual ListConfigurationApplier CreateListConfigurationApplier(ListHeaderAdapterDictionary headerAdapters)
        {
            return new ListConfigurationApplier(headerAdapters);
        }

        /// <summary>
        /// ドキュメントをインポートする前に発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元を指定します。</param>
        /// <param name="e">イベントデータを指定します。</param>
        protected virtual void OnDocumentImporting(object sender, DocumentImportEventArgs e)
        {
        }

        /// <summary>
        /// メインウィンドウを作成します。
        /// </summary>
        /// <returns>作成されたメインウィンドウ。</returns>
        protected virtual MainWindow CreateMainWindow()
        {
            return new MainWindow();
        }

        /// <summary>
        /// プロジェクト設定を適用します。
        /// </summary>
        private void ApplyProjectConfiguration()
        {
            if (null == ProjectDocumentConfiguration) { return; }
            if (null == ProjectDocumentConfiguration.ListDefaults) { return; }

            _xmlListDefaults = ProjectDocumentConfiguration.ListDefaults.ExportDictionary();

            //ListConfigurationApplier applier = new ListConfigurationApplier( _defaultHeaderAdapters );
            ListConfigurationApplier applier = CreateListConfigurationApplier(_defaultHeaderAdapters);

            foreach (Schemas.XmlList xmlList in _xmlListDefaults.Values)
            {
                applier.Apply(xmlList);
            }

            ApplyUriConfigurations();
        }

        /// <summary>
        /// プロジェクト設定を抽出します。
        /// </summary>
        private void ExtractProjectConfiguration()
        {
            if (null == ProjectDocumentConfiguration) { return; }

            //ListConfigurationApplier applier = new ListConfigurationApplier( _defaultHeaderAdapters );
            ListConfigurationApplier applier = CreateListConfigurationApplier(_defaultHeaderAdapters);

            foreach (Schemas.XmlList xmlList in _xmlListDefaults.Values)
            {
                applier.Extract(xmlList);
            }

            ProjectDocumentConfiguration.ListDefaults.List = _xmlListDefaults.Values.ToArray();

            ExtractUriConfigurations();
        }

        private void ApplyUriConfigurations()
        {
            if (ProjectConfiguration.Uris.ContainsKey(UriNames.SoundProjectInitialFilePath))
            {
                ProjectFileQuester.InitialFilePath = ProjectConfiguration.Uris[UriNames.SoundProjectInitialFilePath].Value;
            }

            if (ProjectConfiguration.Uris.ContainsKey(UriNames.SoundSetInitialFilePath))
            {
                SoundSetFileQuester.InitialFilePath = ProjectConfiguration.Uris[UriNames.SoundSetInitialFilePath].Value;
            }

            if (ProjectConfiguration.Uris.ContainsKey(UriNames.BankInitialFilePath))
            {
                BankFileQuester.InitialFilePath = ProjectConfiguration.Uris[UriNames.BankInitialFilePath].Value;
            }

            if (ProjectConfiguration.Uris.ContainsKey(UriNames.SequenceInitialFilePath))
            {
                SeqFileQuester.InitialFilePath = ProjectConfiguration.Uris[UriNames.SequenceInitialFilePath].Value;
            }

            if (ProjectConfiguration.Uris.ContainsKey(UriNames.WaveInitialFilePath))
            {
                WaveFileQuester.InitialFilePath = ProjectConfiguration.Uris[UriNames.WaveInitialFilePath].Value;
            }
        }

        private void ExtractUriConfigurations()
        {
            if (!ProjectConfiguration.Uris.ContainsKey(UriNames.SoundProjectInitialFilePath))
            {
                ProjectConfiguration.Uris.Add(UriNames.SoundProjectInitialFilePath,
                                              new XmlUri()
                                              {
                                                  Name = UriNames.SoundProjectInitialFilePath,
                                                  Value = ProjectFileQuester.InitialFilePath,
                                              });
            }
            else
            {
                ProjectConfiguration.Uris[UriNames.SoundProjectInitialFilePath].Value = ProjectFileQuester.InitialFilePath;
            }

            if (!ProjectConfiguration.Uris.ContainsKey(UriNames.SoundSetInitialFilePath))
            {
                ProjectConfiguration.Uris.Add(UriNames.SoundSetInitialFilePath,
                                              new XmlUri()
                                              {
                                                  Name = UriNames.SoundSetInitialFilePath,
                                                  Value = SoundSetFileQuester.InitialFilePath,
                                              });
            }
            else
            {
                ProjectConfiguration.Uris[UriNames.SoundSetInitialFilePath].Value = SoundSetFileQuester.InitialFilePath;
            }

            if (!ProjectConfiguration.Uris.ContainsKey(UriNames.BankInitialFilePath))
            {
                ProjectConfiguration.Uris.Add(UriNames.BankInitialFilePath,
                                              new XmlUri()
                                              {
                                                  Name = UriNames.BankInitialFilePath,
                                                  Value = BankFileQuester.InitialFilePath,
                                              });
            }
            else
            {
                ProjectConfiguration.Uris[UriNames.BankInitialFilePath].Value = BankFileQuester.InitialFilePath;
            }

            if (!ProjectConfiguration.Uris.ContainsKey(UriNames.SequenceInitialFilePath))
            {
                ProjectConfiguration.Uris.Add(UriNames.SequenceInitialFilePath,
                                              new XmlUri()
                                              {
                                                  Name = UriNames.SequenceInitialFilePath,
                                                  Value = SeqFileQuester.InitialFilePath,
                                              });
            }
            else
            {
                ProjectConfiguration.Uris[UriNames.SequenceInitialFilePath].Value = SeqFileQuester.InitialFilePath;
            }

            if (!ProjectConfiguration.Uris.ContainsKey(UriNames.WaveInitialFilePath))
            {
                ProjectConfiguration.Uris.Add(UriNames.WaveInitialFilePath,
                                              new XmlUri()
                                              {
                                                  Name = UriNames.WaveInitialFilePath,
                                                  Value = WaveFileQuester.InitialFilePath,
                                              });
            }
            else
            {
                ProjectConfiguration.Uris[UriNames.WaveInitialFilePath].Value = WaveFileQuester.InitialFilePath;
            }
        }

        /// <summary>
        /// コンバートダイアログをモーダル表示します。
        /// </summary>
        private void ShowConvertDialog()
        {
            _convertDialog.ShowDialog(_mainWindow);
            _convertDialog = null;

            ApplicationBase.Instance.EnableCommandKeyProcess();
        }

        /// <summary>
        /// プロジェクトが開かれると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントデータ。</param>
        private void OnProjectOpended(object sender, EventArgs e)
        {
            ApplyProjectConfiguration();
        }

        /// <summary>
        /// プロジェクト設定が保存される前に発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントデータ。</param>
        private void OnProjectConfigurationSaving(object sender, EventArgs e)
        {
            ExtractProjectConfiguration();
        }

        /// <summary>
        /// コンバートが開始されると発生します。
        /// </summary>
        /// <param name="sender">コンバートサービス。</param>
        /// <param name="e">サウンドプロジェクトコンバートイベントデータ。</param>
        private void OnBeginConvert(object sender, SoundProjectConvertEventArgs e)
        {
            if (e.IsPartsConvert)
            {
                return;
            }

            var taskbarListType = Type.GetTypeFromCLSID(new Guid("56FDF344-FD6D-11d0-958A-006097C9A090"));
            this.taskbarList3 = Activator.CreateInstance(taskbarListType) as ITaskbarList3;

            _convertDialog = new ConvertDialog(ApplicationBase.Instance.ConvertService);

            ApplicationBase.Instance.DisableCommandKeyProcess();
            _mainWindow.BeginInvoke(new MethodInvoker(ShowConvertDialog));
        }

        /// <summary>
        /// コンバートが終了すると発生します。
        /// </summary>
        /// <param name="sender">コンバートサービス。</param>
        /// <param name="e">サウンドプロジェクトコンバートイベントデータ。</param>
        private void OnEndConvert(object sender, SoundProjectConvertEventArgs e)
        {
            if (this.taskbarList3 != null)
            {
                this.taskbarList3.SetProgressState(this._mainWindow.Handle, TBPFLAG.TBPF_NOPROGRESS);
            }

            this.taskbarList3 = null;
        }

        private void OnConvertOutputLine(object sender, OutputLineEventArgs e)
        {
            if (ApplicationBase.Instance.ConvertService.ProgressMax == 0)
            {
                return;
            }

            if (this.taskbarList3 != null)
            {
                this.taskbarList3.SetProgressState(this._mainWindow.Handle, TBPFLAG.TBPF_NORMAL);

                this.taskbarList3.SetProgressValue(
                    this._mainWindow.Handle,
                    (ulong)ApplicationBase.Instance.ConvertService.ProgressCurrent,
                    (ulong)ApplicationBase.Instance.ConvertService.ProgressMax);
            }
        }

        /// <summary>
        /// ドキュメントをインポートした際に発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元を指定します。</param>
        /// <param name="e">イベントデータを指定します。</param>
        private void OnDocumentImported(object sender, DocumentImportEventArgs e)
        {
            bool fileExists = false;

            StringBuilder confirmMessage = new StringBuilder();
            StringBuilder doneMessage = new StringBuilder();

            foreach (Document document in e.Documents)
            {
                doneMessage.AppendLine(document.Resource.Key);

                if (File.Exists(document.Resource.Key))
                {
                    fileExists = true;
                    confirmMessage.AppendLine(document.Resource.Key);
                }
            }

            TextDisplayMessageBox messageBox =
                new TextDisplayMessageBox(
                                          Resources.MessageResource.Message_ConfirmOverwriteFiles,
                                          confirmMessage.ToString(),
                                          TextDisplayMessageBoxStyle.YesAndNoButton
                                          );

            // 上書き確認
            if (fileExists && messageBox.ShowDialog() != DialogResult.Yes)
            {
                e.Cancel = true;
                return;
            }

            // サウンドセットファイルがないと参照エラーになる仕様なので、
            // サウンドプロジェクトも含め、ここで保存します。
            foreach (Document savingDocument in e.Documents)
            {
                if (savingDocument is TextSequenceSoundDocument)
                {
                    new TextSequenceSoundDocumentWriter(Platforms.Any.TextSequenceSoundDocument)
                        .Write(savingDocument.Resource, savingDocument, ApplicationBase.Instance.DocumentService.Documents);
                }
                else
                {
                    ApplicationBase.Instance.DocumentService.SaveDocument(savingDocument);
                }
            }

            messageBox =
                new TextDisplayMessageBox(
                                          Resources.MessageResource.Message_ImportedFilesSaved,
                                          doneMessage.ToString(),
                                          TextDisplayMessageBoxStyle.OKAndLogButton
                                          );
            messageBox.LogFilePath = ImportLog.LogFilePath;
            ImportLog.OpenAppend(ImportLog.LogFilePath);
            ImportLog.WriteLine(Resources.MessageResource.Message_ImportedFilesSaved);
            ImportLog.WriteLine(doneMessage.ToString());
            ImportLog.WriteLine(MsgRsrc.MessageResource.Message_ImportEnd);
            ImportLog.Close();

            messageBox.ShowDialog();
        }

        /// <summary>
        /// ドキュメントをインポート中にエラーになった場合に発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元を指定します。</param>
        /// <param name="e">イベントデータを指定します。</param>
        private void OnDocumentImportError(object sender, DocumentImportEventArgs e)
        {
            StringBuilder doneMessage = new StringBuilder();

            foreach (Document document in e.Documents)
            {
                doneMessage.AppendLine(document.Resource.Key);
            }

            TextDisplayMessageBox messageBox =
                new TextDisplayMessageBox(
                                          Resources.MessageResource.Message_UnknownError,
                                          doneMessage.ToString(),
                                          TextDisplayMessageBoxStyle.OKAndLogButton
                                          );
            messageBox.LogFilePath = ImportLog.LogFilePath;
            messageBox.ShowDialog();
        }
    }
}
