﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Drawing;
    using System.IO;
    using System.Linq;
    using System.Reflection;
    using System.Runtime.InteropServices;
    using System.Text;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Conversion;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Core.Reflection;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Core.Win32;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.CommandBars;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Utilities;
    using NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers;
    using NintendoWare.SoundMaker.Windows.Forms.CommandHandlers;
    using ToolDevelopmentKit;

    /// <summary>
    /// WindowsForms ベースのメインウィンドウです。
    /// </summary>
    [CommandKeyProcessable]
    public partial class MainWindow : Form, ICommandTarget, IQueryCommandParameter
    {
        ///
        public enum ShowComponentMode
        {
            Auto,
            MultipleList,
            SingleList,
        }

        private const string WindowName = "MainWindow";

        // コマンドバー管理
        private ToolStripAdapter _mainMenuAdapter;
        private ToolStripAdapter _standardToolAdapter;
        private ToolStripAdapter _toolWindowToolAdapter;
        private ToolStripAdapter _tabMenuAdapter;
        private CommandBindingCollection _commandBindings = new CommandBindingCollection();

        // アクティブ管理
        private Control _LastActiveControl = null;
        private Control _LastActivePanel = null;

        // ドッキングコンテナとツールページ
        private ToolPageDictionary _toolPages;
        private DockingToolContainerDictionary _dockingToolContainers;
        private int _leftSplitterPosition = -1;
        private int _rightSplitterPosition = -1;
        private int _bottomSplitterPosition = -1;

        // タブメニュー一時パラメータ
        private TabCtrl _tabMenuTargetCtrl;
        private int _tabMenuTargetIndex = -1;

        // ステータスバー管理
        private string _statusSelectionDescription = string.Empty;

        // 状態
        private bool _loaded = false;
        private bool _isTabClosing = false;
        private bool projectOpened = false;
        private Component oldSelectedComponent = null;

        //
        private FindWindow _FindWindow = null;

        ///
        private IntPtr _NextHandle = IntPtr.Zero;

        ///
        private ProgressDialog checkAndWaitDialog;
        private bool errorCheckAndWait = false;
        private long checkAndWaitTime = 30; // 30秒

        ///
        private EventHandler WindowMinimizing;
        private EventHandler WindowMinimized;
        private EventHandler WindowMaximizing;
        private EventHandler WindowMaximized;
        private EventHandler WindowRestored;
        private EventHandler WindowRestoring;

        private int rightSplitterDistance = 0;
        private int rightSplitterDistanceMaximize = 0;
        private int bottomSplitterDistance = 0;
        private int bottomSplitterDistanceMaximize = 0;

        private int ReloadRetryCountMax = 3;
        private bool _enabledFileChanged = true;

        ///--------------------------------
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public MainWindow()
        {
#if DEBUG
            // WindowsForm デザイナ対応
            FormsApplicationMock.InitializeForFormDesigner();
#endif

            InitializeComponent();

            // パネル間のスプリッタの幅を指定します。
            // デザイナーだと MainWindow を継承する機種依存部分で、
            // 一部反映されないのでここで設定します。
            int width = 6;
            this._rightSplitContainer.SplitterWidth = width;
            this._leftSplitContainer.SplitterWidth = width;
            this._bottomSplitContainer.SplitterWidth = width;
            this._editSplitContainer.SplitterWidth = width;

            Application.FileChanged = this.OnFileChanged;
            SaveHelper.Saving += this.OnSaving;
            SaveHelper.Saved += this.OnSaved;

            this.IsActive = false;

            Icon = Application.Traits.ApplicationIcon;

            if (Application.ConvertService != null)
            {
                Application.ConvertService.EndConvert += OnEndConvert;
            }

            SetCurrentTabCtrl(_TabCtrlMain);

            _editSplitContainer.Panel2Collapsed = true;
            _editSplitContainer.IsSplitterFixed = true;

            _leftDockingToolContainer.Minimizing += OnLeftDockingToolContainerMinimizing;
            _leftDockingToolContainer.SelectedTabChanged += OnLeftDockingToolContainerSelectedTabPageChanged;

            this.WindowMinimizing += OnWindowMinimizing;
            this.WindowMinimized += OnWindowMinimized;
            this.WindowMaximizing += OnWindowMaximizing;
            this.WindowMaximized += OnWindowMaximized;
            this.WindowRestoring += OnWindowRestoring;
            this.WindowRestored += OnWindowRestored;

            this.toolStripSplitButton_PresetListColumns.ButtonClick += OnClickPresetListColumnsOnSplitButton;
            CloseProjectHandler.ClosedDelegate = this.CheckAndWaitMonitoringTargetsIsZero;
        }

        /// <summary>
        /// プロジェクトが再読み込みされると発生します。
        /// </summary>
        public event EventHandler ProjectReloaded;

        /// <summary>
        /// ツールページ一覧を取得します。
        /// </summary>
        public ToolPageDictionary ToolPages
        {
            get { return _toolPages; }
        }

        /// <summary>
        /// ドッキングコンテナ一覧を取得します。
        /// </summary>
        public DockingToolContainerDictionary DockingToolContainers
        {
            get { return _dockingToolContainers; }
        }

        /// <summary>
        /// プロジェクトパネルを取得します。
        /// </summary>
        public ProjectPanel ProjectPanel
        {
            get { return (ToolPages[ProjectTreePanel.PageName] as ProjectTreePanel).ProjectPanel; }
        }

        /// <summary>
        /// ブックマークパネルを取得します。
        /// </summary>
        public BookmarkPanel BookmarkPanel
        {
            get { return (ToolPages[BookmarkTreePanel.PageName] as BookmarkTreePanel).BookmarkPanel; }
        }

        /// <summary>
        /// ブックマークパネルを取得します。
        /// </summary>
        public HistoryPanel HistoryPanel
        {
            get { return (ToolPages[HistoryListPanel.PageName] as HistoryListPanel).HistoryPanel; }
        }

        /// <summary>
        /// ステータスに表示する選択アイテムの説明を取得または設定します。
        /// </summary>
        public string StatusSelectionDescription
        {
            get { return _statusSelectionDescription; }
            set
            {
                if (null == value) { throw new ArgumentNullException("value"); }
                if (value == _statusSelectionDescription) { return; }

                _statusSelectionDescription = value;

                UpdateStatusText(_statusSelectionDescription);
            }
        }

        /// <summary>
        /// アクティブタブページを取得します。
        /// </summary>
        public CommonTabPage ActivePage
        {
            get
            {
                if (null == _CurrentTabCtrl) { return null; }
                return _CurrentTabCtrl.SelectedTab as CommonTabPage;
            }
        }

        /// <summary>
        /// SplitButton に表示するリスト項目設定
        /// </summary>
        public string[] PresetListColumnsPresetNamesOnSplitButton
        {
            set
            {
                this.toolStripSplitButton_PresetListColumns.DropDownItems.Clear();
                this.toolStripSplitButton_PresetListColumns.Enabled = false;
                if (value == null) return;
                else if (value.Length > 0)
                {
                    foreach (string name in value)
                    {
                        ToolStripMenuItem item = new ToolStripMenuItem();
                        item.Name = name;
                        item.Text = name;
                        item.Click += OnClickPresetListColumnsOnSplitButton;
                        this.toolStripSplitButton_PresetListColumns.DropDownItems.Add(item);
                    }
                    this.toolStripSplitButton_PresetListColumns.Enabled = true;
                }
            }
        }

        public string PresetListColumnsPresetCurrentNameOnSplitButton
        {
            set
            {
                this.toolStripSplitButton_PresetListColumns.Text = "----";
                if (value != null && value.Length > 0)
                {
                    ToolStripItem[] items = this.toolStripSplitButton_PresetListColumns.DropDownItems.Find(value, false);
                    if (0 < items.Length)
                    {
                        this.toolStripSplitButton_PresetListColumns.Text = value;
                    }
                    else if (0 < this.toolStripSplitButton_PresetListColumns.DropDownItems.Count)
                    {
                        this.toolStripSplitButton_PresetListColumns.Text = this.toolStripSplitButton_PresetListColumns.DropDownItems[0].Text;
                    }
                }
            }
        }

        /// <summary>
        /// 検索パネルを取得します。(タブ版)
        /// </summary>
        public FindResultPanel2[] FindResultPanels
        {
            get
            {
                FindResultPanel2[] panels = this._TabCtrlMain.TabPages
                    .Concat(this._TabCtrlSub.TabPages)
                    .Cast<CommonTabPage>()
                    .Select(t => t.Panel)
                    .Where(p => p is FindResultPanel2)
                    .Cast<FindResultPanel2>()
                    .ToArray();
                return panels;
            }
        }

        public IEnumerable<SoundSetPanel> SoundSetPanels
        {
            get
            {
                return this._TabCtrlMain.TabPages
                    .Concat(this._TabCtrlSub.TabPages)
                    .OfType<CommonTabPage>()
                    .Select(t => t.Panel)
                    .Where(p => p is SoundSetPanel)
                    .OfType<SoundSetPanel>();
            }
        }

        ///--------------------------------
        /// <summary>
        /// アプリケーションクラスを取得します。
        /// </summary>
        protected FormsApplication Application
        {
            get { return FormsApplication.Instance; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected SoundProjectConfiguration ProjectConfiguration
        {
            get { return FormsApplication.Instance.ProjectConfiguration; }
        }

        /// <summary>
        /// UIサービスを取得します。
        /// </summary>
        protected FormsUIService UIService
        {
            get { return Application.UIService; }
        }

        /// <summary>
        ///
        /// </summary>
        protected SoundProjectService ProjectService
        {
            get { return Application.ProjectService; }
        }

        /// <summary>
        ///
        /// </summary>
        protected bool IsActive { get; private set; }

        /// <summary>
        /// オプションを取得します。
        /// </summary>
        protected XmlOptions Options
        {
            get { return Application.AppConfiguration.Options; }
        }

        /// <summary>
        ///
        /// </summary>
        protected TabCtrl TabMenuTargetCtrl
        {
            get
            {
                return this._tabMenuTargetCtrl;
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected int TabMenuTargetIndex
        {
            get
            {
                return this._tabMenuTargetIndex;
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected ToolStripAdapter MainMenuAdapter
        {
            get { return _mainMenuAdapter; }
            set { _mainMenuAdapter = value; }
        }

        /// <summary>
        /// 全てのサウンドセットドキュメントを取得します。
        /// </summary>
        private SoundSetDocument[] SoundSetDocuments
        {
            get
            {
                return Application.DocumentService.Documents
                    .Where(d => d is SoundSetDocument).Cast<SoundSetDocument>()
                    .ToArray();
            }
        }

        /// <summary>
        /// 全てのバンクドキュメントを取得します。
        /// </summary>
        private BankDocument[] BankDocuments
        {
            get
            {
                return Application.DocumentService.Documents
                    .Where(d => d is BankDocument).Cast<BankDocument>()
                    .ToArray();
            }
        }

        private int LeftSplitterPosition
        {
            get { return _leftSplitterPosition; }
            set
            {
                try
                {
                    _leftSplitContainer.SplitterDistance = value;
                    _leftSplitterPosition = value;
                }
                catch
                {
                    _leftSplitterPosition = _leftSplitContainer.SplitterDistance;
                }
            }
        }

        private int RightSplitterPosition
        {
            get { return _rightSplitterPosition; }
            set
            {
                try
                {
                    _rightSplitContainer.SplitterDistance = _rightSplitContainer.Width - value;
                    _rightSplitterPosition = value;
                }
                catch
                {
                    _rightSplitterPosition = _rightSplitContainer.Width - _rightSplitContainer.SplitterDistance;
                }
            }
        }

        private int BottomSplitterPosition
        {
            get { return _bottomSplitterPosition; }
            set
            {
                try
                {
                    _bottomSplitContainer.SplitterDistance = _bottomSplitContainer.Height - value;
                    _bottomSplitterPosition = value;
                }
                catch
                {
                    _bottomSplitterPosition = _bottomSplitContainer.Height - _bottomSplitContainer.SplitterDistance;
                }
            }
        }

        /// <summary>
        /// コマンドの対象処理プロジェクトサービスを取得します。
        /// </summary>
        private ComponentService TargetComponentService
        {
            get { return FormsApplication.Instance.ProjectService; }
        }

        /// <summary>
        /// コマンドの処理対象ドキュメントを取得します。
        /// </summary>
        private SoundDocument TargetDocument
        {
            get { return FormsApplication.Instance.ProjectService.ProjectDocument; }
        }

        /// <summary>
        /// コマンドの処理対象コンポーネントを取得します。
        /// </summary>
        private Component[] TargetComponents
        {
            get
            {
                if (null == FormsApplication.Instance.ProjectService.ProjectDocument.Project)
                {
                    return new Component[0];
                }
                return new Component[] { FormsApplication.Instance.ProjectService.ProjectDocument.Project };
            }
        }

        /// <summary>
        /// 指定タブの再描画を行いいます。
        /// </summary>
        private void RedrawPanel(CommonTabPage page)
        {
            if (page == null)
            {
                return;
            }

            CommonPanel panel = page.Panel as CommonPanel;
            if (panel != null)
            {
                panel.RedrawPanel();
            }
        }

        /// <summary>
        /// パネルの状態はそのままで再描画だけを行います。
        /// </summary>
        public void RedrawPanels()
        {
            RedrawPanel(_TabCtrlMain.SelectedTab as CommonTabPage);
            RedrawPanel(_TabCtrlSub.SelectedTab as CommonTabPage);
        }

        /// <summary>
        /// 表示されているすべてのパネルの状態を更新し再描画します。
        /// </summary>
        public void RefreshPanels()
        {
            this.DisplayedListPanels.ForEach(p => p.RefreshPanel());
        }

        /// <summary>
        /// カレントタブへのサウンドセットタブページの検索
        /// </summary>
        public CommonTabPage FindPage(SoundSet soundSet)
        {
            return FindOpenedSoundSetTabPage(soundSet, null);
        }

        ///--------------------------------
        /// <summary>
        /// カレントタブへのバンクタブページの検索
        /// </summary>
        public CommonTabPage FindPage(Bank bank)
        {
            return FindOpenedBankTabPage(bank, null);
        }

        ///
        public CommonTabPage FindPage(string filePath)
        {
            return FindOpenedBankTabPage(null, filePath);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public CommonTabPage FindPage(Component component)
        {
            if (component is SoundSet) { return FindPage(component as SoundSet); }
            if (component is Bank) { return FindPage(component as Bank); }
            return null;
        }

        ///--------------------------------
        /// <summary>
        /// カレントタブへのサウンドセットタブページの追加
        /// </summary>
        public CommonTabPage AddPage(SoundSet soundSet)
        {
            return AddPage(soundSet, true);
        }

        ///--------------------------------
        /// <summary>
        /// カレントタブへのサウンドセットタブページの追加
        /// </summary>
        public CommonTabPage AddPage(SoundSet soundSet, bool activate)
        {
            CommonTabPage page = null;

            if ((page = FindOpenedSoundSetTabPage(soundSet, null)) == null)
            {

                using (DocumentReference docRef = GetDocumentReferenceOfSoundSet(soundSet))
                {

                    page = new CommonTabPage();

                    SoundSetPanel panel = CreateSoundSetPanel(docRef);
                    panel.Parent = page;
                    panel.Dock = DockStyle.Fill;
                    panel.Show();

                    _CurrentTabCtrl.TabPages.Add(page);

                }

            }

            if (activate)
            {
                _CurrentTabCtrl.SelectedTab = page;
                page.Panel.ActivatePanel(true, true);
            }

            return page;
        }

        ///--------------------------------
        /// <summary>
        /// カレントタブへのバンクタブページの追加
        /// </summary>
        public CommonTabPage AddPage(SoundSetBankBase soundSetBank)
        {
            return this.OpenFile(soundSetBank.FilePath, soundSetBank);
        }

        ///
        public CommonTabPage AddPage(Bank bank, SoundSetBankBase soundSetBank, bool activate)
        {
            SoundSetBankBase targetSoundSetBank = soundSetBank;

            CommonTabPage page = null;

            if ((page = FindOpenedBankTabPage(bank, null)) == null)
            {
                if (null == targetSoundSetBank)
                {
                    targetSoundSetBank = (from SoundSetBankBase soundSetBankWork in ProjectService.SoundSetBanks
                                          where Application.BankServices.Contains(soundSetBankWork.FilePath)
                                          where Application.BankServices[soundSetBankWork.FilePath].Bank == bank
                                          select soundSetBankWork).FirstOrDefault();
                }

                using (DocumentReference docRef = GetDocumentReferenceOfBank(bank))
                {

                    page = new CommonTabPage();

                    BankPanel panel = CreateBankPanel(docRef.Document.Resource.Key);
                    panel.Parent = page;
                    panel.Dock = DockStyle.Fill;
                    panel.SoundSetBank = soundSetBank;

                    _CurrentTabCtrl.TabPages.Add(page);

                    page.ImageIndex = (int)TabCtrl.IconType.Bank;

                }

            }
            else
            {
                if (null != targetSoundSetBank)
                {
                    (page.Panel as BankPanel).SoundSetBank = soundSetBank;
                }
            }

            if (activate == true)
            {
                _CurrentTabCtrl.SelectedTab = page;
                page.Panel.ActivatePanel(true, true);
            }

            return page;
        }

        /// <summary>
        /// 検索タブを追加します。
        /// </summary>
        public void AddFindTabPage(FindArgs.TargetDocumentKinds targetDocumentKind)
        {
            AddFindTabPage(targetDocumentKind, (FindResultPanelInitializeHandler)null);
        }

        /// <summary>
        /// 検索タブを追加します。
        /// </summary>
        public void AddFindTabPage(FindArgs.TargetDocumentKinds targetDocumentKind, Component[] components)
        {
            AddFindTabPage(targetDocumentKind,
                            delegate (FindResultPanel2 panel)
                                {
                                    if (components != null)
                                    {
                                        panel.SetResult(components);
                                    }
                                });
        }

        /// <summary>
        /// 検索タブを追加します。
        /// </summary>
        public delegate void FindResultPanelInitializeHandler(FindResultPanel2 panel);

        public void AddFindTabPage(FindArgs.TargetDocumentKinds targetDocumentKind, FindResultPanelInitializeHandler initializer)
        {
            CommonTabPage page = new CommonTabPage();
            page.ImageIndex = (int)TabCtrl.IconType.Loupe;

            FindResultPanel2 panel = CreateFindResultPanel();
            panel.Parent = page;
            panel.Dock = DockStyle.Fill;
            panel.TargetDocumentKind = targetDocumentKind;

            if (initializer != null)
            {
                initializer(panel);
            }

            if (panel.TitleNumber <= 0) // initializer() の後でも TitleNumber が未定義なら
            {
                panel.TitleNumber = GetFindTabPageNumber(panel.TitleName);
            }

            panel.Show();

            this._CurrentTabCtrl.TabPages.Add(page);
            this._CurrentTabCtrl.SelectedTab = page;
            page.Panel.ActivatePanel(true, true);
        }

        /// <summary>
        /// 指定タブをアクティブにします。
        /// </summary>
        public void ActivatePage(CommonTabPage page)
        {
            if (page == _CurrentTabCtrl.SelectedTab) { return; }

            if (_CurrentTabCtrl.Contains(page) == true)
            {
                _CurrentTabCtrl.SelectedTab = page;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void ShowPageByComponent(Component component)
        {
            if (ShowSoundSetPageByComponent(component) == true)
            {
                return;
            }

            if (ShowBankPageByComponent(component) == true)
            {
                return;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void ShowPageWithSingleListByComponent(Component component, bool open)
        {
            if (ShowBankPageByComponent(component, open) == true)
            {
                return;
            }

            if (ShowSoundSetPageByComponent(component, ShowComponentMode.SingleList) == true)
            {
                return;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool ShowBankPageByFilePath(string filePath, string name)
        {
            CommonTabPage page = null;
            BankPanel bankPanel = null;
            CommonListCtrl listCtrl = null;
            IListItem listItem = null;

            if ((page = FindPage(filePath)) == null)
            {
                if (OpenFile(filePath) == false)
                {
                    return false;
                }
                page = FindPage(filePath);
            }

            ActivatePage(page);

            //
            bankPanel = page.Panel as BankPanel;

            //
            listCtrl = bankPanel.CurrentListCtrl;
            if ((listItem = listCtrl.GetItem(name)) != null)
            {
                listCtrl.ShowByItem(listItem);
            }

            bankPanel.ActivatePanel(false, true);
            Activate();

            return true;
        }

        /// <summary>
        /// コマンドのテキストを更新します。
        /// </summary>
        public void UpdateCommandTexts()
        {
            CommandTextReplacer.ReplaceColorCommentCommandTexts(ProjectService, this._mainMenuAdapter);
            CommandTextReplacer.ReplaceUserCommandCommandTexts(ProjectService, _toolWindowToolAdapter);
            CommandTextReplacer.ReplaceUserCommandCommandTexts(ProjectService, this._mainMenuAdapter);
        }

        /// <summary>
        /// コマンドバーを更新します。
        /// </summary>
        public void BuildCommandUI()
        {
            if (InvokeRequired == true)
            {
                Invoke((MethodInvoker)delegate
              {
                  if (null != _mainMenuAdapter)
                  {
                      _mainMenuAdapter.BuildUI();
                  }

                  if (null != _standardToolAdapter)
                  {
                      _standardToolAdapter.BuildUI();
                      UpdateConnectToolBar();
                  }

                  if (null != _toolWindowToolAdapter)
                  {
                      _toolWindowToolAdapter.BuildUI();
                  }
              });
            }
            else
            {
                if (null != _mainMenuAdapter)
                {
                    _mainMenuAdapter.BuildUI();
                }

                if (null != _standardToolAdapter)
                {
                    _standardToolAdapter.BuildUI();
                    UpdateConnectToolBar();
                }

                if (null != _toolWindowToolAdapter)
                {
                    _toolWindowToolAdapter.BuildUI();
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public void EnableAllCommands(bool enabled)
        {
            CommandStatus status = CommandStatus.Unsupported;

            if (enabled == false)
            {
                status = CommandStatus.Enabled;
            }

            _mainMenuAdapter.CommandStatusMask = status;
            _standardToolAdapter.CommandStatusMask = status;
            _toolWindowToolAdapter.CommandStatusMask = status;

            BuildCommandUI();
        }

        ///--------------------------------
        /// <summary>
        /// ツールページをアクティブにします。
        /// </summary>
        /// <param name="toolPage">アクティブにするツールページ</param>
        public bool ActivateToolPage(IToolWindowPage toolPage)
        {
            if (null == toolPage) { throw new ArgumentNullException("toolPage"); }
            if (null == toolPage.DockContainer) { return false; }

            toolPage.DockContainer.SelectPage(toolPage);
            return true;
        }

        /// <summary>
        /// プロジェクトパネルをアクティブにします。
        /// </summary>
        public void ActivateProjectPanelToolPage()
        {
            ActivateToolPage(ToolPages[ProjectTreePanel.PageName]);
        }

        /// <summary>
        /// ブックマークパネルをアクティブにします。
        /// </summary>
        public void ActivateBookmarkToolPage()
        {
            ActivateToolPage(ToolPages[BookmarkTreePanel.PageName]);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool CanDropFiles(DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop) == true)
            {
                return true;
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public bool DropFiles(string[] filePaths)
        {
            return DropFiles(filePaths, true);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        public SoundSetDocument GetSoundSetDocument(SoundSet soundSet)
        {
            using (DocumentReference docRef = GetDocumentReferenceOfSoundSet(soundSet))
            {
                return docRef != null ? docRef.Document as SoundSetDocument : null;
            }
        }

        /// <summary>
        /// プロジェクトファイルを開きます。
        /// </summary>
        public bool OpenProjectFile(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            SoundProjectService projectService = FormsApplication.Instance.ProjectService;
            if (projectService.ProjectDocument != null)
            {
                if (CloseProjectHandler.Execute() == false)
                {
                    return false;
                }
            }

            try
            {
                var parameter = new KeyValuePair<string, string>[] { new KeyValuePair<string, string>("TargetFilePath", filePath) };
                if (_mainMenuAdapter.ExecuteCommand(FileCommands.OpenProject, parameter) == false)
                {
                    return false;
                }
            }
            catch (FileNotSupportedException e)
            {
                UIService.ShowMessageBox(string.Format("{0}\n{1}", e.Message, e.FilePath));

                return false;
            }
            catch (FileNotFoundException e)
            {
                UIService.ShowMessageBox(string.Format(Resources.MessageResource.Message_TargetFileNotFound, e.FileName));

                return false;
            }
            catch (FileNotSupportedVersionException e)
            {
                UIService.ShowMessageBox(string.Format("{0}\n{1}", e.Message, e.FilePath));

                return false;
            }
            catch (Exception e)
            {
                string message = e.Message;

                if (string.IsNullOrEmpty(message) == true)
                {
                    message = Resources.MessageResource.Message_UnknownError;
                }
                UIService.ShowMessageBox(message);

                return false;
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        /// ファイルを開きます。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        public bool OpenFile(string filePath)
        {
            CommonTabPage page;

            return OpenFile(filePath, null, out page);
        }

        public CommonTabPage OpenFile(string filePath, SoundSetBankBase soundSetBank)
        {
            CommonTabPage page;

            this.OpenFile(filePath, soundSetBank, out page);

            return page;
        }

        private bool OpenFile(string filePath, SoundSetBankBase soundSetBank, out CommonTabPage page)
        {
            page = null;

            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            try
            {
                // ファイルのバージョンを確認します。
                bool oldFileAllForceSave = false;
                bool isReadOnly = false;
                List<string> newVersionFiles = new List<string>();
                List<string> oldVersionFiles = new List<string>();
                FileVersionUtility.GetDifferentVersionFilesWithBankFile(filePath, ref newVersionFiles, ref oldVersionFiles);
                if (0 < newVersionFiles.Count)
                {
                    string message = string.Format("{0}\n{1}", MessageResource.Message_UnsupportedFileVersion, filePath);
                    ApplicationBase.Instance.UIService.ShowMessageBox(message);

                    return false;
                }
                else if (0 < oldVersionFiles.Count)
                {
                    UpgradeFileDialog dialog = new UpgradeFileDialog();
                    switch (dialog.ShowDialog())
                    {
                        case DialogResult.Cancel:
                            return false;

                        case DialogResult.No: // 読み取り専用で開く
                            isReadOnly = true;
                            break;

                        case DialogResult.OK:
                            oldFileAllForceSave = true;
                            isReadOnly = false;
                            break;
                    }
                }

                using (BankServiceReference bankServiceRef = Application.BankServices.OpenItem(filePath))
                {
                    if (bankServiceRef == null)
                    {
                        return false;
                    }

                    // 旧バージョンを保存する指示があったなら保存する。
                    if (oldFileAllForceSave == true)
                    {
                        SaveHelper.SaveForce(new string[] { filePath });
                    }

                    // 読み取り専用の設定 or 解除
                    bankServiceRef.Target.BankDocument.IsReadOnly = isReadOnly;

                    page = this.AddPage(bankServiceRef.Target.Bank, soundSetBank, true);
                }
            }
            catch (FileNotFoundException exception)
            {
                Application.UIService.ShowMessageBox(
                    string.Format(Resources.MessageResource.Message_FileNotFound, exception.FileName));
                return false;
            }
            catch (FileNotSupportedException exception)
            {
                Application.UIService.ShowMessageBox(string.Format("{0}\n{1}",
                                                                   exception.Message,
                                                                   exception.FilePath));
                return false;
            }
            catch (FileNotSupportedVersionException exception)
            {
                Application.UIService.ShowMessageBox(string.Format("{0}\n{1}",
                                                                   exception.Message,
                                                                   exception.FilePath));
                return false;
            }
            catch (Exception e)
            {
                string message = e.Message;
                if (string.IsNullOrEmpty(message) == true)
                {
                    message = Resources.MessageResource.Message_UnknownError;
                }
                Application.UIService.ShowMessageBox(message);
                return false;
            }

            return true;
        }


        ///--------------------------------
        /// <summary>
        /// ファイルを開きます。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        public bool OpenFile(SoundSetBankBase bank)
        {
            if (null == bank) { throw new ArgumentNullException("bank"); }

            AddPage(bank);
            return true;
        }

        /// <summary>
        /// ステータステキストを更新します。
        /// </summary>
        public void UpdateStatusText(string text)
        {
            if (null == text) { throw new ArgumentNullException("text"); }
            _statusText.Text = text;
        }

        /// <summary>
        /// タイトルバーを更新します。
        /// </summary>
        public void UpdateTitle()
        {
            string text = Application.Traits.ApplicationAssembly.GetTitle();

            if (this.Options.Application.General.ShowProjectFilePath == false)
            {
                if (ProjectService.ProjectDocument != null)
                {
                    text = String.Format("{0} - ",
                                         ProjectService.ProjectDocument.Project.Name) + text;
                }
            }
            else
            {
                if (ProjectService != null)
                {
                    text = String.Format("{0} - ", ProjectService.ProjectFilePath) + text;
                }
            }

            Text = text;
        }

        /// <summary>
        /// ヘッダのユーザーパラメータカラムを更新します。
        /// </summary>
        public void ApplyUserParameterColumns()
        {
            ExecuteForDisplayedPanels
                (delegate (CommonListPanel panel)
                {
                    panel.ApplyUserParameterColumns();
                });
        }

        /// <summary>
        /// ヘッダのテキストを更新します。
        /// </summary>
        public void ApplyColumnsText()
        {
            ExecuteForDisplayedPanels
                (delegate (CommonListPanel panel)
                     {
                         panel.ApplyColumnsText();
                     });
        }

        /// <summary>
        /// オプション設定が変更されたことを通知します。
        /// </summary>
        public void ApplyOptionSettings()
        {
            ExecuteForDisplayedPanels
                (delegate (CommonListPanel panel)
                     {
                         panel.ApplyOptionSettings();
                     });
        }

        /// <summary>
        /// オプション設定が変更された時に呼ばれます。
        /// ApplyOptionSetting()と統合できるのでは？と思っています。(2013.1.24 aoyagi)
        /// </summary>
        public void UpdatedOptions()
        {
            ExecuteForDisplayedPanels
                (delegate (CommonListPanel panel)
                     {
                         panel.UpdatedOptions();
                     });

            //
            ApplyColorComment();
        }

        /// <summary>
        /// カラーコメントが変更されたことを通知します。
        /// </summary>
        public void ApplyColorComment()
        {
            ExecuteForDisplayedPanels
                (delegate (CommonListPanel panel)
                     {
                         panel.ApplySettings(CommonListPanel.SettingKinds.ColorComment);
                     });
        }

        /// <summary>
        ///
        /// </summary>
        public void ClearPresetListColumnsOnSplitButton()
        {
            this.PresetListColumnsPresetNamesOnSplitButton = null;
            this.PresetListColumnsPresetCurrentNameOnSplitButton = string.Empty;
        }

        /// <summary>
        ///
        /// </summary>
        public void UpdatePresetListColumnsOnSplitButton()
        {
            if (ActivePage != null && ActivePage.Panel is CommonListPanel)
            {
                CommonListPanel panel = ActivePage.Panel as CommonListPanel;
                panel.UpdatePresetListColumns();
            }
            else
            {
                ClearPresetListColumnsOnSplitButton();
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void RequestNotifyCommandExecuted(Command command)
        {
            NotifyCommandExecuted(command);
        }

        /// <summary>
        ///
        /// </summary>
        public void HistoryAdd(Component component)
        {
            if (component != null && component != this.oldSelectedComponent)
            {
                this.oldSelectedComponent = component;
                Application.HistoryService.Add(component);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void HistoryAdd(Component component, bool force)
        {
            if (component != null)
            {
                if (force == true)
                {
                    this.oldSelectedComponent = component;
                    Application.HistoryService.Add(component);
                }
                else if (component != this.oldSelectedComponent)
                {
                    this.oldSelectedComponent = component;
                    Application.HistoryService.Add(component);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        bool IQueryCommandParameter.ContainsParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponentService:
                case CommandParameterNames.TargetDocuments:
                case CommandParameterNames.TargetComponents:
                case CommandParameterNames.InsertParentComponent:
                    return true;
            }

            return false;
        }

        object IQueryCommandParameter.GetParameter(string parameterName)
        {
            switch (parameterName)
            {
                case CommandParameterNames.TargetComponentService:
                    return TargetComponentService;

                case CommandParameterNames.TargetDocuments:
                    if (null == TargetDocument) { return new SoundDocument[0]; }
                    return new SoundDocument[] { TargetDocument };

                case CommandParameterNames.TargetComponents:
                    return TargetComponents;

                case CommandParameterNames.InsertParentComponent:
                    return Application.ProjectService.ProjectDocument.Project;
            }

            throw new KeyNotFoundException();
        }

        /// <summary>
        /// 指定コマンドを実行できるコマンドターゲットを検索します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>他のコマンドターゲット。</returns>
        ICommandTarget ICommandTarget.FindTarget(Command command)
        {
            ICommandTarget target = FindOtherTarget(command);
            if (null != target) { return target; }

            return (_commandBindings.Contains(command.ID)) ? this : null;
        }

        /// <summary>
        /// 指定コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <param name="command">実行するコマンド。</param>
        /// <returns>コマンドの状態。</returns>
        CommandStatus ICommandTarget.QueryStatus(Command command)
        {
            if (!_commandBindings.Contains(command.ID)) { return CommandStatus.Unsupported; }
            return _commandBindings[command.ID].QueryStatus(command);
        }

        /// <summary>
        /// 指定コマンドを実行します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>コマンドを実行した場合は true、キャンセルされた場合は false。</returns>
        bool ICommandTarget.Execute(Command command)
        {
            if (!_commandBindings.Contains(command.ID)) { return false; }
            return _commandBindings[command.ID].Execute(command);
        }

        /// <summary>
        /// 指定コマンドを実行できる他のコマンドターゲットを検索します。
        /// </summary>
        /// <param name="command">コマンド。</param>
        /// <returns>他のコマンドターゲット。</returns>
        protected virtual ICommandTarget FindOtherTarget(Command command)
        {
            // MainWindow 以外のクラスでコマンド実行する場合は、ここにコードを追加。

            Control activeControl = this;

            while (this.IsActive && activeControl is ContainerControl)
            {

                Control control = (activeControl as ContainerControl).ActiveControl;
                activeControl = control;

                if (!(control is ICommandTarget)) { continue; }

                ICommandTarget target = (control as ICommandTarget).FindTarget(command);
                if (null != target) { return target; }

            }

            foreach (IToolWindowPage page in ToolPages)
            {

                if (!page.WindowActive) { continue; }
                if (!(page is ICommandTarget)) { continue; }
                if (!(page is Control)) { continue; }
                if (!(page as Control).ContainsFocus) { continue; }

                ICommandTarget target = (page as ICommandTarget).FindTarget(command);
                if (null != target) { return target; }

            }

            return null;
        }

        /// <summary>
        /// サウンドセットパネルを作成します。
        /// </summary>
        protected virtual SoundSetPanel CreateSoundSetPanel(DocumentReference docReference)
        {
            if (null == docReference) { throw new ArgumentNullException("docReference"); }
            return new SoundSetPanel(docReference);
        }

        /// <summary>
        /// バンクパネルを作成します。
        /// </summary>
        protected virtual BankPanel CreateBankPanel(string filePath)
        {
            Ensure.Argument.NotNull(filePath);
            Ensure.Argument.StringNotEmpty(filePath);

            return new BankPanel(filePath);
        }

        /// <summary>
        /// 検索パネルを作成します。
        /// </summary>
        protected virtual FindResultPanel2 CreateFindResultPanel()
        {
            return new FindResultPanel2();
        }

        /// <summary>
        /// メインメニューのコマンドが実行されると発生します。
        /// </summary>
        /// <param name="sender">MainMenuAdapter。</param>
        /// <param name="e">コマンドイベントデータ。</param>
        protected virtual void OnMainWindowCommandExecuted(object sender, CommandEventArgs e)
        {
            if (e.IsExecuted == false)
            {
                return;
            }

            NotifyCommandExecuted(e.Command);

            if (e.Command == ProjectCommands.ShowProjectSettings)
            {
                UpdateConnectMenuTexts();
            }

            if (e.Command == ProjectCommands.ShowProjectSettings ||
                e.Command == ToolCommands.EditUserCommands)
            {
                UpdateCommandTexts();
                BuildCommandUI();
                this.SoundSetPanels.ToList().ForEach(p => p.UpdateTabImage());
            }

            if (e.Command.ID == FileCommands.OpenProject.ID)
            {
                this.RefreshPanels();
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual void UpdateConnectToolBar()
        {
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual void UpdateConnectMenuTexts()
        {
        }

        /// <summary>
        /// Form がロードされるときに実行されます。
        /// </summary>
        /// <param name="e">空のイベントデータ。</param>
        protected override void OnLoad(EventArgs e)
        {
#if DEBUG
            // WindowsForm デザイナ対応
            // FormsApplicationMock のため、Initailize() は行わない。
            if ( null == FormsApplication.Instance.UIService ) {
                base.OnLoad( e );
                return;
            }
#endif

            base.OnLoad(e);

            //クリップボードビューアの設定
            _NextHandle = SetClipboardViewer(Handle);
        }

        /// <summary>
        /// Form が閉じられる前に実行されます。
        /// </summary>
        /// <param name="e">FormClosing イベントデータ。</param>
        protected override void OnFormClosing(FormClosingEventArgs e)
        {
            base.OnFormClosing(e);

            switch (e.CloseReason)
            {
                case CloseReason.UserClosing:
                    // ユーザ操作で閉じる場合は、必ずコマンドを経由する
                    Application.ExecuteCommand(FileCommands.ExitApplication);
                    e.Cancel = true;
                    return;
            }
        }

        /// <summary>
        /// Form が閉じられると実行されます。
        /// </summary>
        /// <param name="e">FormClosed イベントデータ。</param>
        protected override void OnFormClosed(FormClosedEventArgs e)
        {
            ExtractAppConfiguration();

            base.OnFormClosed(e);

            //クリップボードビューアの解除
            ChangeClipboardChain(Handle, _NextHandle);
        }

        protected override void OnActivated(EventArgs e)
        {
            base.OnActivated(e);

            this.IsActive = true;
            UpdatePresetListColumnsOnSplitButton();
        }

        protected override void OnDeactivate(EventArgs e)
        {
            base.OnDeactivate(e);

            this.IsActive = false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void SetVisibleCore(bool value)
        {
            if (!_loaded && value)
            {
#if DEBUG
                // EntryAssembly が null の場合は、
                // Windows Form デザイナによる実行とみなし例外を無視します。
                try
                {
                    Initialize();
                }
                catch
                {
                    if (Assembly.GetEntryAssembly() != null)
                    {
                        throw;
                    }
                }
#else
                Initialize();
#endif
            }

            base.SetVisibleCore(value);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnDragEnter(DragEventArgs drgevent)
        {
            base.OnDragEnter(drgevent);

            if (drgevent.Data.GetDataPresent(DataFormats.FileDrop) == true)
            {
                drgevent.Effect = DragDropEffects.Copy;
            }
            else
            {
                drgevent.Effect = DragDropEffects.None;
            }
        }

        protected override void OnDragOver(DragEventArgs drgevent)
        {
            base.OnDragOver(drgevent);

            if (drgevent.Data.GetDataPresent(DataFormats.FileDrop) == true)
            {
                drgevent.Effect = DragDropEffects.Copy;
            }
            else
            {
                drgevent.Effect = DragDropEffects.None;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnDragDrop(DragEventArgs drgevent)
        {
            base.OnDragDrop(drgevent);

            string[] filePaths = drgevent.Data.GetData(DataFormats.FileDrop, false) as string[];
            DropFiles(filePaths, true);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnKeyDown(KeyEventArgs e)
        {
            if (_LastActivePanel != null)
            {
                if (_LastActivePanel is CommonPanel)
                {
                    ((CommonPanel)_LastActivePanel).KeyDown(e);
                }
                else
                {
                    if (_LastActivePanel is IKeyPreviewReceiver)
                    {
                        ((IKeyPreviewReceiver)_LastActivePanel).KeyDown(e);
                    }
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        protected override void OnKeyUp(KeyEventArgs e)
        {
            if (_LastActivePanel != null)
            {
                if (_LastActivePanel is CommonPanel)
                {
                    ((CommonPanel)_LastActivePanel).KeyUp(e);
                }
                else
                {
                    if (_LastActivePanel is IKeyPreviewReceiver)
                    {
                        ((IKeyPreviewReceiver)_LastActivePanel).KeyUp(e);
                    }
                }
            }
        }

        ///--------------------------------
        /// <summary>
        /// フォーム内のコントロールのフォーカスが移動した時に呼ばれる
        /// </summary>
        protected override void UpdateDefaultButton()
        {
            base.UpdateDefaultButton();

            //
            Control activeControl = _leftSplitContainer.ActiveControl;

            while ((activeControl is CommonPanel) == false &&
                  (activeControl is IKeyPreviewReceiver) == false)
            {
                ContainerControl container = activeControl as ContainerControl;
                if (container == null)
                {
                    break;
                }

                activeControl = container.ActiveControl;
            }

            // ドッキングコンテナ(下)からフォーカスを持っているコントロールを探します。
            if (activeControl == null)
            {
                foreach (DockingPage page in _bottomDockingToolContainer.Pages)
                {
                    if (page.ContainsFocus == true)
                    {
                        activeControl = page;
                        break;
                    }
                }
            }

            //
            if (_LastActiveControl == null ||
                _LastActiveControl != activeControl)
            {

                //Console.WriteLine( "%%%%%%%% {0}", _TababControl);
                //Console.WriteLine( "Active {0}", activeControl);
                //Console.WriteLine( "Last   {0}", _LastActiveControl);

                if (_LastActivePanel != null)
                {
                    if (_LastActivePanel is IPanel)
                    {
                        ((IPanel)_LastActivePanel).Deactivated();
                    }
                }

                // CommonListPanel.Documentが既に消滅しているが、パネルがフォーカスされた場合の
                // 対処になります。(2012.6.19 aoyagi)
                if (activeControl is CommonListPanel)
                {
                    CommonListPanel panel = activeControl as CommonListPanel;
                    if (panel.Document == null)
                    {
                        return;
                    }
                }

                //
                if (null != _TabCtrlMain && _TabCtrlMain.Contains(activeControl) == true)
                {
                    SetCurrentTabCtrl(_TabCtrlMain);
                }

                if (null != _TabCtrlSub && _TabCtrlSub.Contains(activeControl) == true)
                {
                    SetCurrentTabCtrl(_TabCtrlSub);
                }

                //
                _LastActiveControl = activeControl;
                _LastActivePanel = activeControl;

                //
                if (_LastActivePanel != null)
                {
                    if (_LastActivePanel is IPanel)
                    {
                        ((IPanel)_LastActivePanel).Activated();
                    }
                }
            }

            //
            BuildCommandUI();
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual void ApplyAppConfiguration()
        {
            if (null == Application.AppConfiguration.Windows) { return; }
            if (!Application.AppConfiguration.Windows.ContainsKey(WindowName)) { return; }

            XmlFrameWindow xmlWindow = Application.AppConfiguration.Windows[WindowName] as XmlFrameWindow;
            if (null == xmlWindow) { return; }

            WindowConfigurationApplier.Apply(this, xmlWindow);

            XmlWindow xmlFindWindow = ApplicationBase.Instance.AppConfiguration.Windows["FindWindow"];
            if (null != xmlFindWindow)
            {
                WindowConfigurationApplier.Apply(_FindWindow, xmlFindWindow, true);
            }

            ApplyDockingContainersConfiguration(xmlWindow);
            ApplyCommandBarsConfiguration();
        }

        protected virtual void ExtractAppConfiguration()
        {
            if (!Application.AppConfiguration.Windows.ContainsKey(WindowName)) { return; }

            XmlFrameWindow xmlWindow = Application.AppConfiguration.Windows[WindowName] as XmlFrameWindow;
            if (null == xmlWindow) { return; }

            WindowConfigurationApplier.Extract(this, xmlWindow);

            XmlWindow xmlFindWindow = ApplicationBase.Instance.AppConfiguration.Windows["FindWindow"];
            if (null != xmlFindWindow)
            {
                WindowConfigurationApplier.Extract(_FindWindow, xmlFindWindow, true);
            }

            ExtractDockingContainersConfiguration(xmlWindow);
            ExtractCommandBarsConfiguration();
        }

        protected virtual void QueryToolStrips(IList<ToolStrip> result)
        {
            if (null == result) { throw new ArgumentNullException("result"); }

            result.Add(StandardTool);
            result.Add(OtherTool);
            result.Add(ToolWindowTool);
        }

        protected virtual void ApplyProjectConfiguration(bool createFindResultPanel = true)
        {
            XmlEditor xmlEditor = ProjectConfiguration.MainWindow.Editor;

            IDictionary<string, XmlTab> xmlTabs = ProjectConfiguration.
                                                          MainWindow.CreateTabDictionary();

            //
            if (this.Options.Application.General.MaintainTabOpen == true && createFindResultPanel == true)
            {
                if (xmlTabs.ContainsKey("Main"))
                {
                    TabConfigurationApplier.Apply(_TabCtrlMain, this, xmlTabs["Main"]);
                }
                if (xmlTabs.ContainsKey("Sub"))
                {
                    TabConfigurationApplier.Apply(_TabCtrlSub, this, xmlTabs["Sub"]);
                }
            }

            _TabSplitterRatioUD = xmlEditor.HorizontalSplitterPositionRatio;
            _TabSplitterRatioLR = xmlEditor.VerticalSplitterPositionRatio;

            switch (xmlEditor.Split)
            {
                case XmlEditorSplit.Vertical:
                    SplitLR();
                    break;
                case XmlEditorSplit.Horizontal:
                    SplitUD();
                    break;
            }

            SetCurrentTabCtrl(_TabCtrlMain);

            ApplySoundSetConfigurations();
            ApplyFindResultConfigurations();
        }

        protected virtual void ExtractProjectConfiguration()
        {
            IDictionary<string, XmlTab> xmlTabs = ProjectConfiguration.
                                                  MainWindow.CreateTabDictionary();

            // MainWindow.Editor.Tabs
            if (xmlTabs.ContainsKey("Main"))
            {
                TabConfigurationApplier.Extract(_TabCtrlMain, xmlTabs["Main"]);
            }
            if (xmlTabs.ContainsKey("Sub"))
            {
                TabConfigurationApplier.Extract(_TabCtrlSub, xmlTabs["Sub"]);
            }

            // MainWindow.Editor
            XmlEditor xmlEditor = ProjectConfiguration.MainWindow.Editor;

            switch (_SplitState)
            {
                case SplitState.UD:
                    xmlEditor.Split = XmlEditorSplit.Horizontal;
                    break;

                case SplitState.LR:
                    xmlEditor.Split = XmlEditorSplit.Vertical;
                    break;

                default:
                    xmlEditor.Split = XmlEditorSplit.None;
                    break;
            }

            xmlEditor.HorizontalSplitterPositionRatio = _TabSplitterRatioUD;
            xmlEditor.VerticalSplitterPositionRatio = _TabSplitterRatioLR;

            ExtractSoundSetConfigurations();
            ExtractFindResultConfigurations();
        }

        /// <summary>
        /// コマンドの関連付けを行います。
        /// </summary>
        /// <param name="commandBindings">コマンドバインディングコレクション。</param>
        protected virtual void InitializeCommandBindings(CommandBindingCollection commandBindings)
        {
            if (null == commandBindings) { throw new ArgumentNullException("commandBindings"); }

            // ファイル関連コマンドハンドラ
            commandBindings.Add(new CommandBinding(this, FileCommands.OpenFile.ID,
                                      QueryStatusOpenFile, ExecuteOpenFile));
            commandBindings.Add(new CommandBinding(this, FileCommands.ReloadProject.ID,
                                      QueryStatusReloadProject, ExecuteReloadProject));
            commandBindings.Add(new CommandBinding(this, FileCommands.CloseAllDocuments.ID,
                                                    QueryStatusFileCloseAllDocuments, ExecuteFileCloseAllDocuments));

            // 編集関連コマンドハンドラ
            commandBindings.Add
                (new CommandBinding(this, EditCommands.ShowFindWindow.ID,
                                      QueryStatusShowFindWindow,
                                      ExecuteShowFindWindow));
            commandBindings.Add
                (new CommandBinding(this, EditCommands.OpenSoundSetFindTabPage.ID,
                                      QueryStatusOpenSoundSetFindTabPage,
                                      ExecuteOpenSoundSetFindTabPage));
            commandBindings.Add
                (new CommandBinding(this, EditCommands.OpenInstrumentFindTabPage.ID,
                                      QueryStatusOpenInstrumentFindTabPage,
                                      ExecuteOpenInstrumentFindTabPage));

            // 表示関連コマンドハンドラ
            commandBindings.Add(new CommandBinding(this, ViewCommands.SynchronizeTree.ID,
                                                     QueryStatusSynchronizeTree,
                                                     ExecuteSynchronizeTree));

            commandBindings.Add(new CommandBinding
                                 (this, ViewCommands.UpdateWaveFileInformation.ID,
                                   QueryStatusUpdateWaveFileInformation,
                                   ExecuteUpdateWaveFileInformation));

            commandBindings.Add(new CommandBinding(this,
                    new ShowToolPageHandler(ViewCommands.ShowProjectTreeWindow.ID, ProjectTreePanel.PageName)));
            commandBindings.Add(new CommandBinding(this,
                    new ShowToolPageHandler(ViewCommands.ShowBookmarkTreeWindow.ID, BookmarkTreePanel.PageName)));
            commandBindings.Add(new CommandBinding(this,
                    new ShowToolPageHandler(ViewCommands.ShowHistoryWindow.ID, HistoryListPanel.PageName)));
            commandBindings.Add(new CommandBinding(this,
                    new ShowToolPageHandler(ViewCommands.ShowParameterWindow.ID, ParameterPanel.PageName)));
            commandBindings.Add(new CommandBinding(this,
                    new ShowToolPageHandler(ViewCommands.ShowConvertLogWindow.ID, ConvertLogPanel.PageName)));
            commandBindings.Add(new CommandBinding(this,
                    new ShowToolPageHandler(ViewCommands.ShowMeasureWaveReferenceCountWindow.ID,
                                             MeasureWaveReferenceCountPanel.PageName)));

            /// デフォルト非表示です。表示するには派生先で commandBindings.Remove してから構築してください。
            commandBindings.Add(new CommandBinding
                                 (this, ViewCommands.ShowWaveSoundEditorWindow.ID,
                                   (c) => CommandStatus.Supported, (c) => true));

            // プロジェクト関連コマンドハンドラ
            commandBindings.Add(new CommandBinding(this, new QueryAndAddNewSoundSetHandler(this)));
            commandBindings.Add(new CommandBinding(this, new QueryAndAddExistingSoundSetHandler(this)));
            commandBindings.Add(new CommandBinding(this, new ShowProjectSettingsHandler(this)));
            commandBindings.Add(new CommandBinding(this, ProjectCommands.RepairProject.ID,
                                                     QueryStatusRepairProject, ExecuteRepairProject));

            // ツール関連コマンドハンドラ
            commandBindings.Add(new CommandBinding(this, new EditUserCommandsHandler()));

            commandBindings.Add
                (new CommandBinding(this, new UserCommandHandler
                                      (ToolCommands.UserCommand.ID)));

            // ウィンドウ関連コマンドハンドラ
            commandBindings.Add(new CommandBinding(this, WindowCommands.SplitEditHorizontal.ID,
                                                    QueryStatusSplitEditHorizontal, ExecuteSplitEditHorizontal));
            commandBindings.Add(new CommandBinding(this, WindowCommands.SplitEditVertical.ID,
                                                    QueryStatusSplitEditVertical, ExecuteSplitEditVertical));

            // タブ関連コマンドハンドラ
            commandBindings.Add(new CommandBinding(this, TabCommands.SelectDocument.ID,
                                                    QueryStatusSelectDocument, ExecuteSelectDocument));
            commandBindings.Add(new CommandBinding(this, TabCommands.NextDocument.ID,
                                                    QueryStatusNextDocument, ExecuteNextDocument));
            commandBindings.Add(new CommandBinding(this, TabCommands.PreviousDocument.ID,
                                                    QueryStatusPreviousDocument, ExecutePreviousDocument));
            commandBindings.Add(new CommandBinding(this, TabCommands.CloseDocument.ID,
                                                    QueryStatusCloseDocument, ExecuteCloseDocument));
            commandBindings.Add(new CommandBinding(this, TabCommands.CloseSelectedDocument.ID,
                                                    QueryStatusCloseSelectedDocument, ExecuteCloseSelectedDocument));
            commandBindings.Add(new CommandBinding(this, TabCommands.CloseOtherDocuments.ID,
                                                    QueryStatusCloseOtherDocuments, ExecuteCloseOtherDocuments));
            commandBindings.Add(new CommandBinding(this, TabCommands.CloseAllDocuments.ID,
                                                    QueryStatusCloseAllDocuments, ExecuteCloseAllDocuments));
            commandBindings.Add(new CommandBinding(this, TabCommands.OpenDocumentFolder.ID,
                                                    QueryStatusOpenDocumentFolder, ExecuteOpenDocumentFolder));
            commandBindings.Add(new CommandBinding(this, TabCommands.AddBookmark.ID,
                                                    QueryStatusAddBookmark, ExecuteAddBookmark));
            // ヘルプ関連コマンドハンドラ
            commandBindings.Add(new CommandBinding(this,
                             new ShowHelpHandler(HelpCommands.ShowHelp.ID, Application.HelpURL)));
            commandBindings.Add(new CommandBinding(this, new ShowHelpHandler(HelpCommands.ShowTroubleshootingHelp.ID, Application.TroubleshootingHelpURL)));
            commandBindings.Add(new CommandBinding(this, new AboutCommandHandler()));
        }

        /// <summary>
        /// ツールページを初期化します。
        /// </summary>
        protected virtual void InitializeToolPages()
        {
            _dockingToolContainers = new DockingToolContainerDictionary();
            _dockingToolContainers.Add("MainWindowLeft", _leftDockingToolContainer);
            _dockingToolContainers.Add("MainWindowRight", _rightDockingToolContainer);
            _dockingToolContainers.Add("MainWindowBottom", _bottomDockingToolContainer);

            _toolPages = new ToolPageDictionary();

            AddToolPage(new ProjectTreePanel());
            AddToolPage(new BookmarkTreePanel());
            AddToolPage(new HistoryListPanel());
            AddToolPage(new ParameterPanel());
            AddToolPage(new ConvertLogPanel());
            AddToolPage(new MeasureWaveReferenceCountPanel());

            this.ProjectPanel.SelectChanged += OnSelectChangedProjectPanel;

            UpdateDockContainer(_leftDockingToolContainer, _leftSplitContainer, true);
            UpdateDockContainer(_rightDockingToolContainer, _rightSplitContainer, false);
            UpdateDockContainer(_bottomDockingToolContainer, _bottomSplitContainer, false);
        }

        /// <summary>
        /// ツールページの関連付けを有効化します。
        /// </summary>
        protected virtual void ValidateToolPageBindings()
        {
            ValidateToolPageBinding(ToolPages[ProjectTreePanel.PageName], _leftDockingToolContainer);
            ValidateToolPageBinding(ToolPages[BookmarkTreePanel.PageName], _leftDockingToolContainer);
            ValidateToolPageBinding(ToolPages[HistoryListPanel.PageName], _leftDockingToolContainer);
            ValidateToolPageBinding(ToolPages[ParameterPanel.PageName], _rightDockingToolContainer);
            ValidateToolPageBinding(ToolPages[MeasureWaveReferenceCountPanel.PageName], _rightDockingToolContainer);
            ValidateToolPageBinding(ToolPages[ConvertLogPanel.PageName], _bottomDockingToolContainer);
        }

        protected void AddToolPage(IToolWindowPage page)
        {
            if (null == page) { throw new ArgumentNullException("page"); }
            _toolPages.Add(page);
        }

        protected void BindToolPage(IToolWindowPage page, DockingToolContainer container)
        {
            if (null == page) { throw new ArgumentNullException("page"); }
            if (null == container) { throw new ArgumentNullException("container"); }

            bool dockVisible = page.DockVisible;

            if (null != page.DockContainer)
            {
                page.DockContainer.Pages.Remove(page);
            }

            container.Pages.Add(page);
            page.DockVisible = dockVisible;
        }

        protected void ValidateToolPageBinding(IToolWindowPage page, DockingToolContainer container)
        {
            if (null == page) { throw new ArgumentNullException("page"); }
            if (null == container) { throw new ArgumentNullException("container"); }

            if (null != page.DockContainer) { return; }
            BindToolPage(page, container);
        }

        protected void OnProjectReloaded(EventArgs e)
        {
            if (Application.ProjectService.Project != null)
            {
                ProjectPanel.AttachProject(Application.ProjectService.ProjectDocument);
                this.UpdateLeftDockingToolTitle();
            }

            UpdateMruProjects();

            BuildCommandUI();

            Application.ExecuteCommand(ViewCommands.SynchronizeTree);

            _editSplitContainer.Select();

            RemoveGarbagePages();

            if (Application.BookmarkService != null)
            {
                Application.BookmarkService.UpdateTarget();
            }
            if (Application.HistoryService != null)
            {
                Application.HistoryService.UpdateTarget();
            }

            if (null != ProjectReloaded)
            {
                ProjectReloaded(this, e);
            }
        }

        /// <summary>
        /// クリップボード関係
        /// </summary>
        protected override void WndProc(ref Message message)
        {
            switch (message.Msg)
            {
                case WM.WM_DEVICECHANGE:
                    this.OnDeveiceChange(message);
                    break;

                case WM.WM_COPYDATA:
                    bool result = false;

                    COPYDATASTRUCT cds = new COPYDATASTRUCT();
                    cds = (COPYDATASTRUCT)Marshal.PtrToStructure
                        (message.LParam, typeof(COPYDATASTRUCT));
                    if (cds.cbData > 0)
                    {
                        byte[] data = new byte[cds.cbData];
                        Marshal.Copy(cds.lpData, data, 0, (int)cds.cbData);
                        string[] texts = Encoding.Unicode.GetString(data).Split('|');
                        result = ExecuteConsoleCommand(texts);
                    }

                    message.Result = new IntPtr(result == true ? 1 : 0);
                    return;

                case WM.WM_DRAWCLIPBOARD:
                    UpdatedClipboard();

                    if (_NextHandle != IntPtr.Zero)
                    {
                        SendMessage(_NextHandle,
                                     message.Msg, message.WParam, message.LParam);
                    }
                    break;

                case WM.WM_CHANGECBCHAIN:
                    if ((IntPtr)message.WParam == _NextHandle)
                    {
                        _NextHandle = (IntPtr)message.LParam;
                    }
                    else
                    {

                        if (_NextHandle != IntPtr.Zero)
                        {
                            SendMessage(_NextHandle,
                                         message.Msg, message.WParam, message.LParam);
                        }
                    }
                    break;

                case WM.WM_SYSCOMMAND:
                    int wparam = message.WParam.ToInt32() & 0xFFF0;
                    switch (wparam)
                    {
                        // Restore
                        case 0xF120:
                            if (WindowRestoring != null)
                            {
                                WindowRestoring(this, EventArgs.Empty);
                            }
                            base.WndProc(ref message);
                            if (WindowRestored != null)
                            {
                                WindowRestored(this, EventArgs.Empty);
                            }
                            return;

                        // Minimize
                        case 0xF020:
                            if (WindowMinimizing != null)
                            {
                                WindowMinimizing(this, EventArgs.Empty);
                            }
                            base.WndProc(ref message);
                            if (WindowMinimized != null)
                            {
                                WindowMinimized(this, EventArgs.Empty);
                            }
                            return;

                        // Maximize
                        case 0xF030:
                            if (WindowMaximizing != null)
                            {
                                WindowMaximizing(this, EventArgs.Empty);
                            }
                            base.WndProc(ref message);
                            if (WindowMaximized != null)
                            {
                                WindowMaximized(this, EventArgs.Empty);
                            }
                            return;
                    }
                    break;
            }

            base.WndProc(ref message);
        }

        /// <summary>
        ///
        /// </summary>
        private bool ExecuteConsoleCommand(string[] texts)
        {
            if (texts.Length <= 0 ||
                Application.ProjectService.ProjectDocument == null)
            {
                return false;
            }

            string projectFilePath = Application.ProjectService.ProjectDocument.Resource.Key;
            string filePath = texts[0];
            if (String.Compare(projectFilePath, filePath, true) != 0)
            {
                return false;
            }

            if (texts.Length == 8)
            {
                var projectService = Application.ProjectService;
                string command = texts[1];
                string itemName = texts[2];
                string value1 = texts[3];
                string value2 = texts[4];
                string value3 = texts[5];
                string value4 = texts[6];
                string value5 = texts[7];

                switch (command)
                {
                    case "openitem":
                    case "showitem":
                        {
                            Component component = projectService.SoundSetItems
                                .Where(c => String.Compare(c.Name, itemName) == 0)
                                .FirstOrDefault();

                            if (component == null)
                            {
                                Finder finder = new Finder();
                                component = finder.FindInstrumentFromProject
                                    (projectService, itemName);
                            }

                            if (component != null)
                            {
                                bool open = (command == "openitem");
                                ShowPageWithSingleListByComponent(component, open);
                            }
                        }
                        break;

                    case "playitem":
                        {
                            Component component = projectService.SoundSetItems
                                .Where(c => String.Compare(c.Name, itemName) == 0)
                                .FirstOrDefault();

                            if (component != null)
                            {
                                PlayItem(component);
                            }
                            return true;
                        }

                    case "stopallitem":
                        StopAllItem();
                        return true;

                    case "edititem":
                        {
                            Component component = projectService.SoundSetItems
                                .Where(c => String.Compare(c.Name, itemName) == 0)
                                .FirstOrDefault();

#if false               // 今はユーザーパラメータのみなので、バンクを調べない。
                        if (component == null)
                        {
                            Finder finder = new Finder();
                            component = finder.FindInstrumentFromProject(projectService, itemName);
                        }
#endif
                            if (component != null)
                            {
                                ShowPageWithSingleListByComponent(component, true);
                                EditComponentParameterValue(component, value1, value2);
                            }

                            break;
                        }

                    case "outputwavefile":
                        {
                            Component component = projectService.SoundSetItems
                                .Where(c => String.Compare(c.Name, itemName) == 0)
                                .FirstOrDefault();

                            if (component != null)
                            {
                                ShowPageWithSingleListByComponent(component, false);
                                this.OutputWaveFile(component, value1, value2, value3, value4, value5);
                            }

                            break;
                        }

                    case "saveproject":
                        SaveHelper.SaveAll(this);
                        return true;
                }
            }

            // 最小化されていた場合には、通常状態に変更します。
            if (this.WindowState == FormWindowState.Minimized)
            {
                this.WindowState = FormWindowState.Normal;
            }

            // 常に最前面にして、すぐに解除することにより、ウインドウを
            // 最前面に移動します。
            this.TopMost = true;
            this.TopMost = false;

            // アイテムがない場合も true
            return true;
        }

        /// <summary>
        /// パラメータパネルに表示するパラメータ項目を追加します。
        /// </summary>
        protected void InitializeParameterPanel(SoundParameterPanel panel)
        {
            // ストリームサウンド
            panel.AddParameterName(typeof(StreamSoundBase),
                                    ProjectParameterNames.Volume,
                                    MessageResource.HeaderText_Volume);

            panel.AddParameterName(typeof(StreamSoundBase),
                                    ProjectParameterNames.Sound.PlayerPriority,
                                    MessageResource.HeaderText_PlayerPriority);

            panel.AddParameterName(typeof(StreamSoundBase),
                                    ProjectParameterNames.Pitch,
                                    MessageResource.HeaderText_Pitch);

            panel.AddParameterName(typeof(StreamSoundBase),
                                    ProjectParameterNames.Sends.AuxASend,
                                    MessageResource.HeaderText_AuxASend);

            panel.AddParameterName(typeof(StreamSoundBase),
                                    ProjectParameterNames.Sends.AuxBSend,
                                    MessageResource.HeaderText_AuxBSend);

            panel.AddParameterName(typeof(StreamSoundBase),
                                    ProjectParameterNames.Sends.MainSend,
                                    MessageResource.HeaderText_MainSend);

            panel.AddParameterName(typeof(StreamSoundBase),
                                    ProjectParameterNames.Sound3D.Enable3DVolume,
                                    MessageResource.HeaderText_Enable3DVolume);

            panel.AddParameterName(typeof(StreamSoundBase),
                                    ProjectParameterNames.Sound3D.Enable3DPan,
                                    MessageResource.HeaderText_Enable3DPan);

            panel.AddParameterName(typeof(StreamSoundBase),
                                    ProjectParameterNames.Sound3D.Enable3DSurroundPan,
                                    MessageResource.HeaderText_Enable3DSurroundPan);

            panel.AddParameterName(typeof(StreamSoundBase),
                                    ProjectParameterNames.Sound3D.Enable3DPriority,
                                    MessageResource.HeaderText_Enable3DPriority);

            panel.AddParameterName(typeof(StreamSoundBase),
                                    ProjectParameterNames.Sound3D.Enable3DFilter,
                                    MessageResource.HeaderText_Enable3DFilter);

            panel.AddParameterName(typeof(StreamSoundBase),
                                    ProjectParameterNames.Sound3D.DecayRatio3D,
                                    MessageResource.HeaderText_3DDecayRatio);

            panel.AddParameterName(typeof(StreamSoundBase),
                                    ProjectParameterNames.Sound3D.DopplerFactor3D,
                                    MessageResource.HeaderText_3DDopplerFactor);

            // ストリームサウンドトラック
            panel.AddParameterName(typeof(StreamSoundTrackBase),
                                    ProjectParameterNames.Volume,
                                    MessageResource.HeaderText_Volume);

            panel.AddParameterName(typeof(StreamSoundTrackBase),
                                    ProjectParameterNames.Pan,
                                    MessageResource.HeaderText_Pan);

            panel.AddParameterName(typeof(StreamSoundTrackBase),
                                    ProjectParameterNames.SurroundPan,
                                    MessageResource.HeaderText_SurroundPan);

            panel.AddParameterName(typeof(StreamSoundTrackBase),
                                    ProjectParameterNames.FrontBypass,
                                    MessageResource.HeaderText_FrontBypass);

            panel.AddParameterName(typeof(StreamSoundTrackBase),
                                    ProjectParameterNames.LPF,
                                    MessageResource.HeaderText_LPF);

            panel.AddParameterName(typeof(StreamSoundTrackBase),
                                    ProjectParameterNames.Biquad,
                                    MessageResource.HeaderText_Biquad);

            panel.AddParameterName(typeof(StreamSoundTrackBase),
                                    ProjectParameterNames.Sends.AuxASend,
                                    MessageResource.HeaderText_AuxASend);

            panel.AddParameterName(typeof(StreamSoundTrackBase),
                                    ProjectParameterNames.Sends.AuxBSend,
                                    MessageResource.HeaderText_AuxBSend);

            panel.AddParameterName(typeof(StreamSoundTrackBase),
                                    ProjectParameterNames.Sends.MainSend,
                                    MessageResource.HeaderText_MainSend);

            // ウェーブサウンド
            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.Volume,
                                    MessageResource.HeaderText_Volume);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.Sound.ChannelPriority,
                                    MessageResource.HeaderText_ChannelPriority);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.Sound.PlayerPriority,
                                    MessageResource.HeaderText_PlayerPriority);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.Pan,
                                    MessageResource.HeaderText_Pan);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.SurroundPan,
                                    MessageResource.HeaderText_SurroundPan);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.Pitch,
                                    MessageResource.HeaderText_Pitch);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.LPF,
                                    MessageResource.HeaderText_LPF);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.Biquad,
                                    MessageResource.HeaderText_Biquad);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.WaveSound.EnvelopeRelease,
                                    MessageResource.HeaderText_EnvelopeRelease);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.Sends.AuxASend,
                                    MessageResource.HeaderText_AuxASend);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.Sends.AuxBSend,
                                    MessageResource.HeaderText_AuxBSend);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.Sends.MainSend,
                                    MessageResource.HeaderText_MainSend);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.Sound.ReleasePriorityFixed,
                                    MessageResource.HeaderText_ReleasePriorityFixed);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.Sound3D.Enable3DVolume,
                                    MessageResource.HeaderText_Enable3DVolume);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.Sound3D.Enable3DPan,
                                    MessageResource.HeaderText_Enable3DPan);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.Sound3D.Enable3DSurroundPan,
                                    MessageResource.HeaderText_Enable3DSurroundPan);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.Sound3D.Enable3DPriority,
                                    MessageResource.HeaderText_Enable3DPriority);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.Sound3D.Enable3DFilter,
                                    MessageResource.HeaderText_Enable3DFilter);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.Sound3D.DecayRatio3D,
                                    MessageResource.HeaderText_3DDecayRatio);

            panel.AddParameterName(typeof(WaveSoundBase),
                                    ProjectParameterNames.Sound3D.DopplerFactor3D,
                                    MessageResource.HeaderText_3DDopplerFactor);

            // シーケンス
            panel.AddParameterName(typeof(SequenceSoundBase),
                                    ProjectParameterNames.Volume,
                                    MessageResource.HeaderText_Volume);

            panel.AddParameterName(typeof(SequenceSoundBase),
                                    ProjectParameterNames.Sound.ChannelPriority,
                                    MessageResource.HeaderText_ChannelPriority);

            panel.AddParameterName(typeof(SequenceSoundBase),
                                    ProjectParameterNames.Sound.PlayerPriority,
                                    MessageResource.HeaderText_PlayerPriority);

            panel.AddParameterName(typeof(SequenceSoundBase),
                                    ProjectParameterNames.Sound.ReleasePriorityFixed,
                                    MessageResource.HeaderText_ReleasePriorityFixed);

            panel.AddParameterName(typeof(SequenceSoundBase),
                                    ProjectParameterNames.Sound3D.Enable3DVolume,
                                    MessageResource.HeaderText_Enable3DVolume);

            panel.AddParameterName(typeof(SequenceSoundBase),
                                    ProjectParameterNames.Sound3D.Enable3DPan,
                                    MessageResource.HeaderText_Enable3DPan);

            panel.AddParameterName(typeof(SequenceSoundBase),
                                    ProjectParameterNames.Sound3D.Enable3DSurroundPan,
                                    MessageResource.HeaderText_Enable3DSurroundPan);

            panel.AddParameterName(typeof(SequenceSoundBase),
                                    ProjectParameterNames.Sound3D.Enable3DPriority,
                                    MessageResource.HeaderText_Enable3DPriority);

            panel.AddParameterName(typeof(SequenceSoundBase),
                                    ProjectParameterNames.Sound3D.Enable3DFilter,
                                    MessageResource.HeaderText_Enable3DFilter);

            panel.AddParameterName(typeof(SequenceSoundBase),
                                    ProjectParameterNames.Sound3D.DecayRatio3D,
                                    MessageResource.HeaderText_3DDecayRatio);

            panel.AddParameterName(typeof(SequenceSoundBase),
                                    ProjectParameterNames.Sound3D.DopplerFactor3D,
                                    MessageResource.HeaderText_3DDopplerFactor);
        }

        protected virtual void Disconnect()
        {
        }

        protected virtual void PlayItem(Component component)
        {
        }

        protected virtual void StopAllItem()
        {
        }

        protected virtual void OnDeveiceChange(Message message)
        {
        }

        /// <summary>
        ///
        /// </summary>
        [DllImport("user32.dll")]
        private static extern IntPtr SetClipboardViewer(IntPtr hwnd);
        [DllImport("user32.dll")]
        private static extern bool ChangeClipboardChain(IntPtr hwndRemove, IntPtr hwndNext);
        [DllImport("user32.dll")]
        private static extern int SendMessage(IntPtr hwnd, int message,
        IntPtr wparam, IntPtr lparam);

        /// <summary>
        /// 表示されているパネルを取得します。
        /// </summary>
        private CommonListPanel[] DisplayedListPanels
        {
            get
            {
                CommonListPanel[] panels = _TabCtrlMain.TabPages
                    .Concat(_TabCtrlSub.TabPages)
                    .Cast<CommonTabPage>()
                    .Select(t => t.Panel)
                    .Cast<CommonListPanel>()
                    .ToArray();
                return panels;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private delegate void ExecuteForDisplayedPanelsHandler(CommonListPanel panel);
        private void ExecuteForDisplayedPanels(ExecuteForDisplayedPanelsHandler func)
        {
            foreach (CommonListPanel panel in DisplayedListPanels)
            {
                func(panel);
            }
        }

        /// <summary>
        /// 検索タブのタイトル番号を取得します。
        /// 使われていない番号を探して、その番号を返します。
        /// </summary>
        private int GetFindTabPageNumber(string name)
        {
            HashSet<int> numbers = new HashSet<int>();

            foreach (FindResultPanel2 panel in
                     DisplayedListPanels
                     .Where(p => p is FindResultPanel2))
            {
                if (panel.TitleName == name)
                {
                    numbers.Add(panel.TitleNumber);
                }
            }

            int number = 1;
            while (numbers.Contains(number) == true)
            {
                number++;
            }
            return number;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnLeftDockingToolContainerMinimizing(object sender, EventArgs e)
        {
            this.ProjectPanel.ProjectCtrl.EndEdit();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnLeftDockingToolContainerSelectedTabPageChanged(object sender, EventArgs e)
        {
            this.BuildCommandUI();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdatedClipboard()
        {
            CommonListPanel panel = null;

            foreach (CommonTabPage page in _TabCtrlMain.TabPages)
            {
                if ((panel = page.Panel as CommonListPanel) != null)
                {
                    panel.UpdatedClipboard();
                }
            }

            foreach (CommonTabPage page in _TabCtrlSub.TabPages)
            {
                if ((panel = page.Panel as CommonListPanel) != null)
                {
                    panel.UpdatedClipboard();
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool DropFiles(string[] filePaths, bool checkBankFile)
        {
            SoundProjectDocument projectDocument = null;
            bool result = false;

            projectDocument = FormsApplication.Instance.ProjectService.ProjectDocument;

            //プロジェクトファイルなのか？
            foreach (string filePath in filePaths)
            {
                if (IsProjectFile(filePath) == true)
                {
                    if (OpenProjectFile(filePath) == true)
                    {
                        result = true;
                        break;
                    }
                }
            }

            //バンクファイルなのか？
            if (projectDocument != null && checkBankFile == true)
            {
                foreach (string filePath in filePaths)
                {
                    if (IsBankFile(filePath) == true)
                    {
                        OpenFile(filePath);
                        result = true;
                    }
                }
            }

            return result;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool IsProjectFile(string filePath)
        {
            string extension = null;

            extension = Path.GetExtension(filePath).ToLower();
            return extension ==
                ("." + this.Application.Traits.IntermediateOutputTraits.SoundProjectFileExtension);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool IsBankFile(string filePath)
        {
            string extension = null;

            extension = Path.GetExtension(filePath).ToLower();
            return extension ==
                ("." + this.Application.Traits.IntermediateOutputTraits.BankFileExtension);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool CloseAllTabPages()
        {
            _TabCtrlSub.CloseAllTabPages();
            _TabCtrlMain.CloseAllTabPages();
            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void ActivateTabPage(int index, bool subPanel)
        {
            CommonTabPage page = null;
            int pageCount = _CurrentTabCtrl.TabPages.Count;

            if (index < 0 || index >= pageCount)
            {
                return;
            }

            _CurrentTabCtrl.SelectedIndex = index;
            page = _CurrentTabCtrl.SelectedTab as CommonTabPage;
            //page.Panel.ActivatePanel( subPanel);
            ActivatePage(page);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void ActivateTabPage(int index)
        {
            ActivateTabPage(index, false);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void ActivateLastTabPage()
        {
            ActivateTabPage(_CurrentTabCtrl.TabPages.Count - 1);
        }

        ///--------------------------------
        /// <summary>
        /// 開かれている指定パネルを持つタブページの取得
        /// </summary>
        private CommonTabPage[] OpenedAllTabPages()
        {
            return _TabCtrlMain.TabPages.Concat(_TabCtrlSub.TabPages).Cast<CommonTabPage>().ToArray();
        }

        ///--------------------------------
        /// <summary>
        /// 開かれている指定パネルを持つタブページの取得
        /// </summary>
        private CommonTabPage[] OpenedAllTabPages<TPage>()
        {
            return OpenedAllTabPages()
                    .Where(p => p is CommonTabPage && ((CommonTabPage)p).Panel is TPage)
                    .Cast<CommonTabPage>().ToArray();
        }

        ///--------------------------------
        /// <summary>
        /// 開かれている指定パネルを持つタブページの取得
        /// </summary>
        private CommonTabPage[] OpenedTabPage<TPage>(TabCtrl tabCtrl)
        {
            if (tabCtrl == null)
            {
                tabCtrl = _CurrentTabCtrl;
            }

            return tabCtrl.TabPages
                .Where(p => p is CommonTabPage && ((CommonTabPage)p).Panel is TPage)
                .Cast<CommonTabPage>().ToArray();
        }

        ///--------------------------------
        /// <summary>
        /// 開かれているサウンドセットタブページの取得
        /// </summary>
        private CommonTabPage[] OpnedSoundSetTabPage(TabCtrl tabCtrl)
        {
            return OpenedTabPage<SoundSetPanel>(tabCtrl);
        }

        ///--------------------------------
        /// <summary>
        /// 開かれているバンクタブページの取得
        /// </summary>
        private CommonTabPage[] OpenedBankTabPage(TabCtrl tabCtrl)
        {
            return OpenedTabPage<BankPanel>(tabCtrl);
        }

        ///
        private enum SplitState
        {
            LR,
            UD,
            None,
        }

        private SplitState _SplitState = SplitState.None;
        private double _TabSplitterRatioLR = 0.5;
        private double _TabSplitterRatioUD = 0.5;
        private bool _TabSplitterRatioUpdate = true;

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private TabCtrl InactiveTabCtrl
        {
            get { return _CurrentTabCtrl == _TabCtrlMain ? _TabCtrlSub : _TabCtrlMain; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private double TabSplitterRatio
        {
            get
            {
                double ratio = 5.0;

                if (_editSplitContainer != null)
                {
                    ratio = _editSplitContainer.Orientation == Orientation.Horizontal ?
                        _TabSplitterRatioUD : _TabSplitterRatioLR;
                }

                return ratio;
            }

            set
            {
                if (_editSplitContainer != null)
                {
                    if (_editSplitContainer.Orientation == Orientation.Horizontal)
                    {
                        _TabSplitterRatioUD = value;
                    }
                    else
                    {
                        _TabSplitterRatioLR = value;
                    }
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void ChangeTabCtrlState(TabCtrl mController, TabCtrl sController)
        {
            int d = 0;
            int md = 0;
            int sd = 0;

            if (_editSplitContainer.Orientation == Orientation.Horizontal)
            {
                d = _editSplitContainer.Height;
                md = _TabCtrlMain.MinimumSize.Height + 2;
                sd = _TabCtrlSub.MinimumSize.Height + 2;
            }
            else
            {

                d = _editSplitContainer.Width;
                md = _TabCtrlMain.MinimumSize.Width + 2;
                sd = _TabCtrlSub.MinimumSize.Width + 2;
            }

            //
            if (mController.State == TabCtrl.ControlState.Minimized &&
                sController.State == TabCtrl.ControlState.Minimized)
            {
                sController.State = TabCtrl.ControlState.Normal;
            }

            //
            switch (mController.State)
            {
                case TabCtrl.ControlState.Normal:
                    _editSplitContainer.SplitterDistance = (int)(TabSplitterRatio * d);
                    _editSplitContainer.Panel1MinSize = 54;
                    _editSplitContainer.Panel2MinSize = 54;
                    break;

                case TabCtrl.ControlState.Minimized:
                    if (mController == _TabCtrlMain)
                    {
                        _editSplitContainer.Panel1MinSize = md;
                        _editSplitContainer.Panel2MinSize = md;
                        _editSplitContainer.SplitterDistance = md;
                    }
                    else
                    {
                        _editSplitContainer.Panel1MinSize = sd;
                        _editSplitContainer.Panel2MinSize = sd;
                        _editSplitContainer.SplitterDistance = d - sd;
                    }
                    break;
            }

            if (_CurrentTabCtrl.State == TabCtrl.ControlState.Minimized)
            {
                SetCurrentTabCtrl(InactiveTabCtrl);
                ActivateTabPage(_CurrentTabCtrl.SelectedIndex);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void UpdateTabSplitterDistance()
        {
            UpdateTabSplitterDistance(true);
        }

        private bool _UpdatingTabSplitterDistance = false;

        ///
        private void UpdateTabSplitterDistance(bool changeState)
        {
            if (_UpdatingTabSplitterDistance == true) { return; }
            _UpdatingTabSplitterDistance = true;

            int d = 0;
            int md = 0;
            int sd = 0;

            if (_editSplitContainer.Orientation == Orientation.Horizontal)
            {
                //d = _editSplitContainer.Height - _editSplitContainer.SplitterWidth;
                d = _editSplitContainer.Height;
                md = _TabCtrlMain.MinimumSize.Height + 2;
                sd = _TabCtrlSub.MinimumSize.Height + 2;
            }
            else
            {

                //d = _editSplitContainer.Width  - _editSplitContainer.SplitterWidth;
                d = _editSplitContainer.Width;
                md = _TabCtrlMain.MinimumSize.Width + 2;
                sd = _TabCtrlSub.MinimumSize.Width + 2;
            }


            TabCtrl mCtrl = _TabCtrlMain;
            TabCtrl sCtrl = _TabCtrlSub;

            if (changeState == true)
            {
                if (mCtrl.State == TabCtrl.ControlState.Minimized &&
                    sCtrl.State == TabCtrl.ControlState.Minimized)
                {
                    mCtrl.State = TabCtrl.ControlState.Normal;
                }
            }

            if (mCtrl.State == TabCtrl.ControlState.Minimized)
            {
                if (sCtrl.State == TabCtrl.ControlState.Minimized)
                {
                    Debug.Assert(false, "Can not minimized tab main and sub");
                }
                else
                {
                    _editSplitContainer.SplitterDistance = md;
                }
            }
            else
            {

                if (sCtrl.State == TabCtrl.ControlState.Minimized)
                {
                    _editSplitContainer.SplitterDistance = d - sd;
                }
                else
                {
                    _editSplitContainer.SplitterDistance = (int)(TabSplitterRatio * d);
                }
            }

            _UpdatingTabSplitterDistance = false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void VisibleTabControllerMinimizeButton(bool b)
        {
            _TabCtrlMain.Minimizable = b;
            _TabCtrlSub.Minimizable = b;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool Split()
        {
            if (_TabCtrlMain.TabPages.Count <= 0)
            {
                return false;
            }

            if (_TabCtrlSub.TabPages.Count >= 1)
            {
                return true;
            }

            CommonTabPage page = null;
            int selectedIndex = _TabCtrlMain.SelectedIndex;

            page = _TabCtrlMain.TabPages[selectedIndex] as CommonTabPage;

            SetCurrentTabCtrl(_TabCtrlSub);    //SetCurrentTabControllerSub();
            CloneOpenPage(page);
            SetCurrentTabCtrl(_TabCtrlMain);   //SetCurrentTabControllerMain();

            if (_TabCtrlSub.TabPages.Count <= 0)
            {
                return false;
            }

            _CurrentTabCtrl.SelectedTab.Select();

            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void SplitUD()
        {
            if (Split() == false)
            {
                return;
            }

            if (_SplitState == SplitState.UD)
            {
                InactiveTabCtrl.State = (InactiveTabCtrl.State == TabCtrl.ControlState.Normal) ?
                    TabCtrl.ControlState.Minimized : TabCtrl.ControlState.Normal;
                return;
            }

            _SplitState = SplitState.UD;

            _TabCtrlMain.Alignment = TabCtrl.ControlAlignment.Top;
            _TabCtrlSub.Alignment = TabCtrl.ControlAlignment.Bottom;

            _TabSplitterRatioUpdate = false;

            try
            {
                _editSplitContainer.Orientation = Orientation.Horizontal;
                _editSplitContainer.Panel1Collapsed = false;
                _editSplitContainer.Panel2Collapsed = false;
                _editSplitContainer.IsSplitterFixed = false;
            }
            finally { _TabSplitterRatioUpdate = true; }

            UpdateTabSplitterDistance();
            VisibleTabControllerMinimizeButton(true);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void SplitLR()
        {
            if (Split() == false)
            {
                return;
            }

            if (_SplitState == SplitState.LR)
            {
                InactiveTabCtrl.State = (InactiveTabCtrl.State == TabCtrl.ControlState.Normal) ?
                    TabCtrl.ControlState.Minimized : TabCtrl.ControlState.Normal;
                return;
            }

            _SplitState = SplitState.LR;

            _TabCtrlMain.Alignment = TabCtrl.ControlAlignment.Left;
            _TabCtrlSub.Alignment = TabCtrl.ControlAlignment.Right;

            _TabSplitterRatioUpdate = false;
            try
            {
                _editSplitContainer.Orientation = Orientation.Vertical;
                _editSplitContainer.Panel1Collapsed = false;
                _editSplitContainer.Panel2Collapsed = false;
                _editSplitContainer.IsSplitterFixed = false;
            }
            finally { _TabSplitterRatioUpdate = true; }

            UpdateTabSplitterDistance();
            VisibleTabControllerMinimizeButton(true);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void SplitNone(bool judge)
        {
            _SplitState = SplitState.None;

            if (_CurrentTabCtrl == _TabCtrlMain)
            {
                if (judge == true)
                {
                    _TabCtrlSub.ClosePageBySelfJudgement();
                    _TabCtrlSub.CloseAllTabPagesWithoutClosedEvent();
                }

                _editSplitContainer.Panel2Collapsed = true;
                _editSplitContainer.IsSplitterFixed = true;
            }

            if (_CurrentTabCtrl == _TabCtrlSub)
            {
                if (judge == true)
                {
                    _TabCtrlMain.ClosePageBySelfJudgement();
                    _TabCtrlMain.CloseAllTabPagesWithoutClosedEvent();
                }

                MoveAllTabPagesSubToMain();
                _editSplitContainer.Panel2Collapsed = true;
                _editSplitContainer.IsSplitterFixed = true;
            }

            _TabCtrlMain.State = TabCtrl.ControlState.Normal;
            _TabCtrlSub.State = TabCtrl.ControlState.Normal;

            SetCurrentTabCtrl(_TabCtrlMain);
            VisibleTabControllerMinimizeButton(false);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void SplitNone()
        {
            SplitNone(true);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool CloneOpenPage(CommonTabPage page)
        {
            CommonTabPage newPage = null;
            SoundSetPanel soundSetPanel = null;
            SoundSetPanel newSoundSetPanel = null;
            SoundSetDocument soundSetDocument = null;
            BankPanel bankPanel = null;
            BankDocument bankDocument = null;

            if ((soundSetPanel = page.Panel as SoundSetPanel) != null)
            {
                soundSetDocument = soundSetPanel.SoundSetDocument;
                newPage = AddPage(soundSetDocument.SoundSet);
                newSoundSetPanel = newPage.Panel as SoundSetPanel;
                newSoundSetPanel.Show(soundSetPanel.CurrentItem as SoundSetItem);
                return true;
            }

            if ((bankPanel = page.Panel as BankPanel) != null)
            {
                bankDocument = bankPanel.BankDocument;
                newPage = AddPage(bankDocument.Bank, bankPanel.SoundSetBank, true);
            }

            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void MoveAllTabPagesSubToMain()
        {
            CommonTabPage page = null;
            int count = 0;
            int selectedIndex = 0;

            count = _TabCtrlSub.TabPages.Count;
            selectedIndex = _TabCtrlSub.SelectedIndex;

            _TabCtrlSub.SuspendClosedEvent = true;

            for (int loop = 0; loop < count; loop++)
            {
                page = _TabCtrlSub.TabPages[0] as CommonTabPage;
                _TabCtrlSub.TabPages.RemoveAt(0);
                _TabCtrlMain.TabPages.Add(page);
            }

            _TabCtrlSub.SuspendClosedEvent = false;
            _TabCtrlMain.SelectedIndex = selectedIndex;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void RemoveGarbagePages()
        {
            foreach (CommonTabPage page in OpenedAllTabPages())
            {
                if (page.Panel is SoundSetPanel)
                {
                    SoundSetDocument targetDocument = (page.Panel as SoundSetPanel).SoundSetDocument;

                    bool closePage = true;

                    foreach (SoundSetDocument soundSetDocument in ProjectService.SoundSetDocuments)
                    {
                        if (soundSetDocument.Resource.Key != targetDocument.Resource.Key) { continue; }
                        closePage = false;
                        break;
                    }

                    if (!closePage) { continue; }
                }
                else if (page.Panel is BankPanel)
                {
                    BankDocument targetDocument = (page.Panel as BankPanel).BankDocument;

                    bool closePage = true;

                    foreach (SoundSetBankBase bank in ProjectService.SoundSetBanks)
                    {
                        if (bank.FilePath != targetDocument.Resource.Key) { continue; }
                        closePage = false;
                        break;
                    }

                    if (!closePage) { continue; }
                }
                else
                {
                    continue;
                }

                (page.Parent as TabCtrl).TabPages.Remove(page);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void CheckTabPageCondition()
        {
            if (_TabCtrlMain.TabPages.Count <= 0 &&
                _TabCtrlSub.TabPages.Count <= 0)
            {
                ProjectPanel.Focus();
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool RotateSelectingTabPage(TabCtrl tabCtrl, bool forward)
        {
            if (tabCtrl.TabPages.Count <= 0)
            {
                return false;
            }

            SetCurrentTabCtrl(tabCtrl);
            tabCtrl.Focus();

            if (forward == true)
            {
                ActivateTabPage(0);
            }
            else
            {
                ActivateLastTabPage();
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void RotateSelectingTabPage(bool forward)
        {
            TabCtrl tabCtrl = null;
            int index = -1;

            tabCtrl = InactiveTabCtrl;

            index = _CurrentTabCtrl.SelectedIndex;
            if (forward == true)
            {
                if (index + 1 < _CurrentTabCtrl.TabPages.Count)
                {
                    ActivateTabPage(index + 1);
                    return;
                }

                if (tabCtrl.TabPages.Count <= 0)
                {
                    ActivateTabPage(0);
                    return;
                }
            }
            else
            {

                if (index - 1 >= 0)
                {
                    ActivateTabPage(index - 1);
                    return;
                }

                if (tabCtrl.TabPages.Count <= 0)
                {
                    ActivateTabPage(_CurrentTabCtrl.TabPages.Count - 1);
                    return;
                }
            }

            RotateSelectingTabPage(tabCtrl, forward);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnFileDropingTabPage(object sender, EventArgs e)
        {
            SetCurrentTabCtrl(sender as TabCtrl);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnRotateSelectingTabPage(object sender, TabCtrl.RotateSelectionEventArgs e)
        {
            if (RotateSelectingTabPage(sender as TabCtrl, e.Forward) == true)
            {
                e.Cancel = true;
            }
        }

        ///--------------------------------
        /// <summary>
        /// ページの選択が変更された時に呼ばれる
        /// </summary>
        private void OnDeselectingTabPageChanged(object sender, NTabControlCancelEventArgs e)
        {
            //DeactivatedPage( sender as TabCtrl);
        }

        ///--------------------------------
        /// <summary>
        /// ページの選択が変更された時に呼ばれる
        /// </summary>
        private void OnSelectedTabPageChanged(object sender, NTabControlEventArgs e)
        {
            if (this.projectOpened == false)
            {
                CommonTabPage page = e.TabPage as CommonTabPage;
                if (page != null)
                {
                    Component component = null;
                    CommonPanel panel = page.Panel as CommonPanel;
                    if (panel is SoundSetPanel == true)
                    {
                        SoundSetPanel soundSetPanel = panel as SoundSetPanel;
                        component = soundSetPanel.CurrentItem;
                    }
                    else if (panel is BankPanel == true)
                    {
                        BankPanel bankPanel = panel as BankPanel;
                        component = bankPanel.SoundSetBank;
                    }

                    if (component != null)
                    {
                        this.HistoryAdd(component);
                    }
                }
            }
            //ActivatedPage( sender as TabCtrl);
            //SoundMaker.FindWindow.UpdateInnerData();
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnTabPageStateChanged(object sender, EventArgs e)
        {
            if (sender == _TabCtrlMain)
            {
                ChangeTabCtrlState(_TabCtrlMain, _TabCtrlSub);
            }

            if (sender == _TabCtrlSub)
            {
                ChangeTabCtrlState(_TabCtrlSub, _TabCtrlMain);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnTabCtrlDoubleClicked(object sender, EventArgs e)
        {
            TabCtrl tabCtrl = sender as TabCtrl;

            if (tabCtrl.State == TabCtrl.ControlState.Normal)
            {
                SetCurrentTabCtrl(tabCtrl);
                ActivateTabPage(tabCtrl.SelectedIndex);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnTabSplitterDoubleClick(object sender, EventArgs e)
        {
            TabCtrl mController = null;
            TabCtrl sController = null;

            if (_CurrentTabCtrl == _TabCtrlMain)
            {
                mController = _TabCtrlMain;
                sController = _TabCtrlSub;
            }
            else
            {

                mController = _TabCtrlSub;
                sController = _TabCtrlMain;
            }

            //
            switch (_SplitState)
            {
                case SplitState.UD:
                case SplitState.LR:
                    if (mController.State == TabCtrl.ControlState.Normal &&
                        sController.State == TabCtrl.ControlState.Normal)
                    {
                        mController.State = TabCtrl.ControlState.Normal;
                        sController.State = TabCtrl.ControlState.Minimized;
                    }
                    else
                    {

                        mController.State = TabCtrl.ControlState.Normal;
                        sController.State = TabCtrl.ControlState.Normal;
                    }

                    UpdateTabSplitterDistance();

                    SetCurrentTabCtrl(mController);
                    _CurrentTabCtrl.Focus();
                    ActivateTabPage(_CurrentTabCtrl.SelectedIndex);
                    break;

                case SplitState.None:
                    break;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnTabSplitterMoving(object sender, SplitterCancelEventArgs e)
        {
            if (_TabCtrlMain.State == TabCtrl.ControlState.Minimized ||
                 _TabCtrlSub.State == TabCtrl.ControlState.Minimized)
            {
                e.Cancel = true;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnTabSplitterMoved(object sender, EventArgs e)
        {
            int d = 0;

            if (_TabCtrlMain.State == TabCtrl.ControlState.Normal &&
                _TabCtrlSub.State == TabCtrl.ControlState.Normal &&
                _TabSplitterRatioUpdate == true)
            {

                d = _editSplitContainer.Orientation == Orientation.Horizontal ?
                    _editSplitContainer.Height : _editSplitContainer.Width;
                TabSplitterRatio = (double)_editSplitContainer.SplitterDistance / (double)d;
            }
            else
            {

                UpdateTabSplitterDistance(false);
            }
        }

        ///--------------------------------
        /// <summary>
        /// タブコントロールのページが閉じられた時に呼ばれる
        /// </summary>
        private void OnClosedTabPage(object sender, EventArgs e)
        {
            this.oldSelectedComponent = null;

            if (sender == _TabCtrlMain)
            {
                if (_TabCtrlMain.TabPages.Count <= 0)
                {
                    SetCurrentTabCtrl(_TabCtrlMain);
                    SplitNone(false);
                    MoveAllTabPagesSubToMain();
                    CheckTabPageCondition();
                }
            }

            if (sender == _TabCtrlSub)
            {
                if (_TabCtrlSub.TabPages.Count <= 0)
                {
                    SetCurrentTabCtrl(_TabCtrlMain);
                    SplitNone(false);
                    CheckTabPageCondition();
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private DocumentReference GetDocumentReferenceOfSoundSet(SoundSet soundSet)
        {
            foreach (SoundSetDocument soundSetDocument in SoundSetDocuments)
            {
                if (soundSetDocument.SoundSet == soundSet)
                {
                    return Application.DocumentService.OpenDocument(soundSetDocument.Resource);
                }
            }
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private DocumentReference GetDocumentReferenceOfBank(Bank bank)
        {
            foreach (BankDocument bankDocument in BankDocuments)
            {
                if (bankDocument.Bank == bank)
                {
                    return Application.DocumentService.OpenDocument(bankDocument.Resource);
                }
            }
            return null;
        }

        ///--------------------------------
        /// <summary>
        /// 既に開かれているサウンドセットタブページの検索
        /// </summary>
        private CommonTabPage FindOpenedSoundSetTabPage(SoundSet soundSet, TabCtrl tabCtrl)
        {
            SoundSetPanel panel = null;

            foreach (CommonTabPage page in OpnedSoundSetTabPage(tabCtrl))
            {
                panel = page.Panel as SoundSetPanel;
                if (panel.SoundSet == soundSet)
                {
                    return page;
                }
            }
            return null;
        }

        ///--------------------------------
        /// <summary>
        /// 既に開かれているバンクタブページの検索
        /// </summary>
        private CommonTabPage FindOpenedBankTabPage(Bank bank, TabCtrl tabCtrl)
        {
            BankPanel panel = null;

            foreach (CommonTabPage page in OpenedBankTabPage(tabCtrl))
            {
                panel = page.Panel as BankPanel;
                if (panel.Bank == bank)
                {
                    return page;
                }
            }
            return null;
        }

        ///
        private CommonTabPage FindOpenedBankTabPage(TabCtrl tabCtrl, string filePath)
        {
            BankPanel panel = null;

            foreach (CommonTabPage page in OpenedBankTabPage(tabCtrl))
            {
                panel = page.Panel as BankPanel;
                if (panel.BankDocument.Resource.Key == filePath)
                {
                    return page;
                }
            }
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        private bool ShowBankPageByComponent(Component component)
        {
            return ShowBankPageByComponent(component, false);
        }

        private bool ShowBankPageByComponent(Component component, bool open)
        {
            ImaginaryInstrument imaginaryInstrument = null;
            CommonTabPage page = null;
            BankPanel bankPanel = null;
            CommonListCtrl listCtrl = null;
            IListItem listItem = null;
            Bank bank = null;

            if (component is ImaginaryInstrument)
            {
                imaginaryInstrument = component as ImaginaryInstrument;
                return ShowBankPageByFilePath(imaginaryInstrument.BankFilePath,
                                               imaginaryInstrument.Name);
            }

            //
            if ((bank = GetBank(component)) == null)
            {
                if (open == true &&
                    component is SoundSetBankBase == true)
                {
                    SoundSetBankBase soundSetBank = component as SoundSetBankBase;
                    return ShowBankPageByFilePath(soundSetBank.FilePath, string.Empty);
                }

                return false;
            }

            if ((page = FindPage(bank)) == null)
            {
                return false;
            }

            ActivatePage(page);

            //
            bankPanel = page.Panel as BankPanel;

            //
            listCtrl = bankPanel.CurrentListCtrl;
            if ((listItem = listCtrl.GetItem(component)) != null)
            {
                listCtrl.ShowByItem(listItem);
            }

            bankPanel.ActivatePanel(false, true);
            Activate();

            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool ShowSoundSetPageByComponent(Component component)
        {
            return ShowSoundSetPageByComponent(component, ShowComponentMode.Auto);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool ShowSoundSetPageByComponent(Component component, ShowComponentMode showMode)
        {
            SoundSet soundSet = GetSoundSet(component);
            if (null == soundSet) { return false; }

            CommonTabPage page = FindPage(soundSet);
            if (null == page)
            {
                page = AddPage(soundSet);
            }
            else
            {
                ActivatePage(page);
            }


            //
            SoundSetPanel soundSetPanel = page.Panel as SoundSetPanel;
            SoundSetItem targetItem = GetSoundSetItemPack(component);


            bool isSingleListMode = (showMode == ShowComponentMode.SingleList);

            if (showMode == ShowComponentMode.Auto && null != component)
            {
                isSingleListMode = (soundSetPanel.CurrentItem == component.Parent &&
                                    IsSingleListTarget(soundSetPanel.CurrentItem) &&
                                    soundSetPanel.IsSingleList);
            }

            //
            if (isSingleListMode)
            {
                if (targetItem is WaveSoundSetPack ||
                    targetItem is SequenceSoundSetPack)
                {
                    targetItem = component.Parent as SoundSetItem;
                }
            }

            //
            soundSetPanel.Show(targetItem);

            //
            CommonListCtrl listCtrl = null;
            IListItem listItem = null;

            if (targetItem is StreamSoundPack ||
                targetItem is WaveSoundSetPack ||
                targetItem is SequenceSoundSetPack ||
                targetItem is GroupPack)
            {

                listCtrl = soundSetPanel.ListCtrlMain;
                if ((listItem = listCtrl.GetItem(component)) != null)
                {
                    listCtrl.ShowByItem(listItem);
                }
                else
                {

                    listItem = listCtrl.GetItem(component.Parent);
                    listCtrl.ShowByItem(listItem);

                    listCtrl = soundSetPanel.ListCtrlSub;
                    listItem = listCtrl.GetItem(component);
                    listCtrl.ShowByItem(listItem);
                }
            }
            else
            {

                listCtrl = soundSetPanel.ListCtrlMain;
                if ((listItem = listCtrl.GetItem(component)) != null)
                {
                    listCtrl.ShowByItem(listItem);
                }
            }

            soundSetPanel.ActivatePanel(false, IsMainListTarget(targetItem, component));
            Activate();

            return true;
        }

        private bool IsSingleListTarget(Component component)
        {
            if (component is WaveSoundBase) { return true; }
            if (!(component is SequenceSoundBase)) { return true; }
            return false;
        }

        private bool IsMainListTarget(SoundSetItem parentItem, Component component)
        {
            if (null == parentItem) { return true; }
            if (null == component) { return true; }

            if (component is StreamSoundTrackBase) { return false; }
            if (component is WaveSoundBase) { return (parentItem is WaveSoundSetBase); }
            if (component is GroupItemBase) { return (parentItem is GroupBase); }

            if (component is SequenceSoundBase)
            {
                if (component.Parent is SequenceSoundPack) { return true; }
                return (parentItem is SequenceSoundSetBase);
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private SoundSet GetSoundSet(Component component)
        {
            while (component != null)
            {
                if (component is SoundSet)
                {
                    return component as SoundSet;
                }
                component = component.Parent;
            }
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Bank GetBank(Component component)
        {
            while (component != null)
            {
                if (component is Bank)
                {
                    return component as Bank;
                }
                component = component.Parent;
            }
            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private SoundSetItem GetSoundSetItemPack(Component component)
        {
            if (component is StreamSoundPack ||
                component is WaveSoundSetPack ||
                component is SequenceSoundPack ||
                component is SequenceSoundSetPack ||
                component is SoundSetBankPack ||
                component is WaveArchivePack ||
                component is PlayerPack ||
                component is GroupPack)
            {
                return component as SoundSetItem;
            }

            component = component.Parent;

            if (component is WaveSoundSetBase ||
                component is SequenceSoundSetBase)
            {
                return component as SoundSetItem;
            }

            return GetSoundSetItemPack(component);
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private TabCtrl _CurrentTabCtrl = null;

        private void SetCurrentTabCtrl(TabCtrl ctrl)
        {
            if (null != ctrl && TabCtrl.ControlState.Minimized == ctrl.State) { return; }

            if (_CurrentTabCtrl != null)
            {
                _CurrentTabCtrl.SelectionExpression = false;
            }

            _CurrentTabCtrl = ctrl;

            if (_CurrentTabCtrl != null)
            {
                _CurrentTabCtrl.SelectionExpression = true;
            }
        }

        /// <summary>
        /// ドッキングフォームコンテナの状態を更新します。
        /// </summary>
        /// <param name="dockContainer">対象ドッキングフォームコンテナ。</param>
        /// <param name="spliter">対象スプリッタ。</param>
        /// <param name="minimizePanel1">Pane1 が最小化対象の場合は true、それ以外の場合は false。</param>
        private void UpdateDockContainer(DockingToolContainer dockContainer, SplitContainer spliter, bool minimizePanel1)
        {
            Debug.Assert(null != dockContainer, "unexpected error");
            Debug.Assert(null != spliter, "unexpected error");

            bool newState = (0 == dockContainer.DockedCount);

            if (minimizePanel1)
            {
                if (newState == spliter.Panel1Collapsed) { return; }
                spliter.Panel1Collapsed = newState;
                spliter.IsSplitterFixed = newState;
            }
            else
            {
                if (newState == spliter.Panel2Collapsed) { return; }
                spliter.Panel2Collapsed = newState;
                spliter.IsSplitterFixed = newState;
            }

            if (!newState)
            {
                dockContainer.SelectedDockedPage.Select();
            }
        }

        private void ApplyDockingContainersConfiguration(XmlFrameWindow xmlMainWindow)
        {
            if (null == xmlMainWindow) { throw new ArgumentNullException("xmlMainWindow"); }

            IDictionary<string, XmlDockingHostControl> dockingHosts = xmlMainWindow.CreateDockingHostDictionary();
            if (!dockingHosts.ContainsKey("MainWindowLeft") ||
                 !dockingHosts.ContainsKey("MainWindowRight") ||
                 !dockingHosts.ContainsKey("MainWindowBottom")) { return; }

            IDictionary<string, XmlSplitControl> splitControls = xmlMainWindow.CreateSplitDictionary();
            if (!splitControls.ContainsKey("MainWindowLeft") ||
                 !splitControls.ContainsKey("MainWindowRight") ||
                 !splitControls.ContainsKey("MainWindowBottom")) { return; }

            ApplyDockingContainerConfiguration(_leftDockingToolContainer, dockingHosts["MainWindowLeft"]);
            ApplyDockingContainerConfiguration(_rightDockingToolContainer, dockingHosts["MainWindowRight"]);
            ApplyDockingContainerConfiguration(_bottomDockingToolContainer, dockingHosts["MainWindowBottom"]);

            // スプリットコントロール、ドッキングホスト情報
            foreach (XmlSplitControl xmlSplit in splitControls.Values)
            {

                DockingToolContainer dockingToolContainer;

                switch (xmlSplit.Name)
                {
                    case "MainWindowLeft":
                        LeftSplitterPosition = xmlSplit.SplitterPosition;
                        dockingToolContainer = _leftDockingToolContainer;
                        break;

                    case "MainWindowRight":
                        RightSplitterPosition = xmlSplit.SplitterPosition;
                        dockingToolContainer = _rightDockingToolContainer;
                        break;

                    case "MainWindowBottom":
                        BottomSplitterPosition = xmlSplit.SplitterPosition;
                        dockingToolContainer = _bottomDockingToolContainer;
                        break;

                    default:
                        continue;
                }

                if (0 == dockingToolContainer.Pages.Count) { continue; }

                XmlDockingHostControl xmlDockingHost = dockingHosts[xmlSplit.Name];

                if (xmlDockingHost.Minimized)
                {
                    dockingToolContainer.Minimize();
                }

                int selectedIndex = xmlDockingHost.SelectedIndex;

                if (selectedIndex < 0)
                {
                    selectedIndex = 0;
                }
                else if (selectedIndex >= dockingToolContainer.Pages.Count)
                {
                    selectedIndex = dockingToolContainer.Pages.Count - 1;
                }

                dockingToolContainer.SelectedDockedPage = dockingToolContainer.Pages[selectedIndex];
            }

            UpdateDockContainer(_leftDockingToolContainer, _leftSplitContainer, true);
            UpdateDockContainer(_rightDockingToolContainer, _rightSplitContainer, false);
            UpdateDockContainer(_bottomDockingToolContainer, _bottomSplitContainer, false);
        }

        private void ApplyDockingContainerConfiguration(DockingToolContainer dockingContainer, XmlDockingHostControl xmlDockingHost)
        {
            if (null == dockingContainer) { throw new ArgumentNullException("dockingFormContainer"); }
            if (null == xmlDockingHost) { throw new ArgumentNullException("xmlDockingHost"); }

            if (null == xmlDockingHost.ToolPageNames) { return; }
            if (null == xmlDockingHost.ToolPageNames.ToolPageName) { return; }

            foreach (string pageName in xmlDockingHost.ToolPageNames.ToolPageName)
            {
                if (!Application.AppConfiguration.ToolPages.ContainsKey(pageName)) { continue; }
                XmlToolPage xmlPage = Application.AppConfiguration.ToolPages[pageName];

                if (!ToolPages.ContainsKey(pageName)) { continue; }
                IToolWindowPage page = ToolPages[pageName];

                ApplyToolPageConfiguration(page, xmlPage);
                BindToolPage(page, dockingContainer);
            }
        }

        private void ApplyToolPageConfiguration(IToolWindowPage page, XmlToolPage xmlPage)
        {
            if (null == page) { throw new ArgumentNullException("page"); }
            if (null == xmlPage) { throw new ArgumentNullException("xmlPage"); }

            page.DockState = xmlPage.Docked ? DockingPageState.Docked : DockingPageState.Floating;

            // フローティングウィンドウ位置情報
            string toolWindowName = xmlPage.Name + "ToolWindow";

            if (Application.AppConfiguration.Windows.ContainsKey(toolWindowName))
            {

                XmlWindow xmlToolWindow = Application.AppConfiguration.Windows[toolWindowName];

                if (-1 < xmlToolWindow.X && -1 < xmlToolWindow.Y &&
                    0 < xmlToolWindow.Width && 0 < xmlToolWindow.Height)
                {
                    page.FloatingBounds = xmlToolWindow.Bounds;
                }

            }

            // Remove & Add Page の際に、非表示になるので、BindToolPage の後で行う
            page.DockVisible = xmlPage.Visible;
        }

        private void ExtractDockingContainersConfiguration(XmlFrameWindow xmlMainWindow)
        {
            if (null == xmlMainWindow) { throw new ArgumentNullException("xmlMainWindow"); }

            // スプリットコントロール情報
            IDictionary<string, XmlSplitControl> splitControls = xmlMainWindow.CreateSplitDictionary();
            splitControls["MainWindowLeft"].SplitterPosition = LeftSplitterPosition;
            splitControls["MainWindowRight"].SplitterPosition = RightSplitterPosition;
            splitControls["MainWindowBottom"].SplitterPosition = BottomSplitterPosition;
            splitControls["MainWindowLeft"].SplitterPositionSpecified = true;
            splitControls["MainWindowRight"].SplitterPositionSpecified = true;
            splitControls["MainWindowBottom"].SplitterPositionSpecified = true;

            // ドッキングホスト情報
            IDictionary<string, XmlDockingHostControl> dockingHosts = xmlMainWindow.CreateDockingHostDictionary();
            ExtractDockingContainerConfiguration(_leftDockingToolContainer, dockingHosts["MainWindowLeft"]);
            ExtractDockingContainerConfiguration(_rightDockingToolContainer, dockingHosts["MainWindowRight"]);
            ExtractDockingContainerConfiguration(_bottomDockingToolContainer, dockingHosts["MainWindowBottom"]);
        }

        private void ExtractDockingContainerConfiguration(DockingFormContainer dockingFormContainer, XmlDockingHostControl xmlDockingHost)
        {
            if (null == dockingFormContainer) { throw new ArgumentNullException("dockingFormContainer"); }
            if (null == xmlDockingHost) { throw new ArgumentNullException("xmlDockingHost"); }

            List<string> toolPageNames = new List<string>();

            int pageIndex = 0;

            foreach (IToolWindowPage page in dockingFormContainer.Pages)
            {
                ExtractToolPageConfiguration(page);

                if (dockingFormContainer.SelectedDockedPage == page)
                {
                    xmlDockingHost.SelectedIndex = pageIndex;
                }

                toolPageNames.Add(page.PageID);

                pageIndex++;
            }

            if (null == xmlDockingHost.ToolPageNames)
            {
                xmlDockingHost.ToolPageNames = new XmlToolPageNames();
            }

            xmlDockingHost.Minimized = dockingFormContainer.IsMinimized;
            xmlDockingHost.ToolPageNames.ToolPageName = toolPageNames.ToArray();
        }

        private void ExtractToolPageConfiguration(IToolWindowPage page)
        {
            if (null == page) { throw new ArgumentNullException("page"); }

            // ドッキング情報
            if (!Application.AppConfiguration.ToolPages.ContainsKey(page.PageID))
            {
                Application.AppConfiguration.ToolPages.Add(page.PageID,
                                                         new XmlToolPage() { Name = page.PageID });
            }

            XmlToolPage xmlPage = Application.AppConfiguration.ToolPages[page.PageID];

            xmlPage.Docked = (DockingPageState.Docked == page.DockState);
            xmlPage.Visible = page.DockVisible;

            if (page.DockContainer == _leftDockingToolContainer)
            {
                xmlPage.DockedHostName = "MainWindowLeft";
            }
            else if (page.DockContainer == _rightDockingToolContainer)
            {
                xmlPage.DockedHostName = "MainWindowRight";
            }
            else if (page.DockContainer == _bottomDockingToolContainer)
            {
                xmlPage.DockedHostName = "MainWindowBottom";
            }

            // フローティングウィンドウ位置情報
            string toolWindowName = page.PageID + "ToolWindow";

            if (!Application.AppConfiguration.Windows.ContainsKey(toolWindowName))
            {
                Application.AppConfiguration.Windows.Add(toolWindowName,
                                                       new XmlToolWindow() { Name = toolWindowName });
            }

            Application.AppConfiguration.Windows[toolWindowName].Bounds = page.FloatingBounds;
        }

        private void ApplyCommandBarsConfiguration()
        {
            TopToolStripPanel.Controls.Clear();
            LeftToolStripPanel.Controls.Clear();
            BottomToolStripPanel.Controls.Clear();
            RightToolStripPanel.Controls.Clear();

            ToolStripList toolStrips = new ToolStripList();
            QueryToolStrips(toolStrips);

            // XmlCommandBars に項目がない場合は先に追加しておく
            foreach (ToolStrip toolStrip in toolStrips.Reverse())
            {

                if (Application.AppConfiguration.CommandBars.
                      ContainsKey(toolStrip.Tag as string)) { continue; }

                TopToolStripPanel.Join(toolStrip);

            }

            // ToolStripPanel に追加する際、行数の若い順、奥行座標の大きい順にしないと
            // 正しく復元できないので、ここで並び替え。
            XmlCommandBarSortedList xmlCommandBars = new XmlCommandBarSortedList();
            xmlCommandBars.AddRange(Application.AppConfiguration.CommandBars.Values.ToArray());

            foreach (XmlCommandBar xmlCommandBar in xmlCommandBars.Values)
            {
                if (!toolStrips.Contains(xmlCommandBar.Name)) { continue; }
                ApplyCommandBarConfiguration(xmlCommandBar, toolStrips[xmlCommandBar.Name]);
            }
        }

        private void ExtractCommandBarsConfiguration()
        {
            ToolStripList toolStrips = new ToolStripList();
            QueryToolStrips(toolStrips);

            foreach (ToolStrip toolStrip in toolStrips)
            {
                ExtractCommandBarConfiguration(toolStrip);
            }
        }

        private void ApplyCommandBarConfiguration(XmlCommandBar xmlCommandBar, ToolStrip toolStrip)
        {
            if (null == xmlCommandBar) { throw new ArgumentNullException("xmlCommandBar"); }
            if (null == toolStrip) { throw new ArgumentNullException("toolStrip"); }

            GetToolStripPanelFromXmlData(xmlCommandBar.CommandBarContainer).
                                  Join(toolStrip, xmlCommandBar.X, xmlCommandBar.Y);
        }

        private void ExtractCommandBarConfiguration(ToolStrip toolStrip)
        {
            if (null == toolStrip) { throw new ArgumentNullException("toolStrip"); }

            string toolStripName = toolStrip.Tag as string;

            if (null == toolStripName)
            {
                Debug.Fail("nameless toolStrip.");
                return;
            }

            if (Application.AppConfiguration.CommandBars.ContainsKey(toolStripName))
            {
                Application.AppConfiguration.CommandBars.Remove(toolStripName);
            }

            Application.AppConfiguration.CommandBars.Add(
                toolStripName,
                new XmlCommandBar()
                {
                    Name = toolStripName,
                    X = toolStrip.Location.X,
                    Y = toolStrip.Location.Y,
                    CommandBarContainer = GetXmlToolStripContainer(toolStrip),
                });
        }

        private XmlCommandBarContainer GetXmlToolStripContainer(ToolStrip toolStrip)
        {
            if (null == toolStrip) { throw new ArgumentNullException("toolStrip"); }

            if (TopToolStripPanel == toolStrip.Parent)
            {
                return XmlCommandBarContainer.Top;
            }
            if (LeftToolStripPanel == toolStrip.Parent)
            {
                return XmlCommandBarContainer.Left;
            }
            if (BottomToolStripPanel == toolStrip.Parent)
            {
                return XmlCommandBarContainer.Bottom;
            }
            if (RightToolStripPanel == toolStrip.Parent)
            {
                return XmlCommandBarContainer.Right;
            }

            return XmlCommandBarContainer.Top;
        }

        private ToolStripPanel GetToolStripPanelFromXmlData(XmlCommandBarContainer xmlContainer)
        {
            switch (xmlContainer)
            {
                case XmlCommandBarContainer.Top:
                    return TopToolStripPanel;

                case XmlCommandBarContainer.Left:
                    return LeftToolStripPanel;

                case XmlCommandBarContainer.Bottom:
                    return BottomToolStripPanel;

                case XmlCommandBarContainer.Right:
                    return RightToolStripPanel;
            }

            return TopToolStripPanel;
        }

        private void ApplySoundSetConfigurations()
        {
            ResetSoundSetConfigurations();

            foreach (SoundSetDocument document in ProjectService.SoundSetDocuments)
            {

                if (!ProjectConfiguration.DocumentViews.ContainsKey(document.Resource.Key)) { continue; }

                XmlSoundSetDocumentView xmlSoundSetView =
                    ProjectConfiguration.DocumentViews[document.Resource.Key]
                    as XmlSoundSetDocumentView;

                document.SoundSet.HideEmptyNode = xmlSoundSetView.HideEmptyNode;

            }
        }

        private void ExtractSoundSetConfigurations()
        {
            foreach (SoundSetDocument document in ProjectService.SoundSetDocuments)
            {

                if (!ProjectConfiguration.DocumentViews.ContainsKey(document.Resource.Key)) { continue; }

                XmlSoundSetDocumentView xmlSoundSetView =
                    ProjectConfiguration.DocumentViews[document.Resource.Key]
                    as XmlSoundSetDocumentView;

                xmlSoundSetView.HideEmptyNode = document.SoundSet.HideEmptyNode;

            }

            CollectGarbageSoundSetDocumentConfigurations();
            CollectGarbageBankDocumentConfigurations();
        }

        private void ResetSoundSetConfigurations()
        {
            foreach (SoundSetDocument document in ProjectService.SoundSetDocuments)
            {

                if (ProjectConfiguration.DocumentViews.ContainsKey(document.Resource.Key)) { continue; }

                ProjectConfiguration.DocumentViews.Add(
                    document.Resource.Key,
                    XmlSoundSetDocumentView.CreateInstance(document.Resource.Key));

            }
        }

        private void CollectGarbageSoundSetDocumentConfigurations()
        {
            List<XmlDocumentView> garbages = new List<XmlDocumentView>();

            foreach (XmlDocumentView xmlView in ProjectConfiguration.DocumentViews.Values)
            {

                if (!(xmlView is XmlSoundSetDocumentView)) { continue; }

                if ((from Document document in ProjectService.SoundSetDocuments
                     where document.Resource.Key == xmlView.FilePath
                     select document).FirstOrDefault() == null)
                {
                    garbages.Add(xmlView);
                }

            }

            foreach (XmlDocumentView garbage in garbages)
            {
                ProjectConfiguration.DocumentViews.Remove(garbage.FilePath);
            }
        }

        private void CollectGarbageBankDocumentConfigurations()
        {
            List<XmlDocumentView> garbages = new List<XmlDocumentView>();

            foreach (XmlDocumentView xmlView in ProjectConfiguration.DocumentViews.Values)
            {

                if (!(xmlView is XmlBankDocumentView)) { continue; }

                if ((from SoundSetBankBase bank in ProjectService.SoundSetBanks
                     where bank.FilePath == xmlView.FilePath
                     select bank).FirstOrDefault() == null)
                {
                    garbages.Add(xmlView);
                }

            }

            foreach (XmlDocumentView garbage in garbages)
            {
                ProjectConfiguration.DocumentViews.Remove(garbage.FilePath);
            }
        }

        private void ApplyFindResultConfigurations()
        {
            XmlFindResult[] results = ProjectConfiguration.SoundProject.FindResultView.FindResults;
            Dictionary<string, FindResultPanel2> panelsDictionary = new Dictionary<string, FindResultPanel2>();

            foreach (FindResultPanel2 panel in this.FindResultPanels)
            {
                panelsDictionary.Add(panel.TitleName + panel.TitleNumber.ToString(), panel);
            }

            if (results != null && 0 < panelsDictionary.Count)
            {
                foreach (XmlFindResult result in results)
                {
                    string key = result.Name + result.Number;
                    if (panelsDictionary.ContainsKey(key) == true)
                    {
                        FindResultPanel2 panel = panelsDictionary[key];
                        panel.SetHeaderColumnSetting(result.List);
                    }
                }
            }

            ProjectConfiguration.SoundProject.FindResultView.FindResults = null;
        }

        private void ExtractFindResultConfigurations()
        {
            FindResultPanel2[] panels = this.FindResultPanels;

            if (ProjectConfiguration.SoundProject.FindResultView == null)
            {
                ProjectConfiguration.SoundProject.FindResultView = new XmlFindResultView();
            }

            if (panels == null)
            {
                ProjectConfiguration.SoundProject.FindResultView.FindResults = null;
            }
            else
            {
                List<XmlFindResult> results = new List<XmlFindResult>();
                foreach (FindResultPanel2 panel in panels)
                {
                    XmlList list = panel.GetHeaderColumnSetting();
                    XmlFindResult result = new XmlFindResult();
                    result.Name = panel.TitleName;
                    result.Number = panel.TitleNumber.ToString();
                    result.List = list;
                    results.Add(result);
                }
                ProjectConfiguration.SoundProject.FindResultView.FindResults = results.ToArray();
            }
        }

        private void UpdateMruProjects()
        {
            CommandBarPopupItem mruProjectPopupItem = _mainMenuAdapter.CommandBar.FindFirstItem("MruProjects@PopupItem")
                                                      as CommandBarPopupItem;
            if (null == mruProjectPopupItem)
            {
                Debug.Fail("mru project popup item not found.");
                return;
            }

            if (0 < mruProjectPopupItem.Groups.Count)
            {
                mruProjectPopupItem.Groups.Clear();
            }

            if (null == Application.AppConfiguration.MruProjects) { return; }

            mruProjectPopupItem.Groups.Add(new CommandBarItemGroup(string.Empty));

            int index = 1;

            foreach (string filePath in Application.AppConfiguration.MruProjects)
            {

                CommandUri uri = CommandUri.Create(
                                    FileCommands.OpenProject.ID,
                                    new KeyValuePair<string, string>[] {
                                        new KeyValuePair<string, string>( "TargetFilePath", filePath )
                                    });

                Command command = new Command(uri.Uri, FileCommands.OpenProject.Name)
                {
                    Text = FileCommands.OpenProject.Text,
                };

                mruProjectPopupItem.Groups[0].Items.Add(
                    new CommandBarMenuItem(command.Uri, command)
                    {
                        Text = string.Format("{0} {1}(&{2})", index, filePath, index % 10),
                        Image = null,
                    });

                index++;

            }

            _mainMenuAdapter.BuildUI();
        }

        ///--------------------------------
        /// <summary>
        /// メインウィンドウを初期化します。
        /// </summary>
        private void Initialize()
        {
            UpdateTitle();

            InitializeToolPages();

            InitializeCommandBindings(_commandBindings);
            InitializeMenu();
            InitializeTools();
            InitializeControls();

            ApplyAppConfiguration();

            ValidateToolPageBindings();

            UpdateMruProjects();

            Application.AppConfiguration.MruProjects.ItemChanged += OnMruProjectsItemChanged;

            Application.CommandKeyProcessed += OnCommandKeyProcessed;
            Application.CommandKeyProcessing += OnCommandKeyProcessing;

            _leftDockingToolContainer.ShowFloatingForms();
            _rightDockingToolContainer.ShowFloatingForms();
            _bottomDockingToolContainer.ShowFloatingForms();

            _loaded = true;
        }

        /// <summary>
        /// メインメニューを初期化します。
        /// </summary>
        private void InitializeMenu()
        {
            if (null == _mainMenuAdapter)
            {
                _mainMenuAdapter = ToolStripAdapter.FromToolStrip(Application.CommandService, _mainMenu);
                _mainMenuAdapter.CommandTarget = Application;
                _mainMenuAdapter.CommandExecuted += OnMainWindowCommandExecuted;

                //
                CommandTextReplacer.PreserveDefaultColorComment(_mainMenuAdapter);
            }

            if (null == _tabMenuAdapter)
            {
                _tabMenuAdapter = ToolStripAdapter.FromToolStrip(Application.CommandService, _tabMenu);
                _tabMenuAdapter.CommandTarget = Application;
            }
        }

        /// <summary>
        /// ツールバーを初期化します。
        /// </summary>
        private void InitializeTools()
        {
            if (null == _standardToolAdapter)
            {
                _standardToolAdapter = ToolStripAdapter.FromToolStrip(Application.CommandService, StandardTool);
                _standardToolAdapter.CommandTarget = Application;
                _standardToolAdapter.CommandExecuted += OnMainWindowCommandExecuted;

                /// 隠し機能(2012.5.31 aoyagi)
                /// "再コンバート"のツールバーを非表示にします。
                /// メニューは表示します。
                if (this.Options.Application.General.HideReconvertToolBarButton == true)
                {
                    CommandBarItem item = this._standardToolAdapter.CommandBar.FindFirstItem("Project.Reconvert");
                    Debug.Assert(item != null, "Can't find reconvert tool bar button");
                    item.Visible = false;
                }
            }

            if (null == _toolWindowToolAdapter)
            {
                _toolWindowToolAdapter = ToolStripAdapter.FromToolStrip(Application.CommandService, ToolWindowTool);
                _toolWindowToolAdapter.CommandTarget = Application;
                _toolWindowToolAdapter.CommandExecuted += OnMainWindowCommandExecuted;
            }
        }

        /// <summary>
        /// ファイルを開きます。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        private bool OpenFile()
        {
            string filePath = BankFileQuester.QueryLoad();
            if (null == filePath) { return false; }

            return OpenFile(filePath);
        }

        /// <summary>
        /// タブページを閉じます。
        /// </summary>
        /// <param name="tabCtrl">対象タブコントロール。</param>
        /// <param name="targetIndex">対象タブインデックス。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        private bool ClosePage(TabCtrl tabCtrl, int targetIndex)
        {
            if (null == tabCtrl) { return false; }
            if (0 > targetIndex) { return false; }

            tabCtrl.TabPages.RemoveAt(targetIndex);
            return true;
        }

        /// <summary>
        /// 他のタブを全て閉じることができるかどうか調べます。
        /// </summary>
        /// <returns>実行可能な場合は true、実行不可能な場合は false。</returns>
        private bool CanCloseOtherPages()
        {
            if (null == _tabMenuTargetCtrl) { return false; }
            if (0 > _tabMenuTargetIndex) { return false; }
            if (1 >= _tabMenuTargetCtrl.TabPages.Count) { return false; }
            return true;
        }

        /// <summary>
        /// 他のタブを全て閉じます。
        /// </summary>
        /// <param name="tabCtrl">対象タブコントロール。</param>
        /// <param name="targetIndex">対象タブインデックス。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        private bool CloseOtherPages(TabCtrl tabCtrl, int targetIndex)
        {
            if (null == tabCtrl) { return false; }
            if (0 > targetIndex) { return false; }

            while (tabCtrl.TabPages.Count > _tabMenuTargetIndex + 1)
            {
                tabCtrl.TabPages.RemoveAt(_tabMenuTargetIndex + 1);
            }

            for (int i = 0; i < _tabMenuTargetIndex; i++)
            {
                tabCtrl.TabPages.RemoveAt(0);
            }

            return true;
        }

        /// <summary>
        /// タブを全て閉じます。
        /// </summary>
        /// <param name="tabCtrl">対象タブコントロール。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        private bool CloseAllPages(TabCtrl tabCtrl)
        {
            if (null == tabCtrl) { return false; }

            tabCtrl.CloseAllTabPages();
            return true;
        }

        /// <summary>
        /// タブドキュメントのフォルダを開けるかどうか調べます。
        /// </summary>
        /// <returns>実行可能な場合は true、実行不可能な場合は false。</returns>
        private bool CanOpenPageDocumentFolder()
        {
            if (null == _tabMenuTargetCtrl) { return false; }
            if (0 > _tabMenuTargetIndex) { return false; }

            Document document = (_tabMenuTargetCtrl.TabPages[_tabMenuTargetIndex]
                                  as CommonTabPage).Panel.Document;
            if (null == document) { return false; }

            return true;
        }

        /// <summary>
        /// タブドキュメントのフォルダを開きます。
        /// </summary>
        /// <param name="tabCtrl">対象タブコントロール。</param>
        /// <param name="targetIndex">対象タブインデックス。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        private bool OpenPageDocumentFolder(TabCtrl tabCtrl, int targetIndex)
        {
            if (null == tabCtrl) { return false; }
            if (0 > targetIndex) { return false; }

            Document document = (tabCtrl.TabPages[targetIndex] as CommonTabPage).Panel.Document;
            if (null == document) { return false; }

            Process.Start("explorer.exe", "/e,/select, " + document.Resource.Key);

            return true;
        }

        /// <summary>
        /// ブックマークの追加ができるか調べます。
        /// </summary>
        /// <returns>実行可能な場合は true、実行不可能な場合は false。</returns>
        private bool CanAddBookmark()
        {
            if (null == _tabMenuTargetCtrl) { return false; }
            if (0 > _tabMenuTargetIndex) { return false; }

            Component component = null;
            CommonPanel panel = (_tabMenuTargetCtrl.TabPages[_tabMenuTargetIndex] as CommonTabPage).Panel;

            if (panel is SoundSetPanel == true)
            {
                SoundSetPanel soundSetPanel = panel as SoundSetPanel;
                component = soundSetPanel.CurrentItem;
            }
            else if (panel is BankPanel == true)
            {
                BankPanel bankPanel = panel as BankPanel;
                component = bankPanel.SoundSetBank;
            }
            else if (panel is FindResultPanel2 == true)
            {
                FindResultPanel2 findResultPanel2 = panel as FindResultPanel2;
                return true;
            }

            if (null == component) { return false; }

            return true;
        }

        /// <summary>
        /// ブックマークの追加をします。
        /// </summary>
        /// <param name="tabCtrl">対象タブコントロール。</param>
        /// <param name="targetIndex">対象タブインデックス。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        private bool AddBookmark(TabCtrl tabCtrl, int targetIndex)
        {
            if (null == tabCtrl) { return false; }
            if (0 > targetIndex) { return false; }

            Component component = null;
            CommonPanel panel = (tabCtrl.TabPages[targetIndex] as CommonTabPage).Panel;

            if (panel is SoundSetPanel == true)
            {
                SoundSetPanel soundSetPanel = panel as SoundSetPanel;
                component = soundSetPanel.CurrentItem;
            }
            else if (panel is BankPanel == true)
            {
                BankPanel bankPanel = panel as BankPanel;
                component = bankPanel.SoundSetBank;
            }
            else if (panel is FindResultPanel2 == true)
            {
                FindResultPanel2 findResultPanel2 = panel as FindResultPanel2;
                string title = findResultPanel2.TitleName;
                string kind = findResultPanel2.TargetDocumentKind.ToString();
                string keyword = findResultPanel2.Keyword;
                string[] parameters = findResultPanel2.GetTargetParameterNames();
                string[] soundsets = findResultPanel2.GetTargetSoundSetNames();
                string[] filters = findResultPanel2.GetTargetFilterNames();
                Application.BookmarkService.Add(title, kind, keyword, parameters, soundsets, filters);
                return true;
            }

            if (null == component) { return false; }

            Application.BookmarkService.Add(component);

            return true;
        }

        /// <summary>
        /// 編集領域を上下分割可能かどうかを調べます。
        /// </summary>
        /// <param name="command">上下分割コマンド。</param>
        /// <returns>可能な場合は true、不可能な場合は false。</returns>
        private CommandStatus QueryStatusSplitEditHorizontal(Command command)
        {
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// 編集領域を上下分割可能します。
        /// </summary>
        /// <param name="command">上下分割コマンド。</param>
        private bool ExecuteSplitEditHorizontal(Command command)
        {
            SplitUD();
            return true;
        }

        /// <summary>
        /// 編集領域を左右分割可能かどうかを調べます。
        /// </summary>
        /// <param name="command">左右分割コマンド。</param>
        /// <returns>可能な場合は true、不可能な場合は false。</returns>
        private CommandStatus QueryStatusSplitEditVertical(Command command)
        {
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// 編集領域を左右分割可能します。
        /// </summary>
        /// <param name="command">左右分割コマンド。</param>
        private bool ExecuteSplitEditVertical(Command command)
        {
            SplitLR();
            return true;
        }

        private CommandStatus QueryStatusSelectDocument(Command command)
        {
            if (null == _CurrentTabCtrl) { return CommandStatus.SupportedAndVisible; }
            if (0 == _CurrentTabCtrl.TabPages.Count) { return CommandStatus.SupportedAndVisible; }

            int index = _CurrentTabCtrl.SelectedIndex;

            if (command.Parameters.ContainsParameter("Index"))
            {
                index = int.Parse(command.Parameters.GetParameter("Index") as string);
            }

            if (index < -1 || _CurrentTabCtrl.TabPages.Count <= index)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        private bool ExecuteSelectDocument(Command command)
        {
            if (null == _CurrentTabCtrl) { return false; }
            if (0 == _CurrentTabCtrl.TabPages.Count) { return false; }

            int index = _CurrentTabCtrl.SelectedIndex;

            if (command.Parameters.ContainsParameter("Index"))
            {
                index = int.Parse(command.Parameters.GetParameter("Index") as string);
            }

            if (index < -1 || _CurrentTabCtrl.TabPages.Count <= index) { return false; }

            if (0 <= index)
            {
                _CurrentTabCtrl.SelectedIndex = index;
            }
            else
            {
                _CurrentTabCtrl.SelectedIndex = _CurrentTabCtrl.TabPages.Count - 1;
            }

            bool selectMainList = true;

            if (command.Parameters.ContainsParameter("TargetList") &&
                command.Parameters.GetParameter("TargetList") as string == "Sub")
            {
                selectMainList = false;
            }

            ActivePage.Panel.ActivatePanel(false, selectMainList);

            return true;
        }

        private CommandStatus QueryStatusNextDocument(Command command)
        {
            if (null == _CurrentTabCtrl) { return CommandStatus.SupportedAndVisible; }
            if (0 == _CurrentTabCtrl.TabPages.Count) { return CommandStatus.SupportedAndVisible; }
            if (0 == _TabCtrlMain.TabPages.Count && 0 == _TabCtrlSub.TabPages.Count)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        private bool ExecuteNextDocument(Command command)
        {
            bool isActiveControlChange = true;

            if (command.Parameters.ContainsParameter(CommandParameterNames.IsActiveControlChange))
            {
                try
                {
                    isActiveControlChange = bool.Parse(command.Parameters.GetParameter(
                                                CommandParameterNames.IsActiveControlChange) as string);
                }
                catch
                {
                }
            }

            int newIndex = _CurrentTabCtrl.SelectedIndex + 1;

            if (newIndex < _CurrentTabCtrl.TabPages.Count)
            {
                _CurrentTabCtrl.SelectedIndex = newIndex;
                ActivePage.Panel.ActivatePanel(false, true);
                return true;
            }

            TabCtrl targetTabCtrl = null;

            if (isActiveControlChange)
            {
                targetTabCtrl = (_TabCtrlMain == _CurrentTabCtrl) ?
                                                    _TabCtrlSub : _TabCtrlMain;
            }
            else
            {
                targetTabCtrl = _CurrentTabCtrl;
            }

            if (0 < targetTabCtrl.TabPages.Count)
            {
                targetTabCtrl.SelectedIndex = 0;
                SetCurrentTabCtrl(targetTabCtrl);
            }
            else
            {
                _CurrentTabCtrl.SelectedIndex = 0;
            }

            ActivePage.Panel.ActivatePanel(false, true);

            return true;
        }

        private CommandStatus QueryStatusPreviousDocument(Command command)
        {
            if (null == _CurrentTabCtrl) { return CommandStatus.SupportedAndVisible; }
            if (0 == _CurrentTabCtrl.TabPages.Count) { return CommandStatus.SupportedAndVisible; }
            if (0 == _TabCtrlMain.TabPages.Count && 0 == _TabCtrlSub.TabPages.Count)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        private bool ExecutePreviousDocument(Command command)
        {
            bool isActiveControlChange = true;

            if (command.Parameters.ContainsParameter(CommandParameterNames.IsActiveControlChange))
            {
                try
                {
                    isActiveControlChange = bool.Parse(command.Parameters.GetParameter(
                                                CommandParameterNames.IsActiveControlChange) as string);
                }
                catch
                {
                }
            }

            int newIndex = _CurrentTabCtrl.SelectedIndex - 1;

            if (0 <= newIndex)
            {
                _CurrentTabCtrl.SelectedIndex = newIndex;
                ActivePage.Panel.ActivatePanel(false, true);
                return true;
            }

            TabCtrl targetTabCtrl = null;

            if (isActiveControlChange)
            {
                targetTabCtrl = (_TabCtrlMain == _CurrentTabCtrl) ?
                                                    _TabCtrlSub : _TabCtrlMain;
            }
            else
            {
                targetTabCtrl = _CurrentTabCtrl;
            }

            if (0 < targetTabCtrl.TabPages.Count)
            {
                targetTabCtrl.SelectedIndex = targetTabCtrl.TabPages.Count - 1;
                SetCurrentTabCtrl(targetTabCtrl);
            }
            else
            {
                _CurrentTabCtrl.SelectedIndex = _CurrentTabCtrl.TabPages.Count - 1;
            }

            ActivePage.Panel.ActivatePanel(false, true);

            return true;
        }

        private CommandStatus QueryStatusCloseDocument(Command command)
        {
            if (null == _tabMenuTargetCtrl ||
                _tabMenuTargetIndex < 0 || _tabMenuTargetCtrl.TabPages.Count <= _tabMenuTargetIndex)
            {
                return CommandStatus.SupportedAndVisible;
            }
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        private bool ExecuteCloseDocument(Command command)
        {
            if (null == _tabMenuTargetCtrl ||
                _tabMenuTargetIndex < 0 || _tabMenuTargetCtrl.TabPages.Count <= _tabMenuTargetIndex)
            {
                return false;
            }
            return ClosePage(_tabMenuTargetCtrl, _tabMenuTargetIndex);
        }

        private CommandStatus QueryStatusCloseSelectedDocument(Command command)
        {
            if (null == _CurrentTabCtrl || 0 > _CurrentTabCtrl.SelectedIndex)
            {
                return CommandStatus.SupportedAndVisible;
            }
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        private bool ExecuteCloseSelectedDocument(Command command)
        {
            if (null == _CurrentTabCtrl || 0 > _CurrentTabCtrl.SelectedIndex)
            {
                return false;
            }
            return ClosePage(_CurrentTabCtrl, _CurrentTabCtrl.SelectedIndex);
        }

        private CommandStatus QueryStatusCloseOtherDocuments(Command command)
        {
            return CanCloseOtherPages() ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        private bool ExecuteCloseOtherDocuments(Command command)
        {
            return CloseOtherPages(_tabMenuTargetCtrl, _tabMenuTargetIndex);
        }

        private CommandStatus QueryStatusCloseAllDocuments(Command command)
        {
            return (null != _tabMenuTargetCtrl && 0 <= _tabMenuTargetIndex) ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        private bool ExecuteCloseAllDocuments(Command command)
        {
            return CloseAllPages(_tabMenuTargetCtrl);
        }

        private CommandStatus QueryStatusOpenDocumentFolder(Command command)
        {
            return CanOpenPageDocumentFolder() ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        private bool ExecuteOpenDocumentFolder(Command command)
        {
            return OpenPageDocumentFolder(_tabMenuTargetCtrl, _tabMenuTargetIndex);
        }

        private CommandStatus QueryStatusAddBookmark(Command command)
        {
            return (CanAddBookmark() ?
                    CommandStatus.SupportedAndEnabledAndVisible :
                    CommandStatus.SupportedAndVisible);
        }

        private bool ExecuteAddBookmark(Command command)
        {
            return AddBookmark(_tabMenuTargetCtrl, _tabMenuTargetIndex);
        }

        ///--------------------------------
        /// <summary>
        /// コントロールの初期化
        /// タブコントロールなど
        /// </summary>
        private void InitializeControls()
        {
            Application.ProjectService.Opened += OnProjectOpened;
            Application.ProjectService.Closing += OnProjectClosing;
            Application.ProjectService.Closed += OnProjectClosed;
            Application.ProjectService.SoundSetRemoved += OnSoundSetRemoved;
            Application.ProjectService.ComponentsRemoved += OnComponentsRemoved;
            Application.ProjectService.ComponentsAdded += OnComponentsAdded;
            Application.ProjectService.ComponentNameChanged += OnComponentNameChanged;
            Application.BankServices.Removed += OnBankRemoved;
            Application.ProjectConfiguration.Saving += OnProjectConfigurationSaving;

            SetCurrentTabCtrl(_TabCtrlMain);

            //
            _TabCtrlMain.FileDroping += OnFileDropingTabPage;
            _TabCtrlMain.RotateSelecting += OnRotateSelectingTabPage;
            _TabCtrlMain.Selected += OnSelectedTabPageChanged;
            _TabCtrlMain.Deselecting += OnDeselectingTabPageChanged;
            _TabCtrlMain.Closed += OnClosedTabPage;
            _TabCtrlMain.StateChanged += OnTabPageStateChanged;
            _TabCtrlMain.TabDoubleClick += OnTabCtrlDoubleClicked;

            _TabCtrlSub.FileDroping += OnFileDropingTabPage;
            _TabCtrlSub.RotateSelecting += OnRotateSelectingTabPage;
            _TabCtrlSub.Selected += OnSelectedTabPageChanged;
            _TabCtrlSub.Deselecting += OnDeselectingTabPageChanged;
            _TabCtrlSub.Closed += OnClosedTabPage;
            _TabCtrlSub.StateChanged += OnTabPageStateChanged;
            _TabCtrlSub.TabDoubleClick += OnTabCtrlDoubleClicked;

            //
            _editSplitContainer.DoubleClick += OnTabSplitterDoubleClick;
            _editSplitContainer.SplitterMoving += OnTabSplitterMoving;
            _editSplitContainer.SplitterMoved += OnTabSplitterMoved;

            //
            _FindWindow = new FindWindow();
            _FindWindow.Owner = this;
            _FindWindow.DesktopLocation = new Point(0, 0);
            _FindWindow.Hide();
        }

        /// <summary>
        ///
        /// </summary>
        private void NotifyCommandExecuted(Command command)
        {
            if (null == command) { return; }

            CommonTabPage page = null;

            if (ProjectPanel != null)
            {
                ProjectPanel.CommandExecuted(command);
            }

            if ((page = _TabCtrlMain.SelectedTab as CommonTabPage) != null)
            {
                if (page.Panel != null)
                {
                    page.Panel.CommandExecuted(command);
                }
            }

            if ((page = _TabCtrlSub.SelectedTab as CommonTabPage) != null)
            {
                if (page.Panel != null)
                {
                    page.Panel.CommandExecuted(command);
                }
            }

            BuildCommandUI();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnMruProjectsItemChanged(object sender, EventArgs e)
        {
            UpdateMruProjects();
        }

        private void OnCommandKeyProcessed(object sender, CommandKeyProcessEventArgs e)
        {
            if (null == e.Command)
            {

                UpdateStatusText(
                    string.Format(MessageResource.Message_ShortcutCommandNotFound,
                                   e.FirstKey.ToKeys().ToShortcutKeyText(),
                                   ((Keys)e.SecondKey.Key).ToShortcutKeyText()));

            }
            else
            {
                NotifyCommandExecuted(e.Command);
            }
        }

        private void OnCommandKeyProcessing(object sender, CommandKeyProcessEventArgs e)
        {
            if (KeyStroke.NullKeyStroke == e.FirstKey)
            {
                UpdateStatusText(_statusSelectionDescription);
                return;
            }

            UpdateStatusText(
                string.Format(MessageResource.Message_WaitingForShortcutKeyIn,
                               e.FirstKey.ToShortcutKeyText()));
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnProjectOpened(object sender, EventArgs e)
        {
            this.projectOpened = true;

            try
            {
                if (Application.ProjectService.Project != null)
                {
                    ProjectPanel.AttachProject(Application.ProjectService.ProjectDocument);
                    this.UpdateLeftDockingToolTitle();
                    ApplyProjectConfiguration();
                    ProjectPanel.ApplySoundSetConfigurations();
                    this.CheckSoundListOutputs(Application.ProjectService.Project);
                }

                if (Application.BookmarkService != null)
                {
                    BookmarkPanel.AttachBookmark(Application.BookmarkService);
                }

                if (Application.HistoryService != null)
                {
                    HistoryPanel.AttachHistory(Application.HistoryService);
                }

                UpdateTitle();
                UpdateMruProjects();

                UpdateCommandTexts();
                BuildCommandUI();

                Application.ExecuteCommand(ViewCommands.SynchronizeTree);

                _editSplitContainer.Select();
            }
            finally
            {
                this.projectOpened = false;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnProjectClosing(object sender, EventArgs e)
        {
            ProjectPanel.DetachProject();
            this.UpdateLeftDockingToolTitle();
            ExtractProjectConfiguration();

            if (CloseAllTabPages() == false)
            {
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnProjectClosed(object sender, EventArgs e)
        {
            BookmarkPanel.DetachBookmark();
            HistoryPanel.DetachHistory();

            UpdateTitle();
            UpdateStatusText(string.Empty);
        }

        private void OnSoundSetRemoved(object sender, SoundDocumentEventArgs e)
        {
            RemoveGarbagePages();
        }

        private void OnComponentsRemoved(object sender, ComponentEventArgs e)
        {
            foreach (Component component in e.Components)
            {
                if (!(component is SoundSetBankBase)) { continue; }
                RemoveGarbagePages();
                break;
            }

            if (Application.BookmarkService != null)
            {
                Application.BookmarkService.ComponentsRemoved(e.Components);
                Application.BookmarkService.UpdateTarget();
                this.BookmarkPanel.UpdateBookmark();
            }

            if (Application.HistoryService != null)
            {
                Application.HistoryService.ComponentsRemoved(e.Components);
                Application.HistoryService.UpdateTarget();
            }
        }

        private void OnComponentsAdded(object sender, ComponentEventArgs e)
        {
            if (Application.BookmarkService != null)
            {
                Application.BookmarkService.ComponentsAdded(e.Components);
                Application.BookmarkService.UpdateTarget();
                this.BookmarkPanel.UpdateBookmark();
            }

            if (Application.HistoryService != null)
            {
                Application.HistoryService.ComponentsAdded(e.Components);
                Application.HistoryService.UpdateTarget();
            }
        }

        private void OnBankRemoved(object sender, BankServiceEventArgs e)
        {
            if (_isTabClosing) { return; }

            try
            {
                _isTabClosing = true;

                foreach (CommonTabPage page in OpenedAllTabPages<BankPanel>())
                {
                    if ((page.Panel as BankPanel).BankServiceReference.Target == e.BankService)
                    {
                        (page.Parent as TabCtrl).TabPages.Remove(page);
                    }
                }
            }
            finally
            {
                _isTabClosing = false;
            }
        }

        private void OnComponentNameChanged(object sender, ComponentNameChangedEventArgs e)
        {
            if (Application.BookmarkService != null)
            {
                Application.BookmarkService.ComponentNameChanged(e.OldName, e.NewName);
                Application.BookmarkService.UpdateTarget();
                this.BookmarkPanel.UpdateBookmark();
            }

            if (Application.HistoryService != null)
            {
                Application.HistoryService.ComponentNameChanged(e.OldName, e.NewName);
                Application.HistoryService.UpdateName();
                Application.HistoryService.UpdateTarget();
            }
        }

        private void OnApplyProjectConfiguration(object sender, EventArgs e)
        {
            CommonListPanel panel = null;

            foreach (CommonTabPage page in _TabCtrlMain.TabPages)
            {
                if ((panel = page.Panel as CommonListPanel) != null)
                {
                    panel.ApplyProjectConfigurations();
                }
            }

            foreach (CommonTabPage page in _TabCtrlSub.TabPages)
            {
                if ((panel = page.Panel as CommonListPanel) != null)
                {
                    panel.ApplyProjectConfigurations();
                }
            }
        }

        /// <summary>
        /// スプリッタを移動すると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントパラメータ。</param>
        private void OnDockingToolContainerSplitterMoved(object sender, SplitterEventArgs e)
        {
            if (sender == _leftSplitContainer)
            {
                if (_leftDockingToolContainer.IsMinimized) { return; }
                _leftSplitterPosition = _leftSplitContainer.SplitterDistance;
            }
            else if (sender == _rightSplitContainer)
            {
                if (_rightDockingToolContainer.IsMinimized) { return; }
                _rightSplitterPosition = _rightSplitContainer.Width - _rightSplitContainer.SplitterDistance;
            }
            else if (sender == _bottomSplitContainer)
            {
                if (_bottomDockingToolContainer.IsMinimized)
                {
                    return;
                }

                _bottomSplitterPosition = (_bottomSplitContainer.Height -
                                            _bottomSplitContainer.SplitterDistance);

                if (_loaded == false)
                {
                    this.rightSplitterDistance = _rightSplitContainer.SplitterDistance;
                    this.bottomSplitterDistance = _bottomSplitContainer.SplitterDistance;
                }
            }
        }

        /// <summary>
        /// ウインドウが最小化される直前に発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントパラメータ。</param>
        private void OnWindowMinimizing(object sender, EventArgs e)
        {
            switch (this.WindowState)
            {
                case FormWindowState.Normal:
                    this.rightSplitterDistance = _rightSplitContainer.SplitterDistance;
                    this.bottomSplitterDistance = _bottomSplitContainer.SplitterDistance;
                    break;
                case FormWindowState.Maximized:
                    this.rightSplitterDistanceMaximize = _rightSplitContainer.SplitterDistance;
                    this.bottomSplitterDistanceMaximize = _bottomSplitContainer.SplitterDistance;
                    break;
            }
        }

        /// <summary>
        /// ウインドウが最小化された直後に発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントパラメータ。</param>
        private void OnWindowMinimized(object sender, EventArgs e)
        {
        }

        /// <summary>
        /// ウインドウが最大化される直前に発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントパラメータ。</param>
        private void OnWindowMaximizing(object sender, EventArgs e)
        {
            this.rightSplitterDistance = _rightSplitContainer.SplitterDistance;
            this.bottomSplitterDistance = _bottomSplitContainer.SplitterDistance;
        }

        /// <summary>
        /// ウインドウが最大化された直後に発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントパラメータ。</param>
        private void OnWindowMaximized(object sender, EventArgs e)
        {
        }

        /// <summary>
        /// ウインドウが復元される直前に発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントパラメータ。</param>
        private void OnWindowRestoring(object sender, EventArgs e)
        {
            switch (this.WindowState)
            {
                case FormWindowState.Normal:
                    break;
                case FormWindowState.Minimized:
                    break;
                case FormWindowState.Maximized:
                    break;
            }
        }

        /// <summary>
        /// ウインドウが復元された直後に発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントパラメータ。</param>
        private void OnWindowRestored(object sender, EventArgs e)
        {
            switch (this.WindowState)
            {
                case FormWindowState.Normal:
                    _rightSplitContainer.SplitterDistance = this.rightSplitterDistance;
                    _bottomSplitContainer.SplitterDistance = this.bottomSplitterDistance;
                    break;
                case FormWindowState.Minimized:
                    break;
                case FormWindowState.Maximized:
                    _rightSplitContainer.SplitterDistance = this.rightSplitterDistanceMaximize;
                    _bottomSplitContainer.SplitterDistance = this.bottomSplitterDistanceMaximize;
                    break;
            }
        }

        /// <summary>
        /// スプリッタをダブルクリックすると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントパラメータ。</param>
        private void OnDockingToolContainerSplitterDoubleClick(object sender, EventArgs e)
        {
            DockingToolContainer target;

            if (sender == _leftSplitContainer)
            {
                target = _leftDockingToolContainer;
            }
            else if (sender == _rightSplitContainer)
            {
                target = _rightDockingToolContainer;
            }
            else if (sender == _bottomSplitContainer)
            {
                target = _bottomDockingToolContainer;
            }
            else
            {
                return;
            }

            if (target.IsMinimized)
            {
                target.Restore();
            }
            else
            {
                target.Minimize();
            }
        }

        /// <summary>
        /// ドッキングフォームコンテナを最小化すると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントパラメータ。</param>
        private void OnDockingToolContainerMinimized(object sender, EventArgs e)
        {
            if (sender == _leftDockingToolContainer)
            {

                _leftSplitterPosition = _leftSplitContainer.SplitterDistance;

                _leftSplitContainer.SplitterDistance = _leftDockingToolContainer.MinimumSize.Width + 2;
                _leftSplitContainer.IsSplitterFixed = true;

            }
            else if (sender == _rightDockingToolContainer)
            {

                _rightSplitterPosition = _rightSplitContainer.Width - _rightSplitContainer.SplitterDistance;

                _rightSplitContainer.SplitterDistance = _rightSplitContainer.Width - _rightSplitContainer.SplitterWidth
                                                         - _rightDockingToolContainer.MinimumSize.Width - 2;
                _rightSplitContainer.IsSplitterFixed = true;

            }
            else if (sender == _bottomDockingToolContainer)
            {

                _bottomSplitterPosition = _bottomSplitContainer.Height - _bottomSplitContainer.SplitterDistance;

                _bottomSplitContainer.SplitterDistance = _bottomSplitContainer.Height - _bottomSplitContainer.SplitterWidth
                                                         - _bottomDockingToolContainer.MinimumSize.Height - 2;
                _bottomSplitContainer.IsSplitterFixed = true;

            }
        }

        /// <summary>
        /// ドッキングフォームコンテナを元に戻すると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントパラメータ。</param>
        private void OnDockingToolContainerRestored(object sender, EventArgs e)
        {
            if (sender == _leftDockingToolContainer)
            {

                if (_leftSplitContainer.Panel1MinSize <= _leftSplitterPosition &&
                    _leftSplitContainer.Panel2MinSize <= _rightSplitContainer.Width - _rightSplitterPosition)
                {
                    _leftSplitContainer.SplitterDistance = _leftSplitterPosition;
                }

                _leftSplitContainer.IsSplitterFixed = false;

            }
            else if (sender == _rightDockingToolContainer)
            {

                int splitterDistance = _rightSplitContainer.Width - _rightSplitterPosition;

                if (_rightSplitContainer.Panel2MinSize <= _rightSplitterPosition &&
                    _rightSplitContainer.Panel1MinSize <= splitterDistance)
                {
                    _rightSplitContainer.SplitterDistance = splitterDistance;
                }
                _rightSplitContainer.IsSplitterFixed = false;

            }
            else if (sender == _bottomDockingToolContainer)
            {

                int splitterDistance = _bottomSplitContainer.Height - _bottomSplitterPosition;

                if (_bottomSplitContainer.Panel2MinSize <= _bottomSplitterPosition &&
                    _bottomSplitContainer.Panel1MinSize <= splitterDistance)
                {
                    _bottomSplitContainer.SplitterDistance = splitterDistance;
                }
                _bottomSplitContainer.IsSplitterFixed = false;

            }
        }

        /// <summary>
        /// ドッキングコンテナのページ状態が変更されると発生します。
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnDockingToolContainerPageStateChanged(object sender, DockingPageEventArgs e)
        {
            SplitContainer splitContainer;
            bool minimizePanel1 = true;

            if (sender == _leftDockingToolContainer)
            {
                splitContainer = _leftSplitContainer;
            }
            else if (sender == _rightDockingToolContainer)
            {
                splitContainer = _rightSplitContainer;
                minimizePanel1 = false;
            }
            else if (sender == _bottomDockingToolContainer)
            {
                splitContainer = _bottomSplitContainer;
                minimizePanel1 = false;
            }
            else
            {
                return;
            }

            UpdateDockContainer(sender as DockingToolContainer, splitContainer, minimizePanel1);

            /// パネルの表示がトグル式になったのでツールバーのチェック状態を更新する為に
            // BuildCommandUI()が必要になりました。(2012/6/14 aoyagi)
            BuildCommandUI();
        }

        ///--------------------------------
        /// <summary>
        /// タブの選択が変更された時に呼ばれる
        /// </summary>
        private void OnTabSelectedChanged(object sender, EventArgs e)
        {
            _FindWindow.UpdateInnerData();

            //
            if (ActivePage != null)
            {
                ActivePage.Panel.Activated();
            }
        }

        ///--------------------------------
        /// <summary>
        /// タブが閉じられる前に発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">タブコントロールイベントデータ。</param>
        private void OnTabClosing(object sender, NTabControlCancelEventArgs e)
        {
            if (_isTabClosing) { return; }

            CommonPanel panel = (e.TabPage as CommonTabPage).Panel;
            if (!(panel is BankPanel)) { return; }

            BankPanel bankPanel = panel as BankPanel;

            int tabPageCount = OpenedAllTabPages()
                .Where(p => p.Panel is BankPanel).Select(p => p.Panel).Cast<BankPanel>()
                .Where(p => p.BankDocument == bankPanel.BankDocument).Count();

            try
            {
                bankPanel.FinalizeEdit();

                _isTabClosing = true;

                bool result = CloseDocumentHandler.Execute(
                                bankPanel.BankServiceReference, (tabPageCount == 1));

                if (!result)
                {
                    e.Cancel = true;
                }
            }
            finally
            {
                _isTabClosing = false;
            }
        }

        /// <summary>
        /// タブコンテキストメニューが表示されると発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">タブコントロールイベントデータ。</param>
        private void OnTabMenuShown(object sender, NTabControlCancelEventArgs e)
        {
            _tabMenuTargetCtrl = sender as TabCtrl;
            _tabMenuTargetIndex = e.TabPageIndex;

            _tabMenuAdapter.BuildUI();
        }

        /// <summary>
        /// プロジェクト設定が保存される前に発生します。
        /// </summary>
        /// <param name="sender">イベントの送信元。</param>
        /// <param name="e">空のイベントデータ。</param>
        private void OnProjectConfigurationSaving(object sender, EventArgs e)
        {
            ExtractProjectConfiguration();
        }

        private void OnMainMenuMouseActivate(object sender, System.ComponentModel.CancelEventArgs e)
        {
            IntPtr hWnd = NintendoWare.SoundFoundation.Core.Win32.User32.GetActiveWindow();

            // アプリケーションがアクティブでない場合は、アクティブにする
            if (IntPtr.Zero == hWnd)
            {
                e.Cancel = true;
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnEndConvert(object sender, SoundProjectConvertEventArgs e)
        {
            if (e.IsPartsConvert)
            {
                return;
            }

            CommonTabPage page = null;

            if (_TabCtrlMain != null)
            {
                if ((page = _TabCtrlMain.SelectedTab as CommonTabPage) != null)
                {
                    page.Panel.RedrawPanel();
                }
            }

            if (_TabCtrlSub != null)
            {
                if ((page = _TabCtrlSub.SelectedTab as CommonTabPage) != null)
                {
                    page.Panel.RedrawPanel();
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private CommandStatus QueryStatusOpenFile(Command command)
        {
            return (null != Application.ProjectService.ProjectDocument) ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        private bool ExecuteOpenFile(Command command)
        {
            return OpenFile();
        }

        /// <summary>
        /// プロジェクトの再読み込みが可能かどうかを調べます。
        /// </summary>
        /// <param name="command">プロジェクトの再読み込みコマンド。</param>
        /// <returns>コマンドの状態。</returns>
        private CommandStatus QueryStatusReloadProject(Command command)
        {
            return (null != ProjectService.ProjectDocument) ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// プロジェクトの再読み込みを実行します。
        /// </summary>
        /// <param name="command">プロジェクトの再読み込みコマンド。</param>
        /// <returns>実行した場合 true、キャンセルされた、または実行しなかった場合 false。</returns>
        private bool ExecuteReloadProject(Command command)
        {
            if (Application.UIService.ShowMessageBox
                (MessageResource.Message_ConfirmReload,
                 MessageResource.DialogTitle_File_ReloadProject_Text,
                 AppMessageBoxButton.YesNo,
                 AppMessageBoxImage.Question,
                 AppMessageBoxResult.Yes) == AppMessageBoxResult.Yes)
            {
                this.TryReload();
            }

            return true;
        }

        /// <summary>
        /// すべてのドキュメントを閉じるコマンドが可能かどうかを調べます。
        /// </summary>
        /// <param name="command">すべてのドキュメントを閉じるコマンド。</param>
        /// <returns>コマンドの状態。</returns>
        private CommandStatus QueryStatusFileCloseAllDocuments(Command command)
        {
            return (0 < _TabCtrlMain.TabPages.Count || 0 < _TabCtrlSub.TabPages.Count) ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// すべてのドキュメントを閉じるコマンドを実行します。
        /// </summary>
        /// <param name="command">すべてのドキュメントを閉じるコマンド。</param>
        /// <returns>実行した場合 true、キャンセルされた、または実行しなかった場合 false。</returns>
        private bool ExecuteFileCloseAllDocuments(Command command)
        {
            CloseAllPages(_TabCtrlSub);
            CloseAllPages(_TabCtrlMain);
            return true;
        }

        private void TryReload()
        {
            int retryCount = ReloadRetryCountMax;
            bool force = false;
            List<DocumentReference> docRefList = null;

            retry:
            try
            {
                bool oldFileAllForceSave = false;
                bool isReadOnly = false;
                List<string> newVersionFiles = new List<string>();
                List<string> oldVersionFiles = new List<string>();

                if (retryCount == ReloadRetryCountMax)
                {
                    string filePath = ProjectService.ProjectDocument.Resource.Key;

                    // ファイルのバージョンを確認します。
                    docRefList = FileVersionUtility.GetDifferentVersionFilesWithProjectFile(filePath, ref newVersionFiles, ref oldVersionFiles);

                    if (0 < newVersionFiles.Count)
                    {
                        string message = string.Format("{0}\n{1}", MessageResource.Message_UnsupportedFileVersion, filePath);
                        ApplicationBase.Instance.UIService.ShowMessageBox(message);
                        return;
                    }
                    else if (0 < oldVersionFiles.Count)
                    {
                        UpgradeFileDialog dialog = new UpgradeFileDialog();
                        switch (dialog.ShowDialog())
                        {
                            case DialogResult.Cancel:
                                return;

                            case DialogResult.No: // 読み取り専用で開く
                                isReadOnly = true;
                                break;

                            case DialogResult.OK:
                                oldFileAllForceSave = true;
                                isReadOnly = false;
                                break;
                        }
                    }
                }

                Application.ProjectConfiguration.Save();

                Reload(force);

                Application.LoadProjectConfiguration();
                ApplyProjectConfiguration(false); // false : 再読み込みの場合は検索タブを作らない。
                ProjectPanel.ApplySoundSetConfigurations();

                // 旧バージョンを全て保存する指示があったなら保存する。
                if (oldFileAllForceSave == true)
                {
                    SaveHelper.SaveForce(oldVersionFiles.ToArray());
                }

                DocumentService documentService = ApplicationBase.Instance.DocumentService;

                // 全ての読み取り専用をクリア
                foreach (Document document in documentService.Documents)
                {
                    if (document is SoundDocument == true)
                    {
                        (document as SoundDocument).IsReadOnly = false;
                    }
                }

                if (isReadOnly == true)
                {
                    // 旧バージョンのみ読み取り専用に設定
                    foreach (string oldFilePath in oldVersionFiles)
                    {
                        SoundDocument document = documentService.Documents.Where(a => a.Resource.Key == oldFilePath).FirstOrDefault() as SoundDocument;

                        if (document != null)
                        {
                            document.IsReadOnly = true;
                        }
                    }
                }
            }
            catch
            {
                force = true;
                if (0 < retryCount)
                {
                    --retryCount;
                    goto retry;
                }
                else
                {
                    SoundProjectService projectService = Application.ProjectService as SoundProjectService;
                    if (projectService != null)
                    {
                        projectService.Close();
                    }
                    ApplicationBase.Instance.UIService.ShowMessageBox(MessageResource.Message_ReloadError, MessageResource.Message_UnknownError, AppMessageBoxButton.OK, AppMessageBoxImage.Error);
                    return;
                }
            }
            finally
            {
                if (docRefList != null)
                {
                    foreach (DocumentReference docRef in docRefList)
                    {
                        docRef.Close();
                    }
                }
            }
        }

        private void Reload(bool force)
        {
            try
            {
                //
                foreach (FindResultPanel2 panel in this.FindResultPanels)
                {
                    panel.Reloading();
                }

                XmlSndEditList list = new XmlSndEditList();
                list.SndEdit = Application.RealtimeEditService.MonitoringTargets
                    .Select(t => new XmlSndEdit(t))
                    .Cast<XmlSndEdit>()
                    .ToArray();

                foreach (Component component in Application.RealtimeEditService.MonitoringTargets)
                {
                    component.Parameters[ProjectParameterNames.SndEdit].Value = false;
                }

                this.CheckAndWaitMonitoringTargetsIsZero();

                try
                {
                    // プロジェクトファイルに付随したコンポーネントのパラメータ初期値設定ファイルを読み込みます。
                    var defaultparamFilePath = Application.GetDefaultParametersFilePath();
                    Application.CreateComponentService.LoadDefaultParametersFile(defaultparamFilePath);
                }
                catch (Exception e)
                {
                    var title = Resources.MessageResource.Message_ReadErrorDefaultParameterFile;
                    var messageBox = new TextDisplayMessageBox(title, e.Message, TextDisplayMessageBoxStyle.OKButton);
                    messageBox.ShowDialog();
                }
                ProjectService.ReloadProject((time, path) => force == true || time != File.GetLastWriteTime(path));
                ReloadBanks(force);

                XmlSndEdit[] items = list.SndEdit;
                if (items != null)
                {
                    Application.SetSndEditConfiguration(items);
                }

                //
                foreach (CommonListPanel panel in this.DisplayedListPanels)
                {
                    panel.RefreshPanel();
                }

                //
                foreach (FindResultPanel2 panel in this.FindResultPanels)
                {
                    panel.Reloaded();
                }

                //
                OnProjectReloaded(EventArgs.Empty);
            }
            catch (FileNotFoundException e)
            {
                UIService.ShowMessageBox(string.Format("{0}\n{1}", Resources.MessageResource.Message_FileNotFound, e.FileName));
            }
            catch (FileNotSupportedException e)
            {
                UIService.ShowMessageBox(string.Format("{0}\n{1}", e.Message, e.FilePath));
            }
            catch (FileNotSupportedVersionException e)
            {
                UIService.ShowMessageBox(string.Format("{0}\n{1}", e.Message, e.FilePath));
            }
            catch (Exception e)
            {
                string message = e.Message;
                if (string.IsNullOrEmpty(message) == true)
                {
                    message = Resources.MessageResource.Message_UnknownError;
                }
                UIService.ShowMessageBox(message);
                return;
            }
        }

        /// <summary>
        /// サウンドプロジェクトを再読み込みします。
        /// ただし、当面、サウンドプロジェクトファイルは対象外です。
        /// </summary>
        private void ReloadBanks(bool force)
        {
            // バンクファイル
            foreach (BankPanel bankPanel in from tabPage in OpenedAllTabPages()
                                            where tabPage.Panel is BankPanel
                                            select tabPage.Panel)
            {

                if (!Application.BankServices.Contains(bankPanel.BankDocument.Resource.Key))
                {
                    Debug.Fail("unexpected error.");
                    continue;
                }

                BankService bankService = Application.BankServices[bankPanel.BankDocument.Resource.Key];

                if (force == false &&
                    (bankService.BankDocument.Resource as FileResource).LastWriteTime ==
                    File.GetLastWriteTime(bankService.BankDocument.Resource.Key)) { continue; }
                bankService.Reload();
            }
        }

        /// <summary>
        /// 検索ウインドウを開けるか調べます。
        /// </summary>
        private CommandStatus QueryStatusShowFindWindow(Command command)
        {
            CommandStatus status = CommandStatus.SupportedAndEnabledAndVisible;

            if (_FindWindow != null &&
                _FindWindow.Visible == true)
            {
                status |= CommandStatus.Checked;
            }
            return status;
        }

        /// <summary>
        /// 検索ウインドウを開きます。
        /// </summary>
        private bool ExecuteShowFindWindow(Command command)
        {
            if (_FindWindow.Visible == true)
            {
                _FindWindow.Hide();
            }
            else
            {
                _FindWindow.Show();
            }
            return true;
        }

        /// <summary>
        /// サウンドセット検索タブを開けるか調べます。
        /// </summary>
        private CommandStatus QueryStatusOpenSoundSetFindTabPage(Command command)
        {
            CommandStatus status = CommandStatus.SupportedAndEnabledAndVisible;
            return status;
        }

        /// <summary>
        /// サウンドセット検索タブを開きます。
        /// </summary>
        private bool ExecuteOpenSoundSetFindTabPage(Command command)
        {
            AddFindTabPage(FindArgs.TargetDocumentKinds.SoundSet);
            return true;
        }

        /// <summary>
        /// インストルメント検索タブを開けるか調べます。
        /// </summary>
        private CommandStatus QueryStatusOpenInstrumentFindTabPage(Command command)
        {
            CommandStatus status = CommandStatus.SupportedAndEnabledAndVisible;
            return status;
        }

        /// <summary>
        /// インストルメント検索タブを開きます。
        /// </summary>
        private bool ExecuteOpenInstrumentFindTabPage(Command command)
        {
            AddFindTabPage(FindArgs.TargetDocumentKinds.Bank);
            return true;
        }

        ///--------------------------------
        /// <summary>
        /// "ツリーをタブに同期"が実行可能か調べる
        /// </summary>
        private CommandStatus QueryStatusSynchronizeTree(Command command)
        {
            if (ActivePage != null)
            {
                return CommandStatus.SupportedAndEnabledAndVisible;
            }
            else
            {
                return CommandStatus.SupportedAndVisible;
            }
        }

        ///--------------------------------
        /// <summary>
        /// "ツリーをタブに同期"の実行
        /// </summary>
        private bool ExecuteSynchronizeTree(Command command)
        {
            bool ret = false;
            SoundSetPanel soundSetPanel = ActivePage.Panel as SoundSetPanel;

            if (soundSetPanel != null)
            {
                ProjectPanel.ProjectCtrl.SetCaret(soundSetPanel.CurrentItem);
                ret = true;
            }
            else
            {
                BankPanel bankPanel = ActivePage.Panel as BankPanel;
                if (bankPanel != null)
                {
                    if (bankPanel.SoundSetBank != null)
                    {
                        ProjectPanel.ProjectCtrl.SetCaret(bankPanel.SoundSetBank);
                        ret = true;
                    }
                }
            }

            if (ret)
            {
                ActivateProjectPanelToolPage();
            }

            return ret;
        }

        /// <summary>
        /// 参照している波形ファイルの情報を更新が実行できるか判断します。
        /// </summary>
        private CommandStatus QueryStatusUpdateWaveFileInformation(Command command)
        {
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// 参照している波形ファイルの情報を更新します。
        /// 実際には情報をクリアするだけになります。
        /// </summary>
        private bool ExecuteUpdateWaveFileInformation(Command command)
        {
            foreach (CommonTabPage page in OpenedAllTabPages())
            {
                CommonListPanel panel = page.Panel as CommonListPanel;
                if (panel != null)
                {
                    foreach (CommonListCtrl listControl in panel.ListControls)
                    {
                        if (listControl.ItemsSource != null)
                        {
                            foreach (CommonListItem item in listControl.ItemsSource.Items)
                            {
                                item.ResetWaveFile();
                            }
                        }
                    }
                }
            }
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        private SoundSetBankBase FindSoundSetBankByBank(Bank bank)
        {
            SoundSetBankBase soundSetBank = null;

            if (bank != null)
            {
                string filePath = string.Empty;
                foreach (BankService bankService in ApplicationBase.Instance.BankServices.Values)
                {
                    if (bankService.BankDocument.Bank == bank)
                    {
                        filePath = bankService.BankDocument.Resource.Key;
                    }
                }

                SoundSetBankBase[] soundSetBanks =
                    ApplicationBase.Instance.ProjectService.SoundSetBanks;
                foreach (SoundSetBankBase s in soundSetBanks)
                {
                    if (s.FilePath == filePath)
                    {
                        soundSetBank = s;
                        break;
                    }
                }
            }

            return soundSetBank;
        }


        ///--------------------------------
        /// <summary>
        /// "プロジェクトの修復"が実行可能なのか調べる
        /// </summary>
        private CommandStatus QueryStatusRepairProject(Command command)
        {
            return (null != ProjectService.ProjectDocument) ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        ///--------------------------------
        /// <summary>
        /// "プロジェクトの修復"の実行
        /// </summary>
        private bool ExecuteRepairProject(Command command)
        {
            RepairProjectArgs args = new RepairProjectArgs();
            SoundProjectService projectService = null;
            HashSet<string> bankFilePaths = new HashSet<string>();

            projectService = Application.ProjectService as SoundProjectService;

            //
            if (UIService.ShowMessageBox
                (MessageResource.Inquire_RepairProject,
                  MessageResource.DialogTitle_Project_RepairProject_Text,
                  AppMessageBoxButton.YesNo,
                  AppMessageBoxImage.Warning) != AppMessageBoxResult.Yes)
            {
                return false;
            }

            //
            foreach (SoundSetDocument soundSetDocument in projectService.SoundSetDocuments)
            {

                if (soundSetDocument.Resource.CanWrite == false)
                {
                    continue;
                }

                soundSetDocument.OperationHistory.BeginTransaction();

                foreach (Component component in GetRepairCandidateSoundSetItem(soundSetDocument))
                {
                    if (component.Parameters.ContainsKey(ProjectParameterNames.FilePath) == true)
                    {
                        RepairSoundSetItem(args, soundSetDocument, component);

                        if (component is SoundSetBankBase)
                        {
                            var filePath = component.GetFilePath();
                            if (bankFilePaths.Contains(filePath) == false)
                            {
                                bankFilePaths.Add(filePath);
                            }
                        }
                    }
                }

                soundSetDocument.OperationHistory.EndTransaction();
            }

            //
            CommonTabPage page = null;
            FileResource resource = null;
            DocumentReference docRef = null;
            BankDocument bankDocument = null;
            bool repaired = false;

            foreach (string bankFilePath in bankFilePaths)
            {

                if (File.Exists(bankFilePath) == false)
                {
                    continue;
                }

                //読み取り専用のファイルなのか？
                if ((File.GetAttributes(bankFilePath) & FileAttributes.ReadOnly) != 0)
                {
                    continue;
                }

                //
                try
                {
                    resource = new FileResource(bankFilePath);
                    if ((docRef = Application.DocumentService.OpenDocument(resource)) == null)
                    {
                        continue;
                    }

                    bankDocument = docRef.Document as BankDocument;
                    if (bankDocument == null)
                    {
                        continue;
                    }

                    //
                    repaired = false;
                    bankDocument.OperationHistory.BeginTransaction();

                    foreach (Instrument instrument in bankDocument.Bank.Children)
                    {
                        if (RepairInstrument(args, bankDocument, instrument) == true)
                        {
                            repaired = true;
                        }
                    }

                    bankDocument.OperationHistory.EndTransaction();

                    //
                    if (repaired == true && (page = FindPage(bankDocument.Bank)) == null)
                    {
                        AddPage(bankDocument.Bank, null, true);
                    }
                }
                finally
                {
                    if (docRef != null)
                    {
                        docRef.Close();
                    }
                }
            }

            //
            UIService.ShowMessageBox
                (MessageResource.Message_CompleteRepairProject, String.Empty);

            //projectService.SendRepairProjectItemEvent();
            return true;
        }

        private IEnumerable<Component> GetRepairCandidateSoundSetItem(SoundSetDocument soundSetDocument)
        {
            foreach (var item in GetSoundSetItems(soundSetDocument))
            {
                yield return item;

                if (item is StreamSoundBase)
                {
                    foreach (var streamSoundTrack in item.Children)
                    {
                        yield return streamSoundTrack;
                    }
                }
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void RepairSoundSetItem(RepairProjectArgs e, SoundSetDocument soundSetDocument, Component component)
        {
            var componentName = component is StreamSoundTrackBase ? component.Parent.Name : component.Name;
            string newFilePath = null;
            var nowFilePath = component.Parameters[ProjectParameterNames.FilePath].Value as string;
            var message = String.Format(MessageResource.Message_RepairSoundSetItem,
                                        soundSetDocument.SoundSet.Name,
                                        componentName, nowFilePath);

            if (FindNewFilePath(e, message, nowFilePath, ref newFilePath) == true)
            {
                var operation = new SetParameterOperation(component.Parameters, ProjectParameterNames.FilePath, newFilePath);
                operation.Execute();
                soundSetDocument.OperationHistory.AddOperation(operation);
            }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool RepairInstrument(RepairProjectArgs e, BankDocument bankDocument, Instrument instrument)
        {
            SetParameterOperation operation = null;
            string nowFilePath = null;
            string newFilePath = null;
            string message = null;
            bool repaired = false;

            foreach (KeyRegion keyRegion in instrument.Children)
            {
                foreach (VelocityRegion velRegion in keyRegion.Children)
                {
                    nowFilePath = velRegion.Parameters[ProjectParameterNames.FilePath].Value as string;

                    message = String.Format(MessageResource.Message_RepairInstrument,
                                             bankDocument.Resource.Key,
                                             instrument.Name, nowFilePath);

                    if (FindNewFilePath(e, message, nowFilePath, ref newFilePath) == true)
                    {
                        repaired = true;

                        operation = new SetParameterOperation
                            (velRegion.Parameters, ProjectParameterNames.FilePath, newFilePath);
                        operation.Execute();
                        bankDocument.OperationHistory.AddOperation(operation);
                    }
                }
            }
            return repaired;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool FindNewFilePath(RepairProjectArgs e, string message, string nowFilePath, ref string newFilePath)
        {
            if (File.Exists(nowFilePath) == true)
            {
                return false;
            }

            if (e.IgnoreAll == true)
            {
                return false;
            }

            ReferenceFileFinder finder = new ReferenceFileFinder()
            {
                Text = message,
                IsFindAll = e.SpecifyFindDirectory,
                FindTargetDiretory = e.FindDirectory,
            };

            string filePath = finder.Validate(nowFilePath);

            if (null == filePath)
            {
                e.IgnoreAll = finder.IsIgnoreAll;
                e.SpecifyFindDirectory = false;
                return false;
            }

            newFilePath = filePath;

            e.SpecifyFindDirectory = finder.IsFindAll;

            if (finder.IsFindAll)
            {
                e.FindDirectory = finder.FindTargetDiretory;
            }

            return true;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private string FindFile(DirectoryInfo info, string fileName)
        {
            DirectoryInfo[] directoryInfos = null;
            FileInfo[] fileInfos = null;
            string filePath = null;

            if (info.Exists == false)
            {
                return null;
            }

            //
            fileInfos = info.GetFiles();
            foreach (FileInfo fileInfo in fileInfos)
            {
                if (fileInfo.Name == fileName)
                {
                    return fileInfo.FullName;
                }
            }

            //
            directoryInfos = info.GetDirectories();
            foreach (DirectoryInfo directoryInfo in directoryInfos)
            {
                if ((filePath = FindFile(directoryInfo, fileName)) != null)
                {
                    return filePath;
                }
            }

            return null;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private string FindFile(string directory, string filePath)
        {
            try
            {
                DirectoryInfo directoryInfo = new DirectoryInfo(directory);
                string fileName = Path.GetFileName(filePath);
                return FindFile(directoryInfo, fileName);
            }

            catch { return null; }
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private bool AlreadyOpenedBankDocument(string filePath)
        {
            foreach (BankDocument bankDocument in BankDocuments)
            {
                if (bankDocument.Resource.Key == filePath)
                {
                    return true;
                }
            }
            return false;
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private Component[] GetSoundSetItems(SoundSetDocument soundSetDocument)
        {
            List<Component> list = new List<Component>();
            Component[] packs = null;

            packs = soundSetDocument.SoundSet.Children.Cast<Component>().ToArray();
            foreach (Component pack in packs)
            {

                list.AddRange(pack.Children);

                if (pack is WaveSoundSetPack ||
                    pack is SequenceSoundSetPack)
                {
                    foreach (Component item in pack.Children)
                    {
                        list.AddRange(item.Children.Cast<Component>().ToArray());
                    }
                }
            }

            return list.ToArray();
        }

        ///--------------------------------
        /// <summary>
        /// HTML 出力で足りない項目はここで追加する。（プロジェクト読み込み時）
        /// </summary>
        private void CheckSoundListOutputs(SoundProject project)
        {
            if (project == null) { return; }

            foreach (SoundListOutput soundListOutput in project.SoundListOutputSettings.ListOutputs)
            {
                foreach (ListOutput listOutput in soundListOutput.ListOutputs)
                {
                    bool noColumn = true;
                    if (listOutput.Name == "Sounds")
                    {
                        foreach (ListColumnOutput output in listOutput.ColumnOutputs)
                        {
                            if (output.Name == ListTraits.ColumnName_UserParameter0)
                            {
                                noColumn = false;
                                break;
                            }
                        }
                        if (noColumn == true)
                        {
                            listOutput.ColumnOutputs.Add(this.CreateColumnOutput(ListTraits.ColumnName_UserParameter0, false));
                            listOutput.ColumnOutputs.Add(this.CreateColumnOutput(ListTraits.ColumnName_UserParameter1, false));
                            listOutput.ColumnOutputs.Add(this.CreateColumnOutput(ListTraits.ColumnName_UserParameter2, false));
                            listOutput.ColumnOutputs.Add(this.CreateColumnOutput(ListTraits.ColumnName_UserParameter3, false));
                        }
                    }

                    noColumn = true;
                    foreach (ListColumnOutput output in listOutput.ColumnOutputs)
                    {
                        if (output.Name == ListTraits.ColumnName_Comment1)
                        {
                            noColumn = false;
                            break;
                        }
                    }
                    if (noColumn == true)
                    {
                        listOutput.ColumnOutputs.Add(this.CreateColumnOutput(ListTraits.ColumnName_Comment1, false));
                        listOutput.ColumnOutputs.Add(this.CreateColumnOutput(ListTraits.ColumnName_Comment2, false));
                        listOutput.ColumnOutputs.Add(this.CreateColumnOutput(ListTraits.ColumnName_Comment3, false));
                        listOutput.ColumnOutputs.Add(this.CreateColumnOutput(ListTraits.ColumnName_Comment4, false));
                        listOutput.ColumnOutputs.Add(this.CreateColumnOutput(ListTraits.ColumnName_Comment5, false));
                        listOutput.ColumnOutputs.Add(this.CreateColumnOutput(ListTraits.ColumnName_Comment6, false));
                        listOutput.ColumnOutputs.Add(this.CreateColumnOutput(ListTraits.ColumnName_Comment7, false));
                        listOutput.ColumnOutputs.Add(this.CreateColumnOutput(ListTraits.ColumnName_Comment8, false));
                        listOutput.ColumnOutputs.Add(this.CreateColumnOutput(ListTraits.ColumnName_Comment9, false));
                    }
                }
            }
        }

        private ListColumnOutput CreateColumnOutput(string name, bool isOutput)
        {
            return new ListColumnOutput()
            {
                Name = name,
                IsOutput = isOutput,
            };
        }

        ///--------------------------------
        /// <summary>
        ///
        /// </summary>
        private void OnFileChanged(string filePath)
        {
            if (_enabledFileChanged == false)
            {
                return;
            }

            Document[] documents = Application.DocumentService.Documents;
            Document document = null;

            if (ProjectService.Documents.ContainsKey(filePath) == true)
            {
                document = ProjectService.Documents[filePath];
            }
            else if (Application.BankServices.Contains(filePath) == true)
            {
                document = Application.BankServices[filePath].BankDocument;
            }

            if (document != null &&
                ((document.Resource as FileResource).LastWriteTime <
                 File.GetLastWriteTime(filePath)))
            {
                try
                {
                    CancelAllEditing();
                    Application.FileWatcherService.Enabled = false;

                    if (Application.UIService.ShowMessageBox
                        (MessageResource.Message_UpdatedFile + "\n" +
                         MessageResource.Message_ConfirmReload,
                         AppMessageBoxButton.YesNo,
                         AppMessageBoxImage.Warning,
                         AppMessageBoxResult.Yes) == AppMessageBoxResult.Yes)
                    {
                        this.TryReload();
                        return;
                    }
                }
                finally
                {
                    Application.FileWatcherService.Enabled = true;
                }
            }
        }

        private void OnSaving(object sender, SaveHelperEventArgs e)
        {
            _enabledFileChanged = false; // セーブ中はファイル監視を無効にする
        }

        private void OnSaved(object sender, SaveHelperEventArgs e)
        {
            Application.FileWatcherService.CheckClear(); // セーブ後はファイル監視をクリアする。
            _enabledFileChanged = true;                  // ファイル監視を有効にする
        }

        /// <summary>
        ///
        /// </summary>
        private void CancelAllEditing()
        {
            IEditableControlContainer container = GetEditableControlContainer();
            if (container != null)
            {
                container.CancelEdit();
            }
        }

        /// <summary>
        ///
        /// </summary>
        private IEditableControlContainer GetEditableControlContainer()
        {
            //
            Control activeControl = _leftSplitContainer.ActiveControl;

            while ((activeControl is CommonPanel) == false &&
                  (activeControl is IKeyPreviewReceiver) == false)
            {
                ContainerControl container = activeControl as ContainerControl;
                if (container == null)
                {
                    break;
                }

                activeControl = container.ActiveControl;
            }

            // ドッキングコンテナ(下)からフォーカスを持っているコントロールを探します。
            if (activeControl == null)
            {
                foreach (DockingPage page in _bottomDockingToolContainer.Pages)
                {
                    if (page.ContainsFocus == true)
                    {
                        activeControl = page;
                        break;
                    }
                }
            }

            return activeControl as IEditableControlContainer;
        }


        ///--------------------------------
        /// <summary>
        /// リスト項目設定関連
        /// </summary>
        private void OnClickPresetListColumnsOnSplitButton(object sender, EventArgs e)
        {
            string presetName = string.Empty;

            if (sender is ToolStripSplitButton)
            {
                presetName = (sender as ToolStripSplitButton).Text;
            }
            else if (sender is ToolStripMenuItem)
            {
                presetName = (sender as ToolStripMenuItem).Text;
            }

            if (string.IsNullOrEmpty(presetName) == false && ActivePage != null)
            {
                if (ActivePage.Panel is CommonListPanel)
                {
                    CommonListPanel panel = ActivePage.Panel as CommonListPanel;
                    panel.ApplyPresetListColumns(presetName);
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnSelectChangedProjectPanel(object sender, EventArgs e)
        {
            TreeItemSelectedDictionary dic = this.ProjectPanel.ProjectCtrl.GetItemSelecteds();
            if (dic.SelectedItemCount == 1)
            {
                foreach (ProjectTreeItem key in dic.GetAll())
                {
                    Component component = key.Target;
                    if (this.ActivePage != null)
                    {
                        CommonPanel panel = this.ActivePage.Panel;
                        if (panel != null)
                        {
                            if (panel is SoundSetPanel == true)
                            {
                                SoundSetPanel soundSetPanel = panel as SoundSetPanel;
                                if (component == soundSetPanel.CurrentItem)
                                {
                                    this.HistoryAdd(component);
                                    return;
                                }
                            }
                            else if (panel is BankPanel == true)
                            {
                                BankPanel bankPanel = panel as BankPanel;
                                if (component == bankPanel.SoundSetBank)
                                {
                                    this.HistoryAdd(component);
                                    return;
                                }
                            }
                        }
                    }
                    return;
                }
            }
        }

        private bool CheckAndWaitMonitoringTargetsIsZero()
        {
            this.errorCheckAndWait = false;

            this.checkAndWaitDialog = new ProgressDialog();
            this.checkAndWaitDialog.DoWork += DoWork;
            this.checkAndWaitDialog.ProgressChanged += ProgressChanged;
            this.checkAndWaitDialog.RunWorkerCompleted += RunWorkerCompleted;
            this.checkAndWaitDialog.Title = MessageResource.Message_CheckAndWaitTitle;
            this.checkAndWaitDialog.Message = MessageResource.Message_CheckAndWaitText;
            this.checkAndWaitDialog.ShowDialog();

            return this.errorCheckAndWait == false;
        }

        private void DoWork(object sender, System.ComponentModel.DoWorkEventArgs e)
        {
            DateTime time = DateTime.Now;
            int count = Application.RealtimeEditService.MonitoringTargets.Count();
            while (true)
            {
                int count1 = Application.RealtimeEditService.MonitoringTargets.Count();
                if (count1 <= 0)
                {
                    break;
                }
                else if (count != count1)
                {
                    count = count1;
                    time = DateTime.Now;
                }
                else
                {
                    DateTime time1 = DateTime.Now;
                    if (1000 * 10000 * this.checkAndWaitTime < time1.Ticks - time.Ticks) // CheckAndWaitTime秒以上はエラー
                    {
                        this.errorCheckAndWait = true;
                        break;
                    }
                }
            }
        }

        private void ProgressChanged(object sender, System.ComponentModel.ProgressChangedEventArgs e)
        {
        }

        private void RunWorkerCompleted(object sender, System.ComponentModel.RunWorkerCompletedEventArgs e)
        {
            this.checkAndWaitDialog.Close();
            if (this.errorCheckAndWait == true)
            {
                // 切断する。
                this.Disconnect();
                ApplicationBase.Instance.UIService.ShowMessageBox(MessageResource.Message_SndeditDeleteError, MessageResource.Message_UnknownError, AppMessageBoxButton.OK, AppMessageBoxImage.Error);
            }
            Application.RealtimeEditService.Clear();
        }

        private void EditComponentParameterValue(Component component, string paramName, string paramValue)
        {
            // 今はユーザーパラメータのみ。
            Dictionary<string, string> parameterNames
                = new Dictionary<string, string>()
                {
                    { "userparam", ProjectParameterNames.UserParameter },
                    { "userparam1", ProjectParameterNames.UserParameter1 },
                    { "userparam2", ProjectParameterNames.UserParameter2 },
                    { "userparam3", ProjectParameterNames.UserParameter3 },
                };

            paramName = paramName.ToLower();
            if (parameterNames.ContainsKey(paramName) == false ||
                component.Parameters.ContainsKey(parameterNames[paramName]) == false)
            {
                return;
            }

            OperationHistory operationHistory = null;

            SoundSet soundSet = this.GetSoundSet(component);
            if (soundSet != null)
            {
                SoundSetDocument soundSetDocument = GetSoundSetDocument(soundSet);
                if (soundSetDocument != null)
                {
                    operationHistory = soundSetDocument.OperationHistory;
                }
            }
#if false   // 今はユーザーパラメータのみ、バンクは関係ない。
            else
            {
                Bank bank = this.GetBank(component);
                if (bank != null)
                {
                    BankDocument bankDocument = this.BankDocuments.Where((a)=>a.Bank == bank).FirstOrDefault();
                    if (bankDocument != null)
                    {
                        operationHistory = bankDocument.OperationHistory;
                    }
                }
            }
#endif
            if (operationHistory == null)
            {
                return;
            }

            // 今はユーザーパラメータのみ、以下汎用性ない。
            ulong value = 0;
            if (ulong.TryParse(paramValue, out value) == true)
            {
                SetParameterOperation operation = new SetParameterOperation(component.Parameters, parameterNames[paramName], value);
                operation.Execute();
                operationHistory.AddOperation(operation);
            }
        }

        protected virtual void OutputWaveFile(Component component, string maxDuration, string isMonaural, string samplerate, string openFolder, string filePath)
        {
        }

        private void UpdateLeftDockingToolTitle()
        {
            (ToolPages[ProjectTreePanel.PageName] as ProjectTreePanel).UpdateTitle();
            _leftDockingToolContainer.UpdateTitle();
        }
    }

    ///--------------------------------------------------------------------------
    /// <summary>
    ///
    /// </summary>
    public class RepairProjectArgs
    {
        public bool IgnoreAll { get; set; }
        public bool SpecifyFindDirectory { get; set; }
        public string FindDirectory { get; set; }

        public RepairProjectArgs()
        {
            IgnoreAll = false;
            SpecifyFindDirectory = false;
            FindDirectory = null;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public interface IKeyPreviewReceiver
    {
        void KeyDown(KeyEventArgs e);
        void KeyUp(KeyEventArgs e);
    }
}
