﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------


namespace NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers
{
    using System;
    using System.Drawing;
    using System.Linq;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.ToolDevelopmentKit;

    public static class OptionHelper
    {
        /// <summary>
        ///
        /// </summary>
        private static XmlKeyAssignment Find(string mapName, string keyName, bool shiftKey, bool ctrlKey)
        {
            if (Keyboard == null)
            {
                return null;
            }

            XmlKeyAssignmentMap[] maps = Keyboard.KeyAssignmentMaps.KeyAssignmentMap
                .Where(m => m.Name == mapName)
                .ToArray();
            foreach (XmlKeyAssignmentMap map in maps)
            {
                XmlKeyAssignment[] keyAssignments = map.KeyAssignment
                    .Where(ka => (ka.Key == keyName &&
                                   ka.Shift == shiftKey &&
                                   ka.Ctrl == ctrlKey))
                    .ToArray();
                if (keyAssignments.Length > 0)
                {
                    return keyAssignments[0];
                }
            }
            return null;
        }

        /// <summary>
        ///
        /// </summary>
        public static string FunctionOfKeySpace
        {
            get
            {
                XmlKeyAssignment keyAssignment = Find(KeyAssignment.MapNames.SoundList,
                                                       KeyAssignment.Keys.Space,
                                                       false,
                                                       false);
                return (keyAssignment != null ?
                         keyAssignment.Function :
                         KeyAssignment.Functions.Preview_Play);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public static string FunctionOfKeySpaceWithShift
        {
            get
            {
                XmlKeyAssignment keyAssignment = Find(KeyAssignment.MapNames.SoundList,
                                                       KeyAssignment.Keys.Space,
                                                       true,
                                                       false);
                return (keyAssignment != null ?
                         keyAssignment.Function :
                         KeyAssignment.Functions.Preview_TogglePlayStop);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public static bool FunctionOfKeySpaceIsToggle
        {
            get
            {
                return OptionHelper.FunctionOfKeySpace ==
                    KeyAssignment.Functions.Preview_TogglePlayStop ? true : false;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public static bool FunctionOfKeySpaceWithShiftIsToggle
        {
            get
            {
                return OptionHelper.FunctionOfKeySpaceWithShift ==
                    KeyAssignment.Functions.Preview_TogglePlayStop ? true : false;
            }
        }

        /// <summary>
        /// オプション設定を取得します。
        /// </summary>
        private static XmlOptions Options
        {
            get { return ApplicationBase.Instance.AppConfiguration.Options; }
        }

        /// <summary>
        ///
        /// </summary>
        private static XmlOptionsApplicationKeyboard Keyboard
        {
            get
            {
                return Options.Application.Keyboard;
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public static class KeyAssignment
    {
        public static class MapNames
        {
            public static string SoundList
            {
                get { return "SoundList"; }
            }
        }

        public static class Keys
        {
            public static string Space
            {
                get { return "Space"; }
            }
        }

        public static class Functions
        {
            public static string Preview_Play
            {
                get { return "Preview.Play"; }
            }

            public static string Preview_TogglePlayStop
            {
                get { return "Preview.TogglePlayStop"; }
            }
        }
    }
    /// <summary>
    /// オプションを表示します。
    /// </summary>
    public class ShowOptionHandler : CommandHandler
    {
        private string commandID;
        private int tabIndex = 0;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="commandID">処理対象コマンドID。</param>
        /// <param name="helpFilePath">ヘルプファイルパス。</param>
        public ShowOptionHandler(string commandID)
        {
            if (null == commandID) { throw new ArgumentNullException("commandID"); }
            this.commandID = commandID;
        }

        /// <summary>
        /// コマンドハンドラが実行可能なコマンドを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get { return this.commandID; }
        }

        /// <summary>
        /// オプション設定を取得します。
        /// </summary>
        protected XmlOptions Options
        {
            get { return ApplicationBase.Instance.AppConfiguration.Options; }
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            OptionDialog dialog = this.CreateOptionDialog();

            this.LoadOptions(dialog);
            dialog.SelectedTabIndex = tabIndex;

            if (dialog.ShowDialog() == DialogResult.OK)
            {
                tabIndex = dialog.SelectedTabIndex;
                this.ApplyOptions(dialog);
                return true;
            }

            tabIndex = dialog.SelectedTabIndex;
            return false;
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual OptionDialog CreateOptionDialog()
        {
            return new OptionDialog();
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual void LoadOptions(OptionDialog dialog)
        {
            Assertion.Argument.NotNull(dialog);

            dialog.IsProjectAutoLoad =
               (null != this.Options.Application.General.StartupProject &&
               this.Options.Application.General.StartupProject.Enable);

            dialog.IsShowProjectFilePath = this.Options.Application.General.ShowProjectFilePath;
            dialog.IsShowFullPath = this.Options.Application.General.ShowFullFilePath;
            dialog.IsLineNumberVisible = this.Options.Application.General.ShowRowNumber;
            dialog.IsShowParameterBar = this.Options.Application.General.ParameterBar.Enable;
            dialog.IsShowGroupItemTargetComment = this.Options.Application.General.ShowGroupItemTargetComment;
            dialog.PasteToItemLower = this.Options.Application.General.PasteToItemLower;
            dialog.MoveNextLineCaretByEnter = this.Options.Application.General.MoveNextLineByEnter;
            dialog.BeginEditByEnter = this.Options.Application.General.BeginEditByEnter;

            dialog.ParameterBarColor = Color.FromArgb
                (this.Options.Application.General.ParameterBar.Color.Alpha,
                  this.Options.Application.General.ParameterBar.Color.Red,
                  this.Options.Application.General.ParameterBar.Color.Green,
                  this.Options.Application.General.ParameterBar.Color.Blue);

            dialog.IsNotCloseConvertDialog = this.Options.Application.General.NotCloseConvertDialog;
            dialog.ParalellConversionCountMax = this.Options.Application.General.ParallelConversionCountMax;

            dialog.EditorUseNotepad = this.Options.Application.General.EditorUseNotepad;
            dialog.EditorUseEditor = this.Options.Application.General.EditorUseEditor;
            dialog.EditorCommand = this.Options.Application.General.EditorCommand;
            dialog.EditorArguments = this.Options.Application.General.EditorArguments;
            dialog.MidiEditorCommand = this.Options.Application.General.MidiEditorCommand;
            dialog.MidiEditorArguments = this.Options.Application.General.MidiEditorArguments;
            dialog.WavEditorCommand = this.Options.Application.General.WavEditorCommand;
            dialog.WavEditorArguments = this.Options.Application.General.WavEditorArguments;

            dialog.UsePreviewWaveChooseDialog = this.Options.Application.General.UsePreviewWaveChooseDialog;
            dialog.MaintainTabOpen = this.Options.Application.General.MaintainTabOpen;

            dialog.FunctionOfKeySpace = OptionHelper.FunctionOfKeySpace;
            dialog.FunctionOfKeySpaceWithShift = OptionHelper.FunctionOfKeySpaceWithShift;

            dialog.AutoEnableMonitoringTarget = this.Options.Application.InGame.AutoEnableMonitoringTarget;
            dialog.AutoEnableBankBySequenceSound = this.Options.Application.InGame.AutoEnableBankBySequenceSound;
            dialog.EnabledAutoMeasureLoudness = this.Options.Application.Statistics.EnabledAutoMeasureLoudness;
        }

        /// <summary>
        ///
        /// </summary>
        protected virtual void ApplyOptions(OptionDialog dialog)
        {
            Assertion.Argument.NotNull(dialog);

            this.Options.Application.General.StartupProject.Enable = dialog.IsProjectAutoLoad;

            this.Options.Application.General.ShowProjectFilePath = dialog.IsShowProjectFilePath;
            this.Options.Application.General.ShowFullFilePath = dialog.IsShowFullPath;
            this.Options.Application.General.ShowRowNumber = dialog.IsLineNumberVisible;
            this.Options.Application.General.ParameterBar.Enable = dialog.IsShowParameterBar;
            this.Options.Application.General.PasteToItemLower = dialog.PasteToItemLower;
            this.Options.Application.General.ShowGroupItemTargetComment = dialog.IsShowGroupItemTargetComment;
            this.Options.Application.General.MoveNextLineByEnter = dialog.MoveNextLineCaretByEnter;
            this.Options.Application.General.BeginEditByEnter = dialog.BeginEditByEnter;

            this.Options.Application.General.NotCloseConvertDialog = dialog.IsNotCloseConvertDialog;
            this.Options.Application.General.ParallelConversionCountMax = dialog.ParalellConversionCountMax;

            this.Options.Application.General.EditorUseNotepad = dialog.EditorUseNotepad;
            this.Options.Application.General.EditorUseEditor = dialog.EditorUseEditor;
            this.Options.Application.General.EditorCommand = dialog.EditorCommand;
            this.Options.Application.General.EditorArguments = dialog.EditorArguments;
            this.Options.Application.General.MidiEditorCommand = dialog.MidiEditorCommand;
            this.Options.Application.General.MidiEditorArguments = dialog.MidiEditorArguments;
            this.Options.Application.General.WavEditorCommand = dialog.WavEditorCommand;
            this.Options.Application.General.WavEditorArguments = dialog.WavEditorArguments;

            FormsApplication.Instance.UIService.MainWindow.UpdateTitle();
            FormsUIService uiService = ApplicationBase.Instance.UIService as FormsUIService;
            uiService.MainWindow.ApplyColumnsText();
            uiService.MainWindow.ApplyOptionSettings();

            this.Options.Application.General.UsePreviewWaveChooseDialog = dialog.UsePreviewWaveChooseDialog;
            this.Options.Application.General.MaintainTabOpen = dialog.MaintainTabOpen;

            //
            XmlKeyAssignment keyAssign1 = new XmlKeyAssignment();
            keyAssign1.Key = KeyAssignment.Keys.Space;
            keyAssign1.Shift = false;
            keyAssign1.Ctrl = false;
            keyAssign1.Function = dialog.FunctionOfKeySpace;

            XmlKeyAssignment keyAssign2 = new XmlKeyAssignment();
            keyAssign2.Key = KeyAssignment.Keys.Space;
            keyAssign2.Shift = true;
            keyAssign2.Ctrl = false;
            keyAssign2.Function = dialog.FunctionOfKeySpaceWithShift;

            XmlKeyAssignmentMap map = new XmlKeyAssignmentMap();
            map.Name = KeyAssignment.MapNames.SoundList;
            map.KeyAssignment = new XmlKeyAssignment[] { keyAssign1, keyAssign2 };

            XmlKeyAssignmentMaps maps = new XmlKeyAssignmentMaps();
            maps.KeyAssignmentMap = new XmlKeyAssignmentMap[] { map };

            //
            this.Options.Application.Keyboard = new XmlOptionsApplicationKeyboard();
            this.Options.Application.Keyboard.KeyAssignmentMaps = maps;

            FormsApplication.Instance.UIService.MainWindow.UpdatePresetListColumnsOnSplitButton();

            this.Options.Application.InGame.AutoEnableMonitoringTarget = dialog.AutoEnableMonitoringTarget;
            this.Options.Application.InGame.AutoEnableBankBySequenceSound = dialog.AutoEnableBankBySequenceSound;
            this.Options.Application.Statistics.EnabledAutoMeasureLoudness = dialog.EnabledAutoMeasureLoudness;
        }
    }
}
