﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers
{
    using System;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;

    /// <summary>
    /// WindowsForms ベースのプロジェクト設定を表示します。
    /// </summary>
    public class ShowProjectSettingsHandler : CommandHandler
    {
        private static int selectedProjectSettingTabIndex = 0;
        private static int selectedHTMLSoundListTabIndex = 0;
        private static int selectedHTMLSoundListListBoxIndex = -1;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public ShowProjectSettingsHandler(IQueryCommandParameter queryParameter) :
            base(queryParameter)
        {
        }

        /// <summary>
        /// コマンドハンドラが実行可能なコマンドIDを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get { return ProjectCommands.ShowProjectSettings.ID; }
        }

        /// <summary>
        /// プロジェクト設定ダイアログを表示します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        public static bool Execute(SoundProjectService projectService)
        {
            return Execute(projectService, false);
        }

        /// <summary>
        /// プロジェクト設定ダイアログを表示し、ユーザーコマンドタブをアクティブにします。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        public static bool ExecuteActiveUserCommandTab(SoundProjectService projectService)
        {
            return Execute(projectService, true);
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            SoundProjectService projectService = GetTargetProjectService(parameters);
            if (null == projectService) { return CommandStatus.SupportedAndVisible; }

            return (null != projectService.Project) ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            return Execute(GetTargetProjectService(parameters));
        }

        /// <summary>
        /// プロジェクト設定ダイアログを表示します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        private static bool Execute(SoundProjectService projectService, bool activeUserCommandTab)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }

            ProjectSettingsDialog dialog = new ProjectSettingsDialog();

            dialog.LoadSettings(projectService);
            if (activeUserCommandTab == true)
            {
                dialog.SelectUserCommandTab();
            }
            else
            {
                dialog.SelectedProjectSettingTabIndex = selectedProjectSettingTabIndex;
            }
            dialog.SelectedHTMLSoundListTabIndex = selectedHTMLSoundListTabIndex;
            dialog.SelectedHTMLSoundListListBoxIndex = selectedHTMLSoundListListBoxIndex;

            dialog.ReadOnly = projectService.ProjectDocument.IsReadOnly;

            DialogResult result = dialog.ShowDialog();

            selectedProjectSettingTabIndex = dialog.SelectedProjectSettingTabIndex;
            selectedHTMLSoundListTabIndex = dialog.SelectedHTMLSoundListTabIndex;
            selectedHTMLSoundListListBoxIndex = dialog.SelectedHTMLSoundListListBoxIndex;

            if (result != DialogResult.OK)
            {
                return false;
            }

            dialog.SaveSettings(projectService);
            (ApplicationBase.Instance.UIService as FormsUIService).MainWindow.ApplyColumnsText();
            (ApplicationBase.Instance.UIService as FormsUIService).MainWindow.ApplyUserParameterColumns();
            (ApplicationBase.Instance.UIService as FormsUIService).RaiseProjectSettingsChanged();

            return true;
        }

        /// <summary>
        /// 対象プロジェクトサービスを取得します。
        /// </summary>
        private SoundProjectService GetTargetProjectService(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            if (!parameters.ContainsParameter(CommandParameterNames.TargetComponentService))
            {
                return null;
            }

            return parameters.GetParameter(CommandParameterNames.TargetComponentService)
                   as SoundProjectService;
        }
    }
}
