﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;

namespace NintendoWare.SoundMaker.Framework.Projects
{
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundFoundation.Projects;
    using System.Runtime.CompilerServices;

    public static class ComponentEx
    {
        /// <summary>
        /// <see cref="UpdateWaveFile"/> で発生した例外メッセージを保持します。
        /// (余計なコンテキストへの参照が発生しないようにするためメッセージだけ保持します)
        /// </summary>
        private static readonly ConditionalWeakTable<Component, string> WaveFileErrorMessages = new ConditionalWeakTable<Component, string>();

        /// <summary>
        /// WaveFile を更新します。
        /// </summary>
        /// <param name="component"></param>
        /// <param name="force">false を指定した場合、すでに WaveFile が読み込まれていたら更新しません。</param>
        public static void UpdateWaveFile(this Component component, bool force = true)
        {
            WaveFileErrorMessages.Remove(component);

            if (component is StreamSoundTrackBase ||
                component is WaveSoundBase ||
                component is VelocityRegion)
            {
                if (force == false)
                {
                    if ((component as StreamSoundTrackBase)?.WaveFile != null ||
                        (component as WaveSoundBase)?.WaveFile != null ||
                        (component as VelocityRegion)?.WaveFile != null)
                    {
                        return;
                    }
                }

                var filePath = component.GetFilePath();
                if (string.IsNullOrEmpty(filePath) == true)
                {
                    return;
                }

                if (File.Exists(filePath) == false)
                {
                    WaveFileErrorMessages.Add(component, Resources.MessageResource.Message_FileNotFound);
                    return;
                }

                WaveFile waveFile = null;

                try
                {
                    using (WaveFileReader reader = WaveFileReader.CreateInstance(filePath))
                    {
                        waveFile = reader.Open(filePath);
                    }
                }
                catch (Exception ex)
                {
                    if (string.IsNullOrEmpty(ex.Message) == false)
                    {
                        WaveFileErrorMessages.Add(component, string.Format(Resources.MessageResource.ErrorMessage_InvalidWaveFile, ex.Message));
                    }
                }

                if (component is StreamSoundTrackBase)
                {
                    StreamSoundTrackBase track = component as StreamSoundTrackBase;
                    track.WaveFile = waveFile;
                    if (waveFile != null && track.IsMultiChannelAACTrack() == false)
                    {
                        track.ChannelCount = waveFile.ChannelCount;
                    }
                }
                if (component is WaveSoundBase)
                {
                    (component as WaveSoundBase).WaveFile = waveFile;
                }
                if (component is VelocityRegion)
                {
                    (component as VelocityRegion).WaveFile = waveFile;
                }
            }
            else if (component is StreamSoundBase == true)
            {
                string errorMessage = null;

                foreach (Component child in component.Children)
                {
                    UpdateWaveFile(child, force);

                    if (errorMessage == null)
                    {
                        errorMessage = child.GetUpdateWaveFileErrorMessage();
                    }
                }

                if (string.IsNullOrEmpty(errorMessage) == false)
                {
                    WaveFileErrorMessages.Add(component, errorMessage);
                }
            }
        }

        /// <summary>
        /// <see cref="UpdateWaveFile"/> に失敗した場合のエラーメッセージを取得します。
        /// </summary>
        /// <param name="component"></param>
        /// <returns></returns>
        public static string GetUpdateWaveFileErrorMessage(this Component component)
        {
            string message = null;
            WaveFileErrorMessages.TryGetValue(component, out message);
            return message;
        }
    }
}
