﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

namespace NintendoWare.SoundMaker.Framework.Preview.Communications.Tool
{
    using NW4R.ProtocolSound;

    /// <summary>
    /// サウンド設定パケット
    /// </summary>
    public class SetSoundLabelPacket : ToolSoundPacket
    {
        private PreviewPlayerCollection _previewPlayers;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="previewPlayers">対象プレビュープレイヤーのコレクション</param>
        public SetSoundLabelPacket(PreviewPlayerCollection previewPlayers)
        {
            if (null == previewPlayers) { throw new ArgumentNullException("previewPlayers"); }
            _previewPlayers = previewPlayers;
        }

        #region ** プロパティのオーバーライド

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_SET_SOUNDLABELS; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get
            {
                return (ushort)(ToolSoundItemContainer.StructSize + (ToolSoundPreviewSoundLabel.StructSize * _previewPlayers.Count));
            }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// サウンド設定パケットを送信します。
        /// </summary>
        /// <param name="previewPlayers">対象プレビュープレイヤーのコレクション</param>
        public static void Send(PreviewPlayerCollection previewPlayers)
        {
            if (!CommManager.Instance.IsConnected) { return; }
            SendPacket(new SetSoundLabelPacket(previewPlayers));
        }

        #endregion

        #region ** メソッドのオーバーライド

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read(ProtocolSoundReader reader, CommPacketHeader header)
        {
            // TOOL_SOUND_SET_EFFECTPARAMETER は Write のみ
            Debug.Assert(false, "Don't call this method");
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            Debug.Assert(null != writer, "Writer is null");

            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            header.Write(writer);

            Debug.Assert(null != _previewPlayers, "Preview player is null");

            //
            // サウンドラベル一覧を出力する
            //
            // コンテナヘッダ
            ToolSoundItemContainer container = new ToolSoundItemContainer();

            container.offsetForData = ToolSoundItemContainer.StructSize;
            container.itemCount = (uint)_previewPlayers.Count;
            container.itemSize = ToolSoundPreviewSoundLabel.StructSize;
            container.Write(writer);

            // サウンドラベル
            foreach (IPreviewPlayer player in _previewPlayers)
            {

                ToolSoundPreviewSoundLabel outputItem = new ToolSoundPreviewSoundLabel();

                outputItem.index = player.Index;

                if (null != player.Item && 0 < player.Item.Name.Length)
                {

                    byte[] workLabel = Encoding.UTF8.GetBytes(player.Item.Name);

                    if (workLabel.Length >= outputItem.label.Length)
                    {
                        ShowError(FrameworkResources.MessageResource.Message_NameIsTooLong +
                                   " : " + outputItem.label);
                        return null;
                    }

                    workLabel.CopyTo(outputItem.label, 0);

                }
                else
                {
                    outputItem.label[0] = 0;
                }

                outputItem.Write(writer);

            }

            return header;
        }

        #endregion
    }

    /// <summary>
    /// サウンド情報取得パケット
    /// </summary>
    public class GetSoundInformationPacket : ToolSoundPacket
    {
        private PreviewPlayerInformationCollection _informations;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="previewPlayers">対象プレビュープレイヤーのコレクション</param>
        public GetSoundInformationPacket(PreviewPlayerInformationCollection informations)
        {
            if (null == informations) { throw new ArgumentNullException("informations"); }
            _informations = informations;
        }

        #region ** プロパティのオーバーライド

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_GET_SOUNDINFORMATIONS; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get { return 0; }
        }

        #endregion

        #region ** メソッド

        /// <summary>
        /// サウンド情報の取得パケットを送信します。
        /// </summary>
        public static void Send(PreviewPlayerInformationCollection informations)
        {
            if (!CommManager.Instance.IsConnected) { return; }
            SendPacket(new GetSoundInformationPacket(informations));
        }

        #endregion

        #region ** メソッドのオーバーライド

        /// <summary>
        /// 送信パケットを圧縮します。
        /// </summary>
        public override bool Compress(ToolPacket nextPacket)
        {
            return (nextPacket is GetSoundInformationPacket);
        }

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read(ProtocolSoundReader reader, CommPacketHeader header)
        {
            throw new Exception("unexpected call function");
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            Debug.Assert(null != writer, "Writer is null");

            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            header.Write(writer);

            // この１行は仮実装
            // Send用ToolPacket, Recv用ViewerPacketとして１つのクラスにまとめる
            // TODO: 謎 GetSoundInformationPacket.Received = false;

            return header;
        }

        #endregion
    }

#if false
    /// <summary>
    /// サウンド情報表示パケット
    /// </summary>
    public class ShowSoundInformationPacket : ToolSoundPacket
    {
        private IPreviewPlayer _target;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="target">対象プレビュープレイヤー</param>
        public ShowSoundInformationPacket( IPreviewPlayer target )
        {
            if ( null == target ) { throw new ArgumentNullException( "target" ); }
            _target = target;
        }

    #region ** プロパティのオーバーライド

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_SHOWSOUNDINFORMATION; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get { return ToolSoundPreviewPlayerID.StructSize; }
        }

    #endregion

    #region ** メソッド

        /// <summary>
        /// サウンド情報表示パケットを送信します。
        /// </summary>
        public static void Send( IPreviewPlayer target )
        {
            if ( !CommManager.Instance.IsConnected ) { return; }
            SendPacket( new ShowSoundInformationPacket( target ) );
        }

    #endregion

    #region ** メソッドのオーバーライド

        /// <summary>
        /// 送信パケットを圧縮します。
        /// </summary>
        public override bool Compress( ToolPacket nextPacket )
        {
            return ( nextPacket is ShowSoundInformationPacket );
        }

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read( ProtocolSoundReader reader, CommPacketHeader header )
        {
            throw new Exception( "unexpected call function" );
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override CommPacketHeader Write( ProtocolSoundWriter writer )
        {
            Debug.Assert( null != writer, "Writer is null" );

            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            header.Write( writer );

            Debug.Assert( null != _target, "unexpected error" );

            // パラメータを出力する
            ToolSoundPreviewPlayerID parameter = new ToolSoundPreviewPlayerID();
            parameter.index = _target.Index;
            parameter.Write( writer );

            return header;
        }

    #endregion
    }
#endif

#if false
    /// <summary>
    /// プレビュープレイヤーパラメータ設定パケット
    /// </summary>
    public class SetPreviewPlayerParameterPacket : ToolSoundPacket
    {
        private IPreviewPlayer _previewPlayer;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="previewPlayer">対象プレビュープレイヤー</param>
        public SetPreviewPlayerParameterPacket( IPreviewPlayer previewPlayer )
        {
            if ( null == previewPlayer ) { throw new ArgumentNullException( "previewPlayer" ); }
            _previewPlayer = previewPlayer;
        }

    #region ** プロパティのオーバーライド

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.TOOL_SOUND_SET_PREVIEWPLAYERPARAMETER; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get
            {
                return (ushort)ToolSoundPreviewPlayerParameter.StructSize;
            }
        }

    #endregion

    #region ** メソッド

        /// <summary>
        /// 送信パケットの圧縮
        /// </summary>
        public override bool Compress( ToolPacket nextPacket )
        {
            if ( !( nextPacket is SetPreviewPlayerParameterPacket ) ) { return false; }
            return ( _previewPlayer == ( nextPacket as SetPreviewPlayerParameterPacket )._previewPlayer );
        }

        /// <summary>
        /// プレビュープレイヤーパラメータ設定パケットを送信します。
        /// </summary>
        /// <param name="previewPlayer">対象プレビュープレイヤー</param>
        public static void Send( IPreviewPlayer previewPlayer )
        {
            if ( !CommManager.Instance.IsConnected ) { return; }
            SendPacket( new SetPreviewPlayerParameterPacket( previewPlayer ) );
        }

    #endregion

    #region ** メソッドのオーバーライド

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read( ProtocolSoundReader reader, CommPacketHeader header )
        {
            // Write のみ
            Debug.Assert( false );
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override CommPacketHeader Write( ProtocolSoundWriter writer )
        {
            Debug.Assert( null != writer );

            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            header.Write( writer );

            Debug.Assert( null != _previewPlayer );

            //
            // パラメータを出力する
            //
            // コンテナヘッダ
            ToolSoundPreviewPlayerParameter parameter = new ToolSoundPreviewPlayerParameter();

            parameter.previewPlayerIndex = _previewPlayer.Index;

            parameter.volume = _previewPlayer.Paramters.Volume;
            parameter.pitch = _previewPlayer.Paramters.Pitch;
            parameter.pan = _previewPlayer.Paramters.Pan;
            parameter.surroundPan = _previewPlayer.Paramters.SurroundPan;
            parameter.lpfFrequency = _previewPlayer.Paramters.LpfFrequency;
            parameter.biquadType = (ToolSoundBiquadType)_previewPlayer.Paramters.BiquadType;
            parameter.biquadValue = _previewPlayer.Paramters.BiquadValue;
            parameter.startOffsetTime = _previewPlayer.Paramters.StartOffsetTime;

            parameter.mainOutEnabled = _previewPlayer.Paramters.MainOut.Enabled ? 1 : 0;
            parameter.remote0OutEnabled = _previewPlayer.Paramters.RemoteOut[ 0 ].Enabled ? 1 : 0;
            parameter.remote1OutEnabled = _previewPlayer.Paramters.RemoteOut[ 1 ].Enabled ? 1 : 0;
            parameter.remote2OutEnabled = _previewPlayer.Paramters.RemoteOut[ 2 ].Enabled ? 1 : 0;
            parameter.remote3OutEnabled = _previewPlayer.Paramters.RemoteOut[ 3 ].Enabled ? 1 : 0;

            parameter.mainOutVolume = _previewPlayer.Paramters.MainOut.Volume;
            parameter.remote0OutVolume = _previewPlayer.Paramters.RemoteOut[ 0 ].Volume;
            parameter.remote1OutVolume = _previewPlayer.Paramters.RemoteOut[ 1 ].Volume;
            parameter.remote2OutVolume = _previewPlayer.Paramters.RemoteOut[ 2 ].Volume;
            parameter.remote3OutVolume = _previewPlayer.Paramters.RemoteOut[ 3 ].Volume;

            parameter.mainSend = _previewPlayer.Paramters.MainSend;
            parameter.effectAuxASend = _previewPlayer.Paramters.EffectAuxASend;
            parameter.effectAuxBSend = _previewPlayer.Paramters.EffectAuxBSend;
            parameter.effectAuxCSend = _previewPlayer.Paramters.EffectAuxCSend;

            parameter.fadeInFrames = (int)_previewPlayer.Paramters.FadeInFrames;
            parameter.fadeOutFrames = (int)_previewPlayer.Paramters.FadeOutFrames;
            parameter.pauseFadeFrames = (int)_previewPlayer.Paramters.PauseFadeFrames;

            parameter.Write( writer );

            return header;
        }

    #endregion
    }
#endif
}
