﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Linq;
using System.Text;

namespace NintendoWare.SoundMaker.Framework.Preview.Communications.Sre
{
    using NintendoWare.SoundFoundation.Projects;
    using NW4R.ProtocolSound;

    /// <summary>
    ///
    /// </summary>
    public class SndEditPacketHeader : CommPacketHeader
    {
        private readonly char[] Signature = new char[] { 'S', 'E', 'D', 'T' };

        private uint category = 0;
        private ResultTypes resultType = ResultTypes.TRUE;

        public enum ResultTypes : uint
        {
            ERROR_FLAG_MASK = 0x80000000,
            CATEGORY_MASK = 0x7f000000,
            CODE_MASK = 0x00ffffff,

            CATEGORY_SYSTEM = 0x00000000,
            CATEGORY_TARGET = 0x01000000,
            CATEGORY_PC = 0x02000000,

            TRUE = 0x00000000,
            FALSE = 0x00000001,

            ITEM_NOT_FOUND = (0x00000001 | ERROR_FLAG_MASK | CATEGORY_PC),
            ITEM_PREPAREING = (0x00000002 | ERROR_FLAG_MASK | CATEGORY_PC),

            FAILED = 0x000001 + CATEGORY_SYSTEM + ERROR_FLAG_MASK,

            NOT_INITIALIZED = 0x000001 + CATEGORY_TARGET + ERROR_FLAG_MASK,
            OUT_OF_MEMORY = 0x000002 + CATEGORY_TARGET + ERROR_FLAG_MASK,
            NOT_IMPLEMENTED = 0x000003 + CATEGORY_TARGET + ERROR_FLAG_MASK,
            CACHE_OVERFLOW = 0x000004 + CATEGORY_TARGET + ERROR_FLAG_MASK,

            TIMEOUT = 0x000101 + CATEGORY_TARGET + ERROR_FLAG_MASK,
            SEND_ERROR = 0x000102 + CATEGORY_TARGET + ERROR_FLAG_MASK,
            RECIEVE_ERROR = 0x000103 + CATEGORY_TARGET + ERROR_FLAG_MASK,
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SndEditPacketHeader(ProtocolSoundReader reader) : base(reader)
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SndEditPacketHeader(uint packetType, uint size) : base(packetType, size)
        {
        }

        /// <summary>
        /// パケットタイプ
        /// </summary>
        public override uint PacketType
        {
            get
            {
                uint packetType = (uint)(Category | Command);
                if (Reply) { packetType |= Constants.SRE_MESSAGE_REPLY_FLAG; }
                return packetType;
            }
        }

        /// <summary>
        /// カテゴリ
        /// </summary>
        public override uint Category
        {
            get
            {
                return this.category;
            }
            set
            {
                this.category = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public ResultTypes ResultType
        {
            get
            {
                return this.resultType;
            }
            set
            {
                this.resultType = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public ResultTypes ResultCode
        {
            get
            {
                if ((this.resultType & ResultTypes.ERROR_FLAG_MASK) == 0)
                {
                    return ResultTypes.TRUE;
                }
                return this.resultType & ResultTypes.CODE_MASK;
            }
        }

        /// <summary>
        /// 設定
        /// </summary>
        public override void Set(uint packetType, uint size)
        {
            Category = (uint)(packetType & Constants.SRE_CATEGORY_MASK);
            Command = (uint)(packetType & Constants.SRE_MESSAGE_MASK);
            Reply = ((packetType & Constants.SRE_MESSAGE_REPLY_FLAG_MASK) != 0);
            Error = false;
            Size = size;
        }

        /// <summary>
        /// 読み込み
        /// </summary>
        public override void Read(ProtocolSoundReader reader)
        {
            uint signature = reader.ReadUInt32();
            uint packetType = reader.ReadUInt32();
            uint size = reader.ReadUInt32();
            this.resultType = (ResultTypes)reader.ReadUInt32();
            Set(packetType, size);
        }

        /// <summary>
        /// 書き込み
        /// </summary>
        public override void Write(ProtocolSoundWriter writer)
        {
            writer.Write(Signature);
            writer.Write(PacketType);
            writer.Write(Size);
            writer.Write((uint)this.resultType);
            writer.Flush();
        }
    }
}
