﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Linq;
using System.Security.Cryptography;
using System.Text;

namespace NintendoWare.SoundMaker.Framework.Preview.Communications.Func
{
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Sre;
    using NintendoWare.SoundMaker.Preview.Service;
    using NintendoWare.SoundMaker.Windows.Forms;
    using NW4R.ProtocolSound;

    /// <summary>
    ///
    /// </summary>
    public abstract class FuncPacket : CommPacket
    {
        private static IFuncPacketFactoryCollection _factories = new IFuncPacketFactoryCollection();

        /// <summary>
        /// ビューアパケットの作成
        /// </summary>
        public static FuncPacket Create(CommPacketHeader header)
        {
            // 分類分けを行う
            FuncPacket packet = null;

            foreach (IFuncPacketFactory factory in _factories)
            {
                if (factory.Category != (header.PacketType & Constants.SRE_CATEGORY_MASK))
                {
                    continue;
                }

                packet = factory.CreatePacket(header);
                if (null != packet)
                {
                    return packet;
                }
            }

            switch (header.PacketType & Constants.SRE_CATEGORY_MASK)
            {
                case Constants.CATEGORY_TEST:
                    break;

                case Constants.SRE_CATEGORY_SYSTEM:
                    break;
            }

            return packet;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        protected FuncPacket()
        {
            Debug.Assert(!Error, "Error is true");
        }

        #region ** プロパティ

        public static IFuncPacketFactoryCollection Factories
        {
            get { return _factories; }
        }

        /// <summary>
        /// 返信フラグ
        /// </summary>
        public override bool Reply
        {
            get
            {
                return ((PacketType & Constants.SRE_MESSAGE_REPLY_FLAG) != 0);
            }
        }

        /// <summary>
        /// エラーフラグ
        /// </summary>
        public override bool Error
        {
            get
            {
                return false;
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected uint Alignment4(uint value)
        {
            return ((value + 3) / 4) * 4;
        }

        /// <summary>
        ///
        /// </summary>
        protected RealtimeEditService RealtimeEditService
        {
            get
            {
                return ApplicationBase.Instance.RealtimeEditService;
            }
        }

        #endregion
    }

    /// <summary>
    ///
    /// </summary>
    public class FuncPacketHeader : SndEditPacketHeader
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FuncPacketHeader(ProtocolSoundReader reader) : base(reader)
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FuncPacketHeader(uint packetType, uint size) : base(packetType, size)
        {
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class SoundPacketFactory : IFuncPacketFactory
    {
        uint IFuncPacketFactory.Category
        {
            //get { return GetCategory(); }
            get { return Constants.SRE_CATEGORY_SNDEDIT; }
        }

        FuncPacket IFuncPacketFactory.CreatePacket(CommPacketHeader header)
        {
            return InternalCreatePacket(header);
        }

        protected virtual uint GetCategory()
        {
            return Constants.SRE_CATEGORY_SYSTEM;
        }

        protected virtual FuncPacket InternalCreatePacket(CommPacketHeader header)
        {
            switch (header.PacketType & Constants.SRE_MESSAGE_MASK)
            {
                case Constants.SNDEDIT_GET_ITEM_EDIT_INFO:
                    return new SndEditGetItemEditInfoPacket();

                case Constants.SNDEDIT_BEGIN_ITEM_CACHE:
                    return new SndEditBeginItemCachePacket();

                case Constants.SNDEDIT_END_ITEM_CACHE:
                    return new SndEditEndItemCachePacket();
            }
            return null;
        }
    }

    /// <summary>
    /// アイテム情報の要求パケット
    /// </summary>
    public class SndEditGetItemEditInfoPacket : FuncPacket
    {
        private SoundEditInfo soundEditInfo = null;
        private SndEditPacketHeader.ResultTypes resultType = SndEditPacketHeader.ResultTypes.TRUE;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SndEditGetItemEditInfoPacket()
        {
        }

        /// <summary>
        /// ヘッダの作成
        /// </summary>
        public override CommPacketHeader CreateHeader()
        {
            FuncPacketHeader header = null;
            header = new FuncPacketHeader(PacketType | Constants.SRE_CATEGORY_SNDEDIT, Size);
            header.ResultType = this.resultType;
            return header;
        }

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.SNDEDIT_GET_ITEM_EDIT_INFO_REPLY; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get
            {
                return (ushort)(soundEditInfo != null ? soundEditInfo.Size : 0);
            }
        }

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read(ProtocolSoundReader reader, CommPacketHeader header)
        {
            int readLength = 0;

            uint nameLength = reader.ReadUInt32();
            readLength += sizeof(UInt32);

            try
            {
                if (nameLength > 0)
                {
                    byte[] buff = reader.ReadBytes((int)nameLength);
                    readLength += buff.Length;

                    Encoding encoder = Encoding.GetEncoding(932);
                    string name = encoder.GetString(buff, 0, (int)nameLength - 1);

                    SoundEditInfoPack pack = RealtimeEditService.PopSoundEditInfoPack(name);
                    if (pack != null)
                    {
                        this.soundEditInfo = pack.SoundEditInfo;
                        this.resultType = pack.ResultType;
                    }
                }
            }
            finally
            {
                int restLength = (int)header.Size - readLength;

                if (restLength > 0)
                {
                    reader.ReadBytes(restLength);
                }
            }
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            Debug.Assert(null != writer, "Writer is null");

            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            header.Write(writer);

            // パラメータを出力する
            if (this.soundEditInfo != null)
            {
                this.soundEditInfo.Write(writer);
            }

            return header;
        }
    }

    /// <summary>
    /// アイテムのキャッシュ開始パケット
    /// </summary>
    public class SndEditBeginItemCachePacket : FuncPacket
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SndEditBeginItemCachePacket()
        {
        }

        /// <summary>
        /// ヘッダの作成
        /// </summary>
        public override CommPacketHeader CreateHeader()
        {
            FuncPacketHeader header = null;
            header = new FuncPacketHeader(PacketType | Constants.SRE_CATEGORY_SNDEDIT, Size);
            return header;
        }

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.SNDEDIT_BEGIN_ITEM_CACHE_REPLY; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get
            {
                return 0;
            }
        }

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read(ProtocolSoundReader reader, CommPacketHeader header)
        {
            // 読み捨てています。
            int action = (int)reader.ReadByte();
            reader.ReadByte();
            reader.ReadUInt16();
            uint actualNameLength = reader.ReadUInt32();
            uint nameLength = this.Alignment4(actualNameLength);
            byte[] nameArray = reader.ReadBytes((int)nameLength);
            string name = Encoding.GetEncoding(932).GetString
                (nameArray, 0, (int)actualNameLength - 1);

            Debug.WriteLine(String.Format("BeginCache {0} : {1}", name, action));

            RealtimeEditService.BeginItemCache(name);
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            header.Write(writer);
            return header;
        }
    }

    /// <summary>
    /// アイテムのキャッシュ終了パケット
    /// </summary>
    public class SndEditEndItemCachePacket : FuncPacket
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SndEditEndItemCachePacket()
        {
        }

        /// <summary>
        /// ヘッダの作成
        /// </summary>
        public override CommPacketHeader CreateHeader()
        {
            FuncPacketHeader header = null;
            header = new FuncPacketHeader(PacketType | Constants.SRE_CATEGORY_SNDEDIT, Size);
            return header;
        }

        /// <summary>
        /// パケットの種類
        /// </summary>
        public override ushort PacketType
        {
            get { return Constants.SNDEDIT_END_ITEM_CACHE_REPLY; }
        }

        /// <summary>
        /// サイズ
        /// </summary>
        public override ushort Size
        {
            get
            {
                return 0;
            }
        }

        /// <summary>
        /// ストリームからデータを読み込みます。
        /// </summary>
        /// <param name="reader">対象ストリーム</param>
        /// <param name="header">パケットヘッダ</param>
        public override void Read(ProtocolSoundReader reader, CommPacketHeader header)
        {
            FuncPacketHeader funcHeader = header as FuncPacketHeader;
            Debug.Assert(funcHeader != null, "Header is not func packet header");
            RealtimeEditService.ResultTypes result = ResultCodeToResultTypes(funcHeader.ResultType);

            int action = (int)reader.ReadByte();
            bool isTargetAll = reader.ReadByte() != 0 ? true : false;
            reader.ReadUInt16();
            uint actualNameLength = reader.ReadUInt32();
            uint nameLength = this.Alignment4(actualNameLength);
            byte[] nameArray = reader.ReadBytes((int)nameLength);
            string name = Encoding.GetEncoding(932).GetString
                (nameArray, 0, (int)actualNameLength - 1);

            Debug.WriteLine(String.Format("EndCache {0} : {1}", name, action));

            RealtimeEditService.EndItemCache(name, result);
        }

        /// <summary>
        /// ストリームにデータを書き出します。
        /// </summary>
        /// <param name="writer">対象ストリーム</param>
        /// <returns>出力したパケットヘッダ</returns>
        public override CommPacketHeader Write(ProtocolSoundWriter writer)
        {
            // パケットヘッダを出力する
            CommPacketHeader header = CreateHeader();
            header.Write(writer);
            return header;
        }

        /// <summary>
        ///
        /// </summary>
        private RealtimeEditService.ResultTypes ResultCodeToResultTypes(SndEditPacketHeader.ResultTypes resultCode)
        {
            RealtimeEditService.ResultTypes result = RealtimeEditService.ResultTypes.True;

            switch (resultCode)
            {
                case SndEditPacketHeader.ResultTypes.TRUE:
                    result = RealtimeEditService.ResultTypes.True;
                    break;

                case SndEditPacketHeader.ResultTypes.FALSE:
                    result = RealtimeEditService.ResultTypes.False;
                    break;

                case SndEditPacketHeader.ResultTypes.FAILED:
                    result = RealtimeEditService.ResultTypes.False;
                    break;

                case SndEditPacketHeader.ResultTypes.NOT_INITIALIZED:
                    result = RealtimeEditService.ResultTypes.NotInitialized;
                    break;

                case SndEditPacketHeader.ResultTypes.OUT_OF_MEMORY:
                    result = RealtimeEditService.ResultTypes.OutOfMemory;
                    break;

                case SndEditPacketHeader.ResultTypes.CACHE_OVERFLOW:
                    result = RealtimeEditService.ResultTypes.CacheOverflow;
                    break;

                case SndEditPacketHeader.ResultTypes.NOT_IMPLEMENTED:
                    result = RealtimeEditService.ResultTypes.NotImplemented;
                    break;


                case SndEditPacketHeader.ResultTypes.TIMEOUT:
                    result = RealtimeEditService.ResultTypes.TimeOut;
                    break;

                case SndEditPacketHeader.ResultTypes.SEND_ERROR:
                    result = RealtimeEditService.ResultTypes.SendError;
                    break;

                case SndEditPacketHeader.ResultTypes.RECIEVE_ERROR:
                    result = RealtimeEditService.ResultTypes.RecieveError;
                    break;

                default:
                    Debug.Assert(false, "Unknown result type.");
                    result = RealtimeEditService.ResultTypes.False;
                    break;
            }

            return result;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public interface IFuncPacketFactory
    {
        #region ** プロパティ

        uint Category { get; }

        #endregion

        #region ** メソッド

        FuncPacket CreatePacket(CommPacketHeader header);

        #endregion
    }


    /// <summary>
    ///
    /// </summary>
    public class IFuncPacketFactoryCollection : Collection<IFuncPacketFactory>
    {
    }
}
