﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.Preview.Communications
{
    using System;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;
    using NW4R.ProtocolSound;

    #region ** 列挙子

    /// <summary>
    /// 出力モード
    /// </summary>
    public enum OutputMode
    {
        Invalid = 0,
        Monoral = ToolSoundOutputMode.OutputMode_Monoral,
        Stereo = ToolSoundOutputMode.OutputMode_Stereo,
        Surround = ToolSoundOutputMode.OutputMode_Surround,
        Dpl2 = ToolSoundOutputMode.OutputMode_Dpl2
    }

    #endregion

    /// <summary>
    /// サウンド通信クラス
    /// </summary>
    public class CommSound
    {
        private static CommSound instance;
        private static object _instanceLock = new object();

        private string _soundArchivePath = null;
        private bool _soundArchiveDirty = true;

        public CommSound()
        {
            // Viewer 通信の初期化
            this.AddSoundPacketFactory();

            CommManager.Instance = CreateCommManager();
            CommManager.Instance.ConnectionChanged += OnConnectionChanged;
        }

        /// <summary>
        /// インスタンスを取得します。
        /// </summary>
        /// <returns>MCSCommon のインスタンス</returns>
        public static CommSound Instance
        {
            get
            {
                Debug.Assert(instance != null, "instance is null");
                return instance;
            }
            set
            {
                instance = value;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsConnected
        {
            get
            {
                if (CommManager.Instance == null)
                {
                    return false;
                }
                return CommManager.Instance.IsConnected;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsConnecting
        {
            get
            {
                if (CommManager.Instance == null)
                {
                    return false;
                }
                return CommManager.Instance.IsConnecting;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsConnectedSre
        {
            get
            {
                if (CommManager.Instance == null)
                {
                    return false;
                }
                return CommManager.Instance.IsConnectedSndEdit;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsConnectingSre
        {
            get
            {
                if (CommManager.Instance == null)
                {
                    return false;
                }
                return CommManager.Instance.IsConnectingSndEdit;
            }
        }

        /// <summary>
        /// 接続します。
        /// </summary>
        public bool Connect()
        {
            if (CommManager.Instance == null)
            {
                CommManager.Instance = CreateCommManager();
            }

            _soundArchiveDirty = true;
            return CommManager.Instance.Connect();
        }

        /// <summary>
        /// 接続を切断します。
        /// </summary>
        public virtual void Disconnect()
        {
            CommManager.Instance.Disconnect();
        }

        /// <summary>
        /// サウンドアーカイブを読み込みます。
        /// </summary>
        public virtual void TransferSoundArchive(string path)
        {
            _soundArchivePath = path;
            _soundArchiveDirty = true;

            if (IsConnected != false)
            {
                SendSoundArchive();
            }
        }

        /// <summary>
        /// 出力モードを設定します。
        /// </summary>
        /// <param name="outputMode">出力モード</param>
        public void SetOutputMode(ToolSoundOutputMode outputMode)
        {
            SetSoundSystemParametersPacket.Send((ToolSoundOutputMode)outputMode);
        }


        /// <summary>
        /// HACK : sndctrl 暫定対応
        /// </summary>
        public virtual void Play(uint soundHandle, string soundName)
        {
            new NotImplementedException();
        }

        /// <summary>
        /// 再生
        /// </summary>
        public virtual void Play(uint previewPlayerIndex)
        {
            PlaySoundPacket.Send(previewPlayerIndex);
        }

        /// <summary>
        /// 一時停止
        /// </summary>
        public virtual void Pause(uint previewPlayerIndex)
        {
            PauseSoundPacket.Send(previewPlayerIndex);
        }

        /// <summary>
        /// 停止
        /// </summary>
        public virtual void Stop(uint previewPlayerIndex)
        {
            StopSoundPacket.Send(previewPlayerIndex);
        }

        #region ** イベント

        /// <summary>
        /// Viewerとの接続/切断イベントが発生すると実行されます。
        /// </summary>
        public delegate void ViewerConnectionDelegate
            (object sender, ConnectionChangedEventArgs e);
        public event ViewerConnectionDelegate ViewerConnectionChanged;

        /// <summary>
        /// SndEditとの接続/切断イベントが発生すると実行されます。
        /// </summary>
        public delegate void SndEditConnectionDelegate
            (object sender, ConnectionChangedEventArgs e);
        public event SndEditConnectionDelegate SndEditConnectionChanged;

        /// <summary>
        /// サウンドアーカイブの転送が開始されると発生します。
        /// </summary>
        public event EventHandler SoundArchiveSending;

        #endregion

        #region ** イベントハンドラ

        protected virtual void OnConnectionChanged(ConnectionChangedEventArgs e)
        {
            if (null != ViewerConnectionChanged)
            {
                ConnectionChangedEventArgs args =
                    new ConnectionChangedEventArgs(e.State, e.ErrorText);
                ViewerConnectionChanged(this, args);
            }
        }

        /// <summary>
        ///
        /// </summary>
        protected void OnSndEditConnectionChanged(ConnectionChangedEventArgs e)
        {
            if (SndEditConnectionChanged != null)
            {
                ConnectionChangedEventArgs args =
                    new ConnectionChangedEventArgs(e.State, e.ErrorText);
                SndEditConnectionChanged(this, args);
            }
        }

        /// <summary>
        /// Viewerとの接続/切断イベントが発生するとMCSManagerよりコールされます。
        /// </summary>
        protected virtual void OnViewerConnection(ConnectionChangedEventArgs e)
        {
            if (NintendoWare.SoundMaker.Framework.Preview.Communications.ConnectionState.Connected == e.State)
            {
                // Viewer 通信フラグの初期化
            }

            if (null != ViewerConnectionChanged)
            {
                ViewerConnectionChanged(this, new ConnectionChangedEventArgs(e.State, e.ErrorText));
            }
        }

        ///
        protected virtual CommManager CreateCommManager()
        {
            Debug.Assert(false, "Don't call this method");
            return null;
        }

        ///
        protected virtual void OnSoundArchiveSending(EventArgs e)
        {
            if (null != SoundArchiveSending)
            {
                SoundArchiveSending(this, e);
            }
        }

        #endregion

        /// <summary>
        /// SoundPacketFactoryを登録します。
        /// </summary>
        protected virtual void AddSoundPacketFactory()
        {
            Viewer.ViewerPacket.Factories.Add(new Viewer.SoundPacketFactory());
        }

        private void SendSoundArchive()
        {
            if (_soundArchiveDirty == false || _soundArchivePath == null || _soundArchivePath.Length == 0) { return; }

            LoadSarSoundPacket.Send(_soundArchivePath);
            OnSoundArchiveSending(EventArgs.Empty);

            _soundArchiveDirty = false;
        }
    }

    ///
    public interface IPreviewPlayer
    {
        Sound Item { get; }
        uint Index { get; }
    }

    public interface IPreviewPlayerInformation
    {
        IPreviewPlayer PreviewPlayer { get; }
    }

    /// <summary>
    ///
    /// </summary>
    public class PreviewPlayerCollection : Collection<IPreviewPlayer>
    {
    }

    /// <summary>
    ///
    /// </summary>
    public class PreviewPlayerInformationCollection : KeyedCollection<uint, IPreviewPlayerInformation>
    {
        /// <summary>
        /// 指定した要素からキーを抽出します。
        /// </summary>
        /// <param name="item">キーの抽出元要素。</param>
        /// <returns>指定した要素のキー。</returns>
        protected override uint GetKeyForItem(IPreviewPlayerInformation item)
        {
            if (null == item.PreviewPlayer) { throw new Exception("unexpected error"); }
            return item.PreviewPlayer.Index;
        }
    }
}
