﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
//using NW4R.ProtocolSound;

namespace NintendoWare.SoundMaker.Framework.Preview.Communications
{
    /// <summary>
    /// パケットヘッダ
    /// <para>
    /// 通信で送受信されるパケットのヘッダクラスです。
    /// ヘッダの内容に関してはProtocolSample.csをご覧ください。
    /// </para>
    /// </summary>
    public class CommPacketHeader
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CommPacketHeader()
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CommPacketHeader(uint packetType, uint size)
        {
            Set(packetType, size);
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CommPacketHeader(uint category, uint command,
                               bool reply, bool error, uint size)
        {
            Set(category, command, reply, error, size);
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CommPacketHeader(ProtocolSoundReader reader)
        {
            Read(reader);
        }

        /// <summary>
        /// 文字列の取得
        /// </summary>
        public override string ToString()
        {
            string message =
                "[ 0x" + PacketType.ToString("x4") + " " + Size + "Byte";
            if (Reply) { message += " Reply"; }
            if (Error) { message += " Error"; }
            message += " ]";
            return message;
        }

        //---------------------------------------------------------------------
        // アクセサ
        //---------------------------------------------------------------------
        /// <summary>
        /// パケットタイプ
        /// </summary>
        public virtual uint PacketType
        {
            get
            {
                uint packetType = (uint)(Category | Command);
                if (Reply) { packetType |= Constants.HEADER_REPLY_FLAG; }
                if (Error) { packetType |= Constants.HEADER_ERROR_FLAG; }
                return packetType;
            }
        }

        /// <summary>
        /// カテゴリ
        /// </summary>
        public virtual uint Category
        {
            get { return _category; }
            set
            {
                Debug.Assert(_category ==
                             (_category & Constants.HEADER_CATEGORY_MASK), "Unmatch category");
                _category = value;
            }
        }

        /// <summary>
        /// コマンド
        /// </summary>
        public uint Command
        {
            get { return _command; }
            set
            {
                Debug.Assert(_command ==
                             (_command & Constants.HEADER_COMMAND_MASK), "Unmatch command");
                _command = value;
            }
        }

        /// <summary>
        /// 返信
        /// </summary>
        public bool Reply
        {
            get { return _reply; }
            set { _reply = value; }
        }

        /// <summary>
        /// エラー
        /// </summary>
        public bool Error
        {
            get { return _error; }
            set { _error = value; }
        }

        /// <summary>
        /// サイズ。
        /// </summary>
        public uint Size
        {
            get { return _size; }
            set { _size = value; }
        }

        //---------------------------------------------------------------------
        /// <summary>
        /// 設定
        /// </summary>
        public void Set(uint category, uint command,
                        bool reply, bool error, uint size)
        {
            Category = category;
            Command = command;
            Reply = reply;
            Error = error;
            Size = size;
        }

        /// <summary>
        /// 設定
        /// </summary>
        public virtual void Set(uint packetType, uint size)
        {
            Category = (uint)(packetType & Constants.HEADER_CATEGORY_MASK);
            Command = (uint)(packetType & Constants.HEADER_COMMAND_MASK);
            Reply = ((packetType & Constants.HEADER_REPLY_FLAG) != 0);
            Error = ((packetType & Constants.HEADER_ERROR_FLAG) != 0);
            Size = size;
        }

        //---------------------------------------------------------------------
        // 読み書き
        //---------------------------------------------------------------------

        /// <summary>
        /// 読み込み
        /// </summary>
        public virtual void Read(ProtocolSoundReader reader)
        {
            uint packetType = reader.ReadUInt32();
            uint size = reader.ReadUInt32();
            Set(packetType, size);
        }

        /// <summary>
        /// 書き込み
        /// </summary>
        public virtual void Write(ProtocolSoundWriter writer)
        {
            writer.Write(PacketType);
            writer.Write(Size);
            writer.Flush();
        }

        //---------------------------------------------------------------------
        // 非公開メンバ
        //---------------------------------------------------------------------
        // カテゴリ
        private uint _category;
        // コマンド
        private uint _command;
        // サイズ
        private uint _size;
        // 返信
        private bool _reply;
        // エラー
        private bool _error;
    }
}
