﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Framework.FileFormats
{
    using System.IO;
    using SoundFoundation.Conversion;
    using SoundFoundation.Conversion.NintendoWareBinary;
    using ToolDevelopmentKit;

    /// <summary>
    /// Smfを解析します。
    /// </summary>
    public class SmfParser : SequenceSoundFileParser
    {
        /// <summary>
        /// ファイルを解析します。
        /// </summary>
        /// <param name="filePath">ファイルパスを指定します。</param>
        protected override void ParseInternal(string filePath)
        {
            Ensure.Argument.NotNull(filePath);
            Ensure.Argument.StringNotEmpty(filePath);

            // 相対パス指定の#includeが埋め込まれている可能性があるため、
            // smfファイルと同じ場所にテンポラリファイルの作成を試みます。
            string tempFilePath = Path.Combine(
                Path.GetDirectoryName(filePath),
                Path.GetRandomFileName()
            );

            try
            {
                using (FileStream stream = File.Create(tempFilePath)) { }
            }
            catch
            {
                tempFilePath = Path.GetTempFileName();
            }

            try
            {
                // シーケンスサウンドバイナリファイルにコンバートして解析します。
                ISmfConverter converter = ApplicationBase.Instance.Traits.ConversionTraits.SmfConverter;

                converter.InputPath = filePath;
                converter.OutputPath = tempFilePath;

                converter.Run(new ConversionContext(ApplicationBase.Instance.Traits.ConversionTraits));

                SequenceSoundTextParser textParser = new SequenceSoundTextParser()
                {
                    IsLittleEndian = this.IsLittleEndian,
                };
                textParser.Parse(tempFilePath);

                this.Labels = textParser.Labels;

                this.AddDependentFileTimes(filePath);
            }
            finally
            {
                File.Delete(tempFilePath);
            }
        }

        private void AddDependentFileTimes(string filePath)
        {
            Assertion.Argument.NotNull(filePath);
            Assertion.Argument.StringNotEmpty(filePath);

            this.DependentFileTimes[filePath] = File.GetLastWriteTime(filePath);
        }
    }
}
