﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Xml;
    using System.Xml.Schema;
    using System.Xml.Serialization;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using Schemas = NintendoWare.SoundMaker.Framework.Configurations.Schemas;

    /// <summary>
    /// アプリケーション設定を格納します。
    /// </summary>
    public abstract class AppConfiguration : ConfigurationBase
    {
        /// <summary>
        /// プリフェッチストリームが利用可能か調べます。
        /// </summary>
        public static bool EnabledPrefetch
        {
            get
            {
                AppConfiguration appConfig = ApplicationBase.Instance.AppConfiguration;
                return appConfig.EnabledPrefetchInternal;
            }
        }

        /// <summary>
        /// 波形ファイル出力で 48kHz が利用可能か調べます。
        /// </summary>
        public static bool Enabled48kHz
        {
            get
            {
                AppConfiguration appConfig = ApplicationBase.Instance.AppConfiguration;
                return appConfig.Enabled48kHzInternal;
            }
        }

        /// <summary>
        /// AACが利用可能か調べます。
        /// </summary>
        public static bool EnabledAAC
        {
            get
            {
                AppConfiguration appConfig = ApplicationBase.Instance.AppConfiguration;
                return appConfig.EnabledAACInternal;
            }
        }

        /// <summary>
        /// マルチチャンネルAACが利用可能か調べます。
        /// </summary>
        public static bool EnabledMultiChannelAAC
        {
            get
            {
                AppConfiguration appConfig = ApplicationBase.Instance.AppConfiguration;
                return appConfig.EnabledMultiChannelAACInternal;
            }
        }

        /// <summary>
        /// AACファイル拡張子を取得、設定します。
        /// </summary>
        public static string AACFileExtension { get; set; }

        /// <summary>
        /// wav ファイルの最大トラック数を取得、設定します。(aacファイル以外)
        /// </summary>
        public static int WaveFileMultiChannelCountMax { get; set; }

        private const int MruProjectsLimit = 10;

        // コンフィギュレーション情報
        private IDictionary<string, Schemas.XmlWindow> _xmlWindows;
        private IDictionary<string, Schemas.XmlToolPage> _xmlToolPages;
        private IDictionary<string, Schemas.XmlCommandBar> _xmlCommandBars;

        private Schemas.XmlKeyBindingMaps _xmlKeyBindingMaps;
        private Schemas.XmlMruFileList _xmlMruProjects;
        private Schemas.XmlOptions _xmlOptions;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="platform">対象プラットフォーム。</param>
        /// <param name="version">コンフィギュレーションファイルのバージョン。</param>
        public AppConfiguration(string platform, string version) : base(platform, version) { }

        /// <summary>
        /// 全てのウィンドウ設定を取得します。
        /// </summary>
        public IDictionary<string, Schemas.XmlWindow> Windows
        {
            get { return _xmlWindows; }
        }

        /// <summary>
        /// 全てのツールページ設定を取得します。
        /// </summary>
        public IDictionary<string, Schemas.XmlToolPage> ToolPages
        {
            get { return _xmlToolPages; }
        }

        /// <summary>
        /// 全てのコマンドバー設定を取得します。
        /// </summary>
        public IDictionary<string, Schemas.XmlCommandBar> CommandBars
        {
            get { return _xmlCommandBars; }
        }

        /// <summary>
        /// 全ての最近使ったプロジェクトを取得します。
        /// </summary>
        public Schemas.XmlMruFileList MruProjects
        {
            get { return _xmlMruProjects; }
        }

        /// <summary>
        /// オプションを取得します。
        /// </summary>
        public Schemas.XmlOptions Options
        {
            get { return _xmlOptions; }
        }

        public virtual string StreamWaveFileFilter
        {
            get
            {
                string filter;
                if (this.EnabledAACInternal == true)
                {
                    filter = string.Format("Wave Files (*.wav;*.aif;*.aiff;*{0})|*.wav;*.aif;*.aiff;*{0}|All Files (*.*)|*.*", AppConfiguration.AACFileExtension);
                }
                else
                {
                    filter = WaveFileQuester.Filter;
                }

                return filter;
            }
        }

        protected bool Enabled48kHzInternal { get; set; }

        protected bool EnabledPrefetchInternal { get; set; }

        protected bool EnabledAACInternal { get; set; }
        protected bool EnabledMultiChannelAACInternal { get; set; }

        /// <summary>
        /// デフォルト設定を読み込みます。
        /// </summary>
        /// <param name="settings">XmlReader 設定。</param>
        /// <param name="forced">現在の設定を全て破棄する場合は true、読み込み済みの設定を残す場合は false。</param>
        protected override void LoadDefaultsInternal(XmlReaderSettings settings, bool forced)
        {
            if (forced || null == _xmlWindows || null == _xmlToolPages)
            {
                _xmlWindows = LoadWindowsDefault(settings);
                _xmlToolPages = LoadToolPagesDefault(settings);
            }

            if (forced || null == _xmlCommandBars)
            {
                _xmlCommandBars = new Dictionary<string, Schemas.XmlCommandBar>();
            }

            if (forced || null == _xmlMruProjects)
            {
                _xmlMruProjects = new Schemas.XmlMruFileList(MruProjectsLimit);
            }

            if (forced || null == _xmlOptions)
            {
                _xmlOptions = ReadConfigurationPart<Schemas.XmlOptions>(
                                            FileResourcesProxy.OptionConfigurationDefault, settings);
            }
            if (_xmlOptions.Application.General.EditorUseNotepad == false &&
                _xmlOptions.Application.General.EditorUseEditor == false)
            {
                if (string.IsNullOrEmpty(_xmlOptions.Application.General.EditorCommand) == true)
                {
                    _xmlOptions.Application.General.EditorUseNotepad = true;
                }
                else
                {
                    _xmlOptions.Application.General.EditorUseEditor = true;
                }
            }

            if (_xmlOptions.Application.InGame == null)
            {
                _xmlOptions.Application.InGame = new Schemas.XmlOptionsApplicationInGame();
            }

            if (_xmlOptions.Application.Statistics == null)
            {
                _xmlOptions.Application.Statistics = new Schemas.XmlOptionsApplicationStatistics();
            }

            this.InitializeEnabledAAC();
            this.InitializeEnabledPrefetch();
            this.InitializeEnabled48kHz();
            this.InitializePrivateFunctionInformation();
        }

        /// <summary>
        /// スキーマセットを作成します。
        /// </summary>
        /// <returns>スキーマセット。</returns>
        protected override XmlSchemaSet CreateSchemaSet()
        {
            XmlSchemaSet schemas = base.CreateSchemaSet();

            schemas.Add(XmlSchema.Read(new StringReader(FileResourcesProxy.SchemaWindowConfiguration), null));
            schemas.Add(XmlSchema.Read(new StringReader(FileResourcesProxy.SchemaAppConfiguration), null));
            schemas.Add(XmlSchema.Read(new StringReader(FileResourcesProxy.SchemaOptionConfiguration), null));

            return schemas;
        }

        /// <summary>
        /// ウィンドウのデフォルト設定を読み込みます。
        /// </summary>
        /// <param name="settings">XmlReader 設定。</param>
        /// <returns>読み込んだウィンドウのディクショナリ。</returns>
        protected virtual IDictionary<string, Schemas.XmlWindow> LoadWindowsDefault(XmlReaderSettings settings)
        {
            return ReadConfigurationPart<Schemas.XmlWindows>(
                                    FileResourcesProxy.AppConfigurationWindowsDefault, settings).CreateDictionary();
        }

        /// <summary>
        /// ツールページのデフォルト設定を読み込みます。
        /// </summary>
        /// <param name="settings">XmlReader 設定。</param>
        /// <returns>読み込んだツールページのディクショナリ。</returns>
        protected virtual IDictionary<string, Schemas.XmlToolPage> LoadToolPagesDefault(XmlReaderSettings settings)
        {
            return ReadConfigurationPart<Schemas.XmlToolPages>(
                                    FileResourcesProxy.AppConfigurationToolPagesDefault, settings).CreateDictionary();
        }

        /// <summary>
        /// コンフィギュレーション要素を読み込みます。
        /// </summary>
        /// <param name="xmlElement">コンフィギュレーション要素。</param>
        /// <param name="basePath">基準パス。</param>
        protected override void ReadConfigurationElement(XmlElement xmlElement, string basePath)
        {
            if (null == xmlElement) { throw new ArgumentNullException("xmlElement"); }
            if (null == basePath) { throw new ArgumentNullException("basePath"); }

            switch (xmlElement.LocalName)
            {
                case "Windows":
                    _xmlWindows =
                            ReadConfigurationPart<Schemas.XmlWindows>(xmlElement.OuterXml).CreateDictionary();
                    break;

                case "ToolPages":
                    _xmlToolPages =
                            ReadConfigurationPart<Schemas.XmlToolPages>(xmlElement.OuterXml).CreateDictionary();
                    break;

                case "CommandBars":
                    _xmlCommandBars =
                            ReadConfigurationPart<Schemas.XmlCommandBars>(xmlElement.OuterXml).CreateDictionary();
                    break;

                case "KeyBindingMaps":
                    _xmlKeyBindingMaps =
                            ReadConfigurationPart<Schemas.XmlKeyBindingMaps>(xmlElement.OuterXml);
                    break;

                case "MruProjects":
                    _xmlMruProjects =
                            ReadConfigurationPart<Schemas.XmlMruProjects>(xmlElement.OuterXml).
                            CreateList(MruProjectsLimit);
                    break;

                case "Options":
                    _xmlOptions = ReadConfigurationPart<Schemas.XmlOptions>(xmlElement.OuterXml);
                    if (_xmlOptions.Application.InGame == null)
                    {
                        _xmlOptions.Application.InGame = new Schemas.XmlOptionsApplicationInGame();
                    }
                    if (_xmlOptions.Application.Statistics == null)
                    {
                        _xmlOptions.Application.Statistics = new Schemas.XmlOptionsApplicationStatistics();
                    }
                    break;
            }
        }

        /// <summary>
        /// コンフィギュレーション要素を書き出します。
        /// </summary>
        /// <param name="reader">XmlReader。</param>
        /// <param name="basePath">基準パス。</param>
        protected override void WriteConfigurationElements(XmlWriter writer, string basePath)
        {
            if (null == writer) { throw new ArgumentNullException("writer"); }

            WriteConfigurationPart<Schemas.XmlWindows>(writer,
                new Schemas.XmlWindows() { Items = _xmlWindows.Values.ToArray() });

            WriteConfigurationPart<Schemas.XmlToolPages>(writer,
                   new Schemas.XmlToolPages() { ToolPage = _xmlToolPages.Values.ToArray() });

            WriteConfigurationPart<Schemas.XmlCommandBars>(writer,
                   new Schemas.XmlCommandBars() { CommandBar = _xmlCommandBars.Values.ToArray() });

            WriteConfigurationPart<Schemas.XmlMruProjects>(writer,
                new Schemas.XmlMruProjects(_xmlMruProjects, MruProjectsLimit));

            WriteConfigurationPart<Schemas.XmlOptions>(writer, _xmlOptions);
        }

        /// <summary>
        /// コンフィギュレーションルートを書き出します。
        /// </summary>
        /// <param name="writer">XmlWriter。</param>
        protected override void WriteStartConfigurationRoot(XmlWriter writer)
        {
            base.WriteStartConfigurationRoot(writer);

            writer.WriteAttributeString("xmlns", "nwsw", null,
                                         "NintendoWare.SoundMaker.WindowConfiguration");

            writer.WriteAttributeString("xmlns", "nwsa", null,
                                         "NintendoWare.SoundMaker.AppConfiguration");

            writer.WriteAttributeString("xmlns", "nwso", null,
                                         "NintendoWare.SoundMaker.OptionConfiguration");
        }

        /// <summary>
        /// 読み込んだコンフィギュレーションを検証します。
        /// </summary>
        /// <returns>読み込んだ結果が正常な場合は true、それ以外の場合は false。</returns>
        protected override bool Validate()
        {
            if (!_xmlWindows.ContainsKey("MainWindow") ||
                 !_xmlWindows.ContainsKey("ProjectTreeToolWindow") ||
                 !_xmlWindows.ContainsKey("ParameterToolWindow") ||
                 !_xmlWindows.ContainsKey("FindResultToolWindow") ||
                 !_xmlWindows.ContainsKey("ConvertLogToolWindow") ||
                 !_xmlWindows.ContainsKey("ConvertWindow") ||
                 !_xmlWindows.ContainsKey("FindWindow")) { return false; }

            Schemas.XmlFrameWindow xmlMainWindow = _xmlWindows["MainWindow"] as Schemas.XmlFrameWindow;

            IDictionary<string, Schemas.XmlDockingHostControl> dockingHosts = xmlMainWindow.CreateDockingHostDictionary();
            if (!dockingHosts.ContainsKey("MainWindowLeft") ||
                 !dockingHosts.ContainsKey("MainWindowRight") ||
                 !dockingHosts.ContainsKey("MainWindowBottom")) { return false; }

            IDictionary<string, Schemas.XmlSplitControl> splitControls = xmlMainWindow.CreateSplitDictionary();
            if (!splitControls.ContainsKey("MainWindowLeft") ||
                 !splitControls.ContainsKey("MainWindowRight") ||
                 !splitControls.ContainsKey("MainWindowBottom")) { return false; }

            if (null == _xmlOptions.Application) { return false; }
            if (null == _xmlOptions.Application.General) { return false; }
            if (null == _xmlOptions.Application.General.StartupProject) { return false; }
            if (null == _xmlOptions.Application.General.ParameterBar) { return false; }
            if (null == _xmlOptions.Application.Sound) { return false; }
            if (null == _xmlOptions.Application.Midi) { return false; }

            return true;
        }

        protected virtual void InitializeEnabledAAC()
        {
        }

        protected virtual void InitializeEnabledPrefetch()
        {
        }

        protected virtual void InitializeEnabled48kHz()
        {
        }

        protected virtual void InitializePrivateFunctionInformation()
        {
        }
    }
}
