﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using NintendoWare.SoundFoundation.Commands;
using NintendoWare.SoundFoundation.Operations;
using NintendoWare.SoundFoundation.Projects;
using NintendoWare.SoundMaker.Framework.Utilities;

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    /// <summary>
    /// サウンドセットにアイテムを追加します。
    /// </summary>
    public abstract class AddSoundSetItemHandler : AddComponentHandler
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="commandID">対象コマンドID。</param>
        /// <param name="insert">挿入する場合は false、追加する場合は false。</param>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        protected AddSoundSetItemHandler(string commandID, bool insert, IQueryCommandParameter queryParameter)
            : base(commandID, insert, queryParameter) { }

        /// <summary>
        /// サウンドセットにアイテムを追加します。
        /// </summary>
        /// <param name="document">追加先のサウンドセットドキュメント。</param>
        /// <param name="parentItem">追加先のサウンドセットアイテム。</param>
        /// <param name="insertTargetItem">挿入位置のサウンドセットアイテム。</param>
        /// <param name="newItem">追加するサウンドセットアイテム。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        public static bool Execute(SoundSetDocument document, SoundSetItem parentItem,
                                    SoundSetItem insertTargetItem, SoundSetItem newItem)
        {
            if (null == document) { throw new ArgumentNullException("document"); }
            if (null == parentItem) { throw new ArgumentNullException("parentItem"); }
            if (null == newItem) { throw new ArgumentNullException("newItem"); }

            Operation operation = new InsertComponentOperation(parentItem, insertTargetItem, newItem);
            if (!operation.Execute()) { return false; }

            document.OperationHistory.AddOperation(operation);

            return true;
        }

        /// <summary>
        /// グループにグループアイテムを追加します。
        /// </summary>
        /// <param name="document">追加先のサウンドセットドキュメント。</param>
        /// <param name="group">追加先のグループ。</param>
        /// <param name="insertTargetItem">挿入位置のグループアイテム。</param>
        /// <param name="newItem">追加するグループアイテム。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        public static bool Execute(SoundSetDocument document, GroupBase group,
                                    GroupItemBase insertTargetItem, GroupItemBase newItem)
        {
            if (null == document) { throw new ArgumentNullException("document"); }
            if (null == group) { throw new ArgumentNullException("group"); }
            if (null == newItem) { throw new ArgumentNullException("newItem"); }

            Operation operation = new InsertComponentOperation(group, insertTargetItem, newItem);
            if (!operation.Execute()) { return false; }

            document.OperationHistory.AddOperation(operation);

            return true;
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            CommandStatus status = base.QueryStatusInternal(command, parameters);
            if (status.IsEnabled()) { return status; }

            return (GetInsertParentComponent(parameters) != null) ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// 対象バンクドキュメントを取得します。
        /// </summary>
        protected new SoundSetDocument GetTargetDocument(IQueryCommandParameter parameters)
        {
            return base.GetTargetDocument(parameters) as SoundSetDocument;
        }

        protected SoundSet GetTargetSoundSet(IQueryCommandParameter parameters)
        {
            return this.GetTargetDocument(parameters).SoundSet;
        }

        /// <summary>
        /// サウンドセットにアイテムを追加します。
        /// </summary>
        /// <param name="newItem">追加するサウンドセットアイテム。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected virtual bool Execute(IQueryCommandParameter parameters, SoundSetItem newItem)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }
            if (null == newItem) { throw new ArgumentNullException("newItem"); }

            SoundSetDocument targetDocument = GetTargetDocument(parameters);
            Component targetComponent = GetInsertParentComponent(parameters);

            if (null == targetDocument)
            {
                throw new ApplicationException("TargetDocument must not be null.");
            }

            if (!(targetComponent is SoundSetItem) ||
                 (targetComponent as SoundSetItem).SoundSet != targetDocument.SoundSet)
            {
                throw new ArgumentException("target component is invalid.");
            }

            return Execute(targetDocument, targetComponent as SoundSetItem,
                            GetInsertTargetComponent(parameters) as SoundSetItem, newItem);
        }

        /// <summary>
        /// グループにアイテムを追加します。
        /// </summary>
        /// <param name="newItem">追加するグループアイテム。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected virtual bool Execute(IQueryCommandParameter parameters, GroupItemBase newItem)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }
            if (null == newItem) { throw new ArgumentNullException("newItem"); }

            SoundSetDocument targetDocument = GetTargetDocument(parameters);
            Component targetComponent = GetInsertParentComponent(parameters);

            if (!(targetComponent is GroupBase) ||
                 (targetComponent as GroupBase).SoundSet != targetDocument.SoundSet)
            {
                throw new ArgumentException("target component is invalid.");
            }

            return Execute(targetDocument, targetComponent as GroupBase,
                            GetInsertTargetComponent(parameters) as GroupItemBase, newItem);
        }
    }
}
