﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Windows.Forms;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Resources;
    using NintendoWare.SoundMaker.Preview;
    using NintendoWare.SoundMaker.Profile;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

    /// <summary>
    /// コマンドを処理します。
    /// </summary>
    public class QueryAndMeasureHandler : CommandHandler
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="queryParameter">パラメータの問い合わせ先。</param>
        public QueryAndMeasureHandler(IQueryCommandParameter queryParameter)
            : base(queryParameter)
        {
        }

        /// <summary>
        /// コマンドハンドラが実行可能なコマンドIDを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get { return string.Empty; }
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            SoundProjectService projectService = GetTargetProjectService(parameters);
            if (null == projectService) { return CommandStatus.SupportedAndVisible; }
            if (0 == GetTargetSounds(projectService).Length) { return CommandStatus.SupportedAndVisible; }
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected override bool ExecuteInternal(Command command, IQueryCommandParameter parameters)
        {
            SoundProjectService projectService = GetTargetProjectService(parameters);
            return this.ExecuteInternal(projectService);
        }

        /// <summary>
        /// コマンドを実行します。
        /// </summary>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected virtual bool ExecuteInternal(SoundProjectService projectService)
        {
            if (PreExecAction() == false)
            {
                return false;
            }

            Sound[] sounds = GetTargetSounds(projectService);

            ProgressDialog progressDialog = new ProgressDialog();
            progressDialog.Minimum = 0;
            progressDialog.Maximum = sounds.Length;
            progressDialog.Style = ProgressBarStyle.Continuous;
            progressDialog.Message = MessageResource.Message_Measuring;

            progressDialog.DoWork += delegate (object sender, DoWorkEventArgs e)
                {
                    BackgroundWorker worker = sender as BackgroundWorker;
                    worker.WorkerReportsProgress = true;

                    int count = 0;
                    foreach (Sound sound in sounds)
                    {
                        ExecAction(sound);
                        worker.ReportProgress(++count);
                    }
                };

            progressDialog.ProgressChanged += delegate (object sender, ProgressChangedEventArgs e)
                {
                    progressDialog.Value = e.ProgressPercentage;
                };

            progressDialog.RunWorkerCompleted += delegate (object sender, RunWorkerCompletedEventArgs e)
                {
                    progressDialog.Close();
                };

            progressDialog.ShowDialog();

            PostExecAction();

            return true;
        }

        /// <summary>
        /// ExecuteInternal() の開始時に実行します。
        /// </summary>
        protected virtual bool PreExecAction()
        {
            return true;
        }

        /// <summary>
        /// ExecuteInternal() のループ処理内に実行します。
        /// </summary>
        protected virtual void ExecAction(Sound sound)
        {
        }

        /// <summary>
        /// ExecuteInternal() の終了時に実行します。
        /// </summary>
        protected virtual void PostExecAction()
        {
        }

        /// <summary>
        /// 対象プロジェクトサービスを取得します。
        /// </summary>
        private SoundProjectService GetTargetProjectService(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }
            return parameters.GetParameter(CommandParameterNames.TargetComponentService)
                as SoundProjectService;
        }

        /// <summary>
        /// 対象サウンドの一覧を取得します。
        /// </summary>
        private Sound[] GetTargetSounds(SoundProjectService projectService)
        {
            if (projectService.Project == null)
            {
                return new Sound[0];
            }

            return projectService.SequenceSounds
                .Where(s => s.IsEnabled == true)
                .ToArray();
        }
    }
}
