﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Framework.CommandHandlers
{
    using System;
    using System.IO;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundMaker.Framework.Commands;
    using CoreResources = NintendoWare.SoundFoundation.Resources;

    /// <summary>
    /// 新しいプロジェクトを作成します。
    /// </summary>
    public abstract class NewProjectHandler : CommandHandler
    {
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        /// <param name="queryTarget">対象の問い合わせ先。</param>
        protected NewProjectHandler(IQueryCommandParameter queryParameter) :
            base(queryParameter)
        {
        }

        /// <summary>
        /// コマンドハンドラが実行可能なコマンドIDを取得します。
        /// </summary>
        public override string TargetCommandID
        {
            get { return FileCommands.NewProject.ID; }
        }

        /// <summary>
        /// アプリケーションを取得します。
        /// </summary>
        protected ApplicationBase Application
        {
            get { return ApplicationBase.Instance; }
        }

        /// <summary>
        /// コマンドを実行できるかどうか調べます。
        /// </summary>
        /// <returns>コマンドの状態。</returns>
        protected override CommandStatus QueryStatusInternal(Command command, IQueryCommandParameter parameters)
        {
            return (null != GetTargetProjectService(parameters)) ?
                CommandStatus.SupportedAndEnabledAndVisible : CommandStatus.SupportedAndVisible;
        }

        /// <summary>
        /// 対象プロジェクトを取得します。
        /// </summary>
        protected SoundProjectService GetTargetProjectService(IQueryCommandParameter parameters)
        {
            if (null == parameters) { throw new ArgumentNullException("parameters"); }

            if (!parameters.ContainsParameter(CommandParameterNames.TargetComponentService))
            {
                return null;
            }

            return parameters.GetParameter(CommandParameterNames.TargetComponentService)
                   as SoundProjectService;
        }

        /// <summary>
        /// 新しいサウンドプロジェクトを作成します。
        /// </summary>
        /// <param name="filePath">プロジェクトファイルパス。</param>
        /// <returns>実行した場合は true、実行しなかった、キャンセルした場合は false。</returns>
        protected bool NewProject(SoundProjectService projectService, string filePath)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            try
            {
                projectService.Create(filePath);
            }
            catch (FileNotSupportedException exception)
            {
                ApplicationBase.Instance.UIService.ShowMessageBox(string.Format("{0}\n{1}",
                                                                                exception.Message,
                                                                                exception.FilePath));
                return false;
            }
            catch
            {
                ApplicationBase.Instance.UIService.ShowMessageBox(
                    string.Format(Resources.MessageResource.Message_UnknownError, filePath));
                return false;
            }

            return true;
        }

        /// <summary>
        /// 新しいサウンドセットを作成してプロジェクトに追加します。
        /// </summary>
        /// <param name="filePaths">サウンドセットファイルパス。</param>
        protected bool AddNewSoundSet(SoundProjectService projectService, string filePath)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            try
            {
                projectService.AddNewSoundSet(filePath);
            }
            catch
            {
                ApplicationBase.Instance.UIService.ShowMessageBox(
                    Resources.MessageResource.Message_UnknownError);
                return false;
            }

            return true;
        }

        /// <summary>
        /// 既存のサウンドセットをプロジェクトに追加します。
        /// </summary>
        /// <param name="filePaths">サウンドセットファイルパス。</param>
        protected bool AddExistingSoundSet(SoundProjectService projectService, string filePath)
        {
            if (null == projectService) { throw new ArgumentNullException("projectService"); }
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            try
            {
                projectService.AddSoundSet(filePath, projectService.Project);
            }
            catch (FileNotFoundException exception)
            {
                ApplicationBase.Instance.UIService.ShowMessageBox(
                   string.Format(Resources.MessageResource.Message_TargetFileNotFound,
                                  exception.FileName));
                return false;
            }
            catch
            {
                ApplicationBase.Instance.UIService.ShowMessageBox(
                    Resources.MessageResource.Message_UnknownError);
                return false;
            }

            return true;
        }

        /// <summary>
        /// 新しいサウンドセットを作成します。
        /// </summary>
        private SoundSet CreateSoundSet()
        {
            SoundSet soundSet = new SoundSet();

            soundSet.Children.Add(new StreamSoundPack());
            soundSet.Children.Add(new WaveSoundSetPack());
            soundSet.Children.Add(new SequenceSoundPack());
            soundSet.Children.Add(new SequenceSoundSetPack());
            soundSet.Children.Add(new SoundSetBankPack());
            soundSet.Children.Add(new WaveArchivePack());
            soundSet.Children.Add(new PlayerPack());
            soundSet.Children.Add(new GroupPack());

            return soundSet;
        }
    }
}
