﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Windows.Forms
{
    using System;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.FileFormats.Wave;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.FileFormats;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Preview;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

    /// <summary>
    ///
    /// </summary>
    public static class PreviewPlayerOperator
    {
        /// <summary>
        ///
        /// </summary>
        public static Preview.PreviewPlayer SelectedPreviewPlayer
        {
            get
            {
                if (null == PreviewPlayerPanel.SelectedItem) { return null; }
                return PreviewPlayerPanel.SelectedItem.PreviewPlayer;
            }
        }

        /// <summary>
        /// 再生開始時の例外チェックを行います。
        /// </summary>
        /// <param name="playAction"></param>
        public static void CheckExceptionForPlay(Action playAction)
        {
            try
            {
                playAction();
            }
            catch (CompressionAudioPreviewFileNotFoundException exception)
            {
                FormsApplicationCommon.Instance.UIService.ShowMessageBox(
                    string.Format(FrameworkResources.MessageResource.Message_CompressionAudioPreviewFileNotFound,
                                   exception.FileName));
            }
            catch (FileNotFoundException exception)
            {
                FormsApplicationCommon.Instance.UIService.ShowMessageBox(
                    string.Format(FrameworkResources.MessageResource.Message_TargetFileNotFound,
                                   exception.FileName));
            }
            catch (ApplicationException exception)
            {
                FormsApplicationCommon.Instance.UIService.ShowMessageBox(exception.Message);
            }
            catch (WaveFileFormatException exception)
            {
                FormsApplicationCommon.Instance.UIService.ShowMessageBox(
                    string.Format(
                        FrameworkResources.MessageResource.Message_FailedToPlaySoundWithReason,
                        string.Format(
                            FrameworkResources.MessageResource.ErrorMessage_InvalidWaveFile,
                            exception.Message)));
            }
            catch
            {
                FormsApplicationCommon.Instance.UIService.ShowMessageBox(
                    FrameworkResources.MessageResource.Message_FailedToPlaySound);
            }
        }

        /// <summary>
        /// Componentを再生します。
        /// </summary>
        /// <param name="component"></param>
        /// <param name="addPlayMuteParameter">false の場合はストリームサウンドのミュート／ソロ設定を適用しません。</param>
        public static void Play(Component component, bool addPlayMuteParameter = true)
        {
            FormsApplicationCommon.Instance.LoudnessService.Stop();

            if (component is StreamSoundBase)
            {
                try
                {
                    if (addPlayMuteParameter == true)
                    {
                        // ストリームサウンドの場合は Mute, Solo 設定をプレビューに反映させます。
                        // （ラウドネス計算など、他のプレビュー再生に作用しないようにするため、設定は再生開始後ただちに削除します。）
                        PreviewStreamSound.AddPlayMuteParameter(component as StreamSoundBase);
                    }
                    PlayInternal(component as Sound);
                }
                finally
                {
                    if (addPlayMuteParameter == true)
                    {
                        PreviewStreamSound.RemovePlayMuteParameter(component as StreamSoundBase);
                    }
                }
            }
            else
            {
                if (component is Sound)
                {
                    PlayInternal(component as Sound);
                }
            }
        }

        /// <summary>
        /// Soundを再生します。
        /// </summary>
        private static void PlayInternal(Sound sound)
        {
            if (null == sound) { throw new ArgumentNullException("sound"); }

            CheckExceptionForPlay(() =>
            {
                if (AppConfiguration.EnabledAAC == false)
                {
                    var aacFiles = sound.Children
                    .OfType<StreamSoundTrackBase>()
                    .Where(t => AACUtil.IsAACFilePrimitive(t.FilePath));

                    if (aacFiles.Any())
                    {
                        // Siglo では AAC ファイルはノーサポート
                        throw new WaveFileFormatException(aacFiles.First().FilePath);
                    }
                }

                try
                {
                    SelectedPreviewPlayer.Play(sound);
                }
                catch
                {
                    SelectedPreviewPlayer.Item = null;
                    throw;
                }
            });
        }

        /// <summary>
        ///
        /// </summary>
        public static void MuteChannel(Component component, bool value)
        {
            SelectedPreviewPlayer.MuteChannel(component, value);
        }

        /// <summary>
        ///
        /// </summary>
        public static void SetParameter(string parameterName, Sound sound)
        {
            if (sound == null)
            {
                throw new ArgumentNullException("sound");
            }

            Preview.PreviewPlayer previewPlayer = PreviewPlayerPanel.GetPreviewPlayer(sound);
            Debug.Assert(previewPlayer != null, "Sound not regist preview player.");
            previewPlayer.SetParameter(parameterName, sound);
        }

        /// <summary>
        ///
        /// </summary>
        private static MainWindow MainWindow
        {
            get { return FormsApplication.Instance.UIService.MainWindow; }
        }

        /// <summary>
        ///
        /// </summary>
        private static PreviewPlayerPanel PreviewPlayerPanel
        {
            get
            {
                return MainWindow.ToolPages[PreviewPlayerPanel.PageName] as PreviewPlayerPanel;
            }
        }
    }
}
