﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;

namespace NintendoWare.SoundMaker.Framework.Windows.Forms
{
    using Nintendo.AudioToolkit.DomainModels;
    using Nintendo.AudioToolkit.DataModels;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Core.Parameters;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.Forms;
    using NintendoWare.SoundMaker.Configurations;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Projects;
    using NintendoWare.SoundMaker.Windows.Forms;
    using NintendoWare.SoundMakerPlugin;

    public partial class FindResultPanel2Common : FindResultPanel2
    {
        private IParameterValue[] encodingValues = ListCtrlCommon.EncodingValues;
        private readonly CommonListFileWatcher _listFileWatcher;

        public FindResultPanel2Common()
        {
            InitializeComponent();

            if (AppConfigurationCommon.EnabledWaveSoundEdit == true)
            {
                BindCommand(EditCommands.UseWaveSoundWaveFile.ID, QueryStatusUseWaveSoundWaveFile, ExecuteUseWaveSoundWaveFile);
                BindCommand(EditCommands.UseWaveSoundResource.ID, QueryStatusUseWaveSoundResource, ExecuteUseWaveSoundResource);
            }

            this.SetReplaceArgumentDelegate(ProjectParameterNames.WaveEncoding, this.ReplaceArgumentWaveEncoding);
            this.ListDecorationEvaluator.AddDoShowErrorIconDelegate(ProjectParameterNames.WaveEncoding, this.DoShowErrorIconWaveEncoding);
            this.ListDecorationEvaluator.AddDoShowErrorIconDelegate(ProjectParameterNames.SampleRate, this.DoShowErrorIconSampleRate);

            _listFileWatcher = new CommonListFileWatcher(this, this.ListControls);
            _listFileWatcher.FileChanged += (sender, args) => this.RedrawPanel();
        }

        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            _listFileWatcher.Start();
        }

        public override void OnClosed()
        {
            base.OnClosed();

            _listFileWatcher.Stop();
        }

        /// <summary>
        /// "波形ファイルを使用する"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusUseWaveSoundWaveFile(Command command)
        {
            if (this.TargetComponents.All(c => c is WaveSound) == false)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// "波形ファイルを使用する"を実行します。
        /// </summary>
        private bool ExecuteUseWaveSoundWaveFile(Command command)
        {
            this.ExecuteOperation
                (this.TargetComponents,
                 delegate (Component component)
                {
                    WaveSound waveSound = component as WaveSound; // Delegate Captured Variable

                    return this.CreateActionOperation(waveSound, waveSound.WaveSoundResource, null);
                });

            return true;
        }


        /// <summary>
        /// "クリップを使用する"が実行可能なのか調べます。
        /// </summary>
        private CommandStatus QueryStatusUseWaveSoundResource(Command command)
        {
            if (this.TargetComponents.All(c => c is WaveSound) == false)
            {
                return CommandStatus.SupportedAndVisible;
            }

            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// "クリップを使用する"を実行します。
        /// </summary>
        private bool ExecuteUseWaveSoundResource(Command command)
        {
            this.ExecuteOperation
                (this.TargetComponents,
                 delegate (Component component)
                {
                    WaveSound waveSound = component as WaveSound; // Delegate Captured Variable

                    if (waveSound.WaveSoundResource == null)
                    {
                        WaveSoundResourceData waveSoundResourceData = new WaveSoundResourceData();
                        waveSoundResourceData.TrackResources.Add(new WaveSoundTrackResourceData());
                        waveSound.WaveSoundResource = new WaveSoundResource(waveSoundResourceData, (Application.ProjectService as SoundProjectServiceCommon).GetOperationExecutor(this.GetDocument(waveSound) as SoundSetDocument));
                    }

                    return this.CreateActionOperation(waveSound, null, waveSound.WaveSoundResource);
                });

            return true;
        }

        private ActionOperation CreateActionOperation(WaveSound waveSound, WaveSoundResource oldResource, WaveSoundResource newResource)
        {
            WaveSoundEditWindow waveSoundEditWindow = this.WaveSoundEditWindow; // Delegate Captured Variable

            return ActionOperation.CreateActionOperation<WaveSoundResource>(oldResource, newResource, delegate (WaveSoundResource v) { waveSound.WaveSoundResource = v; if (waveSoundEditWindow.WaveSound == waveSound) { waveSoundEditWindow.WaveSound = waveSound; } });
        }

        private FormsApplicationCommon Application
        {
            get
            {
                return FormsApplication.Instance as FormsApplicationCommon;
            }
        }

        private WaveSoundEditWindow WaveSoundEditWindow
        {
            get { return (FormsApplication.Instance.UIService.MainWindow as MainWindowCommon).WaveSoundEditWindow; }
        }

        /// <summary>
        /// 圧縮形式でエラーアイコンの表示非表示判定メソッドです。
        /// </summary>
        private string DoShowErrorIconWaveEncoding(Component component)
        {
            if (component is Instrument == true &&
                0 < component.Children.Count &&
                0 < component.Children[0].Children.Count)
            {
                component = component.Children[0].Children[0]; // VelocityRegion にします。
            }

            if (component is StreamSound == true || component is WaveSound == true || component is VelocityRegion == true)
            {
                var waveEncoding = (WaveEncoding)component.Parameters[ProjectParameterNames.WaveEncoding].Value;
                if (this.Application.CurrentSoundMakerPlugin.IsValidateWaveEncodingValue(waveEncoding) == false)
                {
                    // 波形のエンコード形式が不正です。
                    return string.Format(Resources.MessageResource.ErrorMessage_InvalidWaveEncoding, waveEncoding);
                }
            }

            return null;
        }

        private IInplaceEditorArgument ReplaceArgumentWaveEncoding(ListColumnStyle style,
                                                                   IInplaceEditorArgument origArgs,
                                                                   ComponentListItem item)
        {
            ISoundMakerPlugin plugin = SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin;

            return new DropDownListInplaceEditor.Argument(encodingValues.Where(e => plugin.IsValidateWaveEncodingValue((WaveEncoding)e.Value)).ToArray());
        }

        /// <summary>
        /// サンプルレートのエラーアイコンの表示非表示判定メソッドです。
        /// </summary>
        private string DoShowErrorIconSampleRate(Component component)
        {
            if (component.Parameters.ContainsKey(ProjectParameterNames.IsResampleEnabled) == false ||
                (bool)component.Parameters[ProjectParameterNames.IsResampleEnabled].Value == false)
            {
                return null;
            }

            if (component is StreamSound == true)
            {
                // トラックを抽出します。
                var tracks = component.Children.Cast<StreamSoundTrack>();

                // 波形ファイルの情報がなければ更新します。
                tracks
                    .Where(t => t.WaveFile == null)
                    .ToList()
                    .ForEach(t => t.UpdateWaveFile());

                // １つでもダウンサンプルではなくアップサンプルになっていたらエラーにします。
                if (tracks
                    .Where(t => t.WaveFile != null)
                    .Any(t => t.WaveFile.SampleRate < (component as StreamSound).SampleRate))
                {
                    // アップサンプルはサポートされません。
                    return Resources.MessageResource.ErrorMessage_UpsampleNotSupported;
                }

                return null;
            }

            if (component is WaveSound == true)
            {
                var waveSound = component as WaveSound;

                if (waveSound.WaveFile == null)
                {
                    waveSound.UpdateWaveFile();
                }

                // ダウンサンプルではなくアップサンプルになっていたらエラーにします。
                if (waveSound.WaveFile != null && waveSound.WaveFile.SampleRate < waveSound.SampleRate)
                {
                    // アップサンプルはサポートされません。
                    return Resources.MessageResource.ErrorMessage_UpsampleNotSupported;
                }

                return null;
            }

            if (component is Instrument == true)
            {
                // VelocityRegion を抽出します。
                var velocityRegions = component.Children
                    .Cast<KeyRegion>()
                    .SelectMany(k => k.Children.Cast<VelocityRegion>());

                // 波形ファイルの情報がなければ更新します。
                velocityRegions
                    .Where(v => v.WaveFile == null)
                    .ToList()
                    .ForEach(v => v.UpdateWaveFile());

                // １つでもダウンサンプルではなくアップサンプルになっていたらエラーにします。
                if (velocityRegions
                    .Where(v => v.WaveFile != null)
                    .Any(v => v.WaveFile.SampleRate < (component as Instrument).SampleRate))
                {
                    // アップサンプルはサポートされません。
                    return Resources.MessageResource.ErrorMessage_UpsampleNotSupported;
                }

                return null;
            }

            return null;
        }
    }
}
