﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using NintendoWare.SoundMaker.Configurations;
using NintendoWare.SoundMaker.Configurations.Schemas;
using NintendoWare.SoundMaker.Framework;
using NintendoWare.SoundMaker.Framework.Windows;
using NintendoWare.SoundMaker.Preview;
using NintendoWare.SoundMaker.Preview.Communications;
using NintendoWare.SoundMaker.Preview.Configurations;
using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;
using MCS = NintendoWare.SoundMaker.Preview.MCS;

namespace NintendoWare.SoundMaker.Windows.Forms
{
    public partial class EffectControlPane : UserControl
    {
        private EffectParameterCollection _parameterSets;
        private uint _deviceTypeValue;

        private bool _communicationLocked = false; // Viewer との通信ロック

        public event EventHandler ChangePresets;

        public EffectControlPane()
        {
            InitializeComponent();
            _effectSelectionPane.EffectSelectChanged += OnEffectSelectChanged;
        }

        public PreviewPlayerManager PreviewPlayerManager
        {
            get; set;
        }

        public PresetConfigurationCommon PresetConfiguration
        {
            get; set;
        }

        public EffectParameterSet Parameters
        {
            get
            {
                return _effectParameterPane.Parameters;
            }
            set
            {
                if (value != null)
                {
                    _effectParameterPane.Parameters = value;
                }
            }
        }

        public void ResetItemText()
        {
            _presetControl.ResetItemText();
        }

        public void ClearItemText()
        {
            _presetControl.ClearItemText();
        }

        public void Initialize(uint deviceTypeValue,
                               EffectParameterCollection parameterSets)
        {
            _deviceTypeValue = deviceTypeValue;
            _parameterSets = parameterSets;
            _effectSelectionPane.Initialize(parameterSets);
            _effectParameterPane.Parameters = _effectSelectionPane.SelectedParameters;

            _presetControl.DefaultPresetName = Resources.MessageResource.Label_NewPresetName;
            _presetControl.ResetItemText();
        }

        public void UpdatePresets()
        {
            _presetControl.InitializeItems(PresetConfiguration.Effects.ToArray());
            _presetControl.Enabled = true;

            if (PresetConfiguration.Effects.Contains(_presetControl.SelectedItemName) == false)
            {
                this.ClearItemText();
            }
        }

        public void ClearPresets()
        {
            _presetControl.InitializeItems(new object[0]);
            _presetControl.Enabled = false;
        }

        public void Synchronize()
        {
            if (_communicationLocked == true) { return; }

            // 全エフェクトパラメータをViewerに送信する
            FormsApplicationCommon.Instance.CommunicationService.SetEffectParameters(_parameterSets, _deviceTypeValue);

            if (null == _viewerNotConnectedError) { return; }

            if (this.PreviewPlayerManager != null &&
                this.PreviewPlayerManager.IsSynchronizedViewer == false)
            {
                _viewerNotConnectedError.SetIconAlignment(this, ErrorIconAlignment.TopLeft);
                _viewerNotConnectedError.SetIconPadding(this, -_viewerNotConnectedError.Icon.Width);
                _viewerNotConnectedError.SetError(this, Resources.MessageResource.Message_ThisControlIsViewerOnly);
            }
            else
            {
                _viewerNotConnectedError.Clear();
            }
        }


        #region ** イベントハンドラのオーバーライド / プリセット処理

        private void UpdatePresetsInternal()
        {
            _presetControl.InitializeItems(PresetConfiguration.Effects.ToArray());
            _presetControl.Enabled = true;

            if (this.ChangePresets != null)
            {
                this.ChangePresets(this, EventArgs.Empty);
            }
        }

        private void OnLoadItem(object sender, PresetItemEventArgs e)
        {
            _communicationLocked = true;

            new EffectDataApplier(_parameterSets).Apply(PresetConfiguration.Effects[e.ItemName]);

            _communicationLocked = false;

            // 全エフェクトパラメータをViewerに送信する
            Synchronize();
        }

        private void OnSaveItem(object sender, PresetItemEventArgs e)
        {
            if (PresetConfiguration.Effects.Contains(e.ItemName))
            {

                if (ApplicationBase.Instance.UIService.ShowMessageBox(string.Format(
                                                                        FrameworkResources.MessageResource.Message_ConfirmOverwritePreset,
                                                                        e.ItemName),
                                                          null,
                                                          AppMessageBoxButton.YesNo,
                                                          AppMessageBoxImage.Question,
                                                          AppMessageBoxResult.No
                                                          ) != AppMessageBoxResult.Yes)
                {
                    return;
                }

            }
            else
            {
                PresetConfiguration.Effects.Add(
                                                new XmlEffectData()
                                                {
                                                    Name = e.ItemName,
                                                    Effects = new XmlEffects(),
                                                });
            }

            new EffectDataApplier(_parameterSets).Extract(PresetConfiguration.Effects[e.ItemName]);

            PresetConfiguration.Save();

            UpdatePresetsInternal();
        }

        private void OnDeleteItem(object sender, PresetItemEventArgs e)
        {
            if (ApplicationBase.Instance.UIService.ShowMessageBox(
                                                      string.Format(
                                                                    FrameworkResources.MessageResource.Message_ConfirmDeletePreset,
                                                                    e.ItemName),
                                                      null,
                                                      AppMessageBoxButton.YesNo,
                                                      AppMessageBoxImage.Question,
                                                      AppMessageBoxResult.No
                                                      ) != AppMessageBoxResult.Yes)
            {
                return;
            }

            PresetConfiguration.Effects.Remove(e.ItemName);
            PresetConfiguration.Save();

            UpdatePresetsInternal();
            this.ClearItemText();
        }

        private void OnEffectSelectChanged(object sender, EventArgs e)
        {
            EffectSelectionPane target = sender as EffectSelectionPane;
            Debug.Assert(null != target, "target is null");

            this.Parameters = target.SelectedParameters;
        }

        #endregion

        #region ** メソッド

        #endregion
    }
}
