﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Windows.Forms;

namespace NintendoWare.SoundMaker.Windows.Forms
{
    // テスト
    using Nintendo.AudioToolkit.DataModels;

    using Nintendo.AudioToolkit.DomainModels;
    using Nintendo.AudioToolkit.Operations;
    using Nintendo.AudioToolkit.Views;
    using NintendoWare.SoundFoundation.CommandHandlers;
    using NintendoWare.SoundFoundation.Commands;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.Operations;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.SoundFoundation.Windows.CommandBars;
    using NintendoWare.SoundMaker.Commands;
    using NintendoWare.SoundMaker.Configurations;
    using NintendoWare.SoundMaker.Configurations.Schemas;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.CommandHandlers;
    using NintendoWare.SoundMaker.Framework.Commands;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Windows;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Framework.Windows.Forms.CommandHandlers;
    using NintendoWare.SoundMaker.Preview;
    using NintendoWare.SoundMaker.Preview.Communications;
    using NintendoWare.SoundMaker.Profile;
    using NintendoWare.SoundMaker.Resources;
    using NintendoWare.SoundMaker.Windows.Forms.CommandHandlers;
    using NintendoWare.SoundMakerPlugin;
    using NintendoWare.ToolDevelopmentKit;

    using FR = NintendoWare.SoundMaker.Framework.Resources;
    using FrameworkResources = NintendoWare.SoundMaker.Framework.Resources;

    /// <summary>
    /// WindowsFormsベースのCommonメインウィンドウです。
    /// </summary>
    public partial class MainWindowCommon : MainWindow
    {
        private const int PreviewStatusUpdateInterval = 1000 / 30;

        private const string ImageKeyPreviewLampOn = "LampOn";
        private const string ImageKeyPreviewLampOff = "LampOff";

        private uint CommandOptionPlayPreviewPlayerIndex = 3;

        private Timer _previewStatusUpdateTimer = new Timer();
        private ImageList _previewLampImages = new ImageList();

        private WaveSoundEditWindow waveSoundEditWindow;

        private bool isBeforePlatformCafe = false;
        private bool previewPlayerParameterPanelDockVisible = false;
        private bool effectPanelDockVisible = false;

        public MainWindowCommon()
        {
            TopToolStripPanel.SuspendLayout();

            InitializeComponent();

            // 出力モードのUIを設定します。
            cmb_MainOutputMode.Items.Add(new OutputModeItem(XmlPreviewOutputMode.Monoral));
            cmb_MainOutputMode.Items.Add(new OutputModeItem(XmlPreviewOutputMode.Stereo));
            cmb_MainOutputMode.Items.Add(new OutputModeItem(XmlPreviewOutputMode.Surround));
            //cmb_MainOutputMode.Items.Add( new OutputModeItem(XmlPreviewOutputMode.Dpl2));

            cmb_DrcOutputMode.Items.Add(new OutputModeItem(XmlPreviewOutputMode.Monoral));
            cmb_DrcOutputMode.Items.Add(new OutputModeItem(XmlPreviewOutputMode.Stereo));
            cmb_DrcOutputMode.Items.Add(new OutputModeItem(XmlPreviewOutputMode.Surround));
            //cmb_DrcOutputMode.Items.Add( new OutputModeItem(XmlPreviewOutputMode.Dpl2));

            // 出力モード、Main、Drc両方のComboBoxにアイテムを追加してから
            // 選択を変更する必要があります。
            cmb_MainOutputMode.SelectedIndex = 1;
            cmb_DrcOutputMode.SelectedIndex = 1;

            // 接続関係のツールバーの設定です。
            this.ConnectTool.Enabled = true;
            this.ConnectTool.Visible = true;

            // プラットフォーム切り替えツールバー
            this.toolStripComboBoxPlatform.SelectedIndexChanged += OnPlatformChanged;
            this.toolStripComboBoxPlatform.Items.Clear();
            this.toolStripComboBoxPlatform.Items.AddRange(SoundMakerPluginManager.Instance.Instances.Select(p => new ComboBoxItem(p.Platform, p.PlatformLabel)).ToArray());
            this.SelectedPlatform = SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin.Platform;

            // 接続先切り替えツールバー
            this.cmb_ConnectTarget.SelectedIndexChanged += OnConnectTargetChanged;
            this.cmb_ConnectTarget.SelectedIndex = 0;

#if USE_PCSOUNDPLAYER
            this.tsb_RunPCViewer.Image = ImageResource.BitmapIconRunPCSoundPlayer;
#endif
            this.tsb_RunPCAtkPlayer.Image = ImageResource.BitmapIconRunPCAtkPlayer;
            this.tsb_RunTargetPlayer.Image = ImageResource.BitmapIconRunTargetAtkPlayer;
            this.tsb_ConnectToTarget.Image = ImageResource.BitmapIconConnectToSoundPlayer;
            this.tsb_DisconnectFromTarget.Image = ImageResource.BitmapIconDisconnectFromSoundPlayer;

            this.tsb_RunTargetPlayer.Click += delegate (object sender, EventArgs e)
                {
                    Application.RunTargetPlayer();
                };

#if USE_PCSOUNDPLAYER
            this.tsb_RunPCViewer.Click += delegate( object sender, EventArgs e)
                {
                    Application.RunPCSoundPlayer();
                };
#endif

            this.tsb_RunPCAtkPlayer.Click += delegate (object sender, EventArgs e)
                {
                    Application.RunPCAtkPlayer();
                };

            this.tsb_ConnectToTarget.Click += delegate (object sender, EventArgs e)
                {
                    if (Application.ConnectToTarget() == true)
                    {
                    }
                };

            this.tsb_DisconnectFromTarget.Click += delegate (object sender, EventArgs e)
                {
                    Application.DisconnectFromTarget();
                };


            this.CommunicationService.ViewerConnectionChanged +=
                delegate (object sender, ConnectionChangedEventArgs e)
                {
                    switch (e.State)
                    {
                        case ConnectionState.Connected:
                            this.cmb_ConnectTarget.Enabled = false;
                            this.toolStripComboBoxPlatform.Enabled = false;
                            break;

                        case ConnectionState.Disconnected:
                            this.cmb_ConnectTarget.Enabled = true;
                            this.toolStripComboBoxPlatform.Enabled = true;
                            break;
                    }
                };

            this.CommunicationService.SndEditConnectionChanged += OnConnectionChanged;

            this.ProjectService.Closed += OnProjectClosed;
            this.ProjectService.SoundSetDocumentClosed += OnSoundSetDocumentClosed;

#if !USE_PCSOUNDPLAYER
            this.tsb_RunPCViewer.Enabled = false;
            this.tsb_RunPCViewer.Visible = false;
#endif
            //
            StandardTool.Items.Remove(standardToolItem_RunSoundPlayer);
            StandardTool.Items.Remove(standardToolItem_RunPCSoundPlayer);
            StandardTool.Items.Remove(standardToolItem_ConnectToViewer);
            StandardTool.Items.Remove(standardToolItem_DisconnectFromViewer);
            StandardTool.Items.Remove(standardToolItem_ConnectToPCViewer);
            StandardTool.Items.Remove(standardToolItem_DisconnectFromPCViewer);

            // デフォルトのコマンドバー並び
            TopToolStripPanel.Controls.Clear();
            TopToolStripPanel.Join(_previewStatusTool);
            TopToolStripPanel.Join(ToolWindowTool);
            TopToolStripPanel.Join(OtherTool);
            TopToolStripPanel.Join(ConnectTool);
            TopToolStripPanel.Join(StandardTool);

            TopToolStripPanel.ResumeLayout();

            _previewStatusUpdateTimer.Interval = PreviewStatusUpdateInterval;
            _previewStatusUpdateTimer.Tick += OnPreviewStatusUpdateTimer;

            ProjectService.Opened += OnProjectOpened;

            //
            Bitmap bitmap = NintendoWare.SoundMaker.Resources.ImageResource.BitmapIconWarning;
            bitmap.MakeTransparent();
            pictureBox1.Image = bitmap;

            if (AppConfigurationCommon.EnabledWaveSoundEdit == true)
            {
                this.waveSoundEditWindow = new WaveSoundEditWindow(this.ProjectService as SoundProjectServiceCommon);
            }
        }

        private void OnConnectionChanged(object sender, ConnectionChangedEventArgs e)
        {
            switch (e.State)
            {
                case ConnectionState.Connected:
                    this.cmb_ConnectTarget.Enabled = false;
                    this.toolStripComboBoxPlatform.Enabled = false;
                    break;

                case ConnectionState.Disconnected:
                    this.cmb_ConnectTarget.Enabled = true;
                    this.toolStripComboBoxPlatform.Enabled = true;
                    break;
            }
        }

        /// <summary>
        /// ウェーブサウンド編集ウィンドウを取得します。
        /// </summary>
        public WaveSoundEditWindow WaveSoundEditWindow
        {
            get
            {
                return this.waveSoundEditWindow;
            }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsConnectTargetSoundPlayer
        {
            get
            {
                return this.CommunicationService.IsTargetSoundPlayer(this.SelectedTargetName);
            }
        }

        public string SelectedPlatform
        {
            get
            {
                var item = this.toolStripComboBoxPlatform.SelectedItem as ComboBoxItem;

                Debug.Assert(item != null);

                return item.Value;
            }
            set
            {
                this.SetSelectedValue(this.toolStripComboBoxPlatform, value);
            }
        }

        public string SelectedTargetName
        {
            get
            {
                var item = this.cmb_ConnectTarget.SelectedItem as ComboBoxItem;

                Debug.Assert(item != null);

                return item.Value;
            }
            set
            {
                this.SetSelectedValue(this.cmb_ConnectTarget, value);
            }
        }

        private void SetSelectedValue(ToolStripComboBox comboBox, string value)
        {
            Debug.Assert(string.IsNullOrEmpty(value) == false);

            for (int i = 0; i < comboBox.Items.Count; i++)
            {
                if (((ComboBoxItem)comboBox.Items[i]).Value == value)
                {
                    comboBox.SelectedIndex = i;
                    break;
                }
            }
        }

        public void ShowWaveSoundEditorWindow(WaveSound waveSound = null)
        {
            if (AppConfigurationCommon.EnabledWaveSoundEdit == true)
            {
                this.waveSoundEditWindow.WaveSound = waveSound;
                this.waveSoundEditWindow.Show();
                this.waveSoundEditWindow.Activate();
            }
        }

        /// <summary>
        /// アプリケーションを取得します。
        /// </summary>
        private new FormsApplicationCommon Application
        {
            get { return base.Application as FormsApplicationCommon; }
        }

        /// <summary>
        /// CommunicationService を取得します。
        /// </summary>
        private CommunicationService CommunicationService
        {
            get
            {
                return FormsApplicationCommon.Instance.CommunicationService;
            }
        }

        /// <summary>
        /// プレビューバンクパネルを取得します。
        /// </summary>
        private PreviewBankPanel PreviewBankPanel
        {
            get { return ToolPages[PreviewBankPanel.PageName] as PreviewBankPanel; }
        }

        /// <summary>
        ///
        /// </summary>
        private Preview.PreviewPlayerManager PreviewPlayerManager
        {
            get { return Preview.PreviewManager.Instance.PreviewPlayerManager; }
        }

        /// <summary>
        ///
        /// </summary>
        private new SoundProjectConfigurationCommon ProjectConfiguration
        {
            get { return Application.ProjectConfiguration; }
        }

        /// <summary>
        /// Common 向けオプションを取得します。
        /// </summary>
        private XmlOptionsCommon OptionsCommon
        {
            get { return Application.AppConfiguration.OptionsCommon; }
        }

        /// <summary>
        /// コマンドの関連付けを行います。
        /// </summary>
        /// <param name="commandBindings">コマンドバインディングコレクション。</param>
        protected override void InitializeCommandBindings(CommandBindingCollection commandBindings)
        {
            base.InitializeCommandBindings(commandBindings);

            // 表示関連コマンドハンドラ
            commandBindings.Add(
                new CommandBinding(this, new ShowToolPageHandler(
                                            ViewCommands.ShowPreviewPlayerWindow.ID,
                                            PreviewPlayerPanel.PageName)));
            commandBindings.Add(
                new CommandBinding(this, new ShowToolPageHandler(
                                            ViewCommands.ShowPreviewBankWindow.ID,
                                            PreviewBankPanel.PageName)));

            ShowToolPageHandler hanlder = new ShowToolPageHandler(ViewCommands.ShowPreviewPlayerParameterWindow.ID,
                                                                  PreviewPlayerParameterPanel.PageName);
            hanlder.SetQueryStatusHandler(this.QueryStatusCafeHandler);
            commandBindings.Add(new CommandBinding(this, hanlder));

            commandBindings.Add(
                new CommandBinding(this, new ShowToolPageHandler(
                                            ViewCommands.ShowSequenceVariableWindow.ID,
                                            SequenceVariablePanel.PageName)));

            hanlder = new ShowToolPageHandler(ViewCommands.ShowEffectWindow.ID,
                                              EffectPanel.PageName);
            hanlder.SetQueryStatusHandler(this.QueryStatusCafeHandler);
            commandBindings.Add(new CommandBinding(this, hanlder));
            commandBindings.Add(
               new CommandBinding(this, new ShowToolPageHandler(
                                           ViewCommands.ShowInGameWindow.ID,
                                           InGamePanel.PageName)));

            if (AppConfigurationCommon.EnabledWaveSoundEdit == true)
            {
                if (commandBindings.Contains(ViewCommands.ShowWaveSoundEditorWindow.ID) == true)
                {
                    commandBindings.Remove(ViewCommands.ShowWaveSoundEditorWindow.ID);
                }
                commandBindings.Add(new CommandBinding
                                     (this, ViewCommands.ShowWaveSoundEditorWindow.ID,
                                       QueryShowWaveSoundEditWindow,
                                       ExecuteShowWaveSoundEditWindow));
            }

            // プレビュー関連コマンドハンドラ
            QueryAndMeasureWaveReferenceCountHandler.WaveReferenceCountMeasurer =
                ExecuteMeasureWaveReferenceCount;

            commandBindings.Add
                (new CommandBinding(this,
                                       new QueryAndMeasureWaveReferenceCountHandler
                                           (this)));

            QueryAndMeasureSequenceVoiceCountHandler.SequenceVoiceCountMeasurer =
                ExecuteMeasureSequenceVoiceCount;

            commandBindings.Add
                (new CommandBinding(this,
                                       new QueryAndMeasureSequenceVoiceCountHandler
                                           (this)));

            // ツール関連コマンドハンドラ
            commandBindings.Add(
                new CommandBinding(this, new ShowOptionHandlerCommon(ToolCommands.ShowOptions.ID))
                );

            // タブ関連コマンドハンドラ
            commandBindings.Add
                (new CommandBinding(this, PreviewCommands.RegisterOnPreviewBankFromPanel.ID,
                                      QueryStatusRegisterOnPreviewBank,
                                      ExecuteRegisterOnPreviewBank));

            // ヘルプ関連コマンドハンドラ
            commandBindings.Add(
                new CommandBinding(this, new ShowHelpHandler(
                                            HelpCommands.ShowViewerHelp.ID,
                                            Application.AtkPlayerHelpURL)));

            // プロジェクトツリーへのコマンドハンドラ
            ProjectPanel.BindCommand(new CommandBinding
                                      (ProjectPanel, PreviewCommands.RegisterOnPreviewBankFromPanel.ID,
                                        QueryStatusRegisterOnPreviewBankFromProjectPanel,
                                        ExecuteRegisterOnPreviewBankFromProjectPanel));
        }

        private bool QueryStatusCafeHandler(ref CommandStatus status)
        {
            if (this.IsPlatformCafe() == false) // プラットフォーム Cafe 以外なら非表示
            {
                status = CommandStatus.Supported;
                return true;
            }
            else // プラットフォーム Cafe なら今までどおり
            {
                return false;
            }
        }

        protected override void QueryToolStrips(IList<ToolStrip> result)
        {
            if (null == result) { throw new ArgumentNullException("result"); }

            base.QueryToolStrips(result);
            result.Add(_previewStatusTool);
            result.Add(ConnectTool);
        }

        /// <summary>
        /// ツールページを初期化します。
        /// </summary>
        protected override void InitializeToolPages()
        {
            base.InitializeToolPages();

            AddToolPage(new PreviewPlayerPanel());
            AddToolPage(new PreviewBankPanel());
            AddToolPage(new PreviewPlayerParameterPanel());
            AddToolPage(new SequenceVariablePanel());
            AddToolPage(new EffectPanel());
            AddToolPage(new InGamePanelCommon());
            Application.InGamePanel.SelectChanged += delegate (object sender, EventArgs e)
                {
                    BuildCommandUI();
                };

            Application.RealtimeEditService.Attach(Application.ParameterPanel);

            InitializeParameterPanel();

            this.ChangeCafeUI();
        }

        /// <summary>
        /// ツールページの関連付けを有効化します。
        /// </summary>
        protected override void ValidateToolPageBindings()
        {
            base.ValidateToolPageBindings();

            ValidateToolPageBinding(ToolPages[PreviewPlayerPanel.PageName], DockingToolContainers["MainWindowBottom"]);
            ValidateToolPageBinding(ToolPages[PreviewBankPanel.PageName], DockingToolContainers["MainWindowBottom"]);
            ValidateToolPageBinding(ToolPages[PreviewPlayerParameterPanel.PageName], DockingToolContainers["MainWindowRight"]);
            ValidateToolPageBinding(ToolPages[SequenceVariablePanel.PageName], DockingToolContainers["MainWindowRight"]);
            ValidateToolPageBinding(ToolPages[EffectPanel.PageName], DockingToolContainers["MainWindowRight"]);
            ValidateToolPageBinding(ToolPages[InGamePanel.PageName], DockingToolContainers["MainWindowRight"]);
        }

        /// <summary>
        ///
        /// </summary>
        protected override void UpdateConnectToolBar()
        {
            CommandStatus status;

            status = Application.CanRunTargetPlayer();
            this.tsb_RunTargetPlayer.Enabled = status.IsEnabled();
            this.tsb_RunTargetPlayer.Visible = status.IsVisible();

#if USE_PCSOUNDPLAYER
            status = Application.CanRunPCSoundPlayer();
            this.tsb_RunPCViewer.Enabled = status.IsEnabled();
            this.tsb_RunPCViewer.Visible = status.IsVisible();
#endif
            status = Application.CanRunPCAtkPlayer();
            this.tsb_RunPCAtkPlayer.Enabled = status.IsEnabled();
            this.tsb_RunPCAtkPlayer.Visible = status.IsVisible();

            status = Application.CanConnectToTarget();
            this.tsb_ConnectToTarget.Enabled = status.IsEnabled();
            this.tsb_ConnectToTarget.Visible = status.IsVisible();

            status = Application.CanDisconnectFromTarget();
            this.tsb_DisconnectFromTarget.Enabled = status.IsEnabled();
            this.tsb_DisconnectFromTarget.Visible = status.IsVisible();
        }

        /// <summary>
        ///
        /// </summary>
        protected override void UpdateConnectMenuTexts()
        {
            UpdateConnectTarget();
        }

        /// <summary>
        ///
        /// </summary>
        protected override SoundSetPanel CreateSoundSetPanel(DocumentReference docReference)
        {
            if (null == docReference) { throw new ArgumentNullException("docReference"); }

            SoundSetPanelCommon panel = new SoundSetPanelCommon(docReference);
            panel.SelectChangedListItem += OnSelectChangedListItem;

            return panel;
        }

        /// <summary>
        ///
        /// </summary>
        protected override BankPanel CreateBankPanel(string filePath)
        {
            Ensure.Argument.NotNull(filePath);
            Ensure.Argument.StringNotEmpty(filePath);

            return new BankPanelCommon(filePath);
        }

        /// <summary>
        ///
        /// </summary>
        protected override FindResultPanel2 CreateFindResultPanel()
        {
            FindResultPanel2 panel = new FindResultPanel2Common();

            panel.PreviewPlay += PreviewPlayEventListener.OnPreviewPlay;
            panel.PreviewMute += PreviewPlayEventListener.OnPreviewMute;
            panel.PreviewSoloPlay += PreviewPlayEventListener.OnPreviewSoloPlay;

            panel.SelectChangedListItem += this.OnSelectChangedListItemFindResultPanel2;

            return panel;
        }

        /// <summary>
        ///
        /// </summary>
        protected override void Disconnect()
        {
            this.CommunicationService.Disconnect();
        }

        /// <summary>
        ///
        /// </summary>
        protected override void OnLoad(EventArgs e)
        {
            base.OnLoad(e);

            Initialize();

            PreviewPlayerManager.PreviewPlayerSelectChanged += OnPreviewPlayerSelectChanged;
            PreviewPlayerManager.SelectedPreviewPlayerStateChanged += OnSelectedPreviewPlayerStateChanged;

            this.CommunicationService.ViewerConnectionChanged += OnViewerConnectionChanged;
            this.CommunicationService.SndEditConnectionChanged += OnSreConnectionChanged;

            _previewStatusUpdateTimer.Start();
        }

        protected override void OnFormClosing(FormClosingEventArgs e)
        {
            base.OnFormClosing(e);

            if (e.Cancel == false)
            {
                Application.RealtimeEditService.Dispose();
            }
        }

        protected override void OnFormClosed(FormClosedEventArgs e)
        {
            _previewStatusUpdateTimer.Stop();

            base.OnFormClosed(e);
        }

        protected override void PlayItem(Component component)
        {
            Application.Play(component, this.CommandOptionPlayPreviewPlayerIndex);
        }

        protected override void StopAllItem()
        {
            Application.StopAll();
        }

        protected override void OnDeveiceChange(Message message)
        {
            base.OnDeveiceChange(message);

            if (Application.IsErrorAudioDevice == false)
            {
                string dev = Application.AppConfiguration.Options.Application.Sound.WaveOutDeviceName;

                if (SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin.EnumerateAudioDeviceNames().Contains(dev) == false)
                {
                    Application.IsErrorAudioDevice = true;
                }
            }
        }

        protected override void OutputWaveFile(Component component, string maxDuration, string isMonaural, string samplerate, string openFolder, string output)
        {
            try
            {
                if (string.IsNullOrEmpty(output) == true)
                {
                    string path = Path.GetDirectoryName(ProjectService.ProjectDocument.Resource.Key) + "\\";
                    output = Path.Combine(path, component.Name + ".wav");
                }
                else
                {
                    string ext = Path.GetExtension(output);
                    if (ext == string.Empty || ext != ".wav")
                    {
                        output += ".wav";
                    }
                }

                output = output.Replace('/', '\\');

                string directory = Path.GetDirectoryName(output);

                if (Directory.Exists(directory) == false)
                {
                    Directory.CreateDirectory(directory);
                }

                this.OutputWaveFile(component as Sound, output, int.Parse(maxDuration), isMonaural == "true", samplerate == "48" ? OutputWaveFileRenderType.k48KHz : OutputWaveFileRenderType.k32KHz);

                if (openFolder == "on")
                {
                    Process.Start("explorer.exe", "/e,/select, " + output);
                }
            }
            catch
            {
                FormsApplicationCommon.Instance.UIService.ShowMessageBox(MessageResource.Message_ErrorOutputWaveFile);
            }
        }

        public void OutputWaveFile(Sound sound, string filePath, int maxDuration, bool isMonaural, OutputWaveFileRenderType samplingRate)
        {
            if (null == sound) { throw new ArgumentNullException("sound"); }
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            FormsApplicationCommon.Instance.ExecuteCommand(PreviewCommands.StopAll);
            FormsApplicationCommon.Instance.ExecuteCommand(ToolCommands.StopAllSounds);
            FormsApplicationCommon.Instance.ShutdownWaveOutput();

            Preview.PreviewSound previewSound = Preview.PreviewPlayer.CreatePreviewSound(sound, samplingRate);
            if (null == previewSound)
            {
                throw new ApplicationException(FrameworkResources.MessageResource.Message_UnknownError);
            }
            previewSound.OutputWaveFile(filePath, maxDuration, isMonaural);

            FormsApplicationCommon.Instance.StartupWaveOutput();
        }

        /// <summary>
        /// 現在選択されているプラットフォームが Cafe かどうか取得します。
        /// </summary>
        public bool IsPlatformCafe()
        {
            return SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin.Platform == "Cafe";
        }

        /// <summary>
        ///
        /// </summary>
        private void Initialize()
        {
            _previewLampImages.TransparentColor = UIServiceBase.TransparentColor;
            _previewLampImages.Images.Add(ImageKeyPreviewLampOn, FrameworkResources.ImageResource.BitmapIconLampOn);
            _previewLampImages.Images.Add(ImageKeyPreviewLampOff, FrameworkResources.ImageResource.BitmapIconLampOff);

            UpdatePreviewStatus();
        }

        /// <summary>
        /// パラメータパネルに表示するパラメータ項目を追加します。
        /// </summary>
        private void InitializeParameterPanel()
        {
            SoundParameterPanel panel = Application.ParameterPanel.SoundParameterPanel;
            InitializeParameterPanel(panel);

            // ストリームサウンド
            panel.InsertParameterName(typeof(StreamSoundBase),
                                       ProjectParameterNames.Sound.PlayerPriority,
                                       ProjectParameterNamesCommon.RemoteFilter,
                                       MessageResource.HeaderText_RemoteFilter);

            panel.InsertParameterName(typeof(StreamSoundBase),
                                       ProjectParameterNames.Sends.AuxBSend,
                                       ProjectParameterNamesCommon.Sends.AuxCSend,
                                       MessageResource.HeaderText_AuxCSend);

            panel.InsertParameterName(typeof(StreamSoundTrackBase),
                                       ProjectParameterNames.Sends.AuxBSend,
                                       ProjectParameterNamesCommon.Sends.AuxCSend,
                                       MessageResource.HeaderText_AuxCSend);

            // ウェーブサウンド
            panel.InsertParameterName(typeof(WaveSoundBase),
                                       ProjectParameterNames.Sends.AuxBSend,
                                       ProjectParameterNamesCommon.Sends.AuxCSend,
                                       MessageResource.HeaderText_AuxCSend);

            panel.InsertParameterName(typeof(WaveSoundBase),
                                       ProjectParameterNames.Sends.MainSend,
                                       ProjectParameterNamesCommon.RemoteFilter,
                                       MessageResource.HeaderText_RemoteFilter);

            // シーケンスサウンド
            panel.InsertParameterName(typeof(SequenceSoundBase),
                                       ProjectParameterNames.Sound.PlayerPriority,
                                       ProjectParameterNamesCommon.RemoteFilter,
                                       MessageResource.HeaderText_RemoteFilter);
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdatePreviewStatus()
        {
            _previewStatusTool.SuspendLayout();

            _soundLamp.Image = Application.IsOutputSounds() ?
                                _previewLampImages.Images[ImageKeyPreviewLampOn] :
                                _previewLampImages.Images[ImageKeyPreviewLampOff];

            _midiLamp.Image = Application.IsReceiveMidiSignal(PreviewStatusUpdateInterval + 20) ?
                                _previewLampImages.Images[ImageKeyPreviewLampOn] :
                                _previewLampImages.Images[ImageKeyPreviewLampOff];

            _previewStatusTool.ResumeLayout();
        }

        /// <summary>
        ///
        /// </summary>
        private void SendOutputMode()
        {
            XmlPreviewCommonSystem system = ProjectConfiguration.PreviewCommon.System;
            this.CommunicationService.SetOutputMode
                (new OutputModeItem(system.OutputMode).PreviewOutputMode,
                  new OutputModeItem(system.DrcOutputMode).PreviewOutputMode);
        }

        /// <summary>
        ///
        /// </summary>
        private void OnPreviewStatusUpdateTimer(object sender, EventArgs e)
        {
            if (!InvokeRequired)
            {
                UpdatePreviewStatus();
            }

            Invoke(new MethodInvoker(UpdatePreviewStatus));
        }

        private void OnPreviewPlayerSelectChanged(object sender, EventArgs e)
        {
            BuildCommandUI();
        }

        private void OnSelectedPreviewPlayerStateChanged(Preview.PreviewPlayer sender, EventArgs e)
        {
            BuildCommandUI();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnViewerConnectionChanged(object sender, NintendoWare.SoundMaker.Framework.Preview.Communications.ConnectionChangedEventArgs e)
        {
            BuildCommandUI();

            // 出力モード設定はViewer接続時のみ有効
            if (ConnectionState.Connected == e.State)
            {
                cmb_MainOutputMode.Enabled = true;
                cmb_DrcOutputMode.Enabled = true;

                SendOutputMode();
            }
            else
            {
                cmb_MainOutputMode.Enabled = false;
                cmb_DrcOutputMode.Enabled = false;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void OnSreConnectionChanged(object sender, NintendoWare.SoundMaker.Framework.Preview.Communications.ConnectionChangedEventArgs e)
        {
            BuildCommandUI();
        }

        /// <summary>
        ///
        /// </summary>
        private void OnConnectTargetChanged(object sender, EventArgs e)
        {
            UpdateConnectTarget();
            BuildCommandUI();
        }

        /// <summary>
        ///
        /// </summary>
        private void UpdateConnectTarget()
        {
            ConnectTarget connectTarget = ConnectTarget.App;
            if (this.IsConnectTargetSoundPlayer == true)
            {
                connectTarget = ConnectTarget.SoundPlayer;
            }
            UpdateConnectTarget(connectTarget);
        }

        /// <summary>
        ///
        /// </summary>
        private enum ConnectTarget
        {
            App = 0,
            SoundPlayer,
        }

        private void OnPlatformChanged(object sender, EventArgs e)
        {
            SoundMakerPluginManager.Instance.SetCurrentPlatform(this.SelectedPlatform);
            ISoundMakerPlugin plugin = SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin;

            ITargetPlayer player = plugin.TargetPlayer;
            if (player != null)
            {
                this.tsb_RunTargetPlayer.Text = player.UIRunButtonText;
                this.tsb_RunTargetPlayer.Image = player.UIRunIconImage;
                PreviewCommonCommands.RunTargetPlayer.Text = player.UIRunMenuText;
                PreviewCommonCommands.RunTargetPlayer.ToolTipText = player.UIRunToolTipText;
                PreviewCommonCommands.RunTargetPlayer.Image = player.UIRunIconImage;
                PreviewCommonCommands.ShutdownTargetPlayer.Text = player.UIShutdownMenuText;
            }

            this.cmb_ConnectTarget.Items.Clear();
            this.cmb_ConnectTarget.Items.AddRange(plugin.CommunicationInfo.ConnectionInfos.Select(i => new ComboBoxItem(i.TargetName, i.TargetNameLabel)).ToArray());
            this.cmb_ConnectTarget.SelectedIndex = 0;

            this.ChangeCafeUI();

            if (this.ActivePage != null && this.ActivePage.Panel != null)
            {
                this.ActivePage.Panel.RedrawPanel();
            }

            this.BuildCommandUI();
        }

        private void ChangeCafeUI()
        {
            // Cafe の UI を表示、非表示、関係

            bool visible = this.IsPlatformCafe();

            if (this.ToolPages != null)
            {
                // プレビュープレイヤーパラメータ
                {
                    var panel = this.ToolPages[PreviewPlayerParameterPanel.PageName];
                    if (visible == false) // プラットフォームに Cafe 以外が選ばれた
                    {
                        if (this.isBeforePlatformCafe == true) // プラットフォーム Cafe からの切り替えか？
                        {
                            this.previewPlayerParameterPanelDockVisible = panel.DockVisible; // Cafe の表示状態を保存
                        }
                        panel.DockVisible = false; // 非表示に。
                    }
                    else // プラットフォームに Cafe が選ばれた
                    {
                        panel.DockVisible = this.previewPlayerParameterPanelDockVisible; // Cafe の表示状態を復元
                    }
                }

                // エフェクトパネル
                {
                    var panel = this.ToolPages[EffectPanel.PageName];
                    if (visible == false) // プラットフォームに Cafe 以外が選ばれた
                    {
                        if (this.isBeforePlatformCafe == true) // プラットフォーム Cafe からの切り替えか？
                        {
                            this.effectPanelDockVisible = panel.DockVisible; // Cafe の表示状態を保存
                        }
                        panel.DockVisible = false; // 非表示に。
                    }
                    else // プラットフォームに Cafe が選ばれた
                    {
                        panel.DockVisible = this.effectPanelDockVisible; // Cafe の表示状態を復元
                    }
                }

                this.isBeforePlatformCafe = visible;
            }

            // Output モード (MONO, STEREO, SURROUND)
            cmb_DrcOutputMode.Visible = visible;
            this.toolStripLabel_DRC.Visible = visible;

            // オプションダイアログ
            OptionDialogCommon.VisibleSoundPlayerTab = visible;

            // アイテムの新規作成ダイアログ
            CreateStreamSoundDialog.VisibleWaveEncodingPCM8 = visible;
            CreateWaveSoundDialog.VisibleWaveEncodingPCM8 = visible;
            CreateInstrumentDialog.VisibleWaveEncodingPCM8 = visible;
        }

        private class ComboBoxItem
        {
            public string Value { get; private set; }
            public string Label { get; private set; }

            public ComboBoxItem(string value, string label)
            {
                this.Value = value;
                this.Label = label;
            }

            public override string ToString()
            {
                return this.Label;
            }
        }

        ///--------------------------------
        /// <summary>
        /// サウンドセットドキュメントを閉じると発生します。
        /// </summary>
        private void OnSoundSetDocumentClosed(object sender, SoundDocumentResourceEventArgs e)
        {
            if (this.waveSoundEditWindow == null)
            {
                return;
            }

            this.waveSoundEditWindow.WaveSound = null;
        }

        private class MenuTextData
        {
            public string Name { get; set; }
            public string Text0 { get; set; }
            public string Text1 { get; set; }
        }


        /// <summary>
        ///
        /// </summary>
        private void UpdateConnectTarget(ConnectTarget connectTarget)
        {
            switch (connectTarget)
            {
                case ConnectTarget.App: // Wii U アプリケーション
                    if (ProjectService != null &&
                        ProjectService.Project != null)
                    {
                        this.CommunicationService.SetPCPort(ProjectService.Project.SyncPort,
                                                            ProjectService.Project.FuncPort,
                                                            ProjectService.Project.CtrlPort,
                                                            0,
                                                            0,
                                                            0);
                    }
                    else
                    {
                        this.CommunicationService.SetPCPort(0, 0, 0, 0, 0, 0);
                    }

                    break;

                case ConnectTarget.SoundPlayer: // SoundPlayer
                    XmlOptionsCommon options = Application.AppConfiguration.OptionsCommon;
                    this.CommunicationService.SetPCPort(options.Application.SoundPlayer.Port0,
                                                        options.Application.SoundPlayer.Port1,
                                                        options.Application.SoundPlayer.Port2,
                                                        options.Application.SoundPlayer.Port3,
                                                        options.Application.SoundPlayer.Port4,
                                                        options.Application.SoundPlayer.Port5);
                    break;
            }
        }


        /// <summary>
        /// 出力モードが変更された時に呼ばれます。
        /// </summary>
        private void OnMainOutputModeChanged(object sender, EventArgs e)
        {
            OutputModeItem item = cmb_MainOutputMode.SelectedItem as OutputModeItem;
            ProjectConfiguration.PreviewCommon.System.OutputMode = item.XmlOutputMode;
            SendOutputMode();
        }

        /// <summary>
        /// DRC出力モードが変更された時に呼ばれます。
        /// </summary>
        private void OnDrcOutputModeChanged(object sender, EventArgs e)
        {
            OutputModeItem item = cmb_DrcOutputMode.SelectedItem as OutputModeItem;
            ProjectConfiguration.PreviewCommon.System.DrcOutputMode = item.XmlOutputMode;
            SendOutputMode();
        }

        /// <summary>
        /// プロジェクトが開かれた時に呼ばれます。
        /// </summary>
        private void OnProjectOpened(object sernder, EventArgs e)
        {
            XmlPreviewCommonSystem system = ProjectConfiguration.PreviewCommon.System;
            XmlPreviewOutputMode mainOutputMode = system.OutputMode;
            XmlPreviewOutputMode drcOutputMode = system.DrcOutputMode;

            foreach (OutputModeItem item in cmb_MainOutputMode.Items)
            {
                if (item.XmlOutputMode != mainOutputMode) { continue; }
                cmb_MainOutputMode.SelectedItem = item;
                break;
            }

            //
            foreach (OutputModeItem item in cmb_DrcOutputMode.Items)
            {
                if (item.XmlOutputMode == drcOutputMode)
                {
                    cmb_DrcOutputMode.SelectedItem = item;
                    break;
                }
            }

            //
            UpdateConnectTarget();
            BuildCommandUI();
        }

        /// <summary>
        /// プロジェクトが閉じた時に呼ばれます。
        /// </summary>
        private void OnProjectClosed(object sernder, EventArgs e)
        {
            if (this.waveSoundEditWindow == null)
            {
                return;
            }

            this.waveSoundEditWindow.WaveSound = null;
        }

        /// <summary>
        /// 出力設定を保持するクラスです。
        /// ComboBoxで使用されます。
        /// </summary>
        private class OutputModeItem
        {
            private XmlPreviewOutputMode _outputMode;
            private string _text;

            public OutputModeItem(XmlPreviewOutputMode outputMode)
            {
                _outputMode = outputMode;
                _text = outputMode.ToText();
            }

            public XmlPreviewOutputMode XmlOutputMode
            {
                get { return _outputMode; }
            }

            public OutputMode PreviewOutputMode
            {
                get { return ConvertSystemOutputMode(_outputMode); }
            }

            public override string ToString()
            {
                return _text;
            }

            private OutputMode ConvertSystemOutputMode(XmlPreviewOutputMode xmlData)
            {
                switch (xmlData)
                {
                    case XmlPreviewOutputMode.Monoral:
                        return OutputMode.Monoral;

                    case XmlPreviewOutputMode.Stereo:
                        return OutputMode.Stereo;

                    case XmlPreviewOutputMode.Surround:
                        return OutputMode.Surround;

                    case XmlPreviewOutputMode.Dpl2:
                        return OutputMode.Dpl2;
                }

                return OutputMode.Stereo;
            }
        }

        /// <summary>
        /// タブメニューがターゲットとしているバンクパネルを取得します。
        /// </summary>
        private BankPanel GetTabMenuTargetBankPanel()
        {
            if (0 > TabMenuTargetIndex || TabMenuTargetIndex >= TabMenuTargetCtrl.TabPages.Count)
            {
                return null;
            }

            CommonTabPage page = TabMenuTargetCtrl.TabPages[TabMenuTargetIndex] as CommonTabPage;
            CommonPanel panel = page.Panel;
            return panel as BankPanel;
        }

        /// <summary>
        /// キーの参照回数を計測します。
        /// </summary>
        private void ExecuteMeasureWaveReferenceCount(Sound sound, int measureTime, ProjectProfileService.MeasureWaveReferenceCount.Data data)
        {
            if (sound is SequenceSoundBase)
            {
                PreviewSequenceSound previewSound =
                    PreviewPlayer.CreatePreviewSound(sound) as PreviewSequenceSound;
                previewSound.MeasureWaveReferenceCount(measureTime, data);
            }
        }

        /// <summary>
        /// 最大発音数を計測します。
        /// </summary>
        private void ExecuteMeasureSequenceVoiceCount(Sound sound, int measureTime)
        {
            if (sound is SequenceSoundBase)
            {
                PreviewSequenceSound previewSound = PreviewPlayer.CreatePreviewSound(sound) as PreviewSequenceSound;
                previewSound.MeasureSequenceVoiceCount(measureTime);
            }
        }

        /// <summary>
        /// "プレビューバンクに登録"が実行可能かどうかを調べます。
        /// </summary>
        private CommandStatus QueryStatusRegisterOnPreviewBank(Command command)
        {
            BankPanel panel = GetTabMenuTargetBankPanel();
            if (panel == null)
            {
                return CommandStatus.Supported;
            }
            return CommandStatus.SupportedAndEnabledAndVisible;
        }

        /// <summary>
        /// "プレビューバンクに登録"を実行します。
        /// </summary>
        private bool ExecuteRegisterOnPreviewBank(Command command)
        {
            BankPanel panel = GetTabMenuTargetBankPanel();
            if (panel == null ||
                panel.SoundSetBank == null)
            {
                return false;
            }

            PreviewBankPanel.GetItem(0).Attach(panel.SoundSetBank);
            return true;
        }

        /// <summary>
        /// プロジェクトパネルが選択しているSoundSetBankを取得します。
        /// </summary>
        private SoundSetBankBase GetProjectPanelSelectedSoundSetBank()
        {
            Component[] components = ProjectPanel.TargetComponents;
            if (components == null ||
                components.Length != 1)
            {
                return null;
            }
            return components[0] as SoundSetBankBase;
        }

        /// <summary>
        /// "プレビューバンクに登録"が実行可能かどうかを調べます。
        /// </summary>
        private CommandStatus QueryStatusRegisterOnPreviewBankFromProjectPanel(Command command)
        {
            return GetProjectPanelSelectedSoundSetBank() != null ?
                CommandStatus.SupportedAndEnabledAndVisible :
                CommandStatus.Supported;
        }

        /// <summary>
        /// "プレビューバンクに登録"を実行します。
        /// </summary>
        private bool ExecuteRegisterOnPreviewBankFromProjectPanel(Command command)
        {
            SoundSetBankBase soundSetBank = GetProjectPanelSelectedSoundSetBank();
            if (soundSetBank == null)
            {
                return false;
            }

            PreviewBankPanel.GetItem(0).Attach(soundSetBank);
            return true;
        }

        /// <summary>
        /// ウェーブサウンド編集ウィンドウを表示できるか判断します。
        /// </summary>
        private CommandStatus QueryShowWaveSoundEditWindow(Command command)
        {
            if (AppConfigurationCommon.EnabledWaveSoundEdit == true)
            {
                return CommandStatus.SupportedAndEnabledAndVisible;
            }

            return CommandStatus.Supported;
        }

        /// <summary>
        /// ウェーブサウンド編集ウィンドウを表示します。
        /// </summary>
        private bool ExecuteShowWaveSoundEditWindow(Command command)
        {
            this.ShowWaveSoundEditorWindow();

            return true;
        }

        private void OnSelectChangedListItem(object sender, EventArgs e)
        {
            this.UpdateWaveSoundEditWindow(sender);
        }

        private void OnSelectChangedListItemFindResultPanel2(object sender, EventArgs e)
        {
            this.UpdateWaveSoundEditWindow(sender);
        }

        private void UpdateWaveSoundEditWindow(object sender)
        {
            if (AppConfigurationCommon.EnabledWaveSoundEdit == true)
            {
                if (sender is ListCtrlCommon == true)
                {
                    ListCtrlCommon listCtrl = sender as ListCtrlCommon;
                    if (listCtrl.ContainsFocus == true)
                    {
                        Component target = null;
                        if (listCtrl.SelectedItems.Count() == 1)
                        {
                            CommonListItem item = listCtrl.SelectedItems[0] as CommonListItem;
                            target = item.Target;
                        }

                        this.waveSoundEditWindow.WaveSound = (target as WaveSound);
                    }
                }
            }
        }
    }
}
