﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker
{
    using System;
    using System.IO;
    using System.Reflection;
    using System.Text.RegularExpressions;
    using System.Threading;
    using System.Windows.Forms;
    using Nintendo.Foundation.IO;
    using NintendoWare.SoundFoundation.Core.Threading;
    using NintendoWare.SoundFoundation.Legacies;
    using NintendoWare.SoundFoundation.Utilities;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Resources;
    using NintendoWare.SoundMaker.Windows.Forms;
    using NintendoWare.SoundMakerPlugin;

    internal static class Program
    {
        /// <summary>
        /// アプリケーションのメイン エントリ ポイントです。
        /// </summary>
        /// <param name="args">コマンドライン引数。</param>
        [STAThread]
        public static void Main(string[] args)
        {
            EnvironmentUtility.RemoveMultipleEnvironmentVariables();
            Application.EnableVisualStyles();
            Application.SetCompatibleTextRenderingDefault(false);

            Thread.CurrentThread.InitializeCurrentUICulture();

            if (ExecuteCommandLine(args) == true)
            {
                return;
            }
            // Load Plugin // exe のフォルダ以下の plugins フォルダから読み込みます。
            string exePath = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
            try
            {
                var pm = new SoundMakerPluginManager(exePath + "\\plugins");
            }
            catch
            {
                MessageBox.Show(MessageResource.Message_ErrorLoadPlugin);
                return;
            }
            FormsApplicationCommon.Instance.Run(args);
        }

        private enum ParseResult
        {
            Success,
            HelpOrVersion,
            ParseError,
            ParseCommandError,
        }

        private static bool ExecuteCommandLine(string[] args)
        {
            Settings settings = null;

            if (args.Length < 1)
            {
                return false;
            }

            switch (Parser(out settings, args))
            {
                case ParseResult.HelpOrVersion:
                case ParseResult.ParseError:
                    return true;

                case ParseResult.ParseCommandError:
                    Console.WriteLine(MessageResource.CommandLine_Error);
                    return true;
            }

            return ExecuteCommand(settings);
        }

        private static ParseResult Parser(out Settings settings, string[] args)
        {
            // 共通コマンドラインインターフェイス解析
            try
            {
                CommandLineParserSettings commandLineParserSettings = new CommandLineParserSettings()
                {
                    ApplicationDescription = MessageResource.CommandLine_ApplicationDescription,
                };

                if (new Nintendo.Foundation.IO.CommandLineParser(commandLineParserSettings).ParseArgs(args, out settings) == false)
                {
                    return ParseResult.HelpOrVersion;
                }
            }
            catch
            {
                settings = null;
                return ParseResult.ParseError;
            }

            // --command のコマンド解析
            if (string.IsNullOrEmpty(settings.CommandStrings) == true)
            {
                return ParseResult.Success;
            }

            Regex regex = new Regex("^[^=]+=.+$");
            if (regex.IsMatch(settings.CommandStrings) == true)
            {
                string[] tokens = settings.CommandStrings.Split(new char[] { ' ', '=', ',' }, StringSplitOptions.RemoveEmptyEntries);
                tokens[0] = tokens[0].ToLower();

                switch (tokens[0])
                {
                    case "showitem":
                    case "openitem":
                    case "playitem":
                        settings.Command = tokens[0];
                        settings.Item = tokens[1];
                        return ParseResult.Success;

                    case "edititem":
                        settings.Command = tokens[0];
                        if (tokens.Length == 4)
                        {
                            settings.Item = tokens[1];
                            settings.UserParam = tokens[2];
                            int test;
                            if (int.TryParse(tokens[3], out test) == true)
                            {
                                settings.UserParamValue = tokens[3];
                                return ParseResult.Success;
                            }
                        }
                        break;

                    case "outputwavefile":
                        settings.Command = tokens[0];
                        if (2 <= tokens.Length)
                        {
                            settings.Item = tokens[1];

                            for (int i = 2; i < tokens.Length; i++)
                            {
                                switch (tokens[i].ToLower())
                                {
                                    case "channel":
                                        ++i;
                                        if (tokens.Length <= i)
                                        {
                                            return ParseResult.ParseCommandError;
                                        }
                                        else if (tokens[i].ToLower() == "mono")
                                        {
                                            settings.IsMonaural = "true";
                                        }
                                        else if (tokens[i].ToLower() != "stereo")
                                        {
                                            return ParseResult.ParseCommandError;
                                        }
                                        break;

                                    case "time":
                                        ++i;
                                        if (tokens.Length <= i)
                                        {
                                            return ParseResult.ParseCommandError;
                                        }
                                        int test;
                                        if (int.TryParse(tokens[i], out test) == true)
                                        {
                                            settings.MaxDuration = tokens[i];
                                        }
                                        break;

                                    case "openfolder":
                                        ++i;
                                        if (tokens.Length <= i)
                                        {
                                            return ParseResult.ParseCommandError;
                                        }
                                        else if (tokens[i].ToLower() == "off")
                                        {
                                            settings.OpenFolder = "off";
                                        }
                                        else if (tokens[i].ToLower() != "on")
                                        {
                                            return ParseResult.ParseCommandError;
                                        }
                                        break;

                                    case "samplerate":
                                        ++i;
                                        if (tokens.Length <= i)
                                        {
                                            return ParseResult.ParseCommandError;
                                        }
                                        if (int.TryParse(tokens[i], out test) == true &&
                                            (tokens[i] == "32" || tokens[i] == "48"))
                                        {
                                            settings.Samplerate = tokens[i];
                                        }
                                        else
                                        {
                                            return ParseResult.ParseCommandError;
                                        }
                                        break;

                                    default:
                                        return ParseResult.ParseCommandError;
                                }
                            }
                            return ParseResult.Success;
                        }
                        break;
                }
            }
            else
            {
                var command = settings.CommandStrings.ToLower();
                switch (command)
                {
                    case "stopallitem":
                    case "saveproject":
                        settings.Command = command;
                        return ParseResult.Success;
                }
            }

            return ParseResult.ParseCommandError;
        }

        private static bool ExecuteCommand(Settings settings)
        {
            string assemblyName = Assembly.GetExecutingAssembly().GetName().Name;

            settings.FilePath = settings.FilePath.Replace('/', '\\');

            switch (settings.Command)
            {
                case "showitem":
                case "openitem":
                case "playitem":
                    return FormsApplication.SendToSoundMaker(assemblyName, settings.FilePath, settings.Command, settings.Item);

                case "stopallitem":
                case "saveproject":
                    return FormsApplication.SendToSoundMaker(assemblyName, settings.FilePath, settings.Command);

                case "edititem":
                    return FormsApplication.SendToSoundMaker(assemblyName, settings.FilePath, settings.Command, settings.Item, settings.UserParam, settings.UserParamValue);

                case "outputwavefile":
                    return FormsApplication.SendToSoundMaker(assemblyName, settings.FilePath, settings.Command, settings.Item, settings.MaxDuration, settings.IsMonaural, settings.Samplerate, settings.OpenFolder, settings.Output);
            }

            return false;
        }

        /// <summary>
        /// 設定を格納します。
        /// </summary>
        private class Settings
        {
            public Settings()
            {
                this.FilePath = string.Empty;
                this.Output = string.Empty;
                this.CommandStrings = string.Empty;
                this.Command = string.Empty;
                this.Item = string.Empty;
                this.UserParam = string.Empty;
                this.UserParamValue = string.Empty;
                this.IsMonaural = "false"; // デフォルト：ステレオ出力
                this.MaxDuration = "30";   // デフォルト：最大出力時間３０秒
                this.OpenFolder = "on";  // デフォルト：出力フォルダを開く
                this.Samplerate = "32";    // デフォルト：サンプルレート３２kHz
            }

            [CommandLineValue(0, IsRequired = true, ValueName = "input-filepath", Description = "CommandLine_InputDescription", DescriptionConverterName = "LocalizeDescription")]
            public string FilePath { get; set; }

            [CommandLineOption("command", ValueName = "command-strings", Description = "CommandLine_UsageMessage_Option_command", DescriptionConverterName = "LocalizeDescription")]
            public string CommandStrings { get; set; }

            [CommandLineOption('o', "output", ValueName = "output-filepath", Description = "CommandLine_UsageMessage_Option_output", DescriptionConverterName = "LocalizeDescription")]
            public string Output { get; set; }
            public string Command { get; set; }
            public string Item { get; set; }
            public string UserParam { get; set; }
            public string UserParamValue { get; set; }
            public string IsMonaural { get; set; }
            public string MaxDuration { get; set; }
            public string OpenFolder { get; set; }
            public string Samplerate { get; set; }

            /// <summary>
            /// ローカライズしたオプションの説明を取得します。
            /// </summary>
            /// <param name="description">Descriptionプロパティの値</param>
            /// <param name="valueName">引数名またはオプション名</param>
            /// <returns>ローカライズされたコマンドラインオプションの説明を返します。</returns>
            public static string LocalizeDescription(string description, string valueName)
            {
                string result = MessageResource.ResourceManager.GetString(description, Resources.MessageResource.Culture);
                System.Diagnostics.Debug.Assert(result != null, "コマンドオプションの Description が null");

                return result;
            }
        }
    }
}
