﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundMaker.Preview
{
    using System;
    using System.Collections.ObjectModel;
    using System.Diagnostics;
    using System.Xml;
    using System.Xml.Schema;
    using System.Xml.Serialization;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NW4R.ProtocolSound;

    /// <summary>
    /// エフェクト識別子
    /// </summary>
    public struct EffectID : IComparable<EffectID>
    {
        private ToolSoundAuxBusType _busType;
        private uint _effectIndex;

        public EffectID(ToolSoundAuxBusType busType, uint effectIndex)
        {
            _busType = busType;
            _effectIndex = effectIndex;
        }

        #region ** プロパティ

        public ToolSoundAuxBusType BusType
        {
            get { return _busType; }
        }

        public uint EffectIndex
        {
            get { return _effectIndex; }
        }

        #endregion

        #region ** オペレータ

        public override bool Equals(Object obj)
        {
            if (null == obj) { return false; }
            if (GetType() != obj.GetType()) { return false; }

            EffectID other = (EffectID)obj;
            return this._busType.Equals(other._busType) && this._effectIndex.Equals(other._effectIndex);
        }

        public override int GetHashCode()
        {
            return this._busType.GetHashCode() ^ this._effectIndex.GetHashCode();
        }

        public static bool operator ==(EffectID target1, EffectID target2)
        {
            return (target1._busType == target2._busType) && (target1._effectIndex == target2._effectIndex);
        }

        public static bool operator !=(EffectID target1, EffectID target2)
        {
            return !(target1 == target2);
        }

        #endregion

        #region ** ICompareable の実装

        public int CompareTo(EffectID other)
        {
            if (this._busType < other._busType) { return -1; }
            if (this._busType > other._busType) { return 1; }
            if (this._effectIndex < other._effectIndex) { return -1; }
            if (this._effectIndex > other._effectIndex) { return 1; }
            return 0;
        }

        #endregion
    }

    interface IEffectParameter
    {
        uint Size { get; }
        EffectID EffectID { get; }
        ToolSoundEffectType EffectType { get; }

        void Write(ProtocolSoundWriter writer);
    }

    public abstract class EffectParameter : IEffectParameter, IXmlSerializable
    {
        // パラメータ
        protected EffectID _id;

        protected EffectParameter(EffectID id)
        {
            _id = id;
        }
        protected EffectParameter(EffectParameter src)
        {
            _id = src._id;
        }

        #region ** イベント

        public event EventHandler ParameterChanged;

        #endregion

        #region ** メソッド

        protected void DispatchParameterChangedEvent()
        {
            if (null == ParameterChanged) { return; }
            ParameterChanged(this, new EventArgs());
        }

        protected ToolSoundEffectCommonParameter ToStruct()
        {
            ToolSoundEffectCommonParameter param = new ToolSoundEffectCommonParameter();
            param.busType = _id.BusType;
            param.effectIndex = _id.EffectIndex;
            param.effectType = EffectType;

            return param;
        }

        #endregion

        #region ** イベントハンドラ

        protected virtual void OnReadXmlElement(XmlReader reader) { }
        protected virtual void OnWriteXmlElement(XmlWriter writer) { }

        #endregion

        #region ** IEffectParameter の実装

        public abstract uint Size { get; }

        public EffectID EffectID
        {
            get { return _id; }
        }

        public abstract ToolSoundEffectType EffectType { get; }

        public abstract void Write(ProtocolSoundWriter writer);

        #endregion

        #region ** IXmlSerializable の実装

        /// <summary>
        /// このプロパティは予約されています。代わりに System.Xml.Serialization.XmlSchemaProviderAttribute をクラスに適用します。
        /// </summary>
        /// <returns>XML 表現を記述する System.Xml.Schema.XmlSchema。</returns>
        public XmlSchema GetSchema() { return null; }

        /// <summary>
        /// オブジェクトの XML 表現からオブジェクトを生成します。
        /// </summary>
        /// <param name="reader">オブジェクトの逆シリアル化元である System.Xml.XmlReader ストリーム。</param>
        public virtual void ReadXml(XmlReader reader)
        {
            if (null == reader) { throw new ArgumentNullException("reader"); }

            if (reader.Name != "EffectParameter" || reader.IsEmptyElement)
            {
                reader.Read();
                return;
            }

            reader.Read();

            // AuxBusType, EffectIndex, EffectType は既に読み込み済なので、
            // ここでは何もせず、派生クラスに流す
            OnReadXmlElement(reader);
        }

        /// <summary>
        /// オブジェクトを XML 表現に変換します。
        /// </summary>
        /// <param name="writer">オブジェクトのシリアル化先の System.Xml.XmlWriter ストリーム。</param>
        public virtual void WriteXml(XmlWriter writer)
        {
            writer.WriteStartElement("EffectParameter");
            writer.WriteAttributeString("BusType", ((int)_id.BusType).ToString());	// AuxBus の種類
            writer.WriteAttributeString("Index", (_id.EffectIndex).ToString());	// エフェクトインデックス
            writer.WriteAttributeString("EffectType", ((int)EffectType).ToString());	// エフェクトの種類

            // その他のパラメータを出力する
            OnWriteXmlElement(writer);

            writer.WriteEndElement();
        }

        #endregion
    }


    #region ** エフェクト種類別のパラメータクラス

    public class NoEffectParameter : EffectParameter
    {
        public NoEffectParameter(EffectID id) : base(id) { }
        public NoEffectParameter(NoEffectParameter src) : base(src) { }

        #region ** プロパティのオーバーライド

        public override uint Size
        {
            get { return ToolSoundEffectCommonParameter.StructSize; }
        }

        public override ToolSoundEffectType EffectType
        {
            get { return ToolSoundEffectType.EffectType_NoEffect; }
        }

        #endregion

        #region ** メソッドのオーバーライド

        public override void Write(ProtocolSoundWriter writer)
        {
            ToStruct().Write(writer);
        }

        #endregion
    }

    public class ReverbHiParameter : EffectParameter
    {
        #region ** パラメータフィールド

        private int _earlyMode = 30;
        private float _preDelayTimeMax = 0.15f;
        private float _preDelayTime = 0.02f;
        private ToolSoundFusedMode _fusedMode = ToolSoundFusedMode.FusedMode_OldAxfx;
        private float _fusedTime = 3.00f;
        private float _coloration = 0.60f;
        private float _damping = 0.40f;
        private float _crosstalk = 0.10f;
        private float _earlyGain = 0.00f;
        private float _fusedGain = 1.00f;
        private float _outGain = 1.00f;

        #endregion

        public ReverbHiParameter(EffectID id) : base(id) { }
        public ReverbHiParameter(ReverbHiParameter src) : base(src) { }

        #region ** プロパティ

        public int EarlyMode
        {
            get { return _earlyMode; }
            set
            {
                if (value == _earlyMode) { return; }
                _earlyMode = value;

                DispatchParameterChangedEvent();
            }
        }

        public float PreDelayTimeMax
        {
            get { return _preDelayTimeMax; }
            set
            {
                if (value == _preDelayTimeMax) { return; }

                // PreDelayTimeMax < PreDelayTime の場合、値を補正する
                if (_preDelayTime > value)
                {
                    _preDelayTime = value;
                }

                _preDelayTimeMax = value;

                DispatchParameterChangedEvent();
            }
        }

        public float PreDelayTime
        {
            get { return _preDelayTime; }
            set
            {
                Debug.Assert(_preDelayTimeMax >= value);
                if (value == _preDelayTime) { return; }

                _preDelayTime = value;

                DispatchParameterChangedEvent();
            }
        }

        public ToolSoundFusedMode FusedMode
        {
            get { return _fusedMode; }
            set
            {
                if (value == _fusedMode) { return; }
                _fusedMode = value;

                DispatchParameterChangedEvent();
            }
        }

        public float FusedTime
        {
            get { return _fusedTime; }
            set
            {
                if (value == _fusedTime) { return; }
                _fusedTime = value;

                DispatchParameterChangedEvent();
            }
        }

        public float Coloration
        {
            get { return _coloration; }
            set
            {
                if (value == _coloration) { return; }
                _coloration = value;

                DispatchParameterChangedEvent();
            }
        }

        public float Damping
        {
            get { return _damping; }
            set
            {
                if (value == _damping) { return; }
                _damping = value;

                DispatchParameterChangedEvent();
            }
        }

        public float Crosstalk
        {
            get { return _crosstalk; }
            set
            {
                if (value == _crosstalk) { return; }
                _crosstalk = value;

                DispatchParameterChangedEvent();
            }
        }

        public float EarlyGain
        {
            get { return _earlyGain; }
            set
            {
                if (value == _earlyGain) { return; }
                _earlyGain = value;

                DispatchParameterChangedEvent();
            }
        }

        public float FusedGain
        {
            get { return _fusedGain; }
            set
            {
                if (value == _fusedGain) { return; }
                _fusedGain = value;

                DispatchParameterChangedEvent();
            }
        }

        public float OutGain
        {
            get { return _outGain; }
            set
            {
                if (value == _outGain) { return; }
                _outGain = value;

                DispatchParameterChangedEvent();
            }
        }

        #endregion

        #region ** プロパティのオーバーライド

        public override uint Size
        {
            get { return ToolSoundReverbHiParameter.StructSize; }
        }

        public override ToolSoundEffectType EffectType
        {
            get { return ToolSoundEffectType.EffectType_ReverbHi; }
        }

        #endregion

        #region ** メソッドのオーバーライド

        public override void Write(ProtocolSoundWriter writer)
        {
            ToStruct().Write(writer);
        }

        #endregion

        #region ** メソッドの再定義

        new private ToolSoundReverbHiParameter ToStruct()
        {
            ToolSoundReverbHiParameter param = new ToolSoundReverbHiParameter();
            param.busType = EffectID.BusType;
            param.effectIndex = EffectID.EffectIndex;
            param.effectType = EffectType;

            param.earlyMode = _earlyMode;
            param.preDelayTimeMax = _preDelayTimeMax;
            param.preDelayTime = _preDelayTime;
            param.fusedMode = _fusedMode;
            param.fusedTime = _fusedTime;
            param.coloration = _coloration;
            param.damping = _damping;
            param.crosstalk = _crosstalk;
            param.earlyGain = _earlyGain;
            param.fusedGain = _fusedGain;
            param.outGain = _outGain;

            return param;
        }

        #endregion
    }

    public class ReverbStdParameter : EffectParameter
    {
        #region ** パラメータフィールド

        private int _earlyMode = 30;
        private float _preDelayTimeMax = 0.15f;
        private float _preDelayTime = 0.02f;
        private ToolSoundFusedMode _fusedMode = ToolSoundFusedMode.FusedMode_OldAxfx;
        private float _fusedTime = 3.00f;
        private float _coloration = 0.60f;
        private float _damping = 0.40f;
        private float _earlyGain = 0.00f;
        private float _fusedGain = 1.00f;
        private float _outGain = 1.00f;

        #endregion

        public ReverbStdParameter(EffectID id) : base(id) { }
        public ReverbStdParameter(ReverbHiParameter src) : base(src) { }

        #region ** プロパティ

        public int EarlyMode
        {
            get { return _earlyMode; }
            set
            {
                if (value == _earlyMode) { return; }
                _earlyMode = value;

                DispatchParameterChangedEvent();
            }
        }

        public float PreDelayTimeMax
        {
            get { return _preDelayTimeMax; }
            set
            {
                if (value == _preDelayTimeMax) { return; }

                // PreDelayTimeMax < PreDelayTime の場合、値を補正する
                if (_preDelayTime > value)
                {
                    _preDelayTime = value;
                }

                _preDelayTimeMax = value;

                DispatchParameterChangedEvent();
            }
        }

        public float PreDelayTime
        {
            get { return _preDelayTime; }
            set
            {
                Debug.Assert(_preDelayTimeMax >= value);
                if (value == _preDelayTime) { return; }

                _preDelayTime = value;

                DispatchParameterChangedEvent();
            }
        }

        public ToolSoundFusedMode FusedMode
        {
            get { return _fusedMode; }
            set
            {
                if (value == _fusedMode) { return; }
                _fusedMode = value;

                DispatchParameterChangedEvent();
            }
        }

        public float FusedTime
        {
            get { return _fusedTime; }
            set
            {
                if (value == _fusedTime) { return; }
                _fusedTime = value;

                DispatchParameterChangedEvent();
            }
        }

        public float Coloration
        {
            get { return _coloration; }
            set
            {
                if (value == _coloration) { return; }
                _coloration = value;

                DispatchParameterChangedEvent();
            }
        }

        public float Damping
        {
            get { return _damping; }
            set
            {
                if (value == _damping) { return; }
                _damping = value;

                DispatchParameterChangedEvent();
            }
        }

        public float EarlyGain
        {
            get { return _earlyGain; }
            set
            {
                if (value == _earlyGain) { return; }
                _earlyGain = value;

                DispatchParameterChangedEvent();
            }
        }

        public float FusedGain
        {
            get { return _fusedGain; }
            set
            {
                if (value == _fusedGain) { return; }
                _fusedGain = value;

                DispatchParameterChangedEvent();
            }
        }

        public float OutGain
        {
            get { return _outGain; }
            set
            {
                if (value == _outGain) { return; }
                _outGain = value;

                DispatchParameterChangedEvent();
            }
        }

        #endregion

        #region ** プロパティのオーバーライド

        public override uint Size
        {
            get { return ToolSoundReverbHiParameter.StructSize; }
        }

        public override ToolSoundEffectType EffectType
        {
            get { return ToolSoundEffectType.EffectType_ReverbStd; }
        }

        #endregion

        #region ** メソッドのオーバーライド

        public override void Write(ProtocolSoundWriter writer)
        {
            ToStruct().Write(writer);
        }

        #endregion

        #region ** メソッドの再定義

        new private ToolSoundReverbStdParameter ToStruct()
        {
            ToolSoundReverbStdParameter param = new ToolSoundReverbStdParameter();
            param.busType = EffectID.BusType;
            param.effectIndex = EffectID.EffectIndex;
            param.effectType = EffectType;

            param.earlyMode = _earlyMode;
            param.preDelayTimeMax = _preDelayTimeMax;
            param.preDelayTime = _preDelayTime;
            param.fusedMode = _fusedMode;
            param.fusedTime = _fusedTime;
            param.coloration = _coloration;
            param.damping = _damping;
            param.earlyGain = _earlyGain;
            param.fusedGain = _fusedGain;
            param.outGain = _outGain;

            return param;
        }

        #endregion
    }

    public class ChorusParameter : EffectParameter
    {
        #region ** パラメータフィールド

        private float _delayTime = 10.0f;
        private float _depth = 0.50f;
        private float _rate = 1.00f;
        private float _feedback = 0.00f;
        private float _outGain = 1.00f;

        #endregion

        public ChorusParameter(EffectID id) : base(id) { }
        public ChorusParameter(DelayParameter src) : base(src) { }

        #region ** プロパティ

        public float DelayTime
        {
            get { return _delayTime; }
            set
            {
                if (value == _delayTime) { return; }
                _delayTime = value;

                DispatchParameterChangedEvent();
            }
        }

        public float Depth
        {
            get { return _depth; }
            set
            {
                if (value == _depth) { return; }
                _depth = value;

                DispatchParameterChangedEvent();
            }
        }

        public float Rate
        {
            get { return _rate; }
            set
            {
                if (value == _rate) { return; }
                _rate = value;

                DispatchParameterChangedEvent();
            }
        }

        public float Feedback
        {
            get { return _feedback; }
            set
            {
                if (value == _feedback) { return; }
                _feedback = value;

                DispatchParameterChangedEvent();
            }
        }

        public float OutGain
        {
            get { return _outGain; }
            set
            {
                if (value == _outGain) { return; }
                _outGain = value;

                DispatchParameterChangedEvent();
            }
        }

        #endregion

        #region ** プロパティのオーバーライド

        public override uint Size
        {
            get { return ToolSoundChorusParameter.StructSize; }
        }

        public override ToolSoundEffectType EffectType
        {
            get { return ToolSoundEffectType.EffectType_Chorus; }
        }

        #endregion

        #region ** メソッドのオーバーライド

        public override void Write(ProtocolSoundWriter writer)
        {
            ToStruct().Write(writer);
        }

        #endregion

        #region ** メソッドの再定義

        new private ToolSoundChorusParameter ToStruct()
        {
            ToolSoundChorusParameter param = new ToolSoundChorusParameter();
            param.busType = EffectID.BusType;
            param.effectIndex = EffectID.EffectIndex;
            param.effectType = EffectType;

            param.delayTime = _delayTime;
            param.depth = _depth;
            param.rate = _rate;
            param.feedback = _feedback;
            param.outGain = _outGain;

            return param;
        }

        #endregion
    }

    public class DelayParameter : EffectParameter
    {
        #region ** パラメータフィールド

        private float _maxDelayTime = 500.0f;
        private float _delayTime = 160.0f;
        private float _feedback = 0.40f;
        private float _outGain = 1.00f;
        private float _lpf = 1.00f;

        #endregion

        public DelayParameter(EffectID id) : base(id) { }
        public DelayParameter(DelayParameter src) : base(src) { }

        #region ** プロパティ

        public float MaxDelayTime
        {
            get { return _maxDelayTime; }
            set
            {
                if (value == _maxDelayTime) { return; }

                // MaxDelayTime < DelayTime の場合、値を補正する
                if (_delayTime > value)
                {
                    _delayTime = value;
                }

                _maxDelayTime = value;

                DispatchParameterChangedEvent();
            }
        }

        public float DelayTime
        {
            get { return _delayTime; }
            set
            {
                Debug.Assert(_maxDelayTime >= value);
                if (value == _delayTime) { return; }

                _delayTime = value;

                DispatchParameterChangedEvent();
            }
        }

        public float Feedback
        {
            get { return _feedback; }
            set
            {
                if (value == _feedback) { return; }
                _feedback = value;

                DispatchParameterChangedEvent();
            }
        }

        public float OutGain
        {
            get { return _outGain; }
            set
            {
                if (value == _outGain) { return; }
                _outGain = value;

                DispatchParameterChangedEvent();
            }
        }

        public float LPF
        {
            get { return _lpf; }
            set
            {
                if (value == _lpf) { return; }
                _lpf = value;

                DispatchParameterChangedEvent();
            }
        }

        #endregion

        #region ** プロパティのオーバーライド

        public override uint Size
        {
            get { return ToolSoundDelayParameter.StructSize; }
        }

        public override ToolSoundEffectType EffectType
        {
            get { return ToolSoundEffectType.EffectType_Delay; }
        }

        #endregion

        #region ** メソッドのオーバーライド

        public override void Write(ProtocolSoundWriter writer)
        {
            ToStruct().Write(writer);
        }

        #endregion

        #region ** メソッドの再定義

        new private ToolSoundDelayParameter ToStruct()
        {
            ToolSoundDelayParameter param = new ToolSoundDelayParameter();
            param.busType = EffectID.BusType;
            param.effectIndex = EffectID.EffectIndex;
            param.effectType = EffectType;

            param.maxDelayTime = _maxDelayTime;
            param.delayTime = _delayTime;
            param.feedback = _feedback;
            param.outGain = _outGain;
            param.lpf = _lpf;

            return param;
        }

        #endregion
    }

    public class MultiChannelReverbParameter : EffectParameter
    {
        #region ** パラメータフィールド

        private ToolSoundEarlyMode _earlyMode = ToolSoundEarlyMode.SmallRoom;
        private ToolSoundLateMode _lateMode = ToolSoundLateMode.Room;
        private float _preDelayTime = 0.02f;
        private float _decayTime = 3.0f;
        private float _hfDecayRatio = 0.4f;
        private float _coloration = 0.6f;
        private float _earlyGain = 0.0f;
        private float _lateGain = 1.0f;
        private float _reverbGain = 1.0f;

        #endregion

        public MultiChannelReverbParameter(EffectID id) : base(id) { }
        public MultiChannelReverbParameter(MultiChannelReverbParameter src) : base(src) { }

        #region ** プロパティ

        public ToolSoundEarlyMode EarlyMode
        {
            get { return _earlyMode; }
            set
            {
                if (value == _earlyMode) { return; }

                _earlyMode = value;

                DispatchParameterChangedEvent();
            }
        }

        public ToolSoundLateMode LateMode
        {
            get { return _lateMode; }
            set
            {
                if (value == _lateMode) { return; }

                _lateMode = value;

                DispatchParameterChangedEvent();
            }
        }

        public float PreDelayTime
        {
            get { return _preDelayTime; }
            set
            {
                if (value == _preDelayTime) { return; }

                _preDelayTime = value;

                DispatchParameterChangedEvent();
            }
        }

        public float DecayTime
        {
            get { return _decayTime; }
            set
            {

                if (value == _decayTime) { return; }

                _decayTime = value;

                DispatchParameterChangedEvent();
            }
        }

        public float HfDecayRatio
        {
            get { return _hfDecayRatio; }
            set
            {
                if (value == _hfDecayRatio) { return; }
                _hfDecayRatio = value;

                DispatchParameterChangedEvent();
            }
        }

        public float Coloration
        {
            get { return _coloration; }
            set
            {
                if (value == _coloration) { return; }
                _coloration = value;

                DispatchParameterChangedEvent();
            }
        }

        public float EarlyGain
        {
            get { return _earlyGain; }
            set
            {
                if (value == _earlyGain) { return; }
                _earlyGain = value;

                DispatchParameterChangedEvent();
            }
        }

        public float LateGain
        {
            get { return _lateGain; }
            set
            {
                if (value == _lateGain) { return; }
                _lateGain = value;

                DispatchParameterChangedEvent();
            }
        }

        public float ReverbGain
        {
            get { return _reverbGain; }
            set
            {
                if (value == _reverbGain) { return; }
                _reverbGain = value;

                DispatchParameterChangedEvent();
            }
        }

        #endregion

        #region ** プロパティのオーバーライド

        public override uint Size
        {
            get { return ToolSoundMultiChannelReverbParameter.StructSize; }
        }

        public override ToolSoundEffectType EffectType
        {
            get { return ToolSoundEffectType.EffectType_MultiChannelReverb; }
        }

        #endregion

        #region ** メソッドのオーバーライド

        public override void Write(ProtocolSoundWriter writer)
        {
            ToStruct().Write(writer);
        }

        #endregion

        #region ** メソッドの再定義

        new private ToolSoundMultiChannelReverbParameter ToStruct()
        {
            ToolSoundMultiChannelReverbParameter param = new ToolSoundMultiChannelReverbParameter();
            param.busType = EffectID.BusType;
            param.effectIndex = EffectID.EffectIndex;
            param.effectType = EffectType;

            param.earlyMode = _earlyMode;
            param.lateMode = _lateMode;
            param.preDelayTime = _preDelayTime;
            param.decayTime = _decayTime;
            param.hfDecayRatio = _hfDecayRatio;
            param.coloration = _coloration;
            param.earlyGain = _earlyGain;
            param.lateGain = _lateGain;
            param.reverbGain = _reverbGain;

            return param;
        }

        #endregion
    }

    public class MultiChannelChorusParameter : EffectParameter
    {
        #region ** パラメータフィールド

        private float _delayTime = 10.0f;
        private float _depth = 0.5f;
        private float _rate = 1.0f;
        private float _feedback = 0.0f;
        private float _outGain = 1.0f;

        #endregion

        public MultiChannelChorusParameter(EffectID id) : base(id) { }
        public MultiChannelChorusParameter(MultiChannelChorusParameter src) : base(src) { }

        #region ** プロパティ

        public float DelayTime
        {
            get { return _delayTime; }
            set
            {

                if (value == _delayTime) { return; }

                _delayTime = value;

                DispatchParameterChangedEvent();
            }
        }

        public float Depth
        {
            get { return _depth; }
            set
            {
                if (value == _depth) { return; }
                _depth = value;

                DispatchParameterChangedEvent();
            }
        }

        public float Rate
        {
            get { return _rate; }
            set
            {
                if (value == _rate) { return; }
                _rate = value;

                DispatchParameterChangedEvent();
            }
        }

        public float Feedback
        {
            get { return _feedback; }
            set
            {
                if (value == _feedback) { return; }
                _feedback = value;

                DispatchParameterChangedEvent();
            }
        }

        public float OutGain
        {
            get { return _outGain; }
            set
            {
                if (value == _outGain) { return; }
                _outGain = value;

                DispatchParameterChangedEvent();
            }
        }

        #endregion

        #region ** プロパティのオーバーライド

        public override uint Size
        {
            get { return ToolSoundMultiChannelChorusParameter.StructSize; }
        }

        public override ToolSoundEffectType EffectType
        {
            get { return ToolSoundEffectType.EffectType_MultiChannelChorus; }
        }

        #endregion

        #region ** メソッドのオーバーライド

        public override void Write(ProtocolSoundWriter writer)
        {
            ToStruct().Write(writer);
        }

        #endregion

        #region ** メソッドの再定義

        new private ToolSoundMultiChannelChorusParameter ToStruct()
        {
            ToolSoundMultiChannelChorusParameter param = new ToolSoundMultiChannelChorusParameter();
            param.busType = EffectID.BusType;
            param.effectIndex = EffectID.EffectIndex;
            param.effectType = EffectType;

            param.delayTime = _delayTime;
            param.depth = _depth;
            param.rate = _rate;
            param.feedback = _feedback;
            param.outGain = _outGain;

            return param;
        }

        #endregion
    }


    public class MultiChannelDelayParameter : EffectParameter
    {
        #region ** パラメータフィールド

        private float _delay = 160.0f;
        private float _feedback = 0.4f;
        private float _outGain = 1.0f;
        private float _channelSpread = 0.4f;
        private float _lpf = 1.0f;

        #endregion

        public MultiChannelDelayParameter(EffectID id) : base(id) { }
        public MultiChannelDelayParameter(MultiChannelDelayParameter src) : base(src) { }

        #region ** プロパティ

        public float Delay
        {
            get { return _delay; }
            set
            {

                if (value == _delay) { return; }

                _delay = value;

                DispatchParameterChangedEvent();
            }
        }

        public float Feedback
        {
            get { return _feedback; }
            set
            {
                if (value == _feedback) { return; }
                _feedback = value;

                DispatchParameterChangedEvent();
            }
        }

        public float OutGain
        {
            get { return _outGain; }
            set
            {
                if (value == _outGain) { return; }
                _outGain = value;

                DispatchParameterChangedEvent();
            }
        }

        public float ChannelSpread
        {
            get { return _channelSpread; }
            set
            {
                if (value == _channelSpread) { return; }
                _channelSpread = value;

                DispatchParameterChangedEvent();
            }
        }

        public float LPF
        {
            get { return _lpf; }
            set
            {
                if (value == _lpf) { return; }
                _lpf = value;

                DispatchParameterChangedEvent();
            }
        }

        #endregion

        #region ** プロパティのオーバーライド

        public override uint Size
        {
            get { return ToolSoundMultiChannelDelayParameter.StructSize; }
        }

        public override ToolSoundEffectType EffectType
        {
            get { return ToolSoundEffectType.EffectType_MultiChannelDelay; }
        }

        #endregion

        #region ** メソッドのオーバーライド

        public override void Write(ProtocolSoundWriter writer)
        {
            ToStruct().Write(writer);
        }

        #endregion

        #region ** メソッドの再定義

        new private ToolSoundMultiChannelDelayParameter ToStruct()
        {
            ToolSoundMultiChannelDelayParameter param = new ToolSoundMultiChannelDelayParameter();
            param.busType = EffectID.BusType;
            param.effectIndex = EffectID.EffectIndex;
            param.effectType = EffectType;

            param.delay = _delay;
            param.feedback = _feedback;
            param.outGain = _outGain;
            param.channelSpread = _channelSpread;
            param.lpf = _lpf;

            return param;
        }

        #endregion
    }


    #endregion

    #region ** エフェクトシステムプリセット

    public class EffectSystemPresetLoader
    {
        private static EffectSystemPresetLoader _instance;
        private static object _instanceLock = new object();

        #region ** パラメータ

        private SystemPresetCollection _presets = new SystemPresetCollection();

        #endregion

        private EffectSystemPresetLoader()
        {
            // Reverb Hi
            _presets.Add(new PresetReverbHiOldType1());
            _presets.Add(new PresetReverbHiOldType2());
            _presets.Add(new PresetReverbHiMetalPipe());
            _presets.Add(new PresetReverbHiMetalTank());
            _presets.Add(new PresetReverbHiSmallRoom());
            _presets.Add(new PresetReverbHiMediumRoom());
            _presets.Add(new PresetReverbHiLargeRoom());
            _presets.Add(new PresetReverbHiLongRoom());
            _presets.Add(new PresetReverbHiSmallHall());
            _presets.Add(new PresetReverbHiLargeHall());
            _presets.Add(new PresetReverbHiCavernous());
            _presets.Add(new PresetReverbHiCathedral());

            // Reverb Std
            _presets.Add(new PresetReverbStdOldType1());
            _presets.Add(new PresetReverbStdOldType2());
            _presets.Add(new PresetReverbStdMetalPipe());
            _presets.Add(new PresetReverbStdMetalTank());
            _presets.Add(new PresetReverbStdSmallRoom());
            _presets.Add(new PresetReverbStdMediumRoom());
            _presets.Add(new PresetReverbStdLargeRoom());
            _presets.Add(new PresetReverbStdLongRoom());
            _presets.Add(new PresetReverbStdSmallHall());
            _presets.Add(new PresetReverbStdLargeHall());
            _presets.Add(new PresetReverbStdCavernous());
            _presets.Add(new PresetReverbStdCathedral());

            // Chorus
            _presets.Add(new PresetChorusType1());
            _presets.Add(new PresetChorusType2());
            _presets.Add(new PresetChorusType3());
            _presets.Add(new PresetChorusFlanger1());
            _presets.Add(new PresetChorusFlanger2());
            _presets.Add(new PresetChorusEcho1());
            _presets.Add(new PresetChorusEcho2());
            _presets.Add(new PresetChorusWobble1());
            _presets.Add(new PresetChorusWobble2());

            // Delay
            _presets.Add(new PresetDelayType1());
            _presets.Add(new PresetDelayType2());
            _presets.Add(new PresetDelayFlange1());
            _presets.Add(new PresetDelayFlange2());
        }

        #region ** プロパティ

        public static EffectSystemPresetLoader Instance
        {
            get
            {
                lock (_instanceLock)
                {

                    if (null == _instance)
                    {
                        _instance = new EffectSystemPresetLoader();
                    }

                    return _instance;

                }
            }
        }

        #endregion

        #region ** メソッド

        public void Load(string key, EffectParameter target)
        {
            Debug.Assert(_presets.Contains(key), "unexpected error");
            _presets[key].Load(target);
        }

        public ToolSoundEffectType GetType(string key)
        {
            Debug.Assert(_presets.Contains(key), "unexpected error");
            return _presets[key].Type;
        }

        #endregion

        #region ** プリセットクラス

        #region ** 基本クラス

        private abstract class SystemPreset
        {
            #region ** パラメータ

            private string _key;

            #endregion

            public SystemPreset(string key)
            {
                if (null == key) { throw new ArgumentNullException("key"); }
                if (0 == key.Length) { throw new ArgumentException("invalid key"); }
                _key = key;
            }

            #region ** プロパティ

            public string Key
            {
                get { return _key; }
            }

            public abstract ToolSoundEffectType Type { get; }

            #endregion

            #region ** メソッド

            /// <summary>
            /// パラメータを設定します。
            /// </summary>
            /// <param name="targetParam">パラメータの格納先</param>
            public abstract void Load(EffectParameter target);

            #endregion
        }

        private class SystemPresetCollection : KeyedCollection<string, SystemPreset>
        {
            #region ** KeyedCollection の実装

            /// <summary>
            /// 指定した要素からキーを抽出します。
            /// </summary>
            /// <param name="item">キーの抽出元要素。</param>
            /// <returns>指定した要素のキー。</returns>
            protected override string GetKeyForItem(SystemPreset item)
            {
                if (null == item) { throw new ArgumentNullException("item"); }
                return item.Key;
            }

            #endregion
        }

        private abstract class SystemPresetReverbHi : SystemPreset
        {
            public SystemPresetReverbHi(string key) : base(key) { }

            public override ToolSoundEffectType Type
            {
                get { return ToolSoundEffectType.EffectType_ReverbHi; }
            }
        }

        private abstract class SystemPresetReverbStd : SystemPreset
        {
            public SystemPresetReverbStd(string key) : base(key) { }

            public override ToolSoundEffectType Type
            {
                get { return ToolSoundEffectType.EffectType_ReverbStd; }
            }
        }

        private abstract class SystemPresetChorus : SystemPreset
        {
            public SystemPresetChorus(string key) : base(key) { }

            public override ToolSoundEffectType Type
            {
                get { return ToolSoundEffectType.EffectType_Chorus; }
            }
        }

        private abstract class SystemPresetDelay : SystemPreset
        {
            public SystemPresetDelay(string key) : base(key) { }

            public override ToolSoundEffectType Type
            {
                get { return ToolSoundEffectType.EffectType_Delay; }
            }
        }

        #endregion

        #region ** ReverbHi プリセットクラス

        /// <summary>
        /// ReverbHi / OldType1
        /// </summary>
        private class PresetReverbHiOldType1 : SystemPresetReverbHi
        {
            public PresetReverbHiOldType1() : base("ReverbHi_OldType1") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbHiParameter targetParam = target as ReverbHiParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 30;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.03f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_OldAxfx;
                targetParam.FusedTime = 1.50f;
                targetParam.Coloration = 0.50f;
                targetParam.Damping = 0.50f;
                targetParam.Crosstalk = 0.10f;
                targetParam.EarlyGain = 0.00f;
                targetParam.FusedGain = 0.50f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbHi / OldType2
        /// </summary>
        private class PresetReverbHiOldType2 : SystemPresetReverbHi
        {
            public PresetReverbHiOldType2() : base("ReverbHi_OldType2") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbHiParameter targetParam = target as ReverbHiParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 40;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.05f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_OldAxfx;
                targetParam.FusedTime = 3.00f;
                targetParam.Coloration = 0.50f;
                targetParam.Damping = 0.70f;
                targetParam.Crosstalk = 0.10f;
                targetParam.EarlyGain = 0.40f;
                targetParam.FusedGain = 0.50f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbHi / MetalPipe
        /// </summary>
        private class PresetReverbHiMetalPipe : SystemPresetReverbHi
        {
            public PresetReverbHiMetalPipe() : base("ReverbHi_MetalPipe") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbHiParameter targetParam = target as ReverbHiParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 5;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.00f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_MetalTank;
                targetParam.FusedTime = 0.50f;
                targetParam.Coloration = 0.50f;
                targetParam.Damping = 0.90f;
                targetParam.Crosstalk = 0.10f;
                targetParam.EarlyGain = 0.20f;
                targetParam.FusedGain = 0.30f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbHi / MetalTank
        /// </summary>
        private class PresetReverbHiMetalTank : SystemPresetReverbHi
        {
            public PresetReverbHiMetalTank() : base("ReverbHi_MetalTank") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbHiParameter targetParam = target as ReverbHiParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 35;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.03f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_MetalTank;
                targetParam.FusedTime = 3.00f;
                targetParam.Coloration = 0.50f;
                targetParam.Damping = 0.80f;
                targetParam.Crosstalk = 0.10f;
                targetParam.EarlyGain = 0.25f;
                targetParam.FusedGain = 0.30f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbHi / SmallRoom
        /// </summary>
        private class PresetReverbHiSmallRoom : SystemPresetReverbHi
        {
            public PresetReverbHiSmallRoom() : base("ReverbHi_SmallRoom") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbHiParameter targetParam = target as ReverbHiParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 25;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.01f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_SmallRoom;
                targetParam.FusedTime = 0.50f;
                targetParam.Coloration = 0.60f;
                targetParam.Crosstalk = 0.10f;
                targetParam.Damping = 0.40f;
                targetParam.EarlyGain = 0.30f;
                targetParam.FusedGain = 0.30f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbHi / MediumRoom
        /// </summary>
        private class PresetReverbHiMediumRoom : SystemPresetReverbHi
        {
            public PresetReverbHiMediumRoom() : base("ReverbHi_MediumRoom") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbHiParameter targetParam = target as ReverbHiParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 30;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.02f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_SmallRoom;
                targetParam.FusedTime = 1.00f;
                targetParam.Coloration = 0.60f;
                targetParam.Damping = 0.40f;
                targetParam.Crosstalk = 0.10f;
                targetParam.EarlyGain = 0.30f;
                targetParam.FusedGain = 0.30f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbHi / LargeRoom
        /// </summary>
        private class PresetReverbHiLargeRoom : SystemPresetReverbHi
        {
            public PresetReverbHiLargeRoom() : base("ReverbHi_LargeRoom") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbHiParameter targetParam = target as ReverbHiParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 40;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.03f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_LargeRoom;
                targetParam.FusedTime = 1.50f;
                targetParam.Coloration = 0.60f;
                targetParam.Damping = 0.40f;
                targetParam.Crosstalk = 0.10f;
                targetParam.EarlyGain = 0.25f;
                targetParam.FusedGain = 0.35f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbHi / LongRoom
        /// </summary>
        private class PresetReverbHiLongRoom : SystemPresetReverbHi
        {
            public PresetReverbHiLongRoom() : base("ReverbHi_LongRoom") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbHiParameter targetParam = target as ReverbHiParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 20;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.02f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_LargeRoom;
                targetParam.FusedTime = 2.00f;
                targetParam.Coloration = 0.60f;
                targetParam.Damping = 0.40f;
                targetParam.Crosstalk = 0.10f;
                targetParam.EarlyGain = 0.30f;
                targetParam.FusedGain = 0.45f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbHi / SmallHall
        /// </summary>
        private class PresetReverbHiSmallHall : SystemPresetReverbHi
        {
            public PresetReverbHiSmallHall() : base("ReverbHi_SmallHall") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbHiParameter targetParam = target as ReverbHiParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 25;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.01f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_Hall;
                targetParam.FusedTime = 2.00f;
                targetParam.Coloration = 0.60f;
                targetParam.Damping = 0.40f;
                targetParam.Crosstalk = 0.10f;
                targetParam.EarlyGain = 0.35f;
                targetParam.FusedGain = 0.45f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbHi / LargeHall
        /// </summary>
        private class PresetReverbHiLargeHall : SystemPresetReverbHi
        {
            public PresetReverbHiLargeHall() : base("ReverbHi_LargeHall") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbHiParameter targetParam = target as ReverbHiParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 35;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.02f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_Hall;
                targetParam.FusedTime = 3.00f;
                targetParam.Coloration = 0.60f;
                targetParam.Damping = 0.30f;
                targetParam.Crosstalk = 0.10f;
                targetParam.EarlyGain = 0.30f;
                targetParam.FusedGain = 0.45f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbHi / Cavernous
        /// </summary>
        private class PresetReverbHiCavernous : SystemPresetReverbHi
        {
            public PresetReverbHiCavernous() : base("ReverbHi_Cavernous") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbHiParameter targetParam = target as ReverbHiParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 40;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.05f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_Cavernous;
                targetParam.FusedTime = 4.00f;
                targetParam.Coloration = 0.60f;
                targetParam.Damping = 0.70f;
                targetParam.Crosstalk = 0.10f;
                targetParam.EarlyGain = 0.50f;
                targetParam.FusedGain = 0.50f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbHi / Cathedral
        /// </summary>
        private class PresetReverbHiCathedral : SystemPresetReverbHi
        {
            public PresetReverbHiCathedral() : base("ReverbHi_Cathedral") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbHiParameter targetParam = target as ReverbHiParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 40;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.05f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_Cavernous;
                targetParam.FusedTime = 5.00f;
                targetParam.Coloration = 0.60f;
                targetParam.Damping = 0.30f;
                targetParam.Crosstalk = 0.10f;
                targetParam.EarlyGain = 0.35f;
                targetParam.FusedGain = 0.50f;
                targetParam.OutGain = 1.00f;
            }
        }

        #endregion

        #region ** ReverbStd プリセットクラス

        /// <summary>
        /// ReverbStd / OldType1
        /// </summary>
        private class PresetReverbStdOldType1 : SystemPresetReverbStd
        {
            public PresetReverbStdOldType1() : base("ReverbStd_OldType1") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbStdParameter targetParam = target as ReverbStdParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 30;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.03f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_OldAxfx;
                targetParam.FusedTime = 1.50f;
                targetParam.Coloration = 0.50f;
                targetParam.Damping = 0.50f;
                targetParam.EarlyGain = 0.00f;
                targetParam.FusedGain = 0.50f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbStd / OldType2
        /// </summary>
        private class PresetReverbStdOldType2 : SystemPresetReverbStd
        {
            public PresetReverbStdOldType2() : base("ReverbStd_OldType2") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbStdParameter targetParam = target as ReverbStdParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 40;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.05f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_OldAxfx;
                targetParam.FusedTime = 3.00f;
                targetParam.Coloration = 0.50f;
                targetParam.Damping = 0.70f;
                targetParam.EarlyGain = 0.40f;
                targetParam.FusedGain = 0.50f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbStd / MetalPipe
        /// </summary>
        private class PresetReverbStdMetalPipe : SystemPresetReverbStd
        {
            public PresetReverbStdMetalPipe() : base("ReverbStd_MetalPipe") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbStdParameter targetParam = target as ReverbStdParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 5;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.00f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_MetalTank;
                targetParam.FusedTime = 0.50f;
                targetParam.Coloration = 0.50f;
                targetParam.Damping = 0.90f;
                targetParam.EarlyGain = 0.20f;
                targetParam.FusedGain = 0.30f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbStd / MetalTank
        /// </summary>
        private class PresetReverbStdMetalTank : SystemPresetReverbStd
        {
            public PresetReverbStdMetalTank() : base("ReverbStd_MetalTank") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbStdParameter targetParam = target as ReverbStdParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 35;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.03f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_MetalTank;
                targetParam.FusedTime = 3.00f;
                targetParam.Coloration = 0.50f;
                targetParam.Damping = 0.80f;
                targetParam.EarlyGain = 0.25f;
                targetParam.FusedGain = 0.30f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbStd / SmallRoom
        /// </summary>
        private class PresetReverbStdSmallRoom : SystemPresetReverbStd
        {
            public PresetReverbStdSmallRoom() : base("ReverbStd_SmallRoom") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbStdParameter targetParam = target as ReverbStdParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 25;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.01f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_SmallRoom;
                targetParam.FusedTime = 0.50f;
                targetParam.Coloration = 0.60f;
                targetParam.Damping = 0.40f;
                targetParam.EarlyGain = 0.30f;
                targetParam.FusedGain = 0.30f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbStd / MediumRoom
        /// </summary>
        private class PresetReverbStdMediumRoom : SystemPresetReverbStd
        {
            public PresetReverbStdMediumRoom() : base("ReverbStd_MediumRoom") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbStdParameter targetParam = target as ReverbStdParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 30;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.02f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_SmallRoom;
                targetParam.FusedTime = 1.00f;
                targetParam.Coloration = 0.60f;
                targetParam.Damping = 0.40f;
                targetParam.EarlyGain = 0.30f;
                targetParam.FusedGain = 0.30f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbStd / LargeRoom
        /// </summary>
        private class PresetReverbStdLargeRoom : SystemPresetReverbStd
        {
            public PresetReverbStdLargeRoom() : base("ReverbStd_LargeRoom") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbStdParameter targetParam = target as ReverbStdParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 40;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.03f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_LargeRoom;
                targetParam.FusedTime = 1.50f;
                targetParam.Coloration = 0.60f;
                targetParam.Damping = 0.40f;
                targetParam.EarlyGain = 0.25f;
                targetParam.FusedGain = 0.35f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbStd / LongRoom
        /// </summary>
        private class PresetReverbStdLongRoom : SystemPresetReverbStd
        {
            public PresetReverbStdLongRoom() : base("ReverbStd_LongRoom") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbStdParameter targetParam = target as ReverbStdParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 20;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.02f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_LargeRoom;
                targetParam.FusedTime = 2.00f;
                targetParam.Coloration = 0.60f;
                targetParam.Damping = 0.40f;
                targetParam.EarlyGain = 0.30f;
                targetParam.FusedGain = 0.45f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbStd / SmallHall
        /// </summary>
        private class PresetReverbStdSmallHall : SystemPresetReverbStd
        {
            public PresetReverbStdSmallHall() : base("ReverbStd_SmallHall") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbStdParameter targetParam = target as ReverbStdParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 25;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.02f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_Hall;
                targetParam.FusedTime = 2.00f;
                targetParam.Coloration = 0.60f;
                targetParam.Damping = 0.40f;
                targetParam.EarlyGain = 0.35f;
                targetParam.FusedGain = 0.45f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbStd / LargeHall
        /// </summary>
        private class PresetReverbStdLargeHall : SystemPresetReverbStd
        {
            public PresetReverbStdLargeHall() : base("ReverbStd_LargeHall") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbStdParameter targetParam = target as ReverbStdParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 35;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.03f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_Hall;
                targetParam.FusedTime = 3.00f;
                targetParam.Coloration = 0.60f;
                targetParam.Damping = 0.30f;
                targetParam.EarlyGain = 0.30f;
                targetParam.FusedGain = 0.45f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbStd / Cavernous
        /// </summary>
        private class PresetReverbStdCavernous : SystemPresetReverbStd
        {
            public PresetReverbStdCavernous() : base("ReverbStd_Cavernous") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbStdParameter targetParam = target as ReverbStdParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 40;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.05f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_Cavernous;
                targetParam.FusedTime = 4.00f;
                targetParam.Coloration = 0.60f;
                targetParam.Damping = 0.70f;
                targetParam.EarlyGain = 0.50f;
                targetParam.FusedGain = 0.50f;
                targetParam.OutGain = 1.00f;
            }
        }

        /// <summary>
        /// ReverbStd / Cathedral
        /// </summary>
        private class PresetReverbStdCathedral : SystemPresetReverbStd
        {
            public PresetReverbStdCathedral() : base("ReverbStd_Cathedral") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ReverbStdParameter targetParam = target as ReverbStdParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.EarlyMode = 40;
                targetParam.PreDelayTimeMax = 0.05f;
                targetParam.PreDelayTime = 0.05f;
                targetParam.FusedMode = ToolSoundFusedMode.FusedMode_Cavernous;
                targetParam.FusedTime = 5.00f;
                targetParam.Coloration = 0.60f;
                targetParam.Damping = 0.30f;
                targetParam.EarlyGain = 0.35f;
                targetParam.FusedGain = 0.50f;
                targetParam.OutGain = 1.00f;
            }
        }

        #endregion

        #region ** Chorus プリセットクラス

        /// <summary>
        /// Chorus / Type1
        /// </summary>
        private class PresetChorusType1 : SystemPresetChorus
        {
            public PresetChorusType1() : base("Chorus_Type1") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ChorusParameter targetParam = target as ChorusParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.DelayTime = 20.0f;
                targetParam.Depth = 0.6f;
                targetParam.Rate = 0.1f;
                targetParam.Feedback = 0.0f;
                targetParam.OutGain = 1.0f;
            }
        }

        /// <summary>
        /// Chorus / Type2
        /// </summary>
        private class PresetChorusType2 : SystemPresetChorus
        {
            public PresetChorusType2() : base("Chorus_Type2") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ChorusParameter targetParam = target as ChorusParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.DelayTime = 30.0f;
                targetParam.Depth = 0.3f;
                targetParam.Rate = 0.5f;
                targetParam.Feedback = 0.1f;
                targetParam.OutGain = 1.0f;
            }
        }

        /// <summary>
        /// Chorus / Type3
        /// </summary>
        private class PresetChorusType3 : SystemPresetChorus
        {
            public PresetChorusType3() : base("Chorus_Type3") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ChorusParameter targetParam = target as ChorusParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.DelayTime = 40.0f;
                targetParam.Depth = 0.1f;
                targetParam.Rate = 1.5f;
                targetParam.Feedback = 0.0f;
                targetParam.OutGain = 1.0f;
            }
        }

        /// <summary>
        /// Chorus / Flanger1
        /// </summary>
        private class PresetChorusFlanger1 : SystemPresetChorus
        {
            public PresetChorusFlanger1() : base("Chorus_Flanger1") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ChorusParameter targetParam = target as ChorusParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.DelayTime = 0.8f;
                targetParam.Depth = 0.9f;
                targetParam.Rate = 0.5f;
                targetParam.Feedback = 0.9f;
                targetParam.OutGain = 1.0f;
            }
        }

        /// <summary>
        /// Chorus / Flanger2
        /// </summary>
        private class PresetChorusFlanger2 : SystemPresetChorus
        {
            public PresetChorusFlanger2() : base("Chorus_Flanger2") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ChorusParameter targetParam = target as ChorusParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.DelayTime = 0.3f;
                targetParam.Depth = 0.9f;
                targetParam.Rate = 2.0f;
                targetParam.Feedback = 0.8f;
                targetParam.OutGain = 1.0f;
            }
        }

        /// <summary>
        /// Chorus / Echo1
        /// </summary>
        private class PresetChorusEcho1 : SystemPresetChorus
        {
            public PresetChorusEcho1() : base("Chorus_Echo1") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ChorusParameter targetParam = target as ChorusParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.DelayTime = 30.0f;
                targetParam.Depth = 0.1f;
                targetParam.Rate = 0.5f;
                targetParam.Feedback = 0.7f;
                targetParam.OutGain = 1.0f;
            }
        }

        /// <summary>
        /// Chorus / Echo2
        /// </summary>
        private class PresetChorusEcho2 : SystemPresetChorus
        {
            public PresetChorusEcho2() : base("Chorus_Echo2") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ChorusParameter targetParam = target as ChorusParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.DelayTime = 40.0f;
                targetParam.Depth = 0.8f;
                targetParam.Rate = 1.0f;
                targetParam.Feedback = 0.8f;
                targetParam.OutGain = 1.0f;
            }
        }

        /// <summary>
        /// Chorus / Wobble1
        /// </summary>
        private class PresetChorusWobble1 : SystemPresetChorus
        {
            public PresetChorusWobble1() : base("Chorus_Wobble1") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ChorusParameter targetParam = target as ChorusParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.DelayTime = 40.0f;
                targetParam.Depth = 0.8f;
                targetParam.Rate = 1.5f;
                targetParam.Feedback = 0.0f;
                targetParam.OutGain = 1.0f;
            }
        }

        /// <summary>
        /// Chorus / Wobble2
        /// </summary>
        private class PresetChorusWobble2 : SystemPresetChorus
        {
            public PresetChorusWobble2() : base("Chorus_Wobble2") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                ChorusParameter targetParam = target as ChorusParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.DelayTime = 50.0f;
                targetParam.Depth = 1.0f;
                targetParam.Rate = 2.0f;
                targetParam.Feedback = 0.3f;
                targetParam.OutGain = 1.0f;
            }
        }

        #endregion

        #region ** Delay プリセットクラス

        /// <summary>
        /// Delay / Type1
        /// </summary>
        private class PresetDelayType1 : SystemPresetDelay
        {
            public PresetDelayType1() : base("Delay_Type1") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                DelayParameter targetParam = target as DelayParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.MaxDelayTime = 100.0f;
                targetParam.DelayTime = 80.0f;
                targetParam.Feedback = 0.6f;
                targetParam.LPF = 1.0f;
                targetParam.OutGain = 1.0f;
            }
        }

        /// <summary>
        /// Delay / Type2
        /// </summary>
        private class PresetDelayType2 : SystemPresetDelay
        {
            public PresetDelayType2() : base("Delay_Type2") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                DelayParameter targetParam = target as DelayParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.MaxDelayTime = 100.0f;
                targetParam.DelayTime = 40.0f;
                targetParam.Feedback = 0.5f;
                targetParam.LPF = 0.4f;
                targetParam.OutGain = 1.0f;
            }
        }

        /// <summary>
        /// Delay / Flange1
        /// </summary>
        private class PresetDelayFlange1 : SystemPresetDelay
        {
            public PresetDelayFlange1() : base("Delay_Flange1") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                DelayParameter targetParam = target as DelayParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.MaxDelayTime = 100.0f;
                targetParam.DelayTime = 0.1f;
                targetParam.Feedback = 0.6f;
                targetParam.LPF = 1.0f;
                targetParam.OutGain = 1.0f;
            }
        }

        /// <summary>
        /// Delay / Flange2
        /// </summary>
        private class PresetDelayFlange2 : SystemPresetDelay
        {
            public PresetDelayFlange2() : base("Delay_Flange2") { }

            public override void Load(EffectParameter target)
            {
                if (null == target) { throw new ArgumentNullException("target"); }

                DelayParameter targetParam = target as DelayParameter;
                if (null == targetParam) { throw new ArgumentException("invalid parameter"); }

                targetParam.MaxDelayTime = 100.0f;
                targetParam.DelayTime = 0.5f;
                targetParam.Feedback = 0.6f;
                targetParam.LPF = 0.7f;
                targetParam.OutGain = 1.0f;
            }
        }

        #endregion

        #endregion
    }

    #endregion
}
