﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace Nintendo.Alto.Foundation.Communications
{
    using System;
    using System.IO;

    /// <summary>
    /// HostIO のストリームクラスです。
    /// TODO : CachedStream に改名する。
    /// </summary>
    /// <remarks>
    /// WriteCacheSize 分だけキャッシュします。
    /// </remarks>
    public class HostIOStream : Stream
    {
        private const int WriteCacheSize = 1024;

        private readonly Stream sourceStream;

        private readonly byte[] writeCache = new byte[WriteCacheSize];
        private int writeCacheCurrent = 0;

        //-----------------------------------------------------------------

        public HostIOStream(Stream sourceStream)
        {
            //Assertion.Argument.NotNull(sourceStream);
            this.sourceStream = sourceStream;
        }

        //-----------------------------------------------------------------

        public Stream SourceStream
        {
            get { return this.sourceStream; }
        }

        public override bool CanRead
        {
            get { return this.sourceStream.CanRead; }
        }

        public override bool CanSeek
        {
            get { return this.sourceStream.CanSeek; }
        }

        public override bool CanWrite
        {
            get { return this.sourceStream.CanWrite; }
        }

        public override long Length
        {
            get { return this.sourceStream.Length; }
        }

        public override long Position
        {
            get { return this.sourceStream.Position; }
            set { this.sourceStream.Position = value; }
        }

        //-----------------------------------------------------------------

        public override void Flush()
        {
            this.FlushCache();
            this.sourceStream.Flush();
        }

        public override int Read(byte[] buffer, int offset, int count)
        {
            return this.sourceStream.Read(buffer, offset, count);
        }

        public override long Seek(long offset, SeekOrigin origin)
        {
            return this.sourceStream.Seek(offset, origin);
        }

        public override void SetLength(long value)
        {
            this.sourceStream.SetLength(value);
        }

        public override void Write(byte[] buffer, int offset, int count)
        {
            // 転送回数を削減するためにキャッシュして出力します。
            // バッファサイズを超える場合は、キャッシュ済みの内容を先に出力します。
            // count がバッファより大きい場合は、直接出力します。
            if (this.writeCacheCurrent + count < this.writeCache.Length)
            {
                Array.Copy(buffer, offset, this.writeCache, this.writeCacheCurrent, count);
                this.writeCacheCurrent += count;
                return;
            }

            this.FlushCache();

            if (count > this.writeCache.Length)
            {
                this.sourceStream.Write(buffer, 0, count);
                return;
            }

            Array.Copy(buffer, offset, this.writeCache, 0, count);
        }

        protected override void Dispose(bool disposing)
        {
            base.Dispose(disposing);

            if (disposing && this.sourceStream != null)
            {
                this.sourceStream.Close();
            }
        }

        private void FlushCache()
        {
            if (this.writeCacheCurrent > 0)
            {
                this.sourceStream.Write(this.writeCache, 0, this.writeCacheCurrent);
                this.writeCacheCurrent = 0;
            }
        }
    }
}
