﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace Nintendo.Alto.Foundation.Communications
{
    using System;
    using System.Collections.Generic;
    using System.Net.Sockets;
    using System.Text;
    using System.Text.RegularExpressions;
    using System.Threading;

    /// <summary>
    /// ポートマッピングチャンネルを制御するクラスです。
    /// </summary>
    public sealed class HostIOPortMappingChannel : HostIOChannel
    {
        private const int ChannelPort = 6003;               // ポートマッピングチャンネルのポート番号:6003
        private const char OpenChannelCode = '+';           // チャンネルが開かれたことを示すコード
        private const char CloseChannelCode = '-';          // チャンネルが閉じられたことを示すコード
        private const char ChannelInfoSeparateCode = ':';   // チャンネル情報のセパレータ

        private static readonly char[] OpenCloseCodes = new char[] { OpenChannelCode, CloseChannelCode };

        //-----------------------------------------------------------------

        private Thread workThread;

        //-----------------------------------------------------------------

        /// <summary>
        /// チャンネルコマンドを示します。
        /// </summary>
        private class ChannelCommand
        {
            public ChannelCommand(bool isOpened, HostIOChannelInfo channelInfo)
            {
                //Assertion.Argument.NotNull(channelInfo);
                this.IsOpened = isOpened;
                this.ChannelInfo = channelInfo;
            }

            public bool IsOpened { get; private set; }
            public HostIOChannelInfo ChannelInfo { get; private set; }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <remarks>
        /// Open 前に ChannelOpened イベントを設定しておく必要があるので、
        /// インスタンス生成を許しています。
        /// </remarks>
        public HostIOPortMappingChannel()
            : base(HostIOPortMappingChannel.ChannelPort)
        {
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// チャンネルが開かれると発生します。
        /// </summary>
        public event EventHandler<HostIOChannelEventArgs> ChannelOpend;

        /// <summary>
        /// チャンネルが閉じられると発生します。
        /// </summary>
        public event EventHandler<HostIOChannelEventArgs> ChannelClosed;

        //-----------------------------------------------------------------

        /// <summary>
        /// チャンネルを開きます。
        /// </summary>
        public new void Open()
        {
            base.Open();

            this.workThread = new Thread(this.ChannelMain);
            //Ensure.Operation.NotNull(this.workThread);

            this.workThread.Start();
        }

        protected override void OnClosing()
        {
            Thread thread;
            thread = this.workThread;
            this.workThread = null;

            if (thread != null)
            {
                thread.Join();
            }
        }

        /// <summary>
        /// チャンネル通信スレッドのメイン関数です。
        /// </summary>
        /// <param name="obj">スレッド引数を指定します。</param>
        private void ChannelMain(object obj)
        {
            try
            {
                while (this.IsOpened)
                {
                    string commandText = this.ReadChannelCommandTexts(this.TcpClient);

                    if (!this.IsOpened || this.workThread == null)
                    {
                        break;
                    }

                    if (string.IsNullOrEmpty(commandText))
                    {
                        Thread.Sleep(50);
                        continue;
                    }

                    foreach (var command in this.ExtractChannelCommands(commandText))
                    {
                        if (command.IsOpened)
                        {
                            this.OnChannelOpened(command.ChannelInfo);
                        }
                        else
                        {
                            this.OnChannelClosed(command.ChannelInfo);
                        }
                    }
                }
            }
            catch
            {
            }
        }

        /// <summary>
        /// 指定 TCP クライアントからチャンネルコマンドテキストを読み込みます。
        /// </summary>
        /// <param name="client">TCP クライアントを指定します。</param>
        /// <returns>読み込んだテキストを返します。</returns>
        private string ReadChannelCommandTexts(TcpClient client)
        {
            //Assertion.Argument.NotNull(client);

            if (client.Available == 0)
            {
                return string.Empty;
            }

            // ソケットから呼んだ文字列を this.incomingChannelText に追加します。
            byte[] buffer = new byte[client.Available];
            int readLength = client.GetStream().Read(buffer, 0, buffer.Length);

            string text = Encoding.ASCII.GetString(buffer, 0, readLength);

            // 制御コードから始まっていない場合は、次の制御コードまでスキップさせます。
            // ※通常あり得ないはず
            int channelInfoIndex = text.IndexOfAny(OpenCloseCodes);

            if (channelInfoIndex > 0)
            {
                text = text.Substring(channelInfoIndex);
                //Assertion.Fail("[Warning] unexpected channel command.");
            }

            return text;
        }

        /// <summary>
        /// 指定テキストからチャンネルコマンドを抽出し列挙します。
        /// </summary>
        /// <param name="text">チャンネルコマンドテキストを指定します。</param>
        /// <returns>チャンネルコマンドの列挙子を返します。</returns>
        private IEnumerable<ChannelCommand> ExtractChannelCommands(string text)
        {
            if (string.IsNullOrEmpty(text))
            {
                yield break;
            }

            string workText = text;

            while (true)
            {
                if (workText.Length <= 1)
                {
                    break;
                }

                int nextCommandIndex = workText.IndexOfAny(OpenCloseCodes, 1);

                string commandText = null;

                if (nextCommandIndex > 0)
                {
                    commandText = workText.Substring(0, nextCommandIndex);
                    workText = workText.Substring(nextCommandIndex);
                }
                else
                {
                    commandText = workText;
                    workText = string.Empty;
                }

                ChannelCommand command = this.ParseChannelCommand(commandText);

                if (command != null)
                {
                    yield return command;
                }
            }
        }

        /// <summary>
        /// チャンネルコマンドテキストを解析します。
        /// </summary>
        /// <param name="text">解析するテキストを指定します。</param>
        /// <returns>テキストを解析して生成したチャンネルコマンドを返します。</returns>
        private ChannelCommand ParseChannelCommand(string text)
        {
            if (string.IsNullOrEmpty(text))
            {
                return null;
            }

            // ^([+-])(.*?):(.*)$
            // [+-]  : Open or Close
            // (.*?) : チャンネル名
            // (.*)  : ポート番号
            Match match = Regex.Match(text, @"^([\+\-])(.*?):(.*)$");

            if (match.Groups.Count != 4)
            {
                return null;
            }

            // チャンネル名をチェック
            string channelName = match.Groups[2].Value;

            if (string.IsNullOrEmpty(channelName))
            {
                return null;
            }

            // ポート番号をチェック
            int port = 0;

            if (!int.TryParse(match.Groups[3].Value, out port))
            {
                return null;
            }

            // Open or Close をチェックして、チャンネルコマンドを生成
            if (match.Groups[1].Value.Length != 1)
            {
                return null;
            }

            var channelInfo = new HostIOChannelInfo(channelName, port);

            if (match.Groups[1].Value[0] == OpenChannelCode)
            {
                return new ChannelCommand(true, channelInfo);
            }
            else if (match.Groups[1].Value[0] == CloseChannelCode)
            {
                return new ChannelCommand(false, channelInfo);
            }

            return null;
        }

        private void OnChannelOpened(HostIOChannelInfo channelInfo)
        {
            //Assertion.Argument.NotNull(channelInfo);

            if (this.ChannelOpend != null)
            {
                this.ChannelOpend(this, new HostIOChannelEventArgs(channelInfo));
            }
        }

        private void OnChannelClosed(HostIOChannelInfo channelInfo)
        {
            //Assertion.Argument.NotNull(channelInfo);

            if (this.ChannelClosed != null)
            {
                this.ChannelClosed(this, new HostIOChannelEventArgs(channelInfo));
            }
        }
    }
}
