﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundMaker.Preview.Htcs
{
    using System;
    using System.Diagnostics;
    using System.Linq;
    using System.Net;
    using System.Threading;
    using System.Windows.Forms;
    using Nintendo.InGameEditing.Communication;
    using NintendoWare.SoundMaker.Configurations;
    using NintendoWare.SoundMaker.Framework;
    using NintendoWare.SoundMaker.Framework.Preview.Communications;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Func;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Tool;
    using NintendoWare.SoundMaker.Framework.Windows.Forms;
    using NintendoWare.SoundMaker.Windows.Forms;
    using NintendoWare.Generic.Preview.Htcs.Func;
    using NintendoWare.Generic.Preview.Htcs.Sync;
    using NintendoWare.Generic.Preview.Htcs.Tool;
    using NintendoWare.Generic.Preview.Htcs.Viewer;
    using NintendoWare.Generic.Preview.Htcs.Ping;

    using NintendoWare.Generic.Preview.Htcs.Ctrl;
    using NintendoWare.SoundMaker.Framework.Preview.Communications.Ctrl;
    using NintendoWare.SoundMaker.Resources;

    using NintendoWare.SoundFoundation.Projects;

    using Nintendo.Alto.Foundation.Communications;

    /// <summary>
    /// Htcsマネージャ
    /// <para>
    /// Htcsの管理を行う静的なクラスです。ツールからビューアへパケットを送る接続
    /// （ツール接続）、ビューアからツールへパケットを送る接続（ビューア接続）、
    /// ビューアへのPingを行う接続（Ping接続）の三種類の接続を管理します。
    /// </para>
    /// </summary>
    public class HtcsManager : CommManager
    {
        /// <summary>デフォルトホスト名</summary>
        public const string DefaultHostName = "localhost";

        /// ホスト名
        private string _hostName = DefaultHostName;

        // 切断タイムアウト
        private int timeout = 5; // デフォルト 5 sec

        ///
        private IPEndPoint ipEndPoint = null;

        private const string BaseName = "NN_ATK_";
        private string ToolName = BaseName + "TOOL";
        private string ViewerName = BaseName + "VIEWER";
        private string SyncName = BaseName + "SYNC";
        private string FuncName = BaseName + "FUNC";
        private string CtrlName = BaseName + "CTRL";
        private string PingName = BaseName + "PING";

        ///
        private IComEndPoint currentComEndPoint = null;

        ///
        delegate bool ConnectToSoundPlayerDelegate();

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public HtcsManager()
        {
            this.PeerType = "Generic"; // デフォルト PC 接続
        }

        /// <summary>
        /// 初期化
        /// </summary>
        /// <param name="console">Htcsのメッセージ出力先となるコンソール</param>
        public override void Initialize(Control invokeControl, ICommConsole console)
        {
            base.Initialize(invokeControl, console);

            ToolConnection = new HtcsToolConnection();
            ViewerConnection = new HtcsViewerConnection();
            PingConnection = new HtcsPingConnection();
            SyncConnection = new HtcsSyncConnection();
            FuncConnection = new HtcsFuncConnection();
            CtrlConnection = new HtcsCtrlConnection();

            //
            TargetIsPC();
        }

        /// <summary>
        ///
        /// </summary>
        public SynchronizationContext SynchronizationContext
        {
            get;
            set;
        }

        /// <summary>
        ///
        /// </summary>
        public void TargetIsPC()
        {
            this.currentComEndPoint = HostIOWin.Instance;
        }

        /// <summary>
        /// 接続反応がなくなってから切断するまでの待ち時間（秒）。
        /// </summary>
        public int Timeout
        {
            get
            {
                return this.timeout;
            }
            set
            {
                if (value < 0)
                {
                    throw new ArgumentException(MessageResource.Message_ErrorCanNotSetANegativeValueToTimeOut);
                }

                this.timeout = value;
            }
        }

        /// <summary>
        /// Htcs 接続のための PeerType を取得、設定します。
        /// </summary>
        public string PeerType
        {
            get; set;
        }

        /// <summary>
        ///
        /// </summary>
        private IPHostEntry GetHostEntry(string hostName)
        {
            try
            {
                IPHostEntry entry = Dns.GetHostEntry(hostName);
                if (entry.AddressList.Length <= 0)
                {
                    ShowError(string.Format("DNSによる名前解決に失敗しました。{0}", hostName));
                    return null;
                }
                return entry;
            }
            catch (Exception exception)
            {
                ShowError(string.Format
                           ("DNSによる名前解決に問題が発生しました。{0}\r\n{2}",
                             hostName, exception.Message));
                return null;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private object GetConnectPoint(ConnectionManager manager, string name)
        {
            int maxRetryCount = 5;
            int oneSecond = 1000; // 1000 ms = 1 sec
            int? port = null;
            int count = maxRetryCount;

            do
            {
                var info = manager.TargetInfos.Where(t => t.PeerType.Equals(this.PeerType)).SelectMany(i => i.PortInfos).FirstOrDefault(p => p.PortName.Equals(name));
                if (info != null)
                {
                    port = info.EndPoint.Port;
                    break;
                }

                Thread.Sleep(oneSecond); // 情報が得られなかったら１秒待ってリトライする
            } while (0 < --count);

            return port;
        }

        /// <summary>
        ///
        /// </summary>
        public void ConnectTo6003()
        {
            try
            {
                this.currentComEndPoint.Connect(null);
            }
            catch
            {
            }
        }

        /// <summary>
        /// 接続します。
        /// </summary>
        public override bool Connect()
        {
            ConnectTo6003();

            // 接続されているのか？
            if (IsConnected != false)
            {
                return true;
            }

            // Htcsは接続されているが、チャンネル接続がされていないのか？
            if (IsConnecting == false)
            {
                SetStateSndEdit(ConnectionState.Connecting);
            }

            using (var manager = new ConnectionManager())
            {
                manager.Start();
                if (this.WaitForGetPortInformation(manager) == false)
                {
                    Disconnect();
                    return false;
                }

                // Sync接続の確立
                HtcsSyncConnection.ComEndPoint = this.currentComEndPoint;
                HtcsSyncConnection.Port = GetConnectPoint(manager, SyncName);
                if (HtcsSyncConnection.Port == null || HtcsSyncConnection.Connect(ipEndPoint, _hostName, this.Timeout) == false)
                {
                    Disconnect();
                    return false;
                }

                // Func接続の確立
                HtcsFuncConnection.ComEndPoint = this.currentComEndPoint;
                HtcsFuncConnection.Port = GetConnectPoint(manager, FuncName);
                if (HtcsFuncConnection.Port == null || HtcsFuncConnection.Connect(ipEndPoint, _hostName) == false)
                {
                    Disconnect();
                    return false;
                }

                // Ctrl接続の確立
                HtcsCtrlConnection.ComEndPoint = this.currentComEndPoint;
                HtcsCtrlConnection.Port = GetConnectPoint(manager, CtrlName);
                if (HtcsCtrlConnection.Port == null || HtcsCtrlConnection.Connect(ipEndPoint, _hostName) == false)
                {
                    Disconnect();
                    return false;
                }
            }

            return true;
        }

        protected override void OnDisconnected()
        {
            base.OnDisconnected();

            if (this.currentComEndPoint != null)
            {
                this.currentComEndPoint.Disconnect();
            }
        }

        /// <summary>
        /// SoundPlayerとの接続を行います。
        /// </summary>
        public override bool ConnectToSoundPlayer()
        {
            FormsApplication app = ApplicationBase.Instance as FormsApplication;
            FormsUIService service = app.UIService as FormsUIService;
            MainWindow mainWindow = service.MainWindow;

            if (mainWindow.InvokeRequired != false)
            {
                mainWindow.BeginInvoke
                    (new ConnectToSoundPlayerDelegate(ConnectToSoundPlayerInternal));
                return true;
            }
            else
            {
                return ConnectToSoundPlayerInternal();
            }
        }

        /// <summary>
        /// SoundPlayerとの接続を行います。
        /// </summary>
        private bool ConnectToSoundPlayerInternal()
        {
            SetState(ConnectionState.Connecting);

            using (var manager = new ConnectionManager())
            {
                manager.Start();
                if (this.WaitForGetPortInformation(manager) == false)
                {
                    Disconnect();
                    return false;
                }

                // Ping接続の確立
                HtcsPingConnection.ComEndPoint = this.currentComEndPoint;
                HtcsPingConnection.Port = GetConnectPoint(manager, PingName);
                if (HtcsPingConnection.Port == null || HtcsPingConnection.Connect(ipEndPoint, _hostName) == false)
                {
                    Disconnect();
                    return false;
                }

                // Tool接続の確立
                HtcsToolConnection.ComEndPoint = this.currentComEndPoint;
                HtcsToolConnection.Port = GetConnectPoint(manager, ToolName);
                if (HtcsToolConnection.Port == null || HtcsToolConnection.Connect(ipEndPoint, _hostName) == false)
                {
                    Disconnect();
                    return false;
                }

                // Viewer接続の確立
                HtcsViewerConnection.ComEndPoint = this.currentComEndPoint;
                HtcsViewerConnection.Port = GetConnectPoint(manager, ViewerName);
                if (HtcsViewerConnection.Port == null || HtcsViewerConnection.Connect(ipEndPoint, _hostName) == false)
                {
                    Disconnect();
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// 終了待ち
        /// </summary>
        public override bool IsWantToExit
        {
            get { return HtcsToolConnection.IsWantToExit; }
        }

        /// <summary>
        /// メッセージ表示
        /// </summary>
        protected void ShowMessage(string text)
        {
            Console.WriteMessage(text);
        }

        /// <summary>
        /// エラー表示
        /// </summary>
        protected void ShowError(string text)
        {
            Console.WriteError(text);
        }

        /// <summary>
        /// Tool接続を取得します。
        /// </summary>
        private HtcsToolConnection HtcsToolConnection
        {
            get
            {
                HtcsToolConnection connection = ToolConnection as HtcsToolConnection;
                Debug.Assert(connection != null);
                return connection;
            }
        }

        /// <summary>
        /// Viewer接続を取得します。
        /// </summary>
        private HtcsViewerConnection HtcsViewerConnection
        {
            get
            {
                HtcsViewerConnection connection = ViewerConnection as HtcsViewerConnection;
                Debug.Assert(connection != null);
                return connection;
            }
        }

        /// <summary>
        /// Ping接続を取得します。
        /// </summary>
        private HtcsPingConnection HtcsPingConnection
        {
            get
            {
                HtcsPingConnection connection = PingConnection as HtcsPingConnection;
                Debug.Assert(connection != null);
                return connection;
            }
        }

        /// <summary>
        /// Sync接続を取得します。
        /// </summary>
        private HtcsSyncConnection HtcsSyncConnection
        {
            get
            {
                HtcsSyncConnection connection = SyncConnection as HtcsSyncConnection;
                Debug.Assert(connection != null);
                return connection;
            }
        }

        /// <summary>
        /// Func接続を取得します。
        /// </summary>
        private HtcsFuncConnection HtcsFuncConnection
        {
            get
            {
                HtcsFuncConnection connection = FuncConnection as HtcsFuncConnection;
                Debug.Assert(connection != null);
                return connection;
            }
        }

        /// <summary>
        /// Ctrl接続を取得します。
        /// </summary>
        private HtcsCtrlConnection HtcsCtrlConnection
        {
            get
            {
                HtcsCtrlConnection connection = CtrlConnection as HtcsCtrlConnection;
                Debug.Assert(connection != null);
                return connection;
            }
        }

        /// <summary>
        /// ポート情報が取得できるようになるまで待ちます。
        /// </summary>
        /// <param name="manager"></param>
        /// <returns></returns>
        private bool WaitForGetPortInformation(ConnectionManager manager)
        {
            // 100msec * 50回 = 5秒でタイムアウト
            int retryCount = 50;
            while (manager.TargetInfos == null || manager.TargetInfos.Count == 0)
            {
                if (--retryCount <= 0)
                {
                    return false;
                }
                Thread.Sleep(100);
            }
            return true;
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class HtcsConsole : ICommConsole
    {
        public void WriteMessage(string format, params object[] args)
        {
            Debug.WriteLine(string.Format(format, args));
        }

        public void WriteError(string format, params object[] args)
        {
            Debug.WriteLine(string.Format(format, args));
        }
    }
}
