﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Net;
using System.Runtime.InteropServices;
using System.Text;
using NW4R.ProtocolSound;

namespace NintendoWare.Preview.HIO.Sound
{

    //-----------------------------------------------------------------------
    /** <summary>Constants</summary> */
    public class Constants{

        //-------------------------------------------------------------------
        /*
            バージョン情報の定義
            バージョン番号の運用は各モジュールとツールでの運用ルールにあわせてください
         */
        /** <summary>プロトコルバージョン</summary> */
        public const uint   PROTOCOL_VERSION =  0x00010002;

        //-------------------------------------------------------------------
        /* メッセージのフラグとマスクの定義 */
        /** <summary>エラー用フラグ        </summary> */
        public const ushort HEADER_ERROR_FLAG = 0x8000;
        /** <summary>要返信メッセージフラグ</summary> */
        public const ushort HEADER_REPLY_FLAG = 0x4000;
        /** <summary>カテゴリ用マスク      </summary> */
        public const ushort HEADER_CATEGORY_MASK =  0x3F00;
        /** <summary>コマンド用マスク      </summary> */
        public const ushort HEADER_COMMAND_MASK =   0x00FF;

        //-------------------------------------------------------------------
        /*
            カテゴリの定義
         */
        /** <summary>システム関連カテゴリ </summary> */
        public const ushort CATEGORY_SYSTEM =   0x0000;
        /** <summary>3Dモデル関連カテゴリ </summary> */
        public const ushort CATEGORY_3D =   0x0100;
        /** <summary>サウンド関連カテゴリ </summary> */
        public const ushort CATEGORY_SOUND =    0x0300;
        /** <summary>MCS通信テストカテゴリ</summary> */
        public const ushort CATEGORY_TEST = 0x3F00;

        //-------------------------------------------------------------------
        /* システム関連カテゴリのメッセージ */
        /** <summary>バージョン要求コマンド</summary> */
        public const ushort TOOL_SYSTEM_VERSION =   0x00 + CATEGORY_SYSTEM + HEADER_REPLY_FLAG;
        /** <summary>同期用コマンド</summary> */
        public const ushort TOOL_SYSTEM_SYNC =  0x01 + CATEGORY_SYSTEM + HEADER_REPLY_FLAG;
        /** <summary>モジュールリセットコマンド</summary> */
        public const ushort TOOL_SYSTEM_RESET = 0x02 + CATEGORY_SYSTEM;

        //-------------------------------------------------------------------
        /* 3Dモデル関連カテゴリのメッセージ */
        /** <summary>モデルデータの読み込みコマンド</summary> */
        public const ushort TOOL_3D_LOAD_MODEL =    0x00 + CATEGORY_3D + HEADER_REPLY_FLAG;
        /** <summary>ロードされているモデルリスト取得コマンド</summary> */
        public const ushort TOOL_3D_GET_MODEL_LIST =    0x01 + CATEGORY_3D + HEADER_REPLY_FLAG;
        /** <summary>モデルのノードリスト取得コマンド</summary> */
        public const ushort TOOL_3D_GET_NODE_LIST = 0x02 + CATEGORY_3D + HEADER_REPLY_FLAG;

        //-------------------------------------------------------------------
        /* サウンド関連カテゴリのメッセージ */
        /** <summary>サウンドシステム関連メッセージ範囲</summary> */
        public const ushort TOOL_SOUND_SYSTEMBEGIN =    0x00 + CATEGORY_SOUND;
        /** <summary>サウンドシステム関連メッセージ範囲</summary> */
        public const ushort TOOL_SOUND_SYSTEMEND =  0x39 + CATEGORY_SOUND;
        /** <summary>サウンドモードへの強制切り替えコマンド</summary> */
        public const ushort TOOL_SOUND_FOCUS =  0x01 + CATEGORY_SOUND;
        /** <summary>エコーコマンド</summary> */
        public const ushort TOOL_SOUND_ECHO =   0x02 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>全発音停止</summary> */
        public const ushort TOOL_SOUND_ALLSTOP =    0x03 + CATEGORY_SOUND;
        /** <summary>サウンドシステム設定</summary> */
        public const ushort TOOL_SOUND_SET_SYSTEMPARAMETERS =   0x04 + CATEGORY_SOUND;
        /** <summary>サウンド関連メッセージ範囲</summary> */
        public const ushort TOOL_SOUND_SOUNDBEGIN = 0x40 + CATEGORY_SOUND;
        /** <summary>サウンド関連メッセージ範囲</summary> */
        public const ushort TOOL_SOUND_SOUNDEND =   0x79 + CATEGORY_SOUND;
        /** <summary>サウンドアーカイブの読み込みコマンド</summary> */
        public const ushort TOOL_SOUND_LOADSAR =    0x41 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>サウンドラベルの設定</summary> */
        public const ushort TOOL_SOUND_SET_SOUNDLABELS =    0x42 + CATEGORY_SOUND;
        /** <summary>サウンド状態の取得</summary> */
        public const ushort TOOL_SOUND_GET_SOUNDINFORMATIONS =  0x43 + CATEGORY_SOUND;
        /** <summary>サウンド情報の表示</summary> */
        public const ushort TOOL_SOUND_SHOWSOUNDINFORMATION =   0x44 + CATEGORY_SOUND;
        /** <summary>再生コマンド</summary> */
        public const ushort TOOL_SOUND_PLAY =   0x45 + CATEGORY_SOUND;
        /** <summary>一時停止コマンド</summary> */
        public const ushort TOOL_SOUND_PAUSE =  0x46 + CATEGORY_SOUND;
        /** <summary>停止コマンド</summary> */
        public const ushort TOOL_SOUND_STOP =   0x47 + CATEGORY_SOUND;
        /** <summary>エフェクトパラメータの転送 PC→Viewer</summary> */
        public const ushort TOOL_SOUND_SET_EFFECTPARAMETERS =   0x48 + CATEGORY_SOUND;
        /** <summary>エフェクトパラメータの転送 PC←Viewer</summary> */
        public const ushort TOOL_SOUND_GET_EFFECTPARAMETERS =   0x49 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>シーケンス変数の転送 PC→Viewer</summary> */
        public const ushort TOOL_SOUND_SET_SEQVARIABLES =   0x50 + CATEGORY_SOUND;
        /** <summary>シーケンス変数の転送 PC←Viewer</summary> */
        public const ushort TOOL_SOUND_GET_SEQVARIABLES =   0x51 + CATEGORY_SOUND;
        /** <summary>プレビュープレイヤーパラメータの転送 PC→Viewer</summary> */
        public const ushort TOOL_SOUND_SET_PREVIEWPLAYERPARAMETER = 0x52 + CATEGORY_SOUND;
        /** <summary>アニメーションサウンド関連メッセージ範囲</summary> */
        public const ushort TOOL_SOUND_ANIM_SOUNDBEGIN =    0x80 + CATEGORY_SOUND;
        /** <summary>アニメーションサウンド関連メッセージ範囲</summary> */
        public const ushort TOOL_SOUND_ANIM_SOUNDEND =  0xB9 + CATEGORY_SOUND;
        /** <summary>3Dモデルデータの読み込みコマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_LOAD3DMODEL =   0x81 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>アニメーションデータの読み込みコマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_LOADANIMATION = 0x82 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>アニメーションサウンドデータの読み込みコマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_LOADANIMSOUND = 0x83 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>アニメーションキャッシュの破棄コマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_DISPOSECACHE =  0x84 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>アクティブアニメーションの切り替え</summary> */
        public const ushort TOOL_SOUND_ANIM_ACTIVATEANIMATION = 0x85 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>アニメーションフレーム取得コマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_QUERYFRAME =    0x86 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>アニメーションフレーム設定コマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_SETFRAME =  0x87 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>アニメーション再生</summary> */
        public const ushort TOOL_SOUND_ANIM_PLAY =  0x88 + CATEGORY_SOUND;
        /** <summary>アニメーション停止</summary> */
        public const ushort TOOL_SOUND_ANIM_STOP =  0x89 + CATEGORY_SOUND;
        /** <summary>アニメーション再生レート設定コマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_SETRATE =   0x90 + CATEGORY_SOUND;
        /** <summary>ループ状態設定コマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_SETLOOP =   0x91 + CATEGORY_SOUND;
        /** <summary>アニメーション再生範囲設定コマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_SETRANGE =  0x92 + CATEGORY_SOUND + HEADER_REPLY_FLAG;
        /** <summary>アニメーションサウンドのイベント更新コマンド</summary> */
        public const ushort TOOL_SOUND_ANIM_MODIFIEREVENT = 0x93 + CATEGORY_SOUND;

        //-------------------------------------------------------------------
        /* テストカテゴリのメッセージ */
        /** <summary>EFBカラー変更コマンド</summary> */
        public const ushort TOOL_TEST_EFB_COLOR =   0x00 + CATEGORY_TEST;

        //-------------------------------------------------------------------
        /* ビューア主導 システム関連カテゴリのメッセージ */
        /** <summary>ビューア主導のテストコマンド</summary> */
        public const ushort VIEWER_TEST_NOP =   0x00 + CATEGORY_TEST;

        //-------------------------------------------------------------------
        /* システムエラーカテゴリのメッセージ */
        /** <summary>未定義コマンドエラー</summary> */
        public const ushort ERR_SYSTEM_UNKNOWN_COMMAND =    0x0 + CATEGORY_SYSTEM + HEADER_ERROR_FLAG;

        //-------------------------------------------------------------------
        /* データエラーカテゴリのメッセージ */
        /** <summary>データロードエラー        </summary> */
        public const ushort ERR_3D_LOAD =   0x0 + CATEGORY_3D + HEADER_ERROR_FLAG;

        //-------------------------------------------------------------------
        /* サウンドカテゴリのメッセージ */
        /** <summary>サウンドエラー</summary> */
        public const ushort ERR_SOUND = 0x0 + CATEGORY_SOUND + HEADER_ERROR_FLAG;

        //-------------------------------------------------------------------
        /* エラーコード用フラグの定義 */
        /** <summary>エラーフラグ</summary> */
        public const uint   MCSSOUND_ERRORFLAG =    0x80000000;

        //-------------------------------------------------------------------
        /* エラーコード用カテゴリの定義 */
        /** <summary>一般</summary> */
        public const uint   MCSSOUNDCATEGORY_GENERAL =  0x00000000;
        /** <summary>サウンド</summary> */
        public const uint   MCSSOUNDCATEGORY_SOUND =    0x00010000;
        /** <summary>アニメーションサウンド</summary> */
        public const uint   MCSSOUNDCATEGORY_ANIMSOUND =    0x00020000;

        //-------------------------------------------------------------------
        /* サウンドエラーコード */
        /** <summary>成功。</summary> */
        public const uint   MCSSOUND_S_OK = 0x0000;
        /** <summary>エラーが発生しました。        </summary> */
        public const uint   MCSSOUND_E_FAILED = 0x0000 + MCSSOUNDCATEGORY_GENERAL + MCSSOUND_ERRORFLAG;
        /** <summary>ファイルが見つかりません。    </summary> */
        public const uint   MCSSOUND_E_FILENOTFOUND =   0x0001 + MCSSOUNDCATEGORY_GENERAL + MCSSOUND_ERRORFLAG;
        /** <summary>メモリが不足しています。      </summary> */
        public const uint   MCSSOUND_E_OUTOFMEMORY =    0x0002 + MCSSOUNDCATEGORY_GENERAL + MCSSOUND_ERRORFLAG;
        /** <summary>無効なデータです。            </summary> */
        public const uint   MCSSOUND_E_INVALIDDATA =    0x0003 + MCSSOUNDCATEGORY_GENERAL + MCSSOUND_ERRORFLAG;
        /** <summary>MCS通信で異常が発生しました。 </summary> */
        public const uint   MCSSOUND_E_TRANSFERFAILED = 0x0004 + MCSSOUNDCATEGORY_GENERAL + MCSSOUND_ERRORFLAG;
        /** <summary>無効なアニメーションIDです。      </summary> */
        public const uint   MCSSOUND_E_ANIM_INVALIDCOMPONENTID =    0x0001 + MCSSOUNDCATEGORY_ANIMSOUND + MCSSOUND_ERRORFLAG;
        /** <summary>モデルキャッシュが見つかりません。</summary> */
        public const uint   MCSSOUND_E_ANIM_MODELCACHENOTFOUND =    0x0002 + MCSSOUNDCATEGORY_ANIMSOUND + MCSSOUND_ERRORFLAG;
        /** <summary>アクションキャッシュが見つかりません。</summary> */
        public const uint   MCSSOUND_E_ANIM_ACTIONCACHENOTFOUND =   0x0003 + MCSSOUNDCATEGORY_ANIMSOUND + MCSSOUND_ERRORFLAG;

    }

    //-----------------------------------------------------------------------
    /** <summary>Soundプロトコルのメッセージ共通ヘッダ</summary> */
    public class Header {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+2+2;
        /** <summary>メッセージタイプ   </summary> */
        public ushort   type { get; set; }
        /** <summary>メッセージサイズ   </summary> */
        public ushort   size;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            type = reader.ReadUInt16();
            size = reader.ReadUInt16();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(type);
            writer.Write(size);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
         バージョン要求応答メッセージ
            ツールのプロトコルバージョンを受け取り、ビューアのプロトコルバージョンを返します。
        </summary> */
    public class ToolSystemVersion {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>4B: バージョン番号</summary> */
        public uint version;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            version = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(version);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
        同期メッセージ

              Viewerは、この同期コマンドを受け取ると、指定されたtagName
              名をエコーした応答コマンドをアプリケーションに返信します。

              tagNameは､ツール側が自由に設定できます。
        </summary> */
    public class ToolSystemSync {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>4B: タグ名</summary> */
        public uint tagName;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            tagName = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(tagName);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            モデルデータの読み込みメッセージ

            fileSizeが0の場合には、そのモデルIDを持つデータの破棄とみなします。

            ロードに成功した場合にはTool3DLoadModelReplyを成功応答メッセージとして返します。
            失敗した場合には、ER_DATA_LOADコマンドを返します。
        </summary> */
    public class Tool3DLoadModel {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+1*260;
        /** <summary>  4B: ファイルサイズ       </summary> */
        public uint fileSize;
        /** <summary>260B: PC上のデータのパス名 </summary> */
        public byte[]   pathName = new byte[260];

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            fileSize = reader.ReadUInt32();
            for(int i = 0; i < 260; i++){
                pathName[i] = reader.ReadByte();
            }
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(fileSize);
            for(int i = 0; i < 260; i++){
                writer.Write(pathName[i]);
            }
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            モデルデータの読み込み応答メッセージ
        </summary> */
    public class Tool3DLoadModelReply {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>  4B: ロードされたモデルの管理ID  </summary> */
        public uint id;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            id = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(id);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            ロードされているモデルリスト応答メッセージ
            この後にsize個のモデルIDが続く可変長の応答メッセージです。
        </summary> */
    public class Tool3DGetModelListReply {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>                4B: Viewerにロードされているモデルの数           </summary> */
        public uint modelCount;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            modelCount = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(modelCount);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            モデルリストの要素構造体
        </summary> */
    public class Tool3DGetModelListReplyElement {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+4;
        /** <summary>          4B: モデルの管理ID              </summary> */
        public uint id;
        /** <summary>          4B: モデル名の文字列長(NUL込み) </summary> */
        public uint nameSize;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            id = reader.ReadUInt32();
            nameSize = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(id);
            writer.Write(nameSize);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            モデルのノードリスト取得メッセージ
        </summary> */
    public class Tool3DGetNodeList {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>  4B: モデルの管理ID           </summary> */
        public uint id;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            id = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(id);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            モデルのノードリスト応答メッセージ
            モデルが存在しない場合にはnameLengthが0で返る
        </summary> */
    public class Tool3DGetNodeListReply {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>               4B: モデル内のノード数    </summary> */
        public int  nodeCount;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            nodeCount = reader.ReadInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(nodeCount);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            ノードリストの要素構造体
        </summary> */
    public class Tool3DGetNodeListReplyElement {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+4+4;
        /** <summary>          4B: ノードのインデックス                                 </summary> */
        public int  idx;
        /** <summary>          4B: 親ノードのインデックス(ルートノードの場合には負の値) </summary> */
        public int  parentIdx;
        /** <summary>          4B: ノード名の文字列長(NUL込み)                          </summary> */
        public uint nameSize;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            idx = reader.ReadInt32();
            parentIdx = reader.ReadInt32();
            nameSize = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(idx);
            writer.Write(parentIdx);
            writer.Write(nameSize);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>一括送信用アイテムコンテナ</summary> */
    public class ToolSoundItemContainer {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+4+4;
        /** <summary>データへのオフセット</summary> */
        public uint offsetForData;
        /** <summary>アイテム数</summary> */
        public uint itemCount;
        /** <summary>1アイテムのサイズ</summary> */
        public uint itemSize;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            offsetForData = reader.ReadUInt32();
            itemCount = reader.ReadUInt32();
            itemSize = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(offsetForData);
            writer.Write(itemCount);
            writer.Write(itemSize);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>ファイル情報</summary> */
    public class ToolSoundFileInfo {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+1*260;
        /** <summary>260B: PC上のデータのパス名</summary> */
        public byte[]   fileName = new byte[260];

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            for(int i = 0; i < 260; i++){
                fileName[i] = reader.ReadByte();
            }
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            for(int i = 0; i < 260; i++){
                writer.Write(fileName[i]);
            }
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>サウンドの状態</summary> */
    public enum ToolSoundOutputMode {
        /** <summary>無効値</summary> */
        OutputMode_Invalid =    0,
        /** <summary>モノラル</summary> */
        OutputMode_Monoral =    1,
        /** <summary>ステレオ</summary> */
        OutputMode_Stereo = 2,
        /** <summary>サラウンド</summary> */
        OutputMode_Surround =   3,
        /** <summary>DPL2</summary> */
        OutputMode_Dpl2 =   4,
    }

    //-----------------------------------------------------------------------
    /** <summary>サウンドアーカイブ読み込みメッセージ </summary> */
    public class ToolSoundSystemParameters {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary> 4B: 出力モード</summary> */
        public ToolSoundOutputMode  outputMode;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            outputMode = (ToolSoundOutputMode)reader.ReadInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write((int)outputMode);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>サウンドアーカイブ読み込みメッセージ </summary> */
    public class ToolSoundLoadSar {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+1*260;
        /** <summary>260B: PC上のデータのパス名</summary> */
        public byte[]   pathName = new byte[260];

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            for(int i = 0; i < 260; i++){
                pathName[i] = reader.ReadByte();
            }
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            for(int i = 0; i < 260; i++){
                writer.Write(pathName[i]);
            }
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>サウンドアーカイブ読み込み応答メッセージ</summary> */
    public class ToolSoundLoadSarReply {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>4B: 未使用</summary> */
        public uint id;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            id = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(id);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>サウンドの状態</summary> */
    public enum ToolSoundPreviewSoundState {
        /** <summary>サウンドが指定されていない</summary> */
        SoundState_Disabled =   0,
        /** <summary>停止中</summary> */
        SoundState_Stopped =    1,
        /** <summary>再生準備中</summary> */
        SoundState_Preparing =  2,
        /** <summary>再生中</summary> */
        SoundState_Playing =    3,
        /** <summary>一時停止中</summary> */
        SoundState_Paused = 4,
    }

    //-----------------------------------------------------------------------
    /** <summary>サウンドのラベル</summary> */
    public class ToolSoundPreviewSoundLabel {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+1*260;
        /** <summary>  4B: パラメータの有効フラグ（1:有効, 0:無効）</summary> */
        public uint index;
        /** <summary>260B: サウンドの名前（無効の場合は設定対象としない）</summary> */
        public byte[]   label = new byte[260];

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            index = reader.ReadUInt32();
            for(int i = 0; i < 260; i++){
                label[i] = reader.ReadByte();
            }
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(index);
            for(int i = 0; i < 260; i++){
                writer.Write(label[i]);
            }
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>サウンドの情報</summary> */
    public class ToolSoundPreviewSoundInfomation {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>4B: サウンドの状態</summary> */
        public ToolSoundPreviewSoundState   state;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            state = (ToolSoundPreviewSoundState)reader.ReadInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write((int)state);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>プレビュープレイヤー識別子</summary> */
    public class ToolSoundPreviewPlayerID {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>4B: プレビュープレイヤーのインデックス</summary> */
        public uint index;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            index = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(index);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>バスの種類</summary> */
    public enum ToolSoundAuxBusType {
        /** <summary>Aux A </summary> */
        AuxBusType_AuxA =   0,
        /** <summary>Aux B </summary> */
        AuxBusType_AuxB =   1,
        /** <summary>Aux C </summary> */
        AuxBusType_AuxC =   2,
    }

    //-----------------------------------------------------------------------
    /** <summary>エフェクトの種類</summary> */
    public enum ToolSoundEffectType {
        /** <summary>NO EFFECT  </summary> */
        EffectType_NoEffect =   0,
        /** <summary>REVERB HI  </summary> */
        EffectType_ReverbHi =   1,
        /** <summary>REVERB STD </summary> */
        EffectType_ReverbStd =  2,
        /** <summary>CHORUS     </summary> */
        EffectType_Chorus = 3,
        /** <summary>DELAY      </summary> */
        EffectType_Delay =  4,
    }

    //-----------------------------------------------------------------------
    /** <summary>後部残響音の種類</summary> */
    public enum ToolSoundFusedMode {
        /** <summary>old axfx    </summary> */
        FusedMode_OldAxfx = 0,
        /** <summary>metal tank  </summary> */
        FusedMode_MetalTank =   1,
        /** <summary>small room  </summary> */
        FusedMode_SmallRoom =   2,
        /** <summary>large room  </summary> */
        FusedMode_LargeRoom =   3,
        /** <summary>hall        </summary> */
        FusedMode_Hall =    4,
        /** <summary>cavernous   </summary> */
        FusedMode_Cavernous =   5,
    }

    //-----------------------------------------------------------------------
    /** <summary>各種エフェクトパラメータ共通部</summary> */
    public class ToolSoundEffectCommonParameter {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+4+4+1*44;
        /** <summary> 4B: バスの種類</summary> */
        public ToolSoundAuxBusType  busType;
        /** <summary> 4B: インデックス</summary> */
        public uint effectIndex;
        /** <summary> 4B:エフェクトの種類</summary> */
        public ToolSoundEffectType  effectType;
        /** <summary>44B: reserved</summary> */
        public byte[]   reserved = new byte[44];

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            busType = (ToolSoundAuxBusType)reader.ReadInt32();
            effectIndex = reader.ReadUInt32();
            effectType = (ToolSoundEffectType)reader.ReadInt32();
            for(int i = 0; i < 44; i++){
                reserved[i] = reader.ReadByte();
            }
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write((int)busType);
            writer.Write(effectIndex);
            writer.Write((int)effectType);
            for(int i = 0; i < 44; i++){
                writer.Write(reserved[i]);
            }
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>NO EFFECT パラメータ</summary> */
    public class ToolSoundNoEffectParameter {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+4+4+1*44;
        /** <summary> 4B: バスの種類</summary> */
        public ToolSoundAuxBusType  busType;
        /** <summary> 4B: インデックス</summary> */
        public uint effectIndex;
        /** <summary> 4B:エフェクトの種類</summary> */
        public ToolSoundEffectType  effectType;
        /** <summary>44B: reserved</summary> */
        public byte[]   reserved = new byte[44];

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            busType = (ToolSoundAuxBusType)reader.ReadInt32();
            effectIndex = reader.ReadUInt32();
            effectType = (ToolSoundEffectType)reader.ReadInt32();
            for(int i = 0; i < 44; i++){
                reserved[i] = reader.ReadByte();
            }
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write((int)busType);
            writer.Write(effectIndex);
            writer.Write((int)effectType);
            for(int i = 0; i < 44; i++){
                writer.Write(reserved[i]);
            }
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>REVERB HI パラメータ</summary> */
    public class ToolSoundReverbHiParameter {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+4+4+4+4+4+4+4+4+4+4+4+4+4;
        /** <summary> 4B: バスの種類</summary> */
        public ToolSoundAuxBusType  busType;
        /** <summary> 4B: インデックス</summary> */
        public uint effectIndex;
        /** <summary> 4B: エフェクトの種類</summary> */
        public ToolSoundEffectType  effectType;
        /** <summary> 4B: early mode</summary> */
        public int  earlyMode;
        /** <summary> 4B: pre delay time max</summary> */
        public float    preDelayTimeMax;
        /** <summary> 4B: pre delay time</summary> */
        public float    preDelayTime;
        /** <summary> 4B: fused mode</summary> */
        public ToolSoundFusedMode   fusedMode;
        /** <summary> 4B: fused time</summary> */
        public float    fusedTime;
        /** <summary> 4B: coloration</summary> */
        public float    coloration;
        /** <summary> 4B: damping</summary> */
        public float    damping;
        /** <summary> 4B: crosstalk</summary> */
        public float    crosstalk;
        /** <summary> 4B: early gain</summary> */
        public float    earlyGain;
        /** <summary> 4B: fused gain</summary> */
        public float    fusedGain;
        /** <summary> 4B: out gain</summary> */
        public float    outGain;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            busType = (ToolSoundAuxBusType)reader.ReadInt32();
            effectIndex = reader.ReadUInt32();
            effectType = (ToolSoundEffectType)reader.ReadInt32();
            earlyMode = reader.ReadInt32();
            preDelayTimeMax = reader.ReadSingle();
            preDelayTime = reader.ReadSingle();
            fusedMode = (ToolSoundFusedMode)reader.ReadInt32();
            fusedTime = reader.ReadSingle();
            coloration = reader.ReadSingle();
            damping = reader.ReadSingle();
            crosstalk = reader.ReadSingle();
            earlyGain = reader.ReadSingle();
            fusedGain = reader.ReadSingle();
            outGain = reader.ReadSingle();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write((int)busType);
            writer.Write(effectIndex);
            writer.Write((int)effectType);
            writer.Write(earlyMode);
            writer.Write(preDelayTimeMax);
            writer.Write(preDelayTime);
            writer.Write((int)fusedMode);
            writer.Write(fusedTime);
            writer.Write(coloration);
            writer.Write(damping);
            writer.Write(crosstalk);
            writer.Write(earlyGain);
            writer.Write(fusedGain);
            writer.Write(outGain);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>REVERB STD パラメータ</summary> */
    public class ToolSoundReverbStdParameter {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+4+4+4+4+4+4+4+4+4+4+4+4+1*4;
        /** <summary> 4B: バスの種類</summary> */
        public ToolSoundAuxBusType  busType;
        /** <summary> 4B: インデックス</summary> */
        public uint effectIndex;
        /** <summary> 4B:エフェクトの種類</summary> */
        public ToolSoundEffectType  effectType;
        /** <summary> 4B: early mode</summary> */
        public int  earlyMode;
        /** <summary> 4B: pre delay time max</summary> */
        public float    preDelayTimeMax;
        /** <summary> 4B: pre delay time</summary> */
        public float    preDelayTime;
        /** <summary> 4B: fused mode</summary> */
        public ToolSoundFusedMode   fusedMode;
        /** <summary> 4B: fused time</summary> */
        public float    fusedTime;
        /** <summary> 4B: coloration</summary> */
        public float    coloration;
        /** <summary> 4B: damping</summary> */
        public float    damping;
        /** <summary> 4B: early gain</summary> */
        public float    earlyGain;
        /** <summary> 4B: fused gain</summary> */
        public float    fusedGain;
        /** <summary> 4B: out gain</summary> */
        public float    outGain;
        /** <summary> 4B: reserved</summary> */
        public byte[]   reserved = new byte[4];

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            busType = (ToolSoundAuxBusType)reader.ReadInt32();
            effectIndex = reader.ReadUInt32();
            effectType = (ToolSoundEffectType)reader.ReadInt32();
            earlyMode = reader.ReadInt32();
            preDelayTimeMax = reader.ReadSingle();
            preDelayTime = reader.ReadSingle();
            fusedMode = (ToolSoundFusedMode)reader.ReadInt32();
            fusedTime = reader.ReadSingle();
            coloration = reader.ReadSingle();
            damping = reader.ReadSingle();
            earlyGain = reader.ReadSingle();
            fusedGain = reader.ReadSingle();
            outGain = reader.ReadSingle();
            for(int i = 0; i < 4; i++){
                reserved[i] = reader.ReadByte();
            }
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write((int)busType);
            writer.Write(effectIndex);
            writer.Write((int)effectType);
            writer.Write(earlyMode);
            writer.Write(preDelayTimeMax);
            writer.Write(preDelayTime);
            writer.Write((int)fusedMode);
            writer.Write(fusedTime);
            writer.Write(coloration);
            writer.Write(damping);
            writer.Write(earlyGain);
            writer.Write(fusedGain);
            writer.Write(outGain);
            for(int i = 0; i < 4; i++){
                writer.Write(reserved[i]);
            }
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>CHORUS パラメータ</summary> */
    public class ToolSoundChorusParameter {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+4+4+4+4+4+4+4+1*24;
        /** <summary> 4B: バスの種類</summary> */
        public ToolSoundAuxBusType  busType;
        /** <summary> 4B: インデックス</summary> */
        public uint effectIndex;
        /** <summary> 4B: エフェクトの種類</summary> */
        public ToolSoundEffectType  effectType;
        /** <summary> 4B: delay time</summary> */
        public float    delayTime;
        /** <summary> 4B: depth</summary> */
        public float    depth;
        /** <summary> 4B: rate</summary> */
        public float    rate;
        /** <summary> 4B: feedback</summary> */
        public float    feedback;
        /** <summary> 4B: out gain</summary> */
        public float    outGain;
        /** <summary>24B: reserved</summary> */
        public byte[]   reserved = new byte[24];

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            busType = (ToolSoundAuxBusType)reader.ReadInt32();
            effectIndex = reader.ReadUInt32();
            effectType = (ToolSoundEffectType)reader.ReadInt32();
            delayTime = reader.ReadSingle();
            depth = reader.ReadSingle();
            rate = reader.ReadSingle();
            feedback = reader.ReadSingle();
            outGain = reader.ReadSingle();
            for(int i = 0; i < 24; i++){
                reserved[i] = reader.ReadByte();
            }
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write((int)busType);
            writer.Write(effectIndex);
            writer.Write((int)effectType);
            writer.Write(delayTime);
            writer.Write(depth);
            writer.Write(rate);
            writer.Write(feedback);
            writer.Write(outGain);
            for(int i = 0; i < 24; i++){
                writer.Write(reserved[i]);
            }
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>DELAY パラメータ</summary> */
    public class ToolSoundDelayParameter {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+4+4+4+4+4+4+4+1*24;
        /** <summary> 4B: バスの種類</summary> */
        public ToolSoundAuxBusType  busType;
        /** <summary> 4B: インデックス</summary> */
        public uint effectIndex;
        /** <summary> 4B: エフェクトの種類</summary> */
        public ToolSoundEffectType  effectType;
        /** <summary> 4B: max delay time</summary> */
        public float    maxDelayTime;
        /** <summary> 4B: delay time</summary> */
        public float    delayTime;
        /** <summary> 4B: feedback</summary> */
        public float    feedback;
        /** <summary> 4B: out gain</summary> */
        public float    outGain;
        /** <summary> 4B: LPF</summary> */
        public float    lpf;
        /** <summary>24B: reserved</summary> */
        public byte[]   reserved = new byte[24];

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            busType = (ToolSoundAuxBusType)reader.ReadInt32();
            effectIndex = reader.ReadUInt32();
            effectType = (ToolSoundEffectType)reader.ReadInt32();
            maxDelayTime = reader.ReadSingle();
            delayTime = reader.ReadSingle();
            feedback = reader.ReadSingle();
            outGain = reader.ReadSingle();
            lpf = reader.ReadSingle();
            for(int i = 0; i < 24; i++){
                reserved[i] = reader.ReadByte();
            }
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write((int)busType);
            writer.Write(effectIndex);
            writer.Write((int)effectType);
            writer.Write(maxDelayTime);
            writer.Write(delayTime);
            writer.Write(feedback);
            writer.Write(outGain);
            writer.Write(lpf);
            for(int i = 0; i < 24; i++){
                writer.Write(reserved[i]);
            }
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>シーケンス変数の種類</summary> */
    public enum ToolSoundSeqVariableType {
        /** <summary>グローバル変数</summary> */
        SeqVariableType_Global =    1,
        /** <summary>ローカル変数</summary> */
        SeqVariableType_Local = 2,
        /** <summary>トラック変数</summary> */
        SeqVariableType_Track = 3,
    }

    //-----------------------------------------------------------------------
    /** <summary>シーケンス変数コンテナヘッダ</summary> */
    public class ToolSoundSeqVariableContainerHeader {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+4+4;
        /** <summary> 4B: シーケンス変数の種類</summary> */
        public ToolSoundSeqVariableType variableType;
        /** <summary> 4B: プレビュープレイヤーのインデックス</summary> */
        public uint previewPlayerIndex;
        /** <summary> 4B: トラック番号</summary> */
        public uint trackNo;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            variableType = (ToolSoundSeqVariableType)reader.ReadInt32();
            previewPlayerIndex = reader.ReadUInt32();
            trackNo = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write((int)variableType);
            writer.Write(previewPlayerIndex);
            writer.Write(trackNo);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>シーケンス変数コンテナ</summary> */
    public class ToolSoundSeqVariableContainer {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+ToolSoundSeqVariableContainerHeader.StructSize+4*16;
        /** <summary>12B: コンテナヘッダ</summary> */
        public ToolSoundSeqVariableContainerHeader  header = new ToolSoundSeqVariableContainerHeader();
        /** <summary>64B: 変数の値（ShortMax+1 の場合、無効値）</summary> */
        public int[]    variables = new int[16];

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            header.Read(reader);
            for(int i = 0; i < 16; i++){
                variables[i] = reader.ReadInt32();
            }
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            header.Write(writer);
            for(int i = 0; i < 16; i++){
                writer.Write(variables[i]);
            }
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>バスの種類</summary> */
    public enum ToolSoundBiquadType {
        /** <summary>OFF </summary> */
        BiquadType_Off =    0,
        /** <summary>LPF </summary> */
        BiquadType_Lpf =    1,
        /** <summary>HPF </summary> */
        BiquadType_Hpf =    2,
        /** <summary>BPF512 </summary> */
        BiquadType_Bpf512 = 3,
        /** <summary>BPF1024 </summary> */
        BiquadType_Bpf1024 =    4,
        /** <summary>BPF2048 </summary> */
        BiquadType_Bpf2048 =    5,
    }

    //-----------------------------------------------------------------------
    /** <summary>プレビュープレイヤーパラメータ</summary> */
    public class ToolSoundPreviewPlayerParameter {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+4+4+4+4+4+4+4+4+4+4+4+4+4+4+4+4+4+4+4+4+4+4+4+4+4;
        /** <summary> 4B: 対象プレビュープレイヤーのインデックス</summary> */
        public uint previewPlayerIndex;
        /** <summary> 4B: 音量</summary> */
        public float    volume;
        /** <summary> 4B: ピッチ</summary> */
        public float    pitch;
        /** <summary> 4B: パン</summary> */
        public float    pan;
        /** <summary> 4B: サラウンドパン</summary> */
        public float    surroundPan;
        /** <summary> 4B: LPF周波数</summary> */
        public float    lpfFrequency;
        /** <summary> 4B: BiquadType</summary> */
        public ToolSoundBiquadType  biquadType;
        /** <summary> 4B: BiquadValue</summary> */
        public float    biquadValue;
        /** <summary> 4B: 再生開始位置 [ms]</summary> */
        public uint startOffsetTime;
        /** <summary> 4B: メイン 出力スイッチ</summary> */
        public int  mainOutEnabled;
        /** <summary> 4B: Wiiリモコン1 出力スイッチ</summary> */
        public int  remote0OutEnabled;
        /** <summary> 4B: Wiiリモコン2 出力スイッチ</summary> */
        public int  remote1OutEnabled;
        /** <summary> 4B: Wiiリモコン3 出力スイッチ</summary> */
        public int  remote2OutEnabled;
        /** <summary> 4B: Wiiリモコン4 出力スイッチ</summary> */
        public int  remote3OutEnabled;
        /** <summary> 4B: メイン 出力</summary> */
        public float    mainOutVolume;
        /** <summary> 4B: Wiiリモコン1 出力</summary> */
        public float    remote0OutVolume;
        /** <summary> 4B: Wiiリモコン2 出力</summary> */
        public float    remote1OutVolume;
        /** <summary> 4B: Wiiリモコン3 出力</summary> */
        public float    remote2OutVolume;
        /** <summary> 4B: Wiiリモコン4 出力</summary> */
        public float    remote3OutVolume;
        /** <summary> 4B: メイン センド</summary> */
        public float    mainSend;
        /** <summary> 4B: エフェクト Aux A センド</summary> */
        public float    effectAuxASend;
        /** <summary> 4B: エフェクト Aux B センド</summary> */
        public float    effectAuxBSend;
        /** <summary> 4B: エフェクト Aux C センド</summary> */
        public float    effectAuxCSend;
        /** <summary> 4B: フェードインフレーム</summary> */
        public int  fadeInFrames;
        /** <summary> 4B: フェードアウトフレーム</summary> */
        public int  fadeOutFrames;
        /** <summary> 4B: ポーズフェードフレーム</summary> */
        public int  pauseFadeFrames;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            previewPlayerIndex = reader.ReadUInt32();
            volume = reader.ReadSingle();
            pitch = reader.ReadSingle();
            pan = reader.ReadSingle();
            surroundPan = reader.ReadSingle();
            lpfFrequency = reader.ReadSingle();
            biquadType = (ToolSoundBiquadType)reader.ReadInt32();
            biquadValue = reader.ReadSingle();
            startOffsetTime = reader.ReadUInt32();
            mainOutEnabled = reader.ReadInt32();
            remote0OutEnabled = reader.ReadInt32();
            remote1OutEnabled = reader.ReadInt32();
            remote2OutEnabled = reader.ReadInt32();
            remote3OutEnabled = reader.ReadInt32();
            mainOutVolume = reader.ReadSingle();
            remote0OutVolume = reader.ReadSingle();
            remote1OutVolume = reader.ReadSingle();
            remote2OutVolume = reader.ReadSingle();
            remote3OutVolume = reader.ReadSingle();
            mainSend = reader.ReadSingle();
            effectAuxASend = reader.ReadSingle();
            effectAuxBSend = reader.ReadSingle();
            effectAuxCSend = reader.ReadSingle();
            fadeInFrames = reader.ReadInt32();
            fadeOutFrames = reader.ReadInt32();
            pauseFadeFrames = reader.ReadInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(previewPlayerIndex);
            writer.Write(volume);
            writer.Write(pitch);
            writer.Write(pan);
            writer.Write(surroundPan);
            writer.Write(lpfFrequency);
            writer.Write((int)biquadType);
            writer.Write(biquadValue);
            writer.Write(startOffsetTime);
            writer.Write(mainOutEnabled);
            writer.Write(remote0OutEnabled);
            writer.Write(remote1OutEnabled);
            writer.Write(remote2OutEnabled);
            writer.Write(remote3OutEnabled);
            writer.Write(mainOutVolume);
            writer.Write(remote0OutVolume);
            writer.Write(remote1OutVolume);
            writer.Write(remote2OutVolume);
            writer.Write(remote3OutVolume);
            writer.Write(mainSend);
            writer.Write(effectAuxASend);
            writer.Write(effectAuxBSend);
            writer.Write(effectAuxCSend);
            writer.Write(fadeInFrames);
            writer.Write(fadeOutFrames);
            writer.Write(pauseFadeFrames);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>アニメーションコンポーネントの種類</summary> */
    public enum ToolSoundAnimComponentType {
        /** <summary>3Dモデル</summary> */
        AnimComponentType_3DModel = 1,
        /** <summary>アクション</summary> */
        AnimComponentType_Action =  2,
        /** <summary>アニメーションサウンド</summary> */
        AnimComponentType_AnimSound =   3,
    }

    //-----------------------------------------------------------------------
    /** <summary>アニメーションコンポーネント識別子</summary> */
    public class ToolSoundAnimComponentID {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+4+4+4;
        /** <summary> 4B: コンポーネントの種類</summary> */
        public ToolSoundAnimComponentType   type;
        /** <summary> 4B: モデルID</summary> */
        public uint modelID;
        /** <summary> 4B: アクションID（指定しない場合は -1）</summary> */
        public uint actionID;
        /** <summary> 4B: アニメーションサウンドID（指定しない場合は -1）</summary> */
        public uint animSoundID;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            type = (ToolSoundAnimComponentType)reader.ReadInt32();
            modelID = reader.ReadUInt32();
            actionID = reader.ReadUInt32();
            animSoundID = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write((int)type);
            writer.Write(modelID);
            writer.Write(actionID);
            writer.Write(animSoundID);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>アニメーションコンポーネント追加情報</summary> */
    public class ToolSoundAnimComponentAddInfo {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+ToolSoundAnimComponentID.StructSize+1*260+4;
        /** <summary> 16B: 対象ID</summary> */
        public ToolSoundAnimComponentID id = new ToolSoundAnimComponentID();
        /** <summary>260B: ファイル名</summary> */
        public byte[]   fileName = new byte[260];
        /** <summary>  4B: キャッシュを使用しない場合は 1, キャッシュを使用する場合は 0</summary> */
        public int  reload;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            id.Read(reader);
            for(int i = 0; i < 260; i++){
                fileName[i] = reader.ReadByte();
            }
            reload = reader.ReadInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            id.Write(writer);
            for(int i = 0; i < 260; i++){
                writer.Write(fileName[i]);
            }
            writer.Write(reload);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            アニメーションサウンド読み込みメッセージ
        </summary> */
    public class ToolSoundLoadAsd {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+1*260;
        /** <summary>260B: PC上のデータのパス名</summary> */
        public byte[]   pathName = new byte[260];

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            for(int i = 0; i < 260; i++){
                pathName[i] = reader.ReadByte();
            }
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            for(int i = 0; i < 260; i++){
                writer.Write(pathName[i]);
            }
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            アニメーションサウンド読み込み応答メッセージ
        </summary> */
    public class ToolSoundLoadAsdReply {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>4B: 未使用</summary> */
        public uint id;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            id = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(id);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            アニメーションフレーム設定メッセージ
        </summary> */
    public class ToolSoundAnimFrame {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+4+4;
        /** <summary>4B: アニメーションのフレーム</summary> */
        public float    frame;
        /** <summary>4B: イベントを処理するかどうか</summary> */
        public int  doEvent;
        /** <summary>4B: 同期をとるかどうか</summary> */
        public int  sync;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            frame = reader.ReadSingle();
            doEvent = reader.ReadInt32();
            sync = reader.ReadInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(frame);
            writer.Write(doEvent);
            writer.Write(sync);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            アニメーション再生レート設定メッセージ
        </summary> */
    public class ToolSoundAnimRate {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>4B: アニメーションの再生レート</summary> */
        public float    rate;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            rate = reader.ReadSingle();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(rate);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            アニメーションフレーム取得応答メッセージ
        </summary> */
    public class ToolSoundQueryAnimFrameReply {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>4B: アニメーションのフレーム</summary> */
        public float    frame;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            frame = reader.ReadSingle();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(frame);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            ループ状態設定メッセージ
        </summary> */
    public class ToolSoundSetLoop {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>4B: ループ状態</summary> */
        public int  loopFlag;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            loopFlag = reader.ReadInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(loopFlag);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            アニメーション再生範囲設定メッセージ
        </summary> */
    public class ToolSoundAnimRange {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+4;
        /** <summary>4B: ループ開始位置</summary> */
        public float    start;
        /** <summary>4B: ループ終了位置</summary> */
        public float    end;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            start = reader.ReadSingle();
            end = reader.ReadSingle();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(start);
            writer.Write(end);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            アニメーション再生範囲設定応答メッセージ
        </summary> */
    public class ToolSoundAnimRangeReply {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>4B: 設定変更後の現在のフレーム</summary> */
        public float    frame;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            frame = reader.ReadSingle();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(frame);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            アニメーションサウンドのイベント更新コマンド
        </summary> */
    public class ToolSoundModifierEvent {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4+4+4+1*260+1+1+1+1+4+4+4+1+1+1+1;
        /** <summary>4B: 更新なら0、削除なら1</summary> */
        public int  isDelete;
        /** <summary>4B: インデックス</summary> */
        public int  index;
        /** <summary>4B: サウンドID</summary> */
        public uint soundId;
        /** <summary>260B: 名前</summary> */
        public byte[]   soundName = new byte[260];
        /** <summary>1B: ボリューム</summary> */
        public byte volume;
        /** <summary>1B: パディング</summary> */
        public byte padding1;
        /** <summary>1B: パディング</summary> */
        public byte padding2;
        /** <summary>1B: パディング</summary> */
        public byte padding3;
        /** <summary>4B: ピッチ</summary> */
        public float    pitch;
        /** <summary>4B: 開始フレーム</summary> */
        public int  startFrame;
        /** <summary>4B: 終了フレーム</summary> */
        public int  endFrame;
        /** <summary>1B: フレーム処理のオプションフラグ</summary> */
        public byte frameFlag;
        /** <summary>1B: 再生するループ数</summary> */
        public sbyte    loopOffset;
        /** <summary>1B: loopCount以降で再生するループ間隔</summary> */
        public byte loopInterval;
        /** <summary>1B: パディング</summary> */
        public byte padding4;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            isDelete = reader.ReadInt32();
            index = reader.ReadInt32();
            soundId = reader.ReadUInt32();
            for(int i = 0; i < 260; i++){
                soundName[i] = reader.ReadByte();
            }
            volume = reader.ReadByte();
            padding1 = reader.ReadByte();
            padding2 = reader.ReadByte();
            padding3 = reader.ReadByte();
            pitch = reader.ReadSingle();
            startFrame = reader.ReadInt32();
            endFrame = reader.ReadInt32();
            frameFlag = reader.ReadByte();
            loopOffset = reader.ReadSByte();
            loopInterval = reader.ReadByte();
            padding4 = reader.ReadByte();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(isDelete);
            writer.Write(index);
            writer.Write(soundId);
            for(int i = 0; i < 260; i++){
                writer.Write(soundName[i]);
            }
            writer.Write(volume);
            writer.Write(padding1);
            writer.Write(padding2);
            writer.Write(padding3);
            writer.Write(pitch);
            writer.Write(startFrame);
            writer.Write(endFrame);
            writer.Write(frameFlag);
            writer.Write(loopOffset);
            writer.Write(loopInterval);
            writer.Write(padding4);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
         EFBカラー変更メッセージ
        </summary> */
    public class ToolTestEfbColor {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+1+1+1+1;
        /** <summary>1B: R成分</summary> */
        public byte r;
        /** <summary>1B: G成分</summary> */
        public byte g;
        /** <summary>1B: B成分</summary> */
        public byte b;
        /** <summary>1B: A成分</summary> */
        public byte a;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            r = reader.ReadByte();
            g = reader.ReadByte();
            b = reader.ReadByte();
            a = reader.ReadByte();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(r);
            writer.Write(g);
            writer.Write(b);
            writer.Write(a);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>
            未定義コマンドエラーメッセージ

            受け取った未定義コマンドをcommandパラメータで返します。
        </summary> */
    public class ErrSystemUnknownCommand {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+2;
        /** <summary>  2B: 受け取った未定義コマンド  </summary> */
        public ushort   command;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            command = reader.ReadUInt16();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(command);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>データの読み込みエラー種別</summary> */
    public enum Err3DLoadCode {
        /** <summary>ファイルのロードに失敗しました。        </summary> */
        ERR_3D_LOAD_FILE_NOT_FOUND =    0,
        /** <summary>メモリのアロケーションに失敗しました。  </summary> */
        ERR_3D_LOAD_MEMORY_ALLOCATE =   1,
        /** <summary>有効なデータではありませんでした。      </summary> */
        ERR_3D_LOAD_INVALID_DATA =  2,
        /** <summary>MCS通信で異常が発生しました。           </summary> */
        ERR_3D_LOAD_MCS_STREAM_FAIL =   3,
    }

    //-----------------------------------------------------------------------
    /** <summary>
            データの読み込みエラー
        </summary> */
    public class Err3DLoad {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary>  4B: エラーコード          </summary> */
        public Err3DLoadCode    code;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            code = (Err3DLoadCode)reader.ReadInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write((int)code);
        }

    }

    //-----------------------------------------------------------------------
    /** <summary>サウンドエラー</summary> */
    public class SoundError {
        /// <summary>構造体にした場合のサイズ</summary>
        public const int StructSize =
            0+4;
        /** <summary> 4B: エラーコード</summary> */
        public uint errorCode;

        /// <summary>読み込みメソッド</summary>
        public virtual void Read(ProtocolSoundReader reader){
            errorCode = reader.ReadUInt32();
        }

        /// <summary>書き出しメソッド</summary>
        public virtual void Write(ProtocolSoundWriter writer){
            writer.Write(errorCode);
        }

    }

    //-----------------------------------------------------------------------
    /// <summary>
    /// HioProtocolSoundReader
    /// </summary>
    public class HioProtocolSoundReader : ProtocolSoundReader
    {

        /// <summary>コンストラクタ</summary>
        public HioProtocolSoundReader(Stream input) : base(input){}
        /// <summary>コンストラクタ</summary>
        public HioProtocolSoundReader(Stream input, Encoding encoding) :
            base(input, encoding){}
    }

    //-----------------------------------------------------------------------
    /// <summary>
    /// HioProtocolSoundWriter
    /// </summary>
    public class HioProtocolSoundWriter : ProtocolSoundWriter
    {

        /// <summary>コンストラクタ</summary>
        public HioProtocolSoundWriter() : base(){}
        /// <summary>コンストラクタ</summary>
        public HioProtocolSoundWriter(Stream input) : base(input){}
        /// <summary>コンストラクタ</summary>
        public HioProtocolSoundWriter(Stream input, Encoding encoding) :
            base(input, encoding){}
    }

}
