﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Reflection;
using System.Xml;
using System.Xml.Schema;
using NintendoWare.SoundFoundation;
using NintendoWare.SoundMaker.Configurations.Schemas;
using NintendoWare.SoundMaker.Framework;
using NintendoWare.SoundMaker.Framework.Configurations.Schemas;
using NintendoWare.SoundMaker.Framework.Windows.Forms;
using NintendoWare.SoundMaker.Resources;
using NintendoWare.SoundMakerPlugin;

namespace NintendoWare.SoundMaker.Configurations
{
    /// <summary>
    /// アプリケーション設定を格納します。
    /// </summary>
    public class AppConfigurationCommon : AppConfiguration
    {
        private static Lazy<bool> EnabledWaveSoundEditCache = new Lazy<bool>(() => AppConfigurationCommon.IsWaveSoundEditEnabled());

        /// <summary>
        /// ウェーブサウンド編集が利用可能か調べます。
        /// </summary>
        public static bool EnabledWaveSoundEdit
        {
            get { return EnabledWaveSoundEditCache.Value; }
        }

        private const string Platform = Platforms.Any.PlatformName;
        private const string Version = "1.3.0.0";

        private XmlOptionsCommon _xmlOptionsCommon;

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public AppConfigurationCommon()
            : base(Platform, Version)
        {
            AppConfiguration.AACFileExtension = ".aac";
            AppConfiguration.WaveFileMultiChannelCountMax = 2;
        }

        /// <summary>
        /// Common 向けオプションを取得します。
        /// </summary>
        public XmlOptionsCommon OptionsCommon
        {
            get { return _xmlOptionsCommon; }
        }

        public override string StreamWaveFileFilter
        {
            get
            {
                return string.Format("Wave Files (*.wav;*.aif;*.aiff;*{0})|*.wav;*.aif;*.aiff;*{0}|All Files (*.*)|*.*",
                                     SoundMakerPluginManager.Instance.CurrentSoundMakerPlugin.Platform == "Cafe" ? AppConfiguration.AACFileExtension : ".opus");
            }
        }

        protected override string RootElementName
        {
            get { return "SoundMakerConfiguration"; }
        }

        /// <summary>
        /// デフォルト設定を読み込みます。
        /// </summary>
        /// <param name="settings">XmlReader 設定。</param>
        /// <param name="forced">現在の設定を全て破棄する場合は true、読み込み済みの設定を残す場合は false。</param>
        protected override void LoadDefaultsInternal(XmlReaderSettings settings, bool forced)
        {
            base.LoadDefaultsInternal(settings, forced);

            if (forced || null == _xmlOptionsCommon)
            {
                _xmlOptionsCommon = ReadConfigurationPart<XmlOptionsCommon>(
                                            FileResourcesProxy.OptionConfigurationDefaultCommon, settings);

                ValidateOptions();

            }
        }

        /// <summary>
        /// スキーマセットを作成します。
        /// </summary>
        /// <returns>スキーマセット。</returns>
        protected override XmlSchemaSet CreateSchemaSet()
        {
            XmlSchemaSet schemas = base.CreateSchemaSet();
            schemas.Add(XmlSchema.Read(new StringReader(FileResourcesProxy.SchemaOptionConfigurationCommon), null));

            return schemas;
        }

        /// <summary>
        /// ウィンドウのデフォルト設定を読み込みます。
        /// </summary>
        /// <param name="settings">XmlReader 設定。</param>
        /// <returns>読み込んだウィンドウのディクショナリ。</returns>
        protected override IDictionary<string, XmlWindow> LoadWindowsDefault(XmlReaderSettings settings)
        {
            IDictionary<string, XmlWindow> windows = base.LoadWindowsDefault(settings);

            XmlWindows windowsDefault = ReadConfigurationPart<XmlWindows>(
                                            FileResourcesProxy.AppConfigurationWindowsDefaultCommon, settings);

            foreach (XmlToolWindow window in windowsDefault.Items)
            {
                windows.Add(window.Name, window);
            }

            return windows;
        }

        /// <summary>
        /// ツールページのデフォルト設定を読み込みます。
        /// </summary>
        /// <param name="settings">XmlReader 設定。</param>
        /// <returns>読み込んだツールページのディクショナリ。</returns>
        protected override IDictionary<string, XmlToolPage> LoadToolPagesDefault(XmlReaderSettings settings)
        {
            IDictionary<string, XmlToolPage> toolPages = base.LoadToolPagesDefault(settings);

            XmlToolPages toolPagesDefault = ReadConfigurationPart<XmlToolPages>(
                                            FileResourcesProxy.AppConfigurationToolPagesDefaultCommon, settings);

            foreach (XmlToolPage page in toolPagesDefault.ToolPage)
            {
                toolPages.Add(page.Name, page);
            }

            return toolPages;
        }

        /// <summary>
        /// コンフィギュレーション要素を読み込みます。
        /// </summary>
        /// <param name="xmlElement">コンフィギュレーション要素。</param>
        /// <param name="basePath">基準パス。</param>
        protected override void ReadConfigurationElement(XmlElement xmlElement, string basePath)
        {
            if (null == xmlElement) { throw new ArgumentNullException("xmlElement"); }

            switch (xmlElement.LocalName)
            {
                case "OptionsCommon":
                    _xmlOptionsCommon =
                        ReadConfigurationPart<XmlOptionsCommon>(xmlElement.OuterXml);
                    ValidateOptions();
                    break;

                default:
                    base.ReadConfigurationElement(xmlElement, basePath);
                    break;
            }
        }

        /// <summary>
        /// コンフィギュレーション要素を書き出します。
        /// </summary>
        /// <param name="reader">XmlReader。</param>
        /// <param name="basePath">基準パス。</param>
        protected override void WriteConfigurationElements(XmlWriter writer, string basePath)
        {
            base.WriteConfigurationElements(writer, basePath);

            WriteConfigurationPart<XmlOptionsCommon>(writer, _xmlOptionsCommon);
        }

        /// <summary>
        /// コンフィギュレーションルートを書き出します。
        /// </summary>
        /// <param name="writer">XmlWriter。</param>
        protected override void WriteStartConfigurationRoot(XmlWriter writer)
        {
            base.WriteStartConfigurationRoot(writer);

            writer.WriteAttributeString("xmlns", "nwrsa", null,
                                         "NintendoWare.SoundMaker.AppConfiguration");

            writer.WriteAttributeString("xmlns", "nwrso", null,
                                         "NintendoWare.SoundMaker.OptionConfiguration");
        }

        /// <summary>
        /// 読み込んだコンフィギュレーションを検証します。
        /// </summary>
        /// <returns>読み込んだ結果が正常な場合は true、それ以外の場合は false。</returns>
        protected override bool Validate()
        {
            if (!base.Validate()) { return false; }

            if (!Windows.ContainsKey("PreviewPlayerToolWindow") ||
                 !Windows.ContainsKey("PreviewPlayerParameterToolWindow") ||
                 !Windows.ContainsKey("SequenceVariableToolWindow") ||
                 !Windows.ContainsKey("EffectToolWindow")) { return false; }

            if (null == _xmlOptionsCommon.Application) { return false; }
            if (null == _xmlOptionsCommon.Application.SoundPlayer) { return false; }

            return true;
        }

        /// <summary>
        /// AAC の有効、無効を設定します。
        /// </summary>
        protected override void InitializeEnabledAAC()
        {
#if true
            this.EnabledAACInternal = false;
            this.EnabledMultiChannelAACInternal = false;
            // 注意：コンバーターは AAC_SUPPORTED の定義の有無で判断している。
#else
            string location = FormsApplication.Instance.Traits.ApplicationAssembly.Location;
            string path = Path.GetDirectoryName(location);
            string fileName = Path.GetFileNameWithoutExtension(location);
            path = Path.Combine(path, fileName + "_enable_aac");
            this.EnabledAACInternal = File.Exists(path);
#endif
        }

        /// <summary>
        /// プリフェッチストリームの有効、無効を設定します。
        /// </summary>
        protected override void InitializeEnabledPrefetch()
        {
#if true
            this.EnabledPrefetchInternal = true;
#else
            string location = FormsApplication.Instance.Traits.ApplicationAssembly.Location;
            string path = Path.GetDirectoryName(location);
            string fileName = Path.GetFileNameWithoutExtension(location);
            path = Path.Combine(path, fileName + "_enable_prefetch");
            this.EnabledPrefetchInternal = File.Exists(path);
#endif
        }

        /// <summary>
        /// サンプリングレート48kHz波形出力の有効、無効を設定します。
        /// </summary>
        protected override void InitializeEnabled48kHz()
        {
#if false
            this.Enabled48kHzInternal = true;
#else
            string location = FormsApplication.Instance.Traits.ApplicationAssembly.Location;
            string path = Path.GetDirectoryName(location);
            string fileName = Path.GetFileNameWithoutExtension(location);
            path = Path.Combine(path, fileName + "_enable_48kHz");
            this.Enabled48kHzInternal = File.Exists(path);
#endif
        }

        private string GetPrivateFunctionFilePath(string privateFunctionName)
        {
            string location = FormsApplication.Instance.Traits.ApplicationAssembly.Location;
            string directory = Path.GetDirectoryName(location);
            string fileName = Path.GetFileNameWithoutExtension(location);
            return Path.Combine(directory, fileName + privateFunctionName);
        }

        /// <summary>
        /// ウェーブサウンド編集の有効、無効を取得します。
        /// </summary>
        private static bool IsWaveSoundEditEnabled()
        {
            string location = Assembly.GetEntryAssembly().Location;
            string path = Path.GetDirectoryName(location);
            string fileName = Path.GetFileNameWithoutExtension(location);
            return File.Exists(Path.Combine(path, fileName + "_enable_wsdedit"));
        }

        /// <summary>
        /// 非公開機能の有効、無効を取得します。
        /// </summary>
        public static bool IsHiddenFunctionEnabled()
        {
            string location = Assembly.GetEntryAssembly().Location;
            string path = Path.GetDirectoryName(location);
            string fileName = Path.GetFileNameWithoutExtension(location);
            return File.Exists(Path.Combine(path, fileName + "_enable_hiddenfunc"));
        }

        /// <summary>
        /// オプションを検証します。
        /// </summary>
        private void ValidateOptions()
        {
            // 互換性を保つ
            if (_xmlOptionsCommon.Application.InGame != null)
            {
                ApplicationBase.Instance.AppConfiguration.Options.Application.InGame.AutoEnableMonitoringTarget = _xmlOptionsCommon.Application.InGame.AutoEnableMonitoringTarget;
                _xmlOptionsCommon.Application.InGame = null;
            }
        }
    }
}
