﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using NintendoWare.SoundFoundation.Conversion.NintendoWareReport;
    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.ToolDevelopmentKit;

    public class ComponentConverter
    {
        public static void Converter(Component dst, Component src)
        {
            Converter(dst, src, null);
        }

        public static void Converter(Component dst, Component src, List<string> filters)
        {
            foreach (string key in dst.Parameters.Keys)
            {
                if ((filters != null && filters.Contains(key) == true) ||
                    src.Parameters.ContainsKey(key) == false)
                {
                    continue;
                }

                dst.Parameters[key].Value = src.Parameters[key].Value;
            }
        }
    }


    /// <summary>
    /// CTR サウンドセットドキュメントをサウンドセットドキュメントに変換します。
    /// </summary>
    public class SoundSetCtrToSoundSetDocumentConverter : IDocumentConverter
    {
        private readonly SoundIntermediateOutputTraits intermediateOutputTraits;
        private IObjectFactory<Type, Component> componentFactory;

        public SoundSetCtrToSoundSetDocumentConverter(
            SoundIntermediateOutputTraits intermediateOutputTraits,
            IObjectFactory<Type, Component> componentFactory)
        {
            Ensure.Argument.NotNull(intermediateOutputTraits);
            Ensure.Argument.NotNull(componentFactory);

            this.intermediateOutputTraits = intermediateOutputTraits;
            this.componentFactory = componentFactory;
        }

        public string InputDocumentTypeName
        {
            get { return Platforms.Ctr.SoundSetDocument; }
        }

        public string OutputDocumentTypeName
        {
            get { return this.intermediateOutputTraits.SoundSetDocumentTypeName; }
        }

        public void Convert(Document inputDocument, Document outputDocument)
        {
            Ensure.Argument.NotNull(inputDocument);
            Ensure.Argument.NotNull(outputDocument);
            this.ConvertInternal(
                inputDocument as SoundSetDocument,
                outputDocument as SoundSetDocument);
        }

        private void ConvertInternal(SoundSetDocument inputDocument, SoundSetDocument outputDocument)
        {
            Ensure.Argument.NotNull(inputDocument);
            Ensure.Argument.NotNull(outputDocument);

            outputDocument.SoundSet = this.ConvertSoundSet(inputDocument.SoundSet);

            if (inputDocument.Resource is FileResource)
            {
                outputDocument.Resource = new FileResource
                    (Path.ChangeExtension(inputDocument.Resource.Key,
                                            this.intermediateOutputTraits.SoundSetFileExtension));
            }
        }

        ///
        private SoundSet ConvertSoundSet(SoundSet source)
        {
            Assertion.Argument.NotNull(source);

            SoundSet dest = this.componentFactory.Create(typeof(SoundSet)) as SoundSet;
            Ensure.Operation.ObjectNotNull(dest);

            dest.Children.Add(this.ConvertStreamSoundFolder
                               (GetSoundSetItems<StreamSoundPack, StreamSoundBase>(source)));
            dest.Children.Add(this.ConvertWaveSoundSetFolder
                               (GetSoundSetItems<WaveSoundSetPack, WaveSoundSetBase>(source)));
            dest.Children.Add(this.ConvertSequenceSoundFolder
                               (GetSoundSetItems<SequenceSoundPack, SequenceSoundBase>(source)));
            dest.Children.Add(this.ConvertSequenceSoundSetFolder
                               (GetSoundSetItems<SequenceSoundSetPack, SequenceSoundSetBase>(source)));
            dest.Children.Add(this.ConvertSoundSetBankFolder
                               (GetSoundSetItems<SoundSetBankPack, SoundSetBankBase>(source)));
            dest.Children.Add(this.ConvertWaveArchiveFolder
                               (GetSoundSetItems<WaveArchivePack, WaveArchiveBase>(source)));
            dest.Children.Add(this.ConvertGroupFolder
                               (GetSoundSetItems<GroupPack, GroupBase>(source)));
            dest.Children.Add(this.ConvertPlayerFolder
                               (GetSoundSetItems<PlayerPack, PlayerBase>(source)));

            return dest;
        }

        ///
        private T[] GetSoundSetItems<P, T>(SoundSet soundSet) where P : SoundSetItem where T : SoundSetItem
        {
            foreach (SoundSetItem itemPack in soundSet.Children)
            {
                if (itemPack is P)
                {
                    List<T> items = new List<T>();
                    foreach (T item in itemPack.Children)
                    {
                        items.Add(item);
                    }
                    return items.ToArray();
                }
            }
            return new T[0];
        }

        //-----------------------------------------------------------------
        // サウンドセットアイテムフォルダのコンバート
        //-----------------------------------------------------------------

        ///
        private StreamSoundPack ConvertStreamSoundFolder(StreamSoundBase[] source)
        {
            Assertion.Argument.NotNull(source);
            StreamSoundPack dest = new StreamSoundPack();

            foreach (StreamSoundBase streamSound in source)
            {
                dest.Children.Add(this.ConvertStreamSound(streamSound));
            }
            return dest;
        }

        ///
        private WaveSoundSetPack ConvertWaveSoundSetFolder(WaveSoundSetBase[] source)
        {
            Assertion.Argument.NotNull(source);
            WaveSoundSetPack dest = new WaveSoundSetPack();

            foreach (WaveSoundSetBase sourceWaveSoundSet in source)
            {
                dest.Children.Add(this.ConvertWaveSoundSet(sourceWaveSoundSet));
            }
            return dest;
        }

        ///
        private SequenceSoundSetPack ConvertSequenceSoundSetFolder(SequenceSoundSetBase[] source)
        {
            Assertion.Argument.NotNull(source);
            SequenceSoundSetPack dest = new SequenceSoundSetPack();

            foreach (SequenceSoundSetBase sourceSequenceSoundSet in source)
            {
                dest.Children.Add(this.ConvertSequenceSoundSet(sourceSequenceSoundSet));
            }
            return dest;
        }

        ///
        private SequenceSoundPack ConvertSequenceSoundFolder(SequenceSoundBase[] source)
        {
            Assertion.Argument.NotNull(source);
            SequenceSoundPack dest = new SequenceSoundPack();

            foreach (SequenceSoundBase sourceSequenceSound in source)
            {
                dest.Children.Add(this.ConvertSequenceSound(sourceSequenceSound));
            }
            return dest;
        }

        ///
        private WaveArchivePack ConvertWaveArchiveFolder(WaveArchiveBase[] source)
        {
            Assertion.Argument.NotNull(source);
            WaveArchivePack dest = new WaveArchivePack();

            foreach (WaveArchiveBase sourceWaveArchive in source)
            {
                dest.Children.Add(this.ConvertWaveArchive(sourceWaveArchive));
            }
            return dest;
        }

        ///
        private SoundSetBankPack ConvertSoundSetBankFolder(SoundSetBankBase[] source)
        {
            Assertion.Argument.NotNull(source);
            SoundSetBankPack dest = new SoundSetBankPack();

            foreach (SoundSetBankBase sourceSoundSetBank in source)
            {
                dest.Children.Add(this.ConvertSoundSetBank(sourceSoundSetBank));
            }
            return dest;
        }

        ///
        private PlayerPack ConvertPlayerFolder(PlayerBase[] source)
        {
            Assertion.Argument.NotNull(source);
            PlayerPack dest = new PlayerPack();

            foreach (PlayerBase sourcePlayer in source)
            {
                dest.Children.Add(this.ConvertPlayer(sourcePlayer));
            }

            return dest;
        }

        ///
        private GroupPack ConvertGroupFolder(GroupBase[] source)
        {
            Assertion.Argument.NotNull(source);
            GroupPack dest = new GroupPack();

            foreach (GroupBase sourceGroup in source)
            {
                dest.Children.Add(this.ConvertGroup(sourceGroup));
            }
            return dest;
        }

        //-----------------------------------------------------------------
        // サウンドセットアイテムのコンバート
        //-----------------------------------------------------------------

        /// <summary>
        /// ストリームサウンドの変換
        /// </summary>
        private StreamSoundBase ConvertStreamSound(StreamSoundBase source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Stream Sound: ({0})", source.Name);

            List<StreamSoundTrackBase> list = new List<StreamSoundTrackBase>();
            foreach (StreamSoundTrackBase track in source.Children)
            {
                list.Add(track);
            }

            StreamSoundTrackBase[] tracks = list.ToArray();
            if (tracks.Length <= 0)
            {
                throw new Exception("invalid stream sound found.");
            }

            StreamSoundBase dest = this.componentFactory.Create(typeof(StreamSoundBase)) as StreamSoundBase;
            {
                ComponentConverter.Converter(dest, source, new List<string> { ProjectParameterNames.Sound.Sends });

                dest.Sends = new SendsCommon()
                {
                    AuxASend = source.Sends.AuxASend,
                    AuxBSend = source.Sends.AuxBSend,
                    AuxCSend = 0,
                    MainSend = source.Sends.MainSend,
                };
            }

            foreach (StreamSoundTrackBase track in tracks)
            {
                dest.Children.Add(this.ConvertStreamSoundTrack(track));
            }

            return dest;
        }

        /// <summary>
        /// ストリームサウンドトラックの変換
        /// </summary>
        private StreamSoundTrackBase ConvertStreamSoundTrack(StreamSoundTrackBase source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Stream Sound Track: ({0})", source.FilePath);
            StreamSoundTrackBase dest = this.componentFactory.Create(typeof(StreamSoundTrackBase)) as StreamSoundTrackBase;
            {
                ComponentConverter.Converter(dest, source, new List<string> { ProjectParameterNames.Sound.Sends });

                dest.Sends = new SendsCommon()
                {
                    AuxASend = source.Sends.AuxASend,
                    AuxBSend = source.Sends.AuxBSend,
                    AuxCSend = 0,
                    MainSend = source.Sends.MainSend,
                };
            }

            return dest;
        }

        /// <summary>
        /// ウェーブサウンドセットの変換
        /// </summary>
        private WaveSoundSetBase ConvertWaveSoundSet(WaveSoundSetBase source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Wave Sound Set: ({0})", source.Name);
            WaveSoundSetBase dest = new WaveSoundSetBase();

            ComponentConverter.Converter(dest, source);

            foreach (WaveSoundBase sourceWaveSound in source.Children)
            {
                dest.Children.Add(this.ConvertWaveSound(sourceWaveSound));
            }

            return dest;
        }

        /// <summary>
        /// ウェーブサウンドの変換
        /// </summary>
        private WaveSoundBase ConvertWaveSound(WaveSoundBase source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Wave Sound: ({0})", source.Name);
            WaveSoundBase dest = componentFactory.Create(typeof(WaveSoundBase)) as WaveSoundBase;
            ComponentConverter.Converter(dest, source, new List<string> { ProjectParameterNames.Sound.Sends });
            dest.Sends = new SendsCommon()
            {
                AuxASend = source.Sends.AuxASend,
                AuxBSend = source.Sends.AuxBSend,
                AuxCSend = 0,
                MainSend = source.Sends.MainSend,
            };

            return dest;
        }

        /// <summary>
        /// シーケンスサウンドセットの変換
        /// </summary>
        private SequenceSoundSetBase ConvertSequenceSoundSet(SequenceSoundSetBase source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Sequence Sound Set: ({0})", source.Name);
            SequenceSoundSetBase dest = new SequenceSoundSetBase();
            ComponentConverter.Converter(dest, source);

            foreach (SequenceSoundBase sourceSequenceSound in source.Children)
            {
                dest.Children.Add(this.ConvertSequenceSound(sourceSequenceSound));
            }

            return dest;
        }

        /// <summary>
        /// シーケンスサウンドの変換
        /// </summary>
        private SequenceSoundBase ConvertSequenceSound(SequenceSoundBase source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Sequence Sound: ({0})", source.Name);
            SequenceSoundBase dest = componentFactory.Create(typeof(SequenceSoundBase)) as SequenceSoundBase;
            List<string> filters =
                new List<string>{ ProjectParameterNames.Sound.Sends,
                                  ProjectParameterNames.SequenceSound.SoundSetBankReferences};
            ComponentConverter.Converter(dest, source, filters);

            if (dest.FileType == SequenceSoundFileType.Text)
            {
                dest.FilePath = Path.ChangeExtension(
                    dest.FilePath,
                    this.intermediateOutputTraits.TextSequenceSoundFileExtension);
            }

            //
            foreach (ComponentReference componentRef in source.SoundSetBankReferences)
            {
                dest.SoundSetBankReferences.Add
                    (new ComponentReference()
                    {
                        TargetName = componentRef.TargetName,
                    });
            }

            int addCount = SequenceSoundBase.BankReferenceCount - dest.SoundSetBankReferences.Count;
            for (int count = 0; count < addCount; count++)
            {
                dest.SoundSetBankReferences.Add
                    (new ComponentReference() { TargetName = String.Empty });
            }

            return dest;
        }

        /// <summary>
        /// 波形アーカイブの変換
        /// </summary>
        private WaveArchiveBase ConvertWaveArchive(WaveArchiveBase source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  WaveArchive: ({0})", source.Name);
            WaveArchiveBase dest = new WaveArchiveBase();
            ComponentConverter.Converter(dest, source);

            return dest;
        }

        /// <summary>
        /// バンクの変換
        /// </summary>
        private SoundSetBankBase ConvertSoundSetBank(SoundSetBankBase source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Bank: ({0})", source.Name);
            SoundSetBankBase dest = componentFactory.Create(typeof(SoundSetBankBase)) as SoundSetBankBase;

            ComponentConverter.Converter(dest, source, new List<string> { ProjectParameterNames.FilePath });

            dest.FilePath =
                Path.ChangeExtension(source.FilePath,
                                     this.intermediateOutputTraits.BankFileExtension);

            return dest;
        }

        /// <summary>
        /// プレイヤーの変換
        /// </summary>
        private PlayerBase ConvertPlayer(PlayerBase source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Player: ({0})", source.Name);
            PlayerBase dest = new PlayerBase();

            ComponentConverter.Converter(dest, source);

            return dest;
        }

        /// <summary>
        /// グループの変換
        /// </summary>
        private GroupBase ConvertGroup(GroupBase source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Group: ({0})", source.Name);
            GroupBase dest = new GroupBase();

            ComponentConverter.Converter(dest, source);

            foreach (GroupItemBase groupItem in source.Children)
            {
                dest.Children.Add(this.ConvertGroupItem(groupItem));
            }

            return dest;
        }

        /// <summary>
        /// グループアイテムの変換
        /// </summary>
        private GroupItemBase ConvertGroupItem(GroupItemBase source)
        {
            Assertion.Argument.NotNull(source);

            ImportLog.WriteLine("  Group Item: ({0})", source.Name);
            GroupItemBase dest = new GroupItemBase();

            ComponentConverter.Converter(dest, source);

            return dest;
        }
    }
}
