﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Projects
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Text.RegularExpressions;
    using System.Windows.Forms;

    using NintendoWare.SoundFoundation.Core;
    using NintendoWare.SoundFoundation.Core.Resources;
    using NintendoWare.SoundFoundation.Documents;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareIntermediate;
    using NintendoWare.ToolDevelopmentKit;
    using MsgRsrc = NintendoWare.SoundFoundation.Resources;

    /// <summary>
    /// Nitro バンクをインポートします。
    /// </summary>

    public class BankDocumentImporterCtr : IDocumentImporter
    {
        private readonly DocumentService documentService = null;
        private readonly SoundIntermediateOutputTraits intermediateOutputTraits = null;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="documentService">ドキュメントサービスを指定します。</param>
        /// <param name="intermediateOutputTraits">中間出力特性を指定します。</param>
        public BankDocumentImporterCtr(
                                        DocumentService documentService,
                                        SoundIntermediateOutputTraits intermediateOutputTraits)
        {
            Ensure.Argument.NotNull(documentService);
            Ensure.Argument.NotNull(intermediateOutputTraits);

            this.documentService = documentService;
            this.intermediateOutputTraits = intermediateOutputTraits;
        }

        /// <summary>
        /// インポート元のドキュメントタイプ名を取得します。
        /// </summary>
        public string SourceDocumentTypeName
        {
            get { return Platforms.Ctr.BankDocument; }
        }

        /// <summary>
        /// インポート後のドキュメントタイプ名を取得します。
        /// </summary>
        public string DestinationDocumentTypeName
        {
            get { return this.intermediateOutputTraits.BankDocumentTypeName; }
        }

        /// <summary>
        /// ドキュメントをインポートします。
        /// </summary>
        /// <param name="document">インポートするドキュメントを指定します。</param>
        /// <returns>インポートしたドキュメント参照の配列を返します。</returns>
        public DocumentReference[] Import(Document document)
        {
            Assertion.Argument.NotNull(document);
            Ensure.Argument.True(document is BankDocument);
            Ensure.Argument.True(document.Resource is FileResource);

            List<DocumentReference> results = new List<DocumentReference>();

            try
            {
                DocumentReference importedDocumentRef =
                    this.ImportBankDocument(document as BankDocument);
                results.Add(importedDocumentRef);
            }
            catch
            {
                foreach (DocumentReference bankDocumentRef in results)
                {
                    bankDocumentRef.Close();
                }
                throw;
            }

            return results.ToArray();
        }

        /// <summary>
        /// バンクドキュメントをインポートします。
        /// </summary>
        /// <param name="filePaths">バンクファイルパス。</param>
        /// <returns>インポートしたバンクドキュメントへの参照を返します。</returns>
        public DocumentReference[] Import(string filePath)
        {
            if (null == filePath) { throw new ArgumentNullException("filePath"); }

            List<DocumentReference> results = new List<DocumentReference>();

            if (File.Exists(filePath) == false)
            {
                ImportLog.WriteLine(MsgRsrc.MessageResource.Message_ImportWarning_FileNotFound, filePath);
                return results.ToArray();
            }

            ImportLog.WriteLine(MsgRsrc.MessageResource.Message_ImportFileRead, filePath);

            try
            {
                using (DocumentReference documentRef =
                       documentService.OpenDocument(new FileResource(filePath)))
                {
                    if (documentRef == null)
                    {
                        throw new FileNotSupportedException(filePath);
                    }
                    DocumentReference importedDocumentRef =
                        this.ImportBankDocument(documentRef.Document);

                    results.Add(importedDocumentRef);
                }
            }
            catch (Exception e)
            {
                foreach (DocumentReference documentRef in results)
                {
                    documentRef.Close();
                }

                e.Data["ImportFilePath"] = filePath;
                throw e;
            }

            return results.ToArray();
        }

        /// <summary>
        /// バンクドキュメントをインポートします。
        /// </summary>
        /// <param name="legacySoundSetDocument">
        /// レガシーサウンドセットドキュメントを指定します。
        /// </param>
        /// <returns>インポートしたドキュメントへの参照配列を返します。</returns>
        private DocumentReference ImportBankDocument(Document document)
        {
            Assertion.Argument.NotNull(document);

            string filePath = document.Resource.Key;

            DocumentReference importedDocumentRef = null;
            // レガシーサウンドセットならコンバートします。
            if (document is BankDocument)
            {
                ImportLog.WriteLine(MsgRsrc.MessageResource.Message_ImportFile, filePath);
                importedDocumentRef =
                    documentService.ConvertDocument(document, this.intermediateOutputTraits.BankDocumentTypeName);
            }

            if (importedDocumentRef == null)
            {
                throw new FileNotSupportedException(filePath);
            }

            if (!(importedDocumentRef.Document is BankDocument))
            {
                importedDocumentRef.Close();
                throw new FileNotSupportedException(filePath);
            }

            return importedDocumentRef;
        }
    }
}
