﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using ToolDevelopmentKit;

    /// <summary>
    /// ファイルの識別子です。
    /// IFileID と異なり、このクラスは編集可能です。
    /// </summary>
    public class FileID : ICloneable, IFileID
    {
        public const char Separator = '|';

        public FileID()
        {
        }

        public static FileID Create(string source, string format, int? sampleRate, int? channelCount, string sound)
        {
            return new FileID()
            {
                Source = source,
                Format = format,
                SampleRate = (sampleRate ?? 0).ToString(),
                ChannelCount = (channelCount ?? 0).ToString(),
                Sound = sound,
            };
        }

        public FileID(string value)
        {
            Ensure.Argument.NotNull(value);

            string[] tokens = value.Split(FileID.Separator);

            switch (tokens.Length)
            {
                case 1:
                    this.Source = tokens[0];
                    break;

                case 2:
                    this.Source = tokens[0];
                    this.Format = tokens[1];
                    break;

                case 3:
                    this.Source = tokens[0];
                    this.Format = tokens[1];
                    this.Sound = tokens[3];
                    break;

                case 4:
                    this.Source = tokens[0];
                    this.Format = tokens[1];
                    this.SampleRate = tokens[2];
                    this.Sound = tokens[3];
                    break;

                case 5:
                    this.Source = tokens[0];
                    this.Format = tokens[1];
                    this.SampleRate = tokens[2];
                    this.ChannelCount = tokens[3];
                    this.Sound = tokens[4];
                    break;
            }
        }

        /// <summary>
        /// 識別子を文字列で取得します。
        /// </summary>
        public string Value
        {
            get
            {
                string value = this.Source;

                if (this.Format.Length > 0)
                {
                    value += Separator + this.Format;
                }

                if (this.SampleRate.Length > 0)
                {
                    value += Separator + this.SampleRate;
                }

                if (this.ChannelCount.Length > 0)
                {
                    value += Separator + this.ChannelCount;
                }

                if (this.Sound.Length > 0)
                {
                    value += Separator + this.Sound;
                }

                return value;
            }
        }

        /// <summary>
        /// 識別子に含まれるソース情報を取得または設定します。
        /// </summary>
        public string Source { get; set; } = string.Empty;

        /// <summary>
        /// 識別子に含まれるフォーマット情報を取得または設定します。
        /// </summary>
        public string Format { get; set; } = string.Empty;

        /// <summary>
        /// 識別子に含まれるサンプルレート情報を取得または設定します。
        /// </summary>
        public string SampleRate { get; set; } = string.Empty;

        /// <summary>
        /// 識別子に含まれるチャンネル数情報を取得または設定します。
        /// </summary>
        public string ChannelCount { get; set; } = string.Empty;

        /// <summary>
        /// 識別子に含まれるサウンド種別情報を取得または設定します。
        /// </summary>
        public string Sound { get; set; } = string.Empty;

        public override string ToString()
        {
            return this.Value;
        }

        /// <summary>
        /// 現在のインスタンスのコピーである新しいオブジェクトを作成します。
        /// </summary>
        /// <returns>このインスタンスのコピーである新しいオブジェクトを返します。</returns>
        public FileID Clone()
        {
            return (this as ICloneable).Clone() as FileID;
        }

        /// <summary>
        /// 現在のインスタンスのコピーである新しいオブジェクトを作成します。
        /// </summary>
        /// <returns>このインスタンスのコピーである新しいオブジェクトを返します。</returns>
        object ICloneable.Clone()
        {
            return this.MemberwiseClone();
        }
    }
}
