﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.IO;
    using System.Linq;
    using NintendoWare.SoundFoundation.Codecs;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.WaveElements;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// 波形バイナリ DOM の構築をサポートします。
    /// </summary>
    internal class WaveFileBuilder
    {
        public WaveFileBuilder(string signature, BinaryVersion version)
        {
            Ensure.Argument.NotNull(signature);
            this.Signature = signature;
            this.Version = version;
        }

        public string Signature { get; set; }

        public BinaryVersion Version { get; set; }

        // HACK : ★★ WaveEncoding は CodecOutput.WaveFormat から取得すべきです。
        public WaveBinary Build(WaveEncoding encoding, CodecOutput[] codecOutputs)
        {
            Ensure.Argument.NotNull(codecOutputs);
            Ensure.Argument.True(codecOutputs.Length > 0);

            WaveBinary file = new WaveBinary(
                this.Signature,
                this.Version.Major,
                this.Version.Minor,
                this.Version.Micro,
                this.Version.BinaryBugFix);

            WaveFormat format = codecOutputs[0].Format;

            file.InfoBlock.Body.Encoding = encoding;
            file.InfoBlock.Body.SamplingRate = (uint)format.SamplingRate;
            file.InfoBlock.Body.IsLoop = format.HasLoop;

            if (format.HasLoop)
            {
                file.InfoBlock.Body.LoopStart = (uint)format.LoopStartFrame;
                file.InfoBlock.Body.LoopEnd = (uint)format.LoopEndFrame;
                file.InfoBlock.Body.OriginalLoopStart = (uint)format.OriginalLoopStartFrame;
            }
            else
            {
                file.InfoBlock.Body.LoopStart = 0;
                file.InfoBlock.Body.LoopEnd = (uint)codecOutputs[0].FrameCount;
                file.InfoBlock.Body.OriginalLoopStart = 0;
            }

            for (int i = 0; i < codecOutputs.Length; i++)
            {
                if (codecOutputs[i].CodecData is DspAdpcmData)
                {
                    DspAdpcmData? dspAdpcmData = (DspAdpcmData)codecOutputs[i].CodecData;
                    file.AddChannel(this.CreateChannelInfo(dspAdpcmData, codecOutputs[i].Data));
                }
                else if (codecOutputs[i].CodecData is ImaAdpcmData)
                {
                    ImaAdpcmData? imaAdpcmData = (ImaAdpcmData)codecOutputs[i].CodecData;
                    file.AddChannel(this.CreateChannelInfo(imaAdpcmData, codecOutputs[i].Data));
                }
                else
                {
                    file.AddChannel(this.CreateChannelInfo(null, codecOutputs[i].Data));
                }
            }

            return file;
        }

        private ChannelInfo CreateChannelInfo(object adpcmInformation, byte[] samples)
        {
            ChannelInfo channelInfo = new ChannelInfo()
            {
                Data = new ByteStream(new MemoryStream(samples)),
            };

            if (adpcmInformation is DspAdpcmData)
            {
                DspAdpcmData? dspAdpcmInformation = adpcmInformation as DspAdpcmData?;
                if (dspAdpcmInformation.HasValue)
                {
                    DspAdpcmInfo destAdpcmInformation = new DspAdpcmInfo();

                    for (int i = 0; i < dspAdpcmInformation.Value.Coef.Length; i++)
                    {
                        destAdpcmInformation.Coef[i] = (UInt16)dspAdpcmInformation.Value.Coef[i];
                    }
                    destAdpcmInformation.PredScale = dspAdpcmInformation.Value.Pred_scale;
                    destAdpcmInformation.Yn1 = (UInt16)dspAdpcmInformation.Value.Yn1;
                    destAdpcmInformation.Yn2 = (UInt16)dspAdpcmInformation.Value.Yn2;
                    destAdpcmInformation.LoopPredScale = dspAdpcmInformation.Value.Loop_pred_scale;
                    destAdpcmInformation.LoopYn1 = (UInt16)dspAdpcmInformation.Value.Loop_yn1;
                    destAdpcmInformation.LoopYn2 = (UInt16)dspAdpcmInformation.Value.Loop_yn2;

                    channelInfo.CodecInformation = destAdpcmInformation;
                }
            }
            else if (adpcmInformation is ImaAdpcmData)
            {
                ImaAdpcmData? imaAdpcmInformation = adpcmInformation as ImaAdpcmData?;
                if (imaAdpcmInformation.HasValue)
                {
                    ImaAdpcmInfo destAdpcmInformation = new ImaAdpcmInfo();

                    destAdpcmInformation.OutPrevSample = imaAdpcmInformation.Value.OutPrevSample;
                    destAdpcmInformation.OutPrevTableIndex = imaAdpcmInformation.Value.OutPrevTableIndex;
                    destAdpcmInformation.OutLoopPrevSample = imaAdpcmInformation.Value.OutLoopPrevSample;
                    destAdpcmInformation.OutLoopPrevTableIndex = imaAdpcmInformation.Value.OutLoopPrevTableIndex;

                    channelInfo.CodecInformation = destAdpcmInformation;
                }
            }

            return channelInfo;
        }
    }
}
