﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.StreamSoundElements;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// ストリームサウンドバイナリ DOM の構築をサポートします。
    /// </summary>
    internal class StreamSoundFileBuilder
    {
        public StreamSoundFileBuilder(string signature, BinaryVersion version)
        {
            Ensure.Argument.NotNull(signature);
            this.Signature = signature;
            this.Version = version;
        }

        public string Signature { get; set; }

        public BinaryVersion Version { get; set; }

        public StreamSoundBinary Build(Codecs.CodecOutput codecOutput, uint crc32Value, IList<Codecs.WaveMarkerInfo> markers, IList<Codecs.WaveRegionInfo> regions)
        {
            Ensure.Argument.NotNull(codecOutput);

            // 波形長を超えるリージョンは破棄する
            var validRegions = regions.Where(region => region.EndFrame <= codecOutput.FrameCount).ToArray();

            StreamSoundBinary file = new StreamSoundBinary(
                this.Signature,
                this.Version.Major,
                this.Version.Minor,
                this.Version.Micro,
                this.Version.BinaryBugFix);

            this.BuildInfoBlock(file, codecOutput, crc32Value, validRegions);
            this.BuildSeekBlock(file, codecOutput);
            this.BuildRegionBlock(file, codecOutput, validRegions);
            this.BuildMarkerBlock(file, codecOutput, markers);

            file.DataBlock.Body.Data = file.InfoBlock.Body.StreamSoundInformation.Data;

            return file;
        }

        private void BuildInfoBlock(StreamSoundBinary file, Codecs.CodecOutput codecOutput, uint crc32Value, IList<Codecs.WaveRegionInfo> regions)
        {
            file.InfoBlock.Body.StreamSoundInformation = this.CreateStreamSoundInfo(
                this.ToWaveEncoding(codecOutput.Format),
                regions == null ? 0 : regions.Count,
                codecOutput,
                crc32Value);

            var dspAdpcmStreamData = codecOutput.CodecData as Codecs.DspAdpcmStreamData;

            for (int i = 0; i < codecOutput.Format.ChannelCount; i++)
            {
                Codecs.DspAdpcmData? dspAdpcmData = null;

                if (dspAdpcmStreamData != null &&
                    dspAdpcmStreamData.DspAdpcmData.Length != 0)
                {
                    dspAdpcmData = dspAdpcmStreamData.DspAdpcmData[i];
                }

                ChannelInfo channelInfo = this.CreateChannelInfo(dspAdpcmData);
                file.InfoBlock.Body.ChannelInformationTable.Items.Add(channelInfo);

                if (channelInfo.CodecInformation != null)
                {
                    file.InfoBlock.Body.CodecInformations.Add(channelInfo.CodecInformation);
                }
            }
        }

        private void BuildSeekBlock(StreamSoundBinary file, Codecs.CodecOutput codecOutput)
        {
            var dspAdpcmStreamData = codecOutput.CodecData as Codecs.DspAdpcmStreamData;

            if (dspAdpcmStreamData == null)
            {
                file.SeekBlock = null;
            }
            else
            {
                file.SeekBlock.Body.Data = new ByteStream(dspAdpcmStreamData.DspAdpcmSeekDatas);
            }
        }

        private void BuildRegionBlock(
            StreamSoundBinary file,
            Codecs.CodecOutput codecOutput,
            IList<Codecs.WaveRegionInfo> regions)
        {
            Assertion.Argument.NotNull(file);
            Assertion.Argument.NotNull(codecOutput);
            Assertion.Argument.NotNull(regions);

            if (regions == null || regions.Count == 0)
            {
                file.RegionBlock = null;
                return;
            }

            var regionBlock = new RegionBlock();

            var dspAdpcmStreamData = codecOutput.CodecData as Codecs.DspAdpcmStreamData;
            int regionIndex = 0;

            foreach (var region in regions)
            {
                regionBlock.Body.RegionInfos.Items.Add(
                    this.CreateRegionInfo(region, regionIndex, dspAdpcmStreamData));

                ++regionIndex;
            }

            file.RegionBlock = regionBlock;
            file.InfoBlock.Body.StreamSoundInformation.RegionInfos = regionBlock.Body.RegionInfos;
        }

        private RegionInfo CreateRegionInfo(
            Codecs.WaveRegionInfo region,
            int regionIndex,
            Codecs.DspAdpcmStreamData dspAdpcmStreamData)
        {
            Assertion.Argument.True(regionIndex >= 0);

            // 欠番の場合は、空情報を返します。
            if (region == null)
            {
                return new RegionInfo();
            }

            var result = new RegionInfo()
            {
                StartFrame = region.StartFrame,
                EndFrame = region.EndFrame,
                Name = region.Name,
                IsRegionEnabled = true,
            };

            if (dspAdpcmStreamData == null)
            {
                return result;
            }

            // DspAdpcm の場合は、ADPCM コンテキスト情報を設定します。
            if (dspAdpcmStreamData.DspAdpcmContexts == null ||
                dspAdpcmStreamData.DspAdpcmContexts.Length == 0)
            {
                throw new Exception("DspAdpcmContexts not found.");
            }

            if (dspAdpcmStreamData.DspAdpcmContexts.Length > result.Contexts.Length)
            {
                // NOTE: チャンネル数分の AdpcmContext を格納する領域がなかった
                throw new Exception("StreamSound channels is too many.");
            }

            int contextIndex = 0;

            foreach (var regionAdpcmContexts in dspAdpcmStreamData.DspAdpcmContexts)
            {
                result.Contexts[contextIndex] =
                    this.ToDspAdpcmContextBinary(regionAdpcmContexts[regionIndex]);
                ++contextIndex;
            }

            return result;
        }

        private void BuildMarkerBlock(
            StreamSoundBinary file,
            Codecs.CodecOutput codecOutput,
            IList<Codecs.WaveMarkerInfo> markers)
        {
            Assertion.Argument.NotNull(file);
            Assertion.Argument.NotNull(codecOutput);
            Assertion.Argument.NotNull(markers);

            if (markers == null || markers.Count == 0)
            {
                file.MarkerBlock = null;
                return;
            }

            var markerBlock = new MarkerBlock();

            foreach (var marker in markers)
            {
                markerBlock.Body.MarkerInfos.Add(this.CreateMarkerInfo(marker));
            }
            markerBlock.Body.MarkerCount = markers.Count;

            file.MarkerBlock = markerBlock;
        }

        private MarkerInfo CreateMarkerInfo(
            Codecs.WaveMarkerInfo marker)
        {
            Ensure.Argument.NotNull(marker);

            var result = new MarkerInfo()
            {
                Position = marker.Position,
                Name = marker.Name,
            };

            return result;
        }

        private StreamSoundInfo CreateStreamSoundInfo(WaveEncoding encoding, int regionCount, Codecs.CodecOutput codecOutput, uint crc32Value)
        {
            Ensure.Argument.NotNull(codecOutput);

            StreamSoundInfo streamSoundInfo = new StreamSoundInfo()
            {
                Encoding = encoding,
                IsLoop = codecOutput.Format.HasLoop,
                ChannelCount = (byte)codecOutput.Format.ChannelCount,
                RegionCount = (byte)regionCount,
                SamplingRate = (uint)codecOutput.Format.SamplingRate,
                BlockCount = (uint)codecOutput.StreamData.BlockCount,
                BlockSize = (uint)codecOutput.StreamData.BlockByte,
                SampleCountPerBlock = (uint)codecOutput.StreamData.BlockSamples,
                LastBlockSize = (uint)codecOutput.StreamData.LastBlockByte,
                LastBlockSampleCount = (uint)codecOutput.StreamData.LastBlockSamples,
                LastBlockPaddedLength = (uint)codecOutput.StreamData.LastBlockPaddedByte,
                SeekInformationSize = (uint)4,
                SeekInformationIntervalSampleCount = (uint)codecOutput.StreamData.BlockSamples,
                Data = new ByteStream(codecOutput.Data),
                RegionInfoSize = RegionInfo.Size,
            };

            streamSoundInfo.HashInfo.Crc32Value = crc32Value;

            if (codecOutput.Format.HasLoop)
            {
                // DSPADPCM、LINEARPCM 共に、ループ範囲をフレーム位置で格納します。
                streamSoundInfo.LoopStart = (uint)codecOutput.Format.LoopStartFrame;
                streamSoundInfo.LoopEnd = (uint)codecOutput.Format.LoopEndFrame;
                streamSoundInfo.OriginalLoopStart = (uint)codecOutput.Format.OriginalLoopStartFrame;
                streamSoundInfo.OriginalLoopEnd = (uint)codecOutput.Format.OriginalLoopEndFrame;
            }
            else
            {
                // DSPADPCM、LINEARPCM 共に、終端フレーム位置を格納します。
                streamSoundInfo.LoopStart = 0;
                streamSoundInfo.LoopEnd = (uint)codecOutput.FrameCount;
                streamSoundInfo.OriginalLoopStart = 0;
                streamSoundInfo.OriginalLoopEnd = streamSoundInfo.LoopEnd;
            }

            return streamSoundInfo;
        }

        private ChannelInfo CreateChannelInfo(Codecs.DspAdpcmData? adpcmInformation)
        {
            ChannelInfo channelInfo = new ChannelInfo();

            if (adpcmInformation.HasValue)
            {
                DspAdpcmInfo destAdpcmInformation = new DspAdpcmInfo();

                for (int i = 0; i < adpcmInformation.Value.Coef.Length; i++)
                {
                    destAdpcmInformation.Coef[i] = (UInt16)adpcmInformation.Value.Coef[i];
                }
                destAdpcmInformation.PredScale = adpcmInformation.Value.Pred_scale;
                destAdpcmInformation.Yn1 = (UInt16)adpcmInformation.Value.Yn1;
                destAdpcmInformation.Yn2 = (UInt16)adpcmInformation.Value.Yn2;
                destAdpcmInformation.LoopPredScale = adpcmInformation.Value.Loop_pred_scale;
                destAdpcmInformation.LoopYn1 = (UInt16)adpcmInformation.Value.Loop_yn1;
                destAdpcmInformation.LoopYn2 = (UInt16)adpcmInformation.Value.Loop_yn2;

                channelInfo.CodecInformation = destAdpcmInformation;
            }

            return channelInfo;
        }

        private DspAdpcmContext ToDspAdpcmContextBinary(Codecs.DspAdpcmContext source)
        {
            return new DspAdpcmContext()
            {
                PredScale = source.Pred_scale,
                Yn1 = (UInt16)source.Yn1,
                Yn2 = (UInt16)source.Yn2,
            };
        }

        public WaveEncoding ToWaveEncoding(Codecs.WaveFormat format)
        {
            Ensure.Argument.NotNull(format);

            switch (format.Encoding)
            {
                case Codecs.Encoding.Pcm:
                    switch (format.BitsPerSample)
                    {
                        case 8:
                            return WaveEncoding.Pcm8;

                        case 16:
                            return WaveEncoding.Pcm16;
                    }
                    break;

                case Codecs.Encoding.DspAdpcm:
                    return WaveEncoding.Adpcm;
            }

            throw new Exception("invalid wave format.");
        }
    }
}
