﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary;
    using NintendoWare.SoundFoundation.FileFormats.NintendoWareBinary.GroupElements;
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    /// <summary>
    /// グループバイナリ DOM の構築をサポートします。
    /// </summary>
    internal class GroupFileBuilder
    {
        public GroupFileBuilder(string signature, BinaryVersion version)
        {
            Ensure.Argument.NotNull(signature);
            this.Signature = signature;
            this.Version = version;
        }

        public string Signature { get; set; }

        public BinaryVersion Version { get; set; }

        public GroupBinary Build(
            SoundArchiveContext context,
            GroupBase group,
            IDictionary<IOutputItem, object> fileEntityDictionary,
            object[] fileEntities)
        {
            Ensure.Argument.NotNull(context);
            Ensure.Argument.NotNull(group);
            Ensure.Argument.NotNull(fileEntityDictionary);
            Ensure.Argument.NotNull(fileEntities);

            GroupBinary groupFile = new GroupBinary(
                this.Signature,
                this.Version.Major,
                this.Version.Minor,
                this.Version.Micro,
                this.Version.BinaryBugFix);

            this.BuildInfoBlock(groupFile.InfoBlock.Body, context, group, fileEntityDictionary);
            this.BuildFileBlock(groupFile.FileBlock.Body, fileEntities);
            this.BuildInfoExBlock(groupFile.InfoExBlock.Body, group);

            return groupFile;
        }

        private void BuildInfoBlock(
            InfoBlockBody blockBody, SoundArchiveContext context, GroupBase group,
            IDictionary<IOutputItem, object> fileEntityDictionary)
        {
            Assertion.Argument.NotNull(blockBody);
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(group);
            Assertion.Argument.NotNull(fileEntityDictionary);

            bool isLinkGroup = group.OutputType == GroupOutputType.Link;

            foreach (ComponentFile file in group.GetItemFiles())
            {
                IOutputItem outputItem = this.GetGroupItemOutput(group, file);

                blockBody.Items.Add(this.CreateGroupItemInfo(
                    context,
                    file,
                    (outputItem == null || isLinkGroup) ? null : fileEntityDictionary[outputItem]
                    ));
            }
        }

        private IOutputItem GetGroupItemOutput(GroupBase group, ComponentFile file)
        {
            Assertion.Argument.NotNull(file);

            IOutputItem outputItem = null;
            file.OutputTarget.ItemDictionary.TryGetValue(group.Name, out outputItem);

            if (outputItem != null)
            {
                return outputItem;
            }

            file.OutputTarget.ItemDictionary.TryGetValue(string.Empty, out outputItem);

            return outputItem;
        }

        private GroupItemInfo CreateGroupItemInfo(SoundArchiveContext context, ComponentFile file, object fileEntity)
        {
            Assertion.Argument.NotNull(context);
            Assertion.Argument.NotNull(file);

            int fileIndex = -1;

            // HACK : GroupFileBuilder は並列実行される可能性があるため、
            //        初回の context.Files の Renumber() を排他処理します。
            //        SoundArchiveBuilder で処理すべきかも。
            lock (context.Files)
            {
                fileIndex = context.Files.IndexOf(context.GetFile(file.OutputTarget));
            }

            GroupItemInfo itemInfo = new GroupItemInfo()
            {
                FileID = (UInt32)fileIndex,
                File = fileEntity,
            };

            return itemInfo;
        }

        private void BuildFileBlock(FileBlockBody blockBody, object[] fileEntities)
        {
            Assertion.Argument.NotNull(blockBody);
            Assertion.Argument.NotNull(fileEntities);

            foreach (object fileEntity in fileEntities)
            {
                blockBody.Items.Add(fileEntity);
            }
        }

        private void BuildInfoExBlock(InfoExBlockBody blockBody, GroupBase group)
        {
            Assertion.Argument.NotNull(blockBody);
            Assertion.Argument.NotNull(group);

            foreach (GroupItemBase groupItem in group.GetAllItems().OrderBy(item => item.Target.ID.Value))
            {
                if (!groupItem.IsEnabled || !groupItem.Target.IsHierarchicalConvertTarget())
                {
                    continue;
                }

                blockBody.Items.Add(this.CreateGroupItemInfoEx(groupItem));
            }
        }

        private GroupItemInfoEx CreateGroupItemInfoEx(GroupItemBase groupItem)
        {
            Assertion.Argument.NotNull(groupItem);

            GroupItemInfoEx itemInfo = new GroupItemInfoEx()
            {
                ItemID = groupItem.Target.ID,
                LoadFlag = ConvertGroupItemRegisterType(groupItem.RegisterType),
            };

            return itemInfo;
        }

        private UInt32 ConvertGroupItemRegisterType(GroupItemRegisterType type)
        {
            UInt32 loadFlag = 0;

            switch (type)
            {
                case GroupItemRegisterType.All:
                    loadFlag = (UInt32)LoadFlag.LOAD_ALL;
                    break;

                case GroupItemRegisterType.SequenceAndBank:
                    loadFlag = (UInt32)LoadFlag.LOAD_SEQ | LoadFlag.LOAD_BANK;
                    break;

                case GroupItemRegisterType.SequenceAndWaveArchive:
                    loadFlag = LoadFlag.LOAD_SEQ | LoadFlag.LOAD_WARC;
                    break;

                case GroupItemRegisterType.BankAndWaveArchive:
                    loadFlag = LoadFlag.LOAD_BANK | LoadFlag.LOAD_WARC;
                    break;

                case GroupItemRegisterType.OnlySequence:
                    loadFlag = LoadFlag.LOAD_SEQ;
                    break;

                case GroupItemRegisterType.OnlyWaveSoundSet:
                    loadFlag = LoadFlag.LOAD_WSD;
                    break;

                case GroupItemRegisterType.OnlyBank:
                    loadFlag = LoadFlag.LOAD_BANK;
                    break;

                case GroupItemRegisterType.OnlyWaveArchive:
                    loadFlag = LoadFlag.LOAD_WARC;
                    break;
            }

            return loadFlag;
        }

        private static class LoadFlag
        {
            public static UInt32 LOAD_SEQ { get { return (1 << 0); } }
            public static UInt32 LOAD_WSD { get { return (1 << 1); } }
            public static UInt32 LOAD_BANK { get { return (1 << 2); } }
            public static UInt32 LOAD_WARC { get { return (1 << 3); } }
            public static UInt32 LOAD_ALL { get { return 0xffffffff; } }
        }
    }
}
