﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

//#define AAC_SUPPORTED

namespace NintendoWare.SoundFoundation.Conversion.NintendoWareBinary
{
    using NintendoWare.SoundFoundation.Projects;
    using NintendoWare.ToolDevelopmentKit;

    internal class StreamSoundProcessorFactory : ComponentSetup<SoundArchiveContext, StreamSoundBase>
    {
        public const string OutputID_BinaryForPC = ComponentContext.IntermediateOutputPrefix + "BinaryForPC";

        /// <summary>
        /// コンポーネントを処理します。
        /// </summary>
        /// <param name="context">コンバートコンテキストを指定します。</param>
        /// <param name="component">コンポーネントを指定します。</param>
        protected sealed override void RunInternal(SoundArchiveContext context, StreamSoundBase component)
        {
            IOutput output = component.GetOutputTarget();
            IOutput prefetchOutput = component.GetPrefetchOutputTarget();

            StreamSoundProcessor processor;

            if (context.ContainsProcessor(output, string.Empty))
            {
                processor = context.GetProcessor(output, string.Empty) as StreamSoundProcessor;
                processor.Components.Add(component);
            }
            else
            {
                processor = this.CreateComponentProcessor(context, component, this.GetOutputItem(output)) as StreamSoundProcessor;
                context.AddComponentProcessor(output, string.Empty, processor);
            }

            if (processor.PrefetchOutputItem == null)
            {
                processor.PrefetchOutputItem = this.GetPrefetchOutputItem(prefetchOutput);
            }

#if AAC_SUPPORTED
            // AAC の場合は、PC 用バイナリも出力する
            if (context.Project.DoOutputPCBinariesForAAC && component.GetAdtsHeader() != null)
            {
                context.AddComponentProcessor(
                    output,
                    OutputID_BinaryForPC,
                    this.CreateBinaryForPCComponentProcessor(context, component, this.GetBinaryForPCOutputItem(output))
                    );
            }
#endif
        }

        private IComponentProcessor CreateComponentProcessor(SoundArchiveContext context, StreamSoundBase streamSound, IOutputItem output)
        {
            Assertion.Argument.NotNull(streamSound);
            return new StreamSoundProcessor(streamSound, output, false)
            {
                PreprocessExePath = context.GetWavePreprocessExeFullPath(),
            };
        }

        private IComponentProcessor CreateBinaryForPCComponentProcessor(SoundArchiveContext context, StreamSoundBase streamSound, IOutputItem output)
        {
            Assertion.Argument.NotNull(streamSound);
            return new StreamSoundProcessor(streamSound, output, true)
            {
                PreprocessExePath = context.GetWavePreprocessExeFullPath(),
            };
        }

        private IOutputItem GetOutputItem(IOutput output)
        {
            Assertion.Argument.NotNull(output);
            return output.ItemDictionary[string.Empty];
        }

        private IOutputItem GetPrefetchOutputItem(IOutput output)
        {
            if (output == null)
            {
                return null;
            }

            IOutputItem result = null;
            if (!output.ItemDictionary.TryGetValue(string.Empty, out result))
            {
                return null;
            }

            return result;
        }

        private IOutputItem GetBinaryForPCOutputItem(IOutput output)
        {
            Assertion.Argument.NotNull(output);
            return output.ItemDictionary[OutputID_BinaryForPC];
        }
    }
}
